/**
 * Copyright 2013 Michał Rudewicz
 * 
 * Gimp plugin that fills the top and the bottom empty areas
 * 
 */

#include "config.h"
#include <libgimp/gimp.h>
#include <math.h>

static void query(void);
static void run(const gchar *name,
        gint nparams,
        const GimpParam *param,
        gint *nreturn_vals,
        GimpParam **return_vals);
static void panoramaToZenithAndNadir(GimpDrawable *drawable);

GimpPlugInInfo PLUG_IN_INFO = {
    NULL,
    NULL,
    query,
    run
};

MAIN()

static void
query(void) {
    static GimpParamDef args[] = {
        {
            GIMP_PDB_INT32,
            "run-mode",
            "Run mode"
        },
        {
            GIMP_PDB_IMAGE,
            "image",
            "Input image"
        },
        {
            GIMP_PDB_DRAWABLE,
            "drawable",
            "Input drawable"
        }
    };

    gimp_install_procedure(
            "plug-in-panorama-to-zenith-and-nadir",
            "Panorama to zenith and nadir",
            "Convert 360 panorama to editable zenith and nadir",
            "Michał Rudewicz",
            "Copyright Michał Rudewicz",
            "2013",
            "_Panorama to zenith & nadir",
            "RGB*, GRAY*",
            GIMP_PLUGIN,
            G_N_ELEMENTS(args), 0,
            args, NULL);

    gimp_plugin_menu_register("plug-in-panorama-to-zenith-and-nadir",
            "<Image>/Filters/360 Panorama");
}

static void
run(const gchar *name,
        gint nparams,
        const GimpParam *param,
        gint *nreturn_vals,
        GimpParam **return_vals) {
    static GimpParam values[1];
    GimpPDBStatusType status = GIMP_PDB_SUCCESS;
    GimpDrawable *drawable;
    gint32 imageId;

    /* Setting mandatory output values */
    *nreturn_vals = 1;
    *return_vals = values;

    values[0].type = GIMP_PDB_STATUS;
    values[0].data.d_status = status;

    /*  Get the specified drawable  */

    drawable = gimp_drawable_get(param[2].data.d_drawable);
    imageId = param[1].data.d_image;
    gimp_selection_none(imageId);

    panoramaToZenithAndNadir(drawable);

    gimp_displays_flush();
    gimp_drawable_detach(drawable);

    return;
}

static void
panoramaToZenithAndNadir(GimpDrawable *drawable) {
    gint x, y, channel, channels;
    gint x1, y1, x2, y2, height, width;
    GimpPixelRgn rgn_in, rgn_out;
    guchar *rectIn, *rectOut;
    gint step;


    gimp_drawable_mask_bounds(drawable->drawable_id,
            &x1, &y1,
            &x2, &y2);
    channels = gimp_drawable_bpp(drawable->drawable_id);

    x1 = y1 = 0;
    x2 = gimp_drawable_width(drawable->drawable_id);
    y2 = gimp_drawable_height(drawable->drawable_id);

    gimp_pixel_rgn_init(&rgn_in,
            drawable,
            x1, y1,
            x2 - x1, y2 - y1,
            FALSE, FALSE);
    gimp_pixel_rgn_init(&rgn_out,
            drawable,
            x1, y1,
            x2 - x1, y2 - y1,
            TRUE, TRUE);
    height = y2 - y1;
    width = x2 - x1;
    step = MAX(10, width / 50);
    // Cache doesn't speed-up in this case
    //    gimp_tile_cache_ntiles((((y2 - y1) / gimp_tile_height() + 1)*(width / gimp_tile_width() + 1)));


    /* Allocate memory for Input and output rectangles */
    rectIn = g_new(guchar, channels * width * height);
    rectOut = g_new(guchar, channels * width * height);

    gdouble xNorm, yNorm, xx, yy, rr, phi, c, xxx, yyy, X, Y;
    //    gdouble xCenter = (x1 + x2) / 2.0, yCenter = (y1 + y2) / 2.0;
    gdouble xCenter = (x1 + x2 - 1) / 2.0, yCenter = (y1 + y2 - 1) / 2.0;
    gint inclusiveWidth = width - 1;
    gint sinphi1, inX, inY;
    //    X = ((gdouble) (x2 - xCenter + 0.5))*2 / (gdouble) width;
    //    Y = ((gdouble) (y2 - yCenter + 0.5))*2 / (gdouble) width;
    X = ((gdouble) (x2 - xCenter))*2 / (gdouble) inclusiveWidth;
    Y = ((gdouble) (y2 - yCenter))*2 / (gdouble) inclusiveWidth;

    gimp_progress_init("Converting panorama to zenith and nadir");
    gimp_pixel_rgn_get_rect(&rgn_in, rectIn, x1, y1, width, height);

    for (x = x1; x < x2; x++) {
        //        xNorm = ((gdouble) (x - xCenter + 0.5))*2 / (gdouble) width;
        xNorm = ((gdouble) (x - xCenter))*2 / (gdouble) inclusiveWidth;
        for (y = y1; y < y2; y++) {
            //            yNorm = ((gdouble) (y - yCenter + 0.5))*2 / (gdouble) width;
            yNorm = ((gdouble) (y - yCenter))*2 / (gdouble) inclusiveWidth;
            if (xNorm < 0) {
                sinphi1 = 1;
                xx = xNorm + X / 2;
            } else {
                sinphi1 = -1;
                xx = xNorm - X / 2;
            }

            yy = yNorm;
            rr = sqrt(xx * xx + yy * yy);
            c = atan(rr / Y);
            phi = rr == 0 ? 0.0 : asin(cos(c) * sinphi1);
            xxx = atan2(xx, -yy * sinphi1) * X / M_PI;
            yyy = phi * Y / (M_PI_2);
            //            inX = (width * xxx - 1) / 2 + xCenter;
            //            inY = (width * yyy - 1) / 2 + yCenter;
            inX = round((inclusiveWidth * xxx - 1) / 2 + xCenter);
            if (inX == -1)inX += x2;
            if (inX == x2)inX = 0;
            inY = round((inclusiveWidth * yyy - 1) / 2 + yCenter);
            if (inX >= x1 && inX < x2 && inY >= y1 && inY < y2) {
                for (channel = 0; channel < channels; channel++) {
                    rectOut[channels * (x + width * y) + channel] = rectIn[channels * (inX + width * inY) + channel];
                }
            } else {
                //                g_print("Calculation error: xy:(%i,%i) xxyy: (%f,%f) xxxyyy:(%f,%f) inXinY: (%i,%i)\n", x, y, xx, yy, xxx, yyy, inX, inY);
                for (channel = 0; channel < channels; channel++) {
                    rectOut[channels * (x + width * y) + channel] = 0;
                }
            }
        }
        if (x % step == 0)
            gimp_progress_update((gdouble) (x - x1) / (gdouble) width);
    }
    gimp_progress_update(1);
    gimp_pixel_rgn_set_rect(&rgn_out, rectOut, x1, y1, width, height);

    g_free(rectIn);
    g_free(rectOut);

    gimp_drawable_flush(drawable);
    gimp_drawable_merge_shadow(drawable->drawable_id, TRUE);
    gimp_drawable_update(drawable->drawable_id,
            x1, y1,
            x2 - x1, y2 - y1);
}

