/**
 * Copyright 2013 Michał Rudewicz
 * 
 * Gimp plugin that fills the top and the bottom empty areas
 * 
 */

#include "config.h"
#include <libgimp/gimp.h>
#include <math.h>

static void query(void);
static void run(const gchar *name,
        gint nparams,
        const GimpParam *param,
        gint *nreturn_vals,
        GimpParam **return_vals);
static void ZenithAndNadirToPanorama(GimpDrawable *drawable);

GimpPlugInInfo PLUG_IN_INFO = {
    NULL,
    NULL,
    query,
    run
};

MAIN()

static void
query(void) {
    static GimpParamDef args[] = {
        {
            GIMP_PDB_INT32,
            "run-mode",
            "Run mode"
        },
        {
            GIMP_PDB_IMAGE,
            "image",
            "Input image"
        },
        {
            GIMP_PDB_DRAWABLE,
            "drawable",
            "Input drawable"
        }
    };

    gimp_install_procedure(
            "plug-in-zenith-and-nadir-to-panorama",
            "Zenith and nadir to panorama",
            "Convert editable zenith and nadir to 360 panorama",
            "Michał Rudewicz",
            "Copyright Michał Rudewicz",
            "2013",
            "_Zenith & nadir to panorama",
            "RGB*, GRAY*",
            GIMP_PLUGIN,
            G_N_ELEMENTS(args), 0,
            args, NULL);

    gimp_plugin_menu_register("plug-in-zenith-and-nadir-to-panorama",
            "<Image>/Filters/360 Panorama");
}

static void
run(const gchar *name,
        gint nparams,
        const GimpParam *param,
        gint *nreturn_vals,
        GimpParam **return_vals) {
    static GimpParam values[1];
    GimpPDBStatusType status = GIMP_PDB_SUCCESS;
    GimpDrawable *drawable;
    gint32 imageId;

    /* Setting mandatory output values */
    *nreturn_vals = 1;
    *return_vals = values;

    values[0].type = GIMP_PDB_STATUS;
    values[0].data.d_status = status;

    drawable = gimp_drawable_get(param[2].data.d_drawable);
    imageId = param[1].data.d_image;
    gimp_selection_none(imageId);

    ZenithAndNadirToPanorama(drawable);

    gimp_displays_flush();
    gimp_drawable_detach(drawable);

    return;
}

static void
ZenithAndNadirToPanorama(GimpDrawable *drawable) {
    gint x, y, channel, channels;
    gint x1, y1, x2, y2, height, width;
    GimpPixelRgn rgn_in, rgn_out;
    guchar *rectIn, *rectOut;
    gint step;


    gimp_drawable_mask_bounds(drawable->drawable_id,
            &x1, &y1,
            &x2, &y2);
    channels = gimp_drawable_bpp(drawable->drawable_id);

    x1 = y1 = 0;
    x2 = gimp_drawable_width(drawable->drawable_id);
    y2 = gimp_drawable_height(drawable->drawable_id);

    gimp_pixel_rgn_init(&rgn_in,
            drawable,
            x1, y1,
            x2 - x1, y2 - y1,
            FALSE, FALSE);
    gimp_pixel_rgn_init(&rgn_out,
            drawable,
            x1, y1,
            x2 - x1, y2 - y1,
            TRUE, TRUE);
    height = y2 - y1;
    width = x2 - x1;
    step = MAX(10, width / 50);
    // Cache doesn't speed-up in this case
    //    gimp_tile_cache_ntiles((((y2 - y1) / gimp_tile_height() + 1)*(width / gimp_tile_width() + 1)));


    /* Allocate memory for Input and output rectangles */
    rectIn = g_new(guchar, channels * width * height);
    rectOut = g_new(guchar, channels * width * height);

    gdouble xNorm, yNorm, xx, yy, xxx, yyy, X, Y, cosc, xc;
    //    gdouble xCenter = (x1 + x2) / 2.0, yCenter = (y1 + y2) / 2.0;
    gdouble xCenter = (x1 + x2 - 1) / 2.0, yCenter = (y1 + y2 - 1) / 2.0;
    gint inclusiveWidth = width - 1;
    gint sinphi1, inX, inY, output;
    //    X = ((gdouble) (x2 - xCenter + 0.5))*2 / (gdouble) width;
    //    Y = ((gdouble) (y2 - yCenter + 0.5))*2 / (gdouble) width;
    X = ((gdouble) (x2 - xCenter))*2 / (gdouble) inclusiveWidth;
    Y = ((gdouble) (y2 - yCenter))*2 / (gdouble) inclusiveWidth;

    gimp_progress_init("Converting zenith and nadir to panorama");
    gimp_pixel_rgn_get_rect(&rgn_in, rectIn, x1, y1, width, height);

    for (x = x1; x < x2; x++) {
        //        xNorm = ((gdouble) (x - xCenter + 0.5))*2 / (gdouble) width;
        xNorm = ((gdouble) (x - xCenter))*2 / (gdouble) inclusiveWidth;
        for (y = y1; y < y2; y++) {
            //            yNorm = ((gdouble) (y - yCenter + 0.5))*2 / (gdouble) width;
            yNorm = ((gdouble) (y - yCenter))*2 / (gdouble) inclusiveWidth;

            output = 1;
            if (yNorm > Y / 4) {
                sinphi1 = 1;
                xc = -X / 2;
            } else if (yNorm < (-Y / 4)) {
                sinphi1 = -1;
                xc = X / 2;
            } else {
                output = 0;
            }

            if (output) {
                cosc = sinphi1 * sin(yNorm / Y * M_PI_2);
                xx = cos(yNorm / Y * M_PI_2) * sin(xNorm / X * M_PI) / cosc;
                yy = -sinphi1 * cos(yNorm / Y * M_PI_2) * cos(xNorm / X * M_PI) / cosc;
                if (fabs(xx) >= 0.99) output = 0;
            }
            if (output) {
                xxx = xx * X / 2 + xc;
                yyy = yy*Y;
            //            inX = (width * xxx - 1) / 2 + xCenter;
            //            inY = (width * yyy - 1) / 2 + yCenter;
            inX = round((inclusiveWidth * xxx - 1) / 2 + xCenter);
            inY = round((inclusiveWidth * yyy - 1) / 2 + yCenter);
            }

            if (output && inX >= x1 && inX < x2 && inY >= y1 && inY < y2) {
                for (channel = 0; channel < channels; channel++) {
                    rectOut[channels * (x + width * y) + channel] = rectIn[channels * (inX + width * inY) + channel];
                }
            } else {
                for (channel = 0; channel < channels; channel++) {
                    rectOut[channels * (x + width * y) + channel] = 0;
                }
            }

        }
        if (x % step == 0)
            gimp_progress_update((gdouble) (x - x1) / (gdouble) width);
    }
    gimp_progress_update(1);
    gimp_pixel_rgn_set_rect(&rgn_out, rectOut, x1, y1, width, height);

    g_free(rectIn);
    g_free(rectOut);

    gimp_drawable_flush(drawable);
    gimp_drawable_merge_shadow(drawable->drawable_id, TRUE);
    gimp_drawable_update(drawable->drawable_id,
            x1, y1,
            x2 - x1, y2 - y1);
}

