"""
Pack by filling layers

License:

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 The GNU Public License is available at
 http://www.gnu.org/copyleft/gpl.html
"""
class Sheet:
  """
  Represents a sheet on which to pack fields
  """
  def __init__(self, width, height):
    """
    width: width of the sheet
    height: height of the sheet
    """
    self.width = width
    self.height = height
    self.top = Field(width, 0, self)
    self.left = Field(0, height+1, self)
    self.right = Field(0, height, self)
    self.fields = []
    self.left.place((0, 0), self, (None, self.top))
    self.top.place((0, 0), self, (self.left, self.right))
    self.right.place((width, 0), self, (self.top, None))
  
  def add(self, field):
    """
    Add a field to be packed on the sheet
    field: field to add
    """
    self.fields.append(field)
    
  def __str__(self):
    return ", ".join(map(str, [f for f in self.fields if f not in [self.top, self.left, self.right]]))

  def draw(self):
    """
    Draw all fields packed on the sheet
    """
    for f in self.fields:
      if f not in [self.top, self.left, self.right]:
        f.draw()
      
class Field:
  """
  Represents a field to be packed
  """
  def __init__(self, width, height, packer=None):
    """
    width: width of the field
    height: height of the field
    packer: packer which determines the position of the field on the sheet
    """
    self.packer = packer
    self.size = (width, height)
    self.position = None
    self.sheet = None
    self.unplacable = False
    self.neighbors = None #pair of Fields
    self.below = None # Layer
    
  def unplaced(self):
    return self.position == None and not self.unplacable
  
  def placed(self):
    return self.position != None and not self.unplacable
  
  def place(self, position, sheet, neighbors):
    self.position = position
    self.sheet = sheet
    self.neighbors = neighbors
    sheet.add(self)

  def width(self):
    (w, h) = self.size
    return w

  def height(self):
    (w, h) = self.size
    return h
    
  def leftNeighbor(self):
    (l, r) = self.neighbors
    return l
  
  def rightNeighbor(self):
    (l, r) = self.neighbors
    return r
  
  def left(self):
    if self.unplaced():
      return None
    else:
      (x, y) = self.position
      return x
  
  def right(self):
    if self.unplaced():
      return None
    else:
      (x, y) = self.position
      return x+self.width()
    
  def top(self):
    if self.unplaced():
      return None
    else:
      (x, y) = self.position
      return y

  def bottom(self):
    if self.unplaced():
      return None
    else:
      (x, y) = self.position
      return y+self.height()

  def dificulty(self):
    widthDificulty = self.width()/self.packer.width
    heightDificulty = self.height()/self.packer.height
    if self.packer.layer == None:
      heightDificulty = self.height()/self.packer.height
    else:  
      heightDificulty = self.height()/(self.packer.layer.height)
    if  widthDificulty > heightDificulty:
      return widthDificulty 
    else:
      return heightDificulty
    
  def ord(self, a, b):
    if a < b:
      return -1
    elif a > b:
      return 1
    else:
      return 0
    
  def orderHarder(self, other):
    return self.ord(self.dificulty(), other.dificulty())
  
  def orderWider(self, other):
    return self.ord(self.width(), other.width())

  def orderTaller(self, other):
    return self.ord(self.height(), other.height())
      
  def orderLower(self, other):
    return self.ord(other.bottom(), self.bottom())
  
  def orderHigher(self, other):
    return self.ord(self.bottom(), other.bottom())
    
  def __str__(self):
    return "{"+str(self.position)+", "+str(self.size)+"}"

class Layer:
  def __init__(self, top, sheet):
    self.sheet = sheet
    self.top = top.bottom()
    top.below = self
    self.height = sheet.height-self.top
    (self.left, self.right) = top.neighbors
    while self.left != None and self.left.leftNeighbor() != None and (self.left.below != None and self.top >= self.left.below.bottom() or self.left.below == None and self.top >= self.left.bottom()):
      if self.left.below == None:
        self.left.below = self
      self.left = self.left.leftNeighbor()
      
    while self.right != None and self.right.rightNeighbor() != None and (self.right.below != None and self.top >= self.right.below.bottom() or self.right.below == None and self.top >= self.right.bottom()):
      if self.right.below == None:
        self.right.below = self
      self.right = self.right.rightNeighbor()
    self.width = self.right.left()-self.left.right()
    self.fields = []
    
  def bottom(self):
    result = self.top
    for f in self.fields:
      if f.below == None:
        b = f.bottom()
      else:
        b = f.below.bottom()
      if result < b:
          result = b
    return result
  
  def room(self):
    widths = map(Field.width, self.fields)
    return self.width-sum(widths)
  
  def putIfRoom(self, field):
    if self.room() >= field.width():
      if self.height >= field.height():
        self.fields.append(field)
  
  def place(self):
    if self.left.bottom() > self.right.bottom():
      self.fields.sort(Field.orderLower)
    else:
      self.fields.sort(Field.orderTaller)  
      
    if len(self.fields)>0:
      for i in range(0, len(self.fields)):
        if i == 0:
          left = self.left
          (ln, rn) = left.neighbors
          left.neighbors = (ln, self.fields[0])
        else:
          left = self.fields[i-1]
          
        if i < len(self.fields)-1:
          right = self.fields[i+1]
        else:
          right = self.right
          (ln, rn) = right.neighbors
          right.neighbors = (self.fields[i], rn)
        
        if self.left.bottom() > self.right.bottom() and i==0:
          p = left.right()+self.room()
        else:
          p = left.right()
        self.fields[i].place((p, self.top), self.sheet, (left, right))
        if self.fields[i].bottom() > self.bottom:
          self.bottom = self.fields[i].bottom()
    

class Packer:
  """
  Packer which packs Fields on Sheets by filling layers
  """
  def __init__(self, width, height, pager=Sheet):
    self.width = width
    self.height = height
    self.pager = pager
    self.sheet = pager(width, height)
    self.sheets = [self.sheet]
    self.fields = []
    self.layer = None #Layer(self.top)
    #self.top.below = self.layer
    
  def addField(self, field):
    """
    Add a field to the packer
    """
    self.fields.append(field)
    field.packer = self
    
  def pack(self):
    """
    Pack all fields added to the packer
    """
    self.fields.sort(Field.orderHarder,reverse=True)
    for field in self.fields:
      if field.width()>self.sheet.top.width() or field.height()>self.sheet.left.height():
        field.unplacable = True
    unplaced = filter(Field.unplaced, self.fields)
    remaining = len(unplaced)
    while remaining > 0:
      self.sheet.fields.sort(Field.orderHigher)
      for field in self.sheet.fields:
        if field.below == None and field.leftNeighbor() != None and field.rightNeighbor() != None:
          self.layer = Layer(field, self.sheet)
          break

      if self.layer != None:
        unplaced.sort(Field.orderHarder,reverse=True)
        #unplaced.sort(Field.orderTaller,reverse=True)
        for field in unplaced:
          self.layer.putIfRoom(field)
        self.layer.place()
        self.layer = None
      else:
        print("Page")
        self.sheet = self.pager(self.width, self.height)
        self.sheets.append(self.sheet)
      unplaced = filter(Field.unplaced, self.fields)
      remaining = len(unplaced)
    return [f for f in self.fields if f.unplacable]
    
  def __str__(self):
    return "; ".join(map(str, self.sheets))

  def draw(self):
    """
    Draw all generated sheets
    """
    for s in self.sheets:
      s.draw()
