import os
import gtk
import random

from Packer import *
from gimpfu import *

"""
Pack images respecting their print sizes

License:

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 The GNU Public License is available at
 http://www.gnu.org/copyleft/gpl.html
"""
class Paper(Sheet):
  def __init__(self, width, height, resolution=600):
    """
    width: Width of paper in inches
    height: Height of paper in inches
    resolution: resolution of print in dpi
    """
    Sheet.__init__(self, width, height)
    # Size in pixels
    (widthpx, heightpx) = (width*resolution, height*resolution)
    self.resolution = resolution
    self.background = gimp.Image(widthpx, heightpx, RGB)
    self.layer = gimp.Layer(self.background, "background", widthpx, heightpx, RGB_IMAGE, 0, NORMAL_MODE)
    self.background.add_layer(self.layer, 0)
    pdb.gimp_image_set_resolution(self.background, resolution, resolution)
    disp=gimp.Display(self.background)

  def insertAtPosition(self, x, y, img):
    """
    Insert an image at the specified position
    x: X coordinate in inches
    y: Y coordinate in inches
    img: Image to insert
    """
    widthpx = (img.width()-img.margin)*self.resolution
    heightpx = (img.height()-img.margin)*self.resolution
    pdb.gimp_selection_all(img.img)
    pdb.gimp_edit_copy(img.img.flatten())
    layer = pdb.gimp_edit_paste(self.layer, False)
    layer.name = img.img.name
    #scale and position image according to the output image    
    pdb.gimp_layer_scale(layer, widthpx, heightpx, True)
    layer.set_offsets(x*self.resolution+img.margin/2, y*self.resolution+img.margin/2)
    
class Image(Field):
  def __init__(self, file, margin=0.1):
    """
    file: File to pack
    margin: space between images in inches
    """
    # get image
    self.img = pdb.gimp_file_load(os.path.abspath(file), os.path.basename(file) )
    self.file = file
    self.margin = margin
    # get width and height of image
    (imgresx, imgresy) = pdb.gimp_image_get_resolution(self.img)
    Field.__init__(self, self.img.width/imgresx+margin, self.img.height/imgresy+margin)
    
  def draw(self):
    self.sheet.insertAtPosition(self.left(), self.top(), self)

class ImageFiles:
  def __init__(self):
    """
    File selection dialog for choosing multiple image files
    """
    dialog = gtk.FileChooserDialog("Select Files",
                                    None,
                                    gtk.FILE_CHOOSER_ACTION_OPEN,
                                    (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                    gtk.STOCK_OPEN, gtk.RESPONSE_OK))
    dialog.set_default_response(gtk.RESPONSE_CANCEL)
    dialog.set_select_multiple(True)
    filter = gtk.FileFilter()
    filter.add_mime_type("image/gif")
    filter.add_mime_type("image/jpeg")
    filter.add_mime_type("image/png")
    filter.add_mime_type("image/tiff")
    filter.add_pattern("*.xcf")
    dialog.set_filter(filter)

    response = dialog.run()
    self.files = None
    if response == gtk.RESPONSE_OK:
        self.files = dialog.get_filenames()
        print("Chosen "+str(self.files))
    elif response == gtk.RESPONSE_CANCEL:
        print('Canceled, no files selected')
    else:
        print('Closed, no files selected')
    dialog.destroy()

def papersaver(width, height, margin, res):
    """
    Create a papersaver
    width: width of the resulting images
    height: heighth of the resulting images
    margin: space between the packed image
    res: resolution (dpi) of the resulting images
    """
    paper = Packer(width, height, Paper)
    paper.sheet.resolution = res
    images = ImageFiles()
    for i in images.files:
       paper.addField(Image(i, margin))
    u = paper.pack()
    for f in u:
       pdb.gimp_message(str(f.file)+" could not be placed")
    paper.draw()
    #disp=gimp.Display(paper.sheet)
    return paper.sheet

# Register with The Gimp
register(
    "papersaver",
    "Pack images on a sheet of paper, respecting their print size",
    "Pack images on a sheet of paper, respecting their print size",
    "Christian Clausen",
    "(c) 2010, Christian Clausen",
    "2010-03-09",
    "<Toolbox>/Xtns/Misc/Papersaver",
    "",
    [
    (PF_FLOAT, "width", "width (in)", 7),
    (PF_FLOAT, "height", "height (in)", 11),
    (PF_FLOAT, "margin", "margin (in)", 0.04),
    (PF_INT, "res", "resolution (dpi)", 600),
    ],
    [(PF_IMAGE, "sheet", "sheet")],
    papersaver);

main()
