/*
 * 2009 Lucian Sabo
 *
 * This plugin loads the current image into the RIOT optimizer (DLL version of RIOT required)
 * The DLL version of RIOT can be downloaded from http://luci.criosweb.ro/riot/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/* Many parts of this code are borrowed from other plugins source code. */

/*
 * KNOWN ISSUES:
 * - unicode filenames *seem* not compatible with the way RIOT needs them (ansi encoded or wchar)
 *
 *
 * TODO:
 * Feed RIOT with the visible image (all layers combined)
 * A starting poing would be gimp_layer_new_from_visible(param[1].data.d_int32, image_ID, "riot");
 * Split the menu option to "Save for web with RIOT..." ->
 * 1) Active layer/selection (that's the current behavior)
 * 2) Entire image
 */

#include <windows.h>

#include "libgimp/gimp.h"


/** defines ***********************************************************/

#define PLUG_IN_NAME "plug_in_RIOT"
#define PLUG_IN_VERSION "October 2009, 0.1"

typedef void* (*Dyn_RIOT_LoadFromDIB) (HANDLE hDIB,HWND hwndParent,const char *fileName,const char *iniFile, int flags, const char *buf);
Dyn_RIOT_LoadFromDIB pDyn_RIOT_LoadFromDIB;

/** Plugin interface *********************************************************/
/* How many steps the progress control should do
 */
#define PROGRESS_STEPS 25
#define StepProgress(one,all) \
      (0 == (one % ((all / PROGRESS_STEPS)+1)))
/* FIXME: I'll use -1 as IMAGE_NONE. Is it correct ???
 */
#define IMAGE_NONE -1

void query(void);
void run(const gchar *name, int nparams, const GimpParam *param, int *nreturn_vals, GimpParam **return_vals);


GimpPlugInInfo PLUG_IN_INFO = {
  NULL, /* init_proc */
  NULL, /* quit_proc */
  query,        /* query_proc */
  run   /* run_proc */
};


MAIN()

void
query(void)
{
  /* Definition of parameters */
  static GimpParamDef args[] = {
    { GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_IMAGE, "image", "Input image" },
    { GIMP_PDB_DRAWABLE, "drawable", "Input drawable" }	
  };

  static GimpParamDef *return_vals  = NULL;
  static int        nargs = sizeof(args) / sizeof(args[0]);
  static int        nreturn_vals = 0;

  gimp_install_procedure(
    "plug_in_RIOT",
    "Optimize image for the web",
    "Reduce file size using the RIOT plugin",
    "Lucian Sabo",
    "Lucian Sabo",
    PLUG_IN_VERSION,
	"<Image>/File/Save/Save for web with RIOT...",
    "INDEXED*, RGB*, GRAY*",
    GIMP_PLUGIN,
    nargs,
    nreturn_vals,
    args,
    return_vals);
}

void
run(const gchar *name, int nparams, const GimpParam *param,
    int *nreturn_vals, GimpParam **return_vals)
{
  HINSTANCE hInst;  
  GimpDrawable *drawable;
  GimpImageType drawable_type;
  GimpPixelRgn pixel_rgn;  
  HANDLE hDIB;
  BOOL bRet;
  gint32 image_ID, drawable_ID;
  gint         x1, y1, x2, y2, selWidth, selHeight;  

  int nSizeDIB=0;
  int nSizePal=0;
  int nSizeLine=0; /* DIB lines are 32 bit aligned */  

  static GimpParam values[2];
  GimpRunMode run_mode;  

  run_mode = param[0].data.d_int32;

  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
  
  image_ID = param[1].data.d_int32;
  drawable_ID = param[2].data.d_int32;

  drawable = gimp_drawable_get (drawable_ID);
  drawable_type = gimp_drawable_type (drawable_ID);
 
  if(drawable->bpp > 4) {
	g_message("The input color depth is not supported");
	return;
  }
   
  gimp_drawable_mask_bounds (drawable_ID,
                                   &x1, &y1,
                                   &x2, &y2);
  selWidth = x2 - x1;
  selHeight = y2 - y1;

/* Allocate a big enough tile cache */
        gimp_tile_cache_ntiles (drawable->width /
                                     gimp_tile_width () + 1);


	gimp_pixel_rgn_init (&pixel_rgn,
						drawable,  x1, y1,
                        selWidth, selHeight,
						FALSE, FALSE);

  /* allocate room for DIB */
  if (GIMP_INDEXED_IMAGE == drawable_type || GIMP_GRAY_IMAGE == drawable_type || GIMP_INDEXEDA_IMAGE == drawable_type || GIMP_GRAYA_IMAGE == drawable_type)
    {
      nSizeLine = ((selWidth-1)/4+1)*4;
      nSizeDIB = sizeof(RGBQUAD) * 256 /* always full color map size */
      + nSizeLine * selHeight
      + sizeof (BITMAPINFOHEADER);
    }
  else
    {	 
	  nSizeLine = ((selWidth*drawable->bpp-1)/4+1)*4;
      nSizeDIB = (nSizeLine * selHeight) 
      + sizeof (BITMAPINFOHEADER);
    }

  hDIB = GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE, nSizeDIB);
  if (NULL == hDIB)
    {
      g_message ("Failed to allocate DIB");
      bRet = FALSE;
    }

  /* fill header info */
  if (bRet)
    {
      BITMAPINFOHEADER *pInfo;

      bRet = FALSE;
      pInfo = GlobalLock (hDIB);
      if (pInfo)
      {
        pInfo->biSize   = sizeof(BITMAPINFOHEADER);
        pInfo->biWidth  = selWidth;
        pInfo->biHeight = selHeight;
        pInfo->biPlanes = 1;
		if(GIMP_INDEXED_IMAGE == drawable_type || GIMP_GRAY_IMAGE == drawable_type || GIMP_INDEXEDA_IMAGE == drawable_type || GIMP_GRAYA_IMAGE == drawable_type)
			pInfo->biBitCount = 8;			
		else 
			if(drawable->bpp == 4)
				pInfo->biBitCount = 32;		
		else
		  pInfo->biBitCount = 24;

/*itoa(pInfo->biBitCount, str, 10);
	g_message(str);
	return;*/

        pInfo->biCompression = BI_RGB; /* none */
        pInfo->biSizeImage = 0; /* not calculated/needed */
        pInfo->biXPelsPerMeter =
          pInfo->biYPelsPerMeter = 0;
        /* color map size */
        pInfo->biClrUsed = 
          (GIMP_INDEXED_IMAGE == drawable_type || GIMP_GRAY_IMAGE == drawable_type || GIMP_INDEXEDA_IMAGE == drawable_type || GIMP_GRAYA_IMAGE == drawable_type ? 256 : 0);
        pInfo->biClrImportant = 0; /* all */

        GlobalUnlock (hDIB);
        bRet = TRUE;
      } /* (pInfo) */
      else
      g_message("Failed to lock DIB Header");
    }

/* fill color map */
  if (bRet && (GIMP_INDEXED_IMAGE == drawable_type  || GIMP_GRAY_IMAGE == drawable_type || GIMP_INDEXEDA_IMAGE == drawable_type || GIMP_GRAYA_IMAGE == drawable_type))
    {
      char *pBmp;

      bRet = FALSE;
      pBmp = GlobalLock (hDIB);
      if (pBmp)
      {
        RGBQUAD *pPal;
        int nColors;
        unsigned char *cmap = NULL;
        pPal = (RGBQUAD*)(pBmp + sizeof(BITMAPINFOHEADER));
        nSizePal = sizeof(RGBQUAD) * 256;

        /* get the gimp colormap */
        if (GIMP_GRAY_IMAGE != drawable_type && GIMP_GRAYA_IMAGE != drawable_type)
          cmap = gimp_image_get_colormap (image_ID, &nColors);

        if (cmap)
          {
            int i;
            for (i = 0; (i < 256) && (i < nColors); i++)
            {
              pPal[i].rgbReserved = 0; /* is this alpha? */
              pPal[i].rgbRed      = cmap[3*i];
              pPal[i].rgbGreen    = cmap[3*i+1];
              pPal[i].rgbBlue     = cmap[3*i+2];
            }

            g_free(cmap);
            bRet = TRUE;
          } /* (cmap) */
        else if (GIMP_GRAY_IMAGE == drawable_type || GIMP_GRAYA_IMAGE == drawable_type)
          {
            /* fill with identity palette */
            int i;
            for (i = 0; (i < 256) && (i < nColors); i++)
            {
              pPal[i].rgbReserved = 0; /* is this alpha? */
              pPal[i].rgbRed      = i;
              pPal[i].rgbGreen    = i;
              pPal[i].rgbBlue     = i;
            }

            bRet = TRUE;
          }
        else
          g_message ("Can't get color map");
        GlobalUnlock (hDIB);
      } /* (pBmp) */
      else
      g_message ("Failed to lock DIB Palette");
    } /* indexed or grayscale */

  /* following the slow part ... */
    gimp_progress_init ("Convert to DIB...");

  /* copy data to DIB */
  if (bRet)
    {
      unsigned char *pData;

      bRet = FALSE;
      pData = GlobalLock (hDIB);

      if (pData)
      {
        unsigned char *pLine;

        /* calculate real offset */
        pData += (sizeof(BITMAPINFOHEADER) + nSizePal);

        pLine = g_new (guchar, selWidth * drawable->bpp);

        if (GIMP_INDEXED_IMAGE == drawable_type || GIMP_GRAY_IMAGE == drawable_type || GIMP_INDEXEDA_IMAGE == drawable_type || GIMP_GRAYA_IMAGE == drawable_type)
          {
            int x, y, i, j;
            for (y = y1, i = 0; y < y2; y++, i++)
            {
              if (StepProgress(y, selHeight))
                gimp_progress_update ((double)y / selHeight);

              gimp_pixel_rgn_get_row (&pixel_rgn, pLine, x1,
								y2-i-1, /* invert it */
                                selWidth);
              for (x = x1, j = 0; x < x2; x++, j++)
                pData[j+i*nSizeLine] = pLine[j*drawable->bpp];
            }
          }
        else
          {
            int x, y, i, j;
            for (y = y1, i = 0; y < y2; y++, i++)
            {
              if (StepProgress(y,selHeight))
                gimp_progress_update ((double)y / selHeight);

              gimp_pixel_rgn_get_row (&pixel_rgn, pLine, x1,
                                y2-i-1, /* invert it */
                                selWidth);
              for (x = x1, j = 0; x < x2; x++, j++)
                {
                  /* RGBQUAD: blue, green, red, reserved */
					pData[j*drawable->bpp+i*nSizeLine]   = pLine[j*drawable->bpp+2]; /* blue */                  
					pData[j*drawable->bpp+i*nSizeLine+1] = pLine[j*drawable->bpp+1]; /* green */
					pData[j*drawable->bpp+i*nSizeLine+2] = pLine[j*drawable->bpp];   /* red */
					pData[j*drawable->bpp+i*nSizeLine+3] = pLine[j*drawable->bpp+3];   /* alpha */                  					
                }
            }
          }

        g_free (pLine);
        bRet = TRUE;

        GlobalUnlock (hDIB);
      } /* (pData) */
      else
      g_message("Failed to lock DIB Data");
    } /* copy data to DIB */
  

  hInst = LoadLibrary(L"Riot.dll"); //incarcam libraria in memorie
         
		if(hInst) {						
                    pDyn_RIOT_LoadFromDIB = (Dyn_RIOT_LoadFromDIB) GetProcAddress(hInst,"RIOT_LoadFromDIB");                        
										
					if(pDyn_RIOT_LoadFromDIB) {						
    					pDyn_RIOT_LoadFromDIB(hDIB, NULL, gimp_image_get_filename (image_ID), "", 0, "");						
					}

               FreeLibrary(hInst);                       
               }
				else  					
					if(MessageBox(NULL,
						L"Could not load Riot.dll\n\nThe latest version of Riot.dll is freely available at:\nhttp://luci.criosweb.ro/riot/\n\nPress OK to go to the official website and download the DLL version of RIOT",
								L"Component required",
								MB_OKCANCEL|MB_ICONEXCLAMATION) == IDOK) {

							ShellExecute(NULL, NULL, L"http://luci.criosweb.ro/riot/download/",NULL,NULL,SW_SHOW);
					}

 
  if(hDIB)
    GlobalFree(hDIB);  

  gimp_drawable_detach (drawable);
  if(bRet)    
		values[0].data.d_status = GIMP_PDB_SUCCESS;
  else
      values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
}
