#!/usr/bin/python
# =====================================================================
# Bloom for GIMP Python-Fu
# =====================================================================
#
# N-...n-...n-need m-m... m-more... Uh, oh, sorry, just one of the
# dark sides of my TES4:Oblivion/Zelda: Twilight Princess
# addiction. Well, one thing the game industry has shown is that we
# need more Bloom.  And this script will hopefully help you get
# that.
#
from string import join, split
helptext = """Apply light bloom effect on an image. Simply put, this script will
make shiny parts of the image even shinier and makes the light creep over the
edges of darker objects, sort of like what usually happens when you take a photo
against the light. The script will create a flattened copy of the image, and add
a layer of light on top of it. You can play with the growth settings in the
script and the layer opacity and mode to achieve certain effects. Suggested
modes are Soft Light, which creates a very subtle effect, and Hard Light, which
creates a stylish, exaggeratedly shiny effect."""
helptext = join(split(helptext,"\n"))
#
# This script is, as it is, a quick hack based on Matthew Welch's tutorial
# <URL:http://www.squaregear.net/gimptips/blom.shtml>. There are many
# ways to achieve the same effect with slightly different outcomes;
# (all depending on what to blur and when); perhaps in the future
# I'll make a little bit different versions. =)
#
# =====================================================================
# Copyright (C) 2008 Urpo Lankinen.
# This script is distributed under GNU General Public License v2 or
# later, which is the same license as The GIMP itself uses. You should
# have received a copy of this license with the GIMP. As usual, there
# is NO WARRANTY whatsoever, expressed or implied.
# =====================================================================
# This is a port of my earlier Perl-Fu script. Complete release history:
release_history = """2008-04-12, Release 1 - in Perl-Fu
2008-06-30, Release 1 [yes, again] - in Python-Fu
2008-07-04, Release 1.0.1 - cleanup, no major functional changes"""
# =====================================================================
# E-mail: <wwwwolf@iki.fi> Homepage: <URL:http://www.iki.fi/wwwwolf/>
# =====================================================================
# SVN: $Id: bloom.py 42 2008-07-04 16:21:46Z wwwwolf $
# =====================================================================

from gimpfu import *

def python_wolfy_bloom(sourceimg, drawable,
	bloom_amount=10, bloom_opacity=70, hard_light=0, flatten_final=0):
	
	pdb.gimp_context_push()
	
	img = sourceimg.duplicate()
	orig_layer = img.flatten()
	width = img.width
	height = img.height
	pdb.gimp_display_new(img)
	
	img.undo_group_start()
	
	# Add a new layer here. We need to add an alpha channel to it and clear
	# its contents, and it's tricky to do that later when we're supposed to
	# have the fillable content at hand, so we do it here in the beginning.
	if hard_light:
		bloom_mode = HARDLIGHT_MODE
	else:
		bloom_mode = SOFTLIGHT_MODE
	bloom_layer = gimp.Layer(img,"Bloom",width,height,RGB_IMAGE,
		bloom_opacity,bloom_mode)
	img.add_layer(bloom_layer,0)
	bloom_layer.add_alpha()
	pdb.gimp_selection_all(img)
	pdb.gimp_edit_clear(bloom_layer)
	
	# Copy image
	img.active_layer = orig_layer
	pdb.gimp_edit_copy(img.active_layer)
	
	# Create a new channel
	# For some reason hardly disclosed, the OO version doesn't work.
	#selchn = gimp.Channel(img,"For selection",width,height,100,(0,0,0))
	selchn = pdb.gimp_channel_new(img,width,height,"For selection",100,(0,0,0))
	img.add_channel(selchn,0)
	img.active_channel = selchn
	
	# Paste to the new channel
	fs = pdb.gimp_edit_paste(selchn,1)
	# fs.anchor() # Unlike Perl, Python doesn't have this. 
	pdb.gimp_floating_sel_anchor(fs) # We'll do this the stupid way.
	# If there's an OO way to do this, please do tell.
	
	# Starken the layer's colours.
	# Possible alternative method: Threshold/Blur.
	pdb.gimp_curves_spline(selchn,0,4,[127,0,254,254])
	
	# Take the new channel and turn it into a selection.
	pdb.gimp_selection_none(img)
	pdb.gimp_selection_load(selchn)
	
	# Get rid of the channel in question.
	#img.active_channel = None # This doesn't work, even as specced.
	img.remove_channel(selchn)
	
	# Grow the selection by the user-chosen bloom amount. This is where the
	# magic happens. =)
	pdb.gimp_selection_grow(img,bloom_amount)
	
	# Okay, let's add the bloom effect on the bloom layer.
	img.active_layer = bloom_layer
	
	# Flood-fill with white
	gimp.set_foreground(255,255,255)
	pdb.gimp_edit_bucket_fill(bloom_layer,FG_BUCKET_FILL,NORMAL_MODE,
		100.0, 0, 1, 0, 0)
	
	# Blur
	pdb.gimp_selection_all(img)
	pdb.plug_in_gauss_rle(img,bloom_layer,bloom_amount*2,1,1)
	# ...and that's essentially the whole bloom process.
	# Time for post-processing.
	
	# If we flatten the image, we flatten the image. Can't get more
	# specific than that!
	if flatten_final:
		img.flatten()
	
	# Done
	pdb.gimp_selection_none(img)
	img.undo_group_end()
	pdb.gimp_context_pop()
	
	return img


##########################################################################

register("wolfy-bloom",
	"Apply light bloom effect on an image.",
	helptext,
	"Urpo Lankinen <wwwwolf@iki.fi>, based on a tutorial by Matthew Welch",
	u"\u00A9 2008 Urpo Lankinen. Distributed under GNU GPL v2 or later.",
	release_history,
	"<Image>/Filters/Light and Shadow/Light/Bloom...",
	"*",
	[
		(PF_INT32, "bloom_amount", "Bloom growth in pixels", 10),
		(PF_SLIDER, "bloom_opacity", "Bloom opacity", 70, [0,100,1]),
		(PF_TOGGLE, "hard_light", "Hard instead of soft light", 0),
		(PF_TOGGLE, "flatten_final", "Flatten final image", 0)
	],
	[
		(PF_IMAGE,"image","A copy of the source image with effect applied.")
	],
	python_wolfy_bloom)

main()


