#!/usr/bin/env python
# -*- coding: utf-8 -*-

# GIMP script to add a muti layers calendar outside an untouched image
# Copyright under GPL v2

# Plug-in history:
# ° 2007, Script para GIMP en Python : Crear calendario de imagen
# Realizado por Fco. Javier Pérez Pacheco como ejercicio.
#
# ° Dec. 3, 2009, Translate the menu to English, add some language mods and other changes to increase flexibility, I hope;
#    tested then with GIMP 2.6.7 and Python 2.6.2; by Robert Brizard.
# ° July 14, 2010, Version 0.2 with 'date comments'.
# ° May 16, 2011, version 0.3 with second stage dialogs, like a date comments editor.

# importamos los módulos necesarios
from gimpfu import *
import gtk, locale
import datetime, gettext
import os, sys, platform

locale_directory = gimp.locale_directory
gettext.install( "gimp20-python", locale_directory, unicode=True )

def rank2date(cell_date_shift, daysInMonth, rank):
    """
    Converts a date by position like fist Monday = '|12' (rank) to a date number.
    The algorithm below is based on a month's table where the first of the month is in the first cell position,
    the weekdays are reorganized (but preserve week cycle) to accommodate.
    Return a date number (integer) or -1 if rank is not valid.
    """
    rank = int(rank)
    day_nr =  rank % 10
    if day_nr > 7 or day_nr < 1: date = -1
    else: 
        temp = (rank/10 - 1)*7 + day_nr - cell_date_shift
        if day_nr > cell_date_shift:
            date = temp
        else:
            date = temp + 7

        if daysInMonth < date or 0 > date: date = -1
    return(date)

def syntax_com(task_line, cell_date_shift, daysInMonth):
    """
    Verifies the syntax of the comment control line, it has been amputated of trailing empty field.
    Return a tuple: number (0, 1, 2, 3) for (ok, warning, error, not it) and a text string for a message line.
    """

    # the not it
    try:
        entries = eval(task_line)
    except:
        return(3, _("line# %d don't have the basic structure of a control line! Edit here if it should."))

    # the error
    len_entries = len(entries)
    # verifies the required fields first: date and at least one line
    if len_entries < 2:  return(2, _("missing fields in line# %d"))
    date = entries[0].strip()
    if date.find('|') == 0:
        rank = date[1:]
        if len(date) != 3 or not rank.isdigit(): return(2, _("problem in line# %d after '|' in the Date field: 2 digits are needed."))
        elif rank2date(cell_date_shift, daysInMonth, rank) == -1: return(2, _("problem in line# %d after '|' in the Date field: \
2 digits, first is week in the month, second weekday number."))
    else:
        if len(date) > 2 or not date.isdigit(): return(2, _("problem with number in line# %d, in the Date field."))
        elif int(date) > daysInMonth: return(2, _("not a realistic number in line# %d, in the Date field."))
    lines = entries[1].strip()
    if len_entries > 2: lines +=  entries[2].strip()
    if lines == "": return(2, _("a non empty field 'Line 1' or 'Line 2' is needed for line# %d."))
    if len_entries > 4: return(2, _("error in line# %d because of too many fields (max = 4)."))

    # the warning
    if entries[1].find("\"") != -1: return(1, _("line# %d: the char \" may cause problem later."))
    if len_entries > 2:
        if entries[2].find("\"") != -1: return(1, _("line# %d: the char \" may cause problem later."))
    if len_entries == 4:
        other = entries[3].strip()
        if len(other) != 1 or not other.isdigit(): return(1, _("line# %d: problem with the number of digit in optional Repeat field."))
        elif int(other) > 4: return(1, _("not a realistic number (limit is 4 weeks) in line# %d in the optional Repeat field."))

    # ok
    return(0, _("line# %d seems OK"))

def text_buffer_op(buffer_com, marks, line_nr, button, mess_line):
    start_iter = buffer_com.get_iter_at_offset(0)
    buffer_com.delete_interactive(start_iter, buffer_com.get_iter_at_mark(marks[0]), True)
    buffer_com.insert_with_tags_by_name(start_iter, button+mess_line%line_nr, "blue_foreground")

def buffer2tasks(buffer_com, marks, cell_date_shift, daysInMonth):
    tasks_iter = buffer_com.get_iter_at_mark(marks[1])
    tasks_line = tasks_iter.get_visible_text(buffer_com.get_end_iter())
    tasks = tasks_line.splitlines()
    syn_bin = [[], [], [], []]
    if len(tasks) > 0:
        buffer_com.delete_interactive(tasks_iter, buffer_com.get_end_iter(), True)
        syns = [syntax_com(tasks[i], cell_date_shift, daysInMonth)[0] for i in range(len(tasks))]
        # puts in 'syn_bin' (4), only tasks with warning or ok will be passed along later
        for i in range(len(tasks)): syn_bin[syns[i]].append(tasks[i])
    return(syn_bin)

#************** GTK PARTS DIALOG *************

def messBox(message, gtk_type, modal):
    """
     Halt the script and display instructions concerning what is expected next.
    """
    if modal == 0:
        flag = gtk.DIALOG_DESTROY_WITH_PARENT
    else:
        flag = gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT
    msgBox = gtk.MessageDialog(None, flag, gtk_type, gtk.BUTTONS_OK, message)
    msgBox.run()
    msgBox.destroy()

def commentBox(message, txt_lengh, tasks_line, cell_date_shift, daysInMonth, mess_font):
    """
    Halt the script and display instructions concerning what is expected next;
    edit comments interactively and return the possibly modified tasks_line for 'Calendar()'.
    """

    gray50_width  = 2
    gray50_height = 2
    gray50_bits   = '\x02\x01'

    PROCESS = _("Process entries")
    BACK = _("Back to entries")
    INFO = _("Update list")
    dialog = gtk.Dialog(_("Calendar date comments editor"), None,
                       gtk.DIALOG_MODAL,        #  | gtk.DIALOG_DESTROY_WITH_PARENT
                       (str(PROCESS), 70, str(BACK), 71, str(INFO), 72, gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))    

    hbox = gtk.HBox(False, 8)
    hbox.set_border_width(8)
    dialog.vbox.pack_start(hbox, False, False, 0)

    label = gtk.Label(message) # text in Label is left justify by default
    label.set_alignment(0.05, 0.1) # but not the box who is centered in the window

    # gtk.Table(rows=1, columns=1, homogeneous=False)
    table_com = gtk.Table(3, 4)
    table_com.set_row_spacings(4)
    table_com.set_col_spacings(4)
    fields_name = [_('Date'), _('Line 1'), _('Line 2'), _('Repeat')]
    fields_type = [_('int. or |int.'), _('text'), _('text'), _('digit')]
    fields_val = ['', '', '', '']
    entry_width = [3, txt_lengh+3, txt_lengh+3, 1]
    for i in range(0, 4):   # to populate the columns of the table
        table_name = gtk.Label(fields_name[i])
        table_com.attach(table_name, i, i+1, 0, 1)
        table_type = gtk.Label(fields_type[i])
        table_com.attach(table_type, i, i+1, 1, 2)
        fields_val[i] = gtk.Entry(entry_width[i])
        fields_val[i].set_width_chars(entry_width[i]+1)     # to set the width of Entry boxes
        table_com.attach(fields_val[i], i, i+1, 2, 3)
    frame = gtk.Frame(_("Entries:"))
    frame.add(table_com)
    
    label1 = gtk.Label("↑↑↑↑↑↑↑↑↑↑ "+_("Place text cursor in a line below and click on 'Back to entries'.\n\
                    \t\t\t       Check syntax of entries and convert to control line by button 'Process entries'.")+" ↓↓↓↓↓↓↓↓↓↓")
    label1.set_alignment(0.05, 0.1)
    frame1 = gtk.Frame(_("List of control lines:"))    
    view_com = gtk.TextView()
    buffer_com = view_com.get_buffer()
    view_com.set_wrap_mode(gtk.WRAP_WORD)
    #view_com.set_border_window_size(gtk.TEXT_WINDOW_TOP, 1)
    sw = gtk.ScrolledWindow()
    sw.set_shadow_type(gtk.SHADOW_ETCHED_IN)
    sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.add(view_com)
    frame1.add(sw)
    #frame1.set_resize_mode(gtk.RESIZE_IMMEDIATE) cause problem
    entry2 = gtk.Entry() # Click on a line to re-edit.
    entry2.set_editable(False)
    frame2 = gtk.Frame(_("Font attributes: "))
    table_font = gtk.Table(1, 2)
    entry3 = gtk.Entry()
    entry3.set_editable(False)
    font_attr = gtk.FontSelection()
    font_colo = gtk.ColorSelection()
    table_font.attach(font_attr, 0, 1, 0, 1)
    table_font.attach(font_colo, 1, 2, 0, 1)
    frame2.add(table_font)

    # in the vbox area 
    dialog.vbox.pack_start(label)
    dialog.vbox.pack_start(frame)
    dialog.vbox.pack_start(label1)
    dialog.vbox.pack_start(frame1)
    dialog.vbox.pack_start(entry2)
    dialog.vbox.pack_start(frame2)
    dialog.vbox.pack_start(entry3)
    
    # add none in the action area
    dialog.show_all()

    # get start of buffer; each insertion will revalidated the
    #   iterator to point to just after the inserted text. Need to use tag I think
    stipple = gtk.gdk.bitmap_create_from_data(None,
            gray50_bits, gray50_width, gray50_height)
    tag_bg = ["green_background", "yellow_background", "red_background", "blue_background"]
    # text_tags =  gtk.TextTag().set_property("language-set", True, "language", lang_list[mess_font[3]] )     #  don't work
    for i in range(4): buffer_com.create_tag(tag_bg[i], background=tag_bg[i].partition('_')[0]) 
    buffer_com.create_tag("blue_foreground", foreground="blue")
    buffer_com.create_tag("not_editable", editable=False)
    buffer_com.create_tag("right_justify", justification=gtk.JUSTIFY_RIGHT)
    buffer_com.create_tag("background_stipple", background_stipple=stipple)
    
    # put a special line at the beginning for interactive messaging with name mark after
    iter = buffer_com.get_iter_at_offset(0)
    mess_line = _("MESSAGE LINE: if you 'copy & paste', do it below. ")
    end_mess_line = len(mess_line)-1  # 47
    buffer_com.insert_with_tags_by_name(iter, mess_line, "blue_foreground")
    end_mess_mark = buffer_com.create_mark("end_mess", buffer_com.get_iter_at_line_offset(0, end_mess_line), False)
    buffer_com.insert(iter, "\n"+tasks_line)
    tasks_mark = buffer_com.create_mark("tasks", buffer_com.get_iter_at_line_offset(1, 0), True)
    marks = (end_mess_mark, tasks_mark)

    process_mess = _("'Process entries': place the result of the entry fields in List frame; green background in list (OK), yellow \
(WARNING) and red (ERROR).")
    back_mess = _("'Back to entries': typical use is after an edit in the List frame, a particular instance of red (ERROR) or \
blue (STRUCTURE) will abort that operation.")
    update_mess = _("'Update list': transform the valid lines to a basic form for the month; gives a date list to check date \
collision. Valid lines are the yellow and green ones.")

    entry2.set_text(process_mess)
    entry3.set_text(mess_font[0]+mess_font[1])
    response = dialog.run()
    while response in [70, 71, 72, gtk.RESPONSE_ACCEPT]:

        if response == 70:  # PROCESS
            entry2.set_text(process_mess)
            
            task_entry = [fields_val[i].get_text() for i in range(4)]
            tasks_iter = buffer_com.get_iter_at_mark(tasks_mark)
            list_nr = buffer_com.get_line_count() - 2
            stop = False
            if list_nr > 0: 
                tasks_line = tasks_iter.get_visible_text(buffer_com.get_end_iter())
                tasks_date = tasks_line.splitlines()
                if len(task_entry[0]) == 1: task_entry[0] = '0' + task_entry[0]
                for i in range(len(tasks_date)):  # to stop simple date duplication, but the list is not necessarily ok
                    pos_coma = tasks_date[i].find(',')
                    if pos_coma == -1: continue
                    tasks_date[i] = tasks_date[i][2:pos_coma-1]
                    if len(tasks_date[i]) == 1: tasks_date[i] = '0' + tasks_date[i]
                    if task_entry[0] == tasks_date[i]:
                        text_buffer_op(buffer_com, marks, -1, _("LAST 'Process...': "), _("will not process because there \
is already a line with that date field. Suggestion: bring that line 'Back to entries' instead. %d"))
                        stop = True
                        break
            
            if task_entry[0] == '': text_buffer_op(buffer_com, marks, -1, _("LAST 'Process...': "), _("will not process because \
a non empty date field is required. %d"))
            elif stop == False:
                trail_ind = 4     # removing trailing empty field
                for i in range(3, -1, -1):
                    fields_val[i].set_text('')  # clear the entries
                    if task_entry[i] == '' and not stop: trail_ind = i
                    else: stop = True
                    
                # verifies syntax and add tag_bg[0] (green), .[1] (yellow), .[2] (red), .[3] (blue)
                task_entry = repr(task_entry[:trail_ind])
                state = syntax_com(task_entry, cell_date_shift, daysInMonth)     # 'state' is a tuple with 2 elements
                # put 'textiter' at the end of buffer
                iter = buffer_com.get_end_iter()
                buffer_com.insert_with_tags_by_name(iter, task_entry+"\n", tag_bg[state[0]], "background_stipple")
                text_buffer_op(buffer_com, marks, buffer_com.get_line_count() - 2, _("LAST 'Process...': "), state[1])

        if response == 71:  # BACK
            entry2.set_text(back_mess)
            # point textiter to text cursor
            iter = buffer_com.get_iter_at_mark(buffer_com.get_insert())
            line_nr = iter.get_line()
            start_iter = buffer_com.get_iter_at_line_offset(line_nr, 0)
            end_iter = buffer_com.get_iter_at_line_offset(line_nr+1, 0)
            task_line = buffer_com.get_text(start_iter, end_iter, False)[:-1]
            state = syntax_com(task_line, cell_date_shift, daysInMonth)
            try:
                entries = eval(task_line)
                # gimp.message('The 'textview' cursor was at line = %s'%task_line)
                if len(entries) < 5:    # ok
                    for i in range(len(entries)): fields_val[i].set_text(entries[i])
                    buffer_com.delete_interactive(start_iter, end_iter, True)
                    text_buffer_op(buffer_com, marks, line_nr, _("BEFORE 'Back...' it was ")+task_line+\
                                   _(" with a syntax comment: "), state[1])
                else:   # error too many fields for entries
                    buffer_com.delete_interactive(start_iter, end_iter, True)
                    buffer_com.insert_with_tags_by_name(buffer_com.get_end_iter(), task_line+"\n", tag_bg[2],
                                                        "background_stipple")
                    line_nr =  buffer_com.get_line_count() - 2
                    mess_line = _("error in line# %d because of too many fields, unable to comply. (Edit below)")
                    text_buffer_op(buffer_com, marks, line_nr, _("LAST 'Back...': "), mess_line)
            except:
                mess_line = _("sorry, line# %d don't have the basic structure of a control line!\
 Edit below if it should.")
                if line_nr == 0:  mess_line = _("line# %d is not part of the list of control lines!")
                text_buffer_op(buffer_com, marks, line_nr, _("LAST 'Back...': "), mess_line)
                
        if response == 72 or response == gtk.RESPONSE_ACCEPT:   # common code for the two
            # transform the valid lines to a basic form for the month and give general info on the one that will be kept;
            #   give the longest line, etc...
            entry2.set_text(update_mess)
            syn_bin = buffer2tasks(buffer_com, marks, cell_date_shift, daysInMonth)
            if syn_bin != []:
                # list of date in the month with comment, for valid control lines, if collision, user choose which to keep
                gtk.ColorButton(color=gtk.gdk.Color(0,0,0))
                task_valid = syn_bin[0] + syn_bin[1]
                if task_valid != []:   # else please enter a valid control line!
                    date_valid = [] #; date_can = []     # lists of date characters
                    tasks_add = []   # tasks to replace 'task_valid' because of transformation or not
                    max_txt_line = 0
                    for task in task_valid:
                        task_e = eval(task)
                        #  check date field
                        if len(task_e[0]) == 1:
                            task_e[0] = '0' + task_e[0]
                            date_can = task_e[0]
                        elif len(task_e[0]) < 3: date_can = task_e[0] # int(task_e[0])
                        else:   # date by rank
                            task_e[0] = repr(rank2date(cell_date_shift, daysInMonth, task_e[0][1:]))
                            if len(task_e[0]) == 1: task_e[0] = '0'+task_e[0]
                            date_can = task_e[0] + '*|'
                        # find the largest number of char in comment text line
                        if len(task_e) > 2: cand_max = max(len(task_e[1]), len(task_e[2]))
                        else: cand_max = len(task_e[1])
                        if cand_max > max_txt_line: max_txt_line = cand_max
                        # add the repeat dates
                        if len(task_e) == 4:
                            field_re = task_e[3]
                            task_e = task_e[:3]
                            if len(field_re) == 1 and field_re.isdigit() and int(field_re) < 5: # if repeat param OK
                                limit = daysInMonth + 1 - int(field_re)*7
                                date_rep = int(date_can[:2])
                                while date_rep < limit:
                                    date_rep += 7*int(field_re)
                                    if date_rep < 10: date_str = '0'+repr(date_rep)
                                    else: date_str = repr(date_rep)
                                    date_valid.append(date_str+'*r')
                                    tasks_add.append(repr([date_str, task_e[1], task_e[2]]))
                        tasks_add.append(repr(task_e))
                        date_valid.append(date_can)

                    # test for a collision by pair comparison of first 2 chars of each element in 'date_valid'
                    date_valid.sort()   # key=int
                    coll_list = []
                    for i in range(len(date_valid)-1):
                        if date_valid[i][0:2] == date_valid[i+1][0:2]: coll_list += i+1, i+2 # remenber line numbers
                    date_list = ', '.join(date_valid)
                    # sort list by date
                    tasks_add.sort()
                
                    # write all tasks to 'buffer_com'
                    tasks_add_line = '\n'.join(tasks_add)
                    iter = buffer_com.get_iter_at_mark(tasks_mark)
                    for i in range(3, 1, -1):
                        if len(syn_bin[i]) == 0: continue
                        for task in syn_bin[i]: buffer_com.insert_with_tags_by_name(iter, task+"\n", tag_bg[i], "background_stipple")
                    for i in range(len(tasks_add)):
                        state = syntax_com(tasks_add[i], cell_date_shift, daysInMonth)
                        if i+1 in coll_list: bckg = ""
                        else: bckg = "background_stipple"
                        buffer_com.insert_with_tags_by_name(iter, tasks_add[i]+"\n", tag_bg[state[0]], bckg)
                                
                    if response == 72:  # UPDATE
                        
                        if date_list.find('*') != -1: text_buffer_op(buffer_com, marks, len(tasks_add), _("LAST 'Update...': "),\
                                                                     _("Number of valid lines = %d. The date list \
is : ")+date_list+_(" . Symbol: *| => date by rank and *r => repeat series. The maximum number of text characters\
 in List: %d")%max_txt_line)
                        else: text_buffer_op(buffer_com, marks, len(tasks_add), _("LAST 'Update...': "), \
                                             _("Number of valid lines = %d. The date list \
is : ")+date_list+_(" . The maximum number of text characters in List: %d . ")% max_txt_line)
                        if max_txt_line > txt_lengh + 3: entry3.set_text(_("Your maximum number of text characters is\
 greater than the estimate, the choice of a 'narrow' font is prudent! ")+mess_font[1])
                        elif txt_lengh < 14:  entry3.set_text(_("The estimate %d is low, to have longer text on the line\
 use a 'narrow' font. ")+mess_font[1]%txt_lengh)
                        else:
                            font_comm_words = font_attr.get_font_name().split()    # split on space to permit next lines
                            font_comm_name = ' '.join(font_comm_words[:len(font_comm_words)-1])
                            # construct the list of comment text lines and find the longest (space)
                            text_lines_list = []
                            for task in tasks_add: text_lines_list.extend(eval(task)[1:])
                            index_max = max_text_width(text_lines_list, font_comm_name)
                            max_size_comm = Size_line(mess_font[2][0], mess_font[2][1], text_lines_list[index_max], font_comm_name)
                            entry3.set_text(_("With the selected font it's about 6<font_size<%d PX (constrained by %s), for '%s', your longest line. ") \
                                             %(max_size_comm[0], max_size_comm[1], text_lines_list[index_max]) +mess_font[1])
                                            
                        if coll_list != []:
                            iter = buffer_com.get_iter_at_mark(end_mess_mark)
                            buffer_com.insert_with_tags_by_name(iter, _("There's a date collision, please edit further to remove it."),\
                                                                 tag_bg[2], "blink")
                        
                    if response == gtk.RESPONSE_ACCEPT:   # if OK return

                        # stop on date collision between valid line for that month (reuse part of response = 72),
                        if coll_list != []:
                            entry2.set_text(_("OK: there is a problem you have to resolve!"))
                            start_iter = buffer_com.get_iter_at_offset(0)
                            buffer_com.delete_interactive(start_iter, buffer_com.get_iter_at_mark(end_mess_mark), True)
                            buffer_com.insert_with_tags_by_name(start_iter, _("After OK: Eliminates date collision between lines: %s, before \
normal exit, please.")%repr(coll_list), tag_bg[2], "background_stipple")
                        else:   # normal exit
                            name_font = font_attr.get_font_name()
                            color_font = font_colo.get_current_color().to_string()
                            # transform to a color tuple, directly usable by GIMP 2.6 pdb procedures
                            color_font = (int(color_font[1:5], 16)/257, int(color_font[5:9], 16)/257, int(color_font[9:], 16)/257)
                            dialog.destroy()
                            return(tasks_add_line, name_font, color_font)
                else:
                    # else a message to please enter a valid control line!
                    start_iter = buffer_com.get_iter_at_offset(0)
                    buffer_com.delete_interactive(start_iter, buffer_com.get_iter_at_mark(end_mess_mark), True)
                    buffer_com.insert_with_tags_by_name(start_iter, _("MESSAGE LINE: for this you need a valid control line. You can quit \
by clicking on the [x] in the title bar."), "blue_foreground")

        response = dialog.run()
        
    dialog.destroy()
    return('')
    
def colorsBox(defaults=''):
    """
    Replacement of four previous color choices (GIMP tuple) in the register dialog by a secondary dialog.
    
    Input: the default GIMP color tuples list to use or none
    Output: a list of four chosen colors, compatible with GIMP script.
    """
    colour_val = [] # to store the user's choice of color
    color_list = [] # return list of string of hexadecimal characters that represents the colors 
    colors = []
    
    dialog = gtk.Dialog(_("Calendar colour chooser"), None,
                       gtk.DIALOG_MODAL,        #  | gtk.DIALOG_DESTROY_WITH_PARENT
                       (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))    
    hbox = gtk.HBox(False, 4)
    hbox.set_border_width(4)
    dialog.vbox.pack_start(hbox, False, False, 0)

    label = gtk.Label(_("Click on the colour button to choose a RGB colour for each item.\n To better judge colours \
drag this window between your image and the background.\n")) # text in Label is left justify by default
    label.set_alignment(0.05, 0.1) # but not the box who is centered in the window

    # gtk.Table(rows=1, columns=1, homogeneous=False)
    table_col = gtk.Table(1, 5, True)
    table_col.set_row_spacings(4)
    table_col.set_col_spacings(8)
    fields_name = [_('Grid'), _('Weekday_tags'), _('Holiday_dates'), _('Other_dates')]
    # gtk.gdk.Color(r, g, b) where r, g, & b are between 0->65535
    if len(defaults) == 4:
        for i in range(4):
            defaults[i] = (defaults[i][0]*257, defaults[i][1]*257, defaults[i][2]*257) 
            colors.append(gtk.gdk.Color(defaults[i][0], defaults[i][1], defaults[i][2]))
    else: colors = [gtk.gdk.Color(51400,51400,51400), gtk.gdk.Color(0,0,0), gtk.gdk.Color(0,0,59110), gtk.gdk.Color(0,0,0)]
    for i in range(4):   # to populate the columns of the table
        table_name = gtk.Label(fields_name[i])
        table_col.attach(table_name, i, i+1, 0, 1)
        colour_val.append(gtk.ColorButton(colors[i]))
        colour_val[i].set_title(fields_name[i])
        table_col.attach(colour_val[i], i, i+1, 1, 2)

    # in the vbox area
    dialog.vbox.pack_start(label)
    dialog.vbox.pack_start(table_col)
    dialog.show_all()

    response = dialog.run()
    if response == gtk.RESPONSE_ACCEPT:
        for i in range(4):
            color_list.append(colour_val[i].get_color().to_string())
            color_list[i] = (int(color_list[i][1:5], 16)/257, int(color_list[i][5:9], 16)/257, int(color_list[i][9:], 16)/257)
    dialog.destroy()
    return(color_list)

def headerBox(weekdays,  lang,  header):
    """
    Secondary dialog to edit weekday header tags, need function 'get_resource_file()'
    
    Input: the default tag list to use + other tag list
    Output: the custom tag list.
    """
    no_go = False
    no_file = False
    title = _("Calendar: edit weekday tags")
    label_val = _("Enter the tags you want.\n")
    if header == '':
        # no proper resource file
        no_file = True
        mess = _("To navigate tag use the <tab> key. 'Save tags' and 'Load next' will not operate because there is no proper \
'year_comments...' file.")
    else:
        nr_header = cycle = len(header)
        if nr_header > 0:
            mess = _("To navigate tag use the <tab> key. The number of header presets is %d ; use the button 'Load next' to cycle.")%nr_header
        else:
            mess = _("To navigate tag use the <tab> key. 'Load next' is not operative (preset = %d). Quitting (clicking [x]) results \
in the tags display at first.")%nr_header
    next = _("Load next")
    save = _("Save tags")
    dialog = gtk.Dialog(title, None, gtk.DIALOG_MODAL,  # | gtk.DIALOG_DESTROY_WITH_PARENT
                       (str(save),  70,  str(next),  71,  gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))    
    hbox = gtk.HBox(False, 4)
    hbox.set_border_width(3)
    dialog.vbox.pack_start(hbox, False, False, 0)

    label = gtk.Label(label_val)

    # gtk.Table(rows=1, columns=1, homogeneous=False)
    table_col = gtk.Table(2, 7, True)
    table_col.set_row_spacings(4)
    table_col.set_col_spacings(3)
    fields_val = ['', '', '', '', '', '', '']
    for i in range(7):
        table_name = gtk.Label(str(i+1)+': '+weekdays[0][i])
        table_col.attach(table_name, i, i+1, 0, 1)
        fields_val[i] = gtk.Entry(11)
        fields_val[i].set_width_chars(11)     # to set the width of Entry boxes
        table_col.attach(fields_val[i], i, i+1, 1, 2)
    label1 = gtk.Label('')
    entry_mess = gtk.Entry()
    entry_mess.set_editable(False)
    # in the vbox area
    dialog.vbox.pack_start(label)
    dialog.vbox.pack_start(table_col)
    dialog.vbox.pack_start(label1)
    dialog.vbox.pack_start(entry_mess)
    dialog.show_all()

    entry_mess.set_text(mess)
    for i in range(7): fields_val[i].set_text(weekdays[1][i])

    mess1 = _("PROBLEM: at least a tag in your weekday header is missing. Please correct that!")
    response = dialog.run()
    while response in  [70,  71,  gtk.RESPONSE_ACCEPT]:
        if response == 70  and not no_file :      
            # Save
            if nr_header > 9:
                entry_mess.set_text(_("Over the limit of 10, no more saving!"))
            else:
                weekdays_cus = [fields_val[i].get_text() for i in range(7)]
                for tag in weekdays_cus:
                    if len(tag) == 0: no_go = True
                if no_go == True:
                    entry_mess.set_text(mess1)
                    no_go = False
                else:   
                    # proceed, see if not there already
                    if weekdays_cus in header:
                        entry_mess.set_text(_("Not save because it's already there!"))
                    else:
                        # call get_resource_file with the header parameter
                        error_file = get_resource_file(1, lang, 0,  weekdays_cus)[1:3]
                        file_abrev = error_file[1][len(error_file[1])-20:len(error_file[1])]
                        if error_file[0] != 5: mess = _("Unable to save in '...%s'. Error number: %s .")%(file_abrev,  str(error_file[0]))
                        else: 
                            nr_header += 1
                            mess = _("Has been save in '...%s'. The number of header presets is now %d .")%(file_abrev,  nr_header)
                            header.append(weekdays_cus)
                        entry_mess.set_text(mess)
            
        if response == 71  and not no_file and nr_header > 0:      
            # Next preset
            cycle -= 1
            if cycle < 0: cycle =nr_header-1
            # verify if not the same as in the entries
            try:
                weekdays_cus = [fields_val[i].get_text() for i in range(7)]
                if weekdays_cus == header[cycle] and nr_header > 1: cycle -= 1
                for i in range(7): fields_val[i].set_text(header[cycle][i])
                entry_mess.set_text(_("Header preset #%d, out of %d, is in the entry fields.")%(cycle+1, nr_header))
            except:
                header.remove(header[cycle])
                nr_header -= 1
                if nr_header < 1: entry_mess.set_text(_("Header preset #%d has an error! No preset left, edit or quit?")%(cycle+1))
                else: entry_mess.set_text(_("Header preset #%d has an error!")%(cycle+1))
        
        if response == gtk.RESPONSE_ACCEPT:
            weekdays_cus = [fields_val[i].get_text() for i in range(7)]
            for tag in weekdays_cus:
                if len(tag) == 0: no_go = True
            if no_go == True:
                entry_mess.set_text(mess1)
                no_go = False
            else:
                dialog.destroy()
                return(weekdays_cus)
        response = dialog.run()
    dialog.destroy()

    return(weekdays[1])

#********************************************** end GTK PARTS: DIALOG

def Size_font(cell_w, cell_h, text, font):
    """
    To select a size for the font so the text (one string) will fit inside a cell, priority given to width.

    Input variables: cell_w, the cell width
                     cell_h, the cell height
                     text, the string to fit inside the cell
                     font, the font for the string
    Output:     the font size for the fitted string
    """
    size_es = 5.0
    size_font = 7.0
    while (size_font - size_es) > 0.05:     # for the width of the cell, should be absolute value but too much time!  
        size_es = size_font
        text_width, text_height, text_ascent, text_descent = pdb.gimp_text_get_extents_fontname(text, size_es, PIXELS, font)
        size_font = size_es * 0.9 *  cell_w / text_width
    # verify if the height is OK
    height_font = text_height + text_ascent
    while height_font/cell_h > 1.0:
        size_font = size_font * 0.95
        text_width, text_height, text_ascent, text_descent = pdb.gimp_text_get_extents_fontname(text, size_font, PIXELS, font)
        height_font = text_height + text_ascent
    return(size_font)

def Size_line(cell_w, cell_h, text, font):
    """
    To select a size for the font so the text line will fit inside a cell, priority given to height.

    Input variables: cell_w, the cell width
                     cell_h, the cell height
                     text, the string to fit inside the cell
                     font, the font for the string
    Output:     the integer font size for the fitted string and the constraint
    """
    size_es = 5.0
    size_comm_adj = 7.0
    while (size_comm_adj - size_es) > 0.05:
        size_es = size_comm_adj
        text_width, text_height = pdb.gimp_text_get_extents_fontname(text, round(size_es),
                                                                     PIXELS, font)[0:2]
        size_comm_adj = size_es * 1.0 *  cell_h / text_height
    constraint = _('height')
    if text_width/cell_w > 0.95:
        constraint = _('width')
        # verify if the width is OK
        while text_width/cell_w > 0.95:
            size_comm_adj *=  0.95
            text_width = pdb.gimp_text_get_extents_fontname(text, round(size_comm_adj),
                                                            PIXELS, font)[0]
    size_comm_adj = int(round(size_comm_adj - 0.3))
    return(size_comm_adj, constraint)

def max_text_width(text_list, font):
    """
    Return the index of the string which occupies the max. width of the list
    The caller is responsible for a non empty test_list.

    Input variables: text_list: the list of strings to test
                     font: the font to use for the strings
    Output:     the list index of the longest string of the lot
    """
    cont = 0
    cont_max = 0
    text_width_max = 0
    for item in text_list:
        if item == [] or '': continue
        text_width = pdb.gimp_text_get_extents_fontname(item, 40, POINTS, font)[0]
        if text_width > text_width_max:
            text_width_max = text_width
            cont_max = cont
        cont += 1
    return(cont_max)

def get_resource_file(month, lang, first_weekday,  header=[]):
    """
    Return date comment control line for the month, found in a standard yearly comment file 
 
    Input variables: month: the month number
                            lang: the index in the global language list,
                            first_weekday: the index for Monday or Sunday
                            header: the weekday header to save in the file
    Output:  a tuple of the month comments, standard file existence, the path name of that file
    """
    month = str(month)
    tags = []
    file_status = 0  # error in dir
    year_comments = _("... don't know!")
    comments_text = ""
    prog_param = (lang, first_weekday)

    try:
        year_comments =  os.path.dirname(os.path.abspath(__file__))+r"%scalendar%syear_comments-%s.txt"\
                        %(os.sep, os.sep, lang_list[lang])  # 'lang_list' is a global var
        file_status = 1  # error in existence
        if platform.system() == 'Windows':
            # not necessary for a file in this place, since GIMP 2.6 on 'Win' is not adapted to the next char set,
            #   so it will not be able to find this script if non ascii in the path, here anyway if there is a change.
            f = open(year_comments.encode(sys.getfilesystemencoding()), 'r')  # the char set for 'Windows' file
        else :
            f = open(year_comments, 'r')
        lines_file = f.readlines()
        f.close()
        
        file_status = 2  # error in content
        if  header == []:
            cntr =  cr = len(lines_file)
            stop = False
            for i in range(len(lines_file)):
                lines_file[i] = lines_file[i].strip(' ')
                if lines_file[i][0] != "#":
                    if lines_file[i] == "0:\n": test_ind = i+1
                    # read comment line
                    if lines_file[i] == month+":\n": cntr = i
                    if i > cntr and lines_file[i] == "\n": stop = True
                    if i > cntr and not stop: 
                        comments_text += lines_file[i]
                        cntr += 1
                    elif i > cntr:
                        # read header
                        if lines_file[i] == "tags:\n": cr = i
                        if i > cr and  lines_file[i] != "\n": 
                            tags.append(eval(lines_file[i]))
                            #cr += 1
        
            test_param = eval(lines_file[test_ind])  
            # evaluate to a tuple with index to language and first_weekday values
            if test_param != prog_param:
                comments_text = ""
                file_status = 3  # error in suitability
            else: 
                file_status = 4  # ok for suitability
                if cr < len(lines_file): file_status = 5     # "tags:\n" is there
        else:                
            # write custom header at the end
            f = open(year_comments, 'a+')
            f.writelines(str(header)+"\n")
            f.close()
            file_status = 5 
    except: comments_text = ""            

    return(comments_text, file_status, year_comments,  tags)
    

#=============================================================================================================
##############################################################################################################

# Global variables, active before procedure register():
default_font = "Sans"
pt2px_conv = 300.0/72.0   # to print calendar it's 300 px/inch. (To convert from mm to pixel * by 118/9.99)
rel_size_my = 0.06           # relative size of month & year font to the height of the image except in calendar

weekdays_l = []           # list of weekday tags in different languages; the two letters is ISO 639-1 codes for language
weekdays_en = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat']; weekdays_l.append(weekdays_en)
weekdays_fr = ['dim', 'lun', 'mar', 'mer', 'jeu', 'ven', 'sam']; weekdays_l.append(weekdays_fr)
weekdays_de = ['Son', 'Mon', 'Die', 'Mit', 'Don', 'Fre', 'Sam']; weekdays_l.append(weekdays_de)
weekdays_it = ['Dom', 'Lun', 'Mar', 'Mer', 'Gio', 'Ven', 'Sab']; weekdays_l.append(weekdays_it)
weekdays_es = ['Dom', 'Lun', 'Mar', 'Mie', 'Jue', 'Vie', 'Sab']; weekdays_l.append(weekdays_es)
weekdays_ru = ['Вск', 'Пнд', 'Втр', 'Срд', 'Чтв', 'Птн', 'Сбт']; weekdays_l.append(weekdays_ru)
weekdays_pt = ['Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sáb']; weekdays_l.append(weekdays_pt)
                # <- add your language here if not in the above list, add also the month name in section |1
lang_list = ['en', 'fr', 'de', 'it', 'es', 'ru', 'pt']

path_list = str(__file__).split(os.sep)
if len(path_list) > 3: from_file = '...'+os.sep+os.sep.join(path_list[len(path_list)-3:])
else: from_file = str(__file__)

def Calendar(img, drawable, PhoCal_sep, cal_size_w, cal_size_h, first_weekday, weekday_header,
             font_weekdays, font_number, size_font_cal, month,year, holidays_om, lang, 
             font_month, space_month, print_ratio, comment_op):
    """
    Main procedure to add a calendar to a mainly untouched image.
    """

    # comenzamos a agrupar el UNDO                                                                                       
    img.undo_group_start()
    gimp.context_push()
    # checking starting conditions: 
    #   a) if more than one layer, 'flatten'? If name 'img.dirty' or not (you can undo)
    num_layers = pdb.gimp_image_get_layers(img)[0]
    if num_layers > 1 : drawable = img.flatten()
    
    #0 GESTION OF VALUES FOR THE SECONDARY DIALOGS ============================

    #####  In the next class
    # To have a gestion of the secondary dialog info by the section number, in a permanent parasite,
    #   so for now to preserve some data after an undo of 'calendar', similar to the primary dialog.
    #
    # The string in the parasite will be in the form of a dictionary with section number as keys,
    #   there value is a list of a tuple {check impacting choice} + other {value}.
    #
    # Methods: import from other month image? and save_all.
    #####################################

    class Paras_info(object):
        """
        A class for the gestion of the previous choices in secondary dialogs.
        Organize the data into a dictionary with section number as key; with a old and new dict.
        """
        new_dic = {'s1': [], 's3': [], 's11': []}
        cale_parasit = img.parasite_find('calendar-input')  # it is the object with the string data of the parasite
        if cale_parasit: old_dic = eval(cale_parasit.data)
        else: old_dic = {'s1': [], 's3': [], 's11': []}

        def __init__(self):
            self.old_dic
            self.new_dic
        
        # store the actual input data of the secondary dialogs in a persistent parasite at the end
        def save_all(self):
            img.parasite_detach('calendar-input')
            img.attach_new_parasite('calendar-input', 1, str(self.new_dic))

    gimp.progress_init(_("Calendar with image: sections 1-3 out of 12"))
    pdb.gimp_progress_pulse()

    #1 OUTPUT LANGUAGE & HEADER TYPE ======================================

    ##### variables defines in this section
    month = int(month); year = int(year)    # PF_SPINNER return a float but these are realy integers
    months_l = []   # the list of list for the name of the month in different languages
    # months : list of names of the months used in the rest of the code, start with '' for month 0
    weekdays_cus = ['', '', '', '', '', '', ''] # for user choice: custom
    weekdays_list = []  # for user choice: custom
    # mess = ""       # store text to communicate to the user by the 'Errors Console'
    # text_width : the width of the text box for a tag in weekdays
    # cont_max : the index of the weekdays list that has the longest text
    #####################################
    
    instance_paras = Paras_info()    # instance for the calendar parasite
    # user choice for the language to appear on the calendar
    months_en = ['en', 'January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', \
                 'November', 'December']; months_l.append(months_en)        # English
    months_fr = ['fr', 'Janvier', 'Février', 'Mars', 'Avril', 'Mai', 'Juin', 'Juillet', 'Août', 'Septembre', 'Octobre', \
                 'Novembre', 'Décembre']; months_l.append(months_fr)        # French
    months_de = ['de', 'Januar', 'Februar ', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', \
                 'November', 'Dezember']; months_l.append(months_de)        # German    
    months_it = ['it', 'Gennaio', 'Febbraio', 'Marzo', 'Aprile', 'Maggio', 'Giugno', 'luglio', 'Agosto', 'Settembre', 'Ottobre', \
                 'Novembre', 'Dicembre']; months_l.append(months_it)        # Italian
    months_es = ['es', 'Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio', 'Julio', 'Agosto', 'Septiembre', 'Octubre', \
                 'Noviembre', 'Diciembre']; months_l.append(months_es)      # Spanish
    months_ru = ['ru', 'Январь', 'Февраль', 'Март', 'Апрель', 'Май ', 'Июнь', 'Июль', 'Август', 'Сентябрь', 'Октябрь', \
                 'Ноябрь', 'Декабрь']; months_l.append(months_ru)           # Russian
    months_pt = ['pt', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 'Julho', 'Agosto', 'Setembro', 'Outubro', \
                 'Novembro', 'Dezembro']; months_l.append(months_pt)        # Portuguese
                # <- add your language here if not in the above list, then modify '(PF_OPTION, "lang" ...' in the register( ) function
    months = months_l[lang]
 
    # consult resource file
    if weekday_header == 2 or comment_op > 0:
        comment_file = get_resource_file(month, lang, first_weekday)    # , header

    # user's presentation of the weekday headers
    # bilingual_en_fr = ['M/L', 'T/M', 'W/M', 'T/J', 'F/V', 'S/S', 'S/D'] # and others variations
    weekdays = weekdays_l[lang]
    if first_weekday == 1:       
        # the first day is Monday
        weekdays.append(weekdays[0])
        del weekdays[0]
    if weekday_header == 0:        
        # Default
        cont_max = max_text_width(weekdays, font_weekdays)
    elif weekday_header == 1:     
        # Default (all caps)
        for i in range(0, len(weekdays)): weekdays[i] = weekdays[i].upper()
        cont_max = max_text_width(weekdays, font_weekdays)
    elif weekday_header == 2:     
        # Custom
        lang_interface = _('en')    # language display in dialogs to identify weekday
        try:  lang_interface = lang_list.index(lang_interface)
        except: lang_interface = 0
        weekdays_list.append(weekdays_l[lang_interface])
        for i in range(7): weekdays_cus[i] = weekdays[i].upper()
        check_s1 = (first_weekday, lang)   # for the validity of stored values in parasite
        weekdays_pre = instance_paras.old_dic.get('s1')
        if len(weekdays_pre) != 0 and check_s1 == weekdays_pre[0]:
            weekdays_cus = weekdays_pre[1]
        weekdays_list.append(weekdays_cus)

        if comment_file[1] > 4: tags_preset = comment_file[3]
        else: tags_preset = ''
        # elif comment_file[1] == 3:  
            # first_weekday is the other one
            
        weekdays = headerBox(weekdays_list, lang,  tags_preset)
        # find the largest tag of the the weekdays header. This result will also be used later in section 4.
        cont_max = max_text_width(weekdays, font_weekdays)

        instance_paras.new_dic['s1'] = [check_s1, weekdays]
    # month 13 is for first semester of the year and 14 is the second semester?
    
    #2 PRESIZE CALENDAR PAGE ==============================================

    ##### variables defines in this section
    layer_photo = drawable          # the layer of the starting image
    widthPho = img.width    # the width of the starting image in px
    heightPho = img.height  # height of the photo in px
    widthCal = cal_size_w*widthPho  # the width of the calendar box
    sep = PhoCal_sep*heightPho      # separation in px between photo & calendar box
    # w : it's the widest dim of the calendar box or photo
    dx = 0.0                        # delta x for calendar box
    offx = 0                        # offset along x for the image
    #####################################

    pdb.gimp_image_set_resolution(img, 300, 300)    # this is for printing.
    #img.unit = 1   # unit 1 is inch? (no effect alone)

    layer_photo.add_alpha()

    if widthCal < 300:
        gimp.message(_("ERROR:\n The calendar is too small for displaying weekday.\nTry with a larger calendar\
 or image, termination of the plug-in."))
        return
    heightCal = widthCal        # for now, will probably change later  
    if cal_size_w < 1.0:
        w = widthPho
        dx = (w-widthCal)/2.0
    else:
        w = widthCal
        offx = (w - widthPho)/2.0
    pcy = heightPho + sep       # position of the top of the calendar box along y    
    h = pcy + heightCal         # h, the provisional height of calendar page
    pdb.gimp_image_resize(img, w, h, offx, 0)
    drawa_photo = pdb.gimp_image_get_active_drawable(img)
    pdb.gimp_drawable_set_name( drawa_photo, _("starting image"))

    #3 CREATE CALENDAR'S LAYER BACKGROUND & GET COLOR =====================

    ##### variables defines in this section
    # layer_bg : ID of the background layer
    colo_list = []   # four RGB GIMP colors choosen by the user
    default_color = ''  # list of 4 default colors
    comments_text = ''  # a string of comments applicable in section 11
    # bg_weekdays : color for weekday background & grid lines
    # color_weekdays : color for weekday's tags except Sunday
    # color_festive : color for Sunday and selected holiday
    # color_number : color for ordinary date
    #####################################
    layer_bg = pdb.gimp_layer_new(img, w, h, RGB_IMAGE, _("background to calendar"), 100, NORMAL_MODE)
    img.add_layer(layer_bg, -1)
    pdb.gimp_image_lower_layer(img, layer_bg)

    layer_bg.fill(BACKGROUND_FILL)

    # color dialog: if there is a parasite get the 'default_color' & the rest (comments line)
    default_color = instance_paras.old_dic.get('s3')
    if len(default_color) != 0: default_color = default_color[1]
    # get color of 'bg_weekdays', 'color_weekdays', 'color_festive' and 'color_number'
    gimp.displays_flush()
    colo_list = colorsBox(default_color) # function define above for second stage dialog
    if len(colo_list) != 4:
        color_weekdays = color_festive = color_number = (0, 0, 0)
        bg_weekdays = (255, 255, 255)
        # no choices -> no parasite data
    else:
        bg_weekdays, color_weekdays, color_festive, color_number = colo_list[0], colo_list[1], colo_list[2], colo_list[3]
        # replace & store the chosen colors in the persistent parasite later at the end
        instance_paras.new_dic['s3'] = [(), colo_list]

    #4 FONT SIZES FOR CALENDAR BOX ========================================

    ##### variables defines in this section
    # calculate the size of the font for weekday by finding the cell size.
    heightCal = widthCal * cal_size_h   # choice of the user for the height of the calendar box
    sep_w = widthCal/7  # distance between the column of the dates table
    sep_h = heightCal/5.5 # vertical distance, 5 weeks and 1 header (½ height of week), temporary
    # size_font_weekdays : size of the font for days of the week, found in relation to the cell size
    # height_layer_weekdays : height of the banner behind the days of the week
    # size_font_number : size of the font for date of the month
    #####################################

    pdb.gimp_progress_set_text(_("Calendar with image: sections 4-6 out of 12"))    
    pdb.gimp_progress_pulse()
    # adjust the largest tag to one cell
    size_font_weekdays = Size_font(sep_w, sep_h*(size_font_cal*0.77+0.1), weekdays[cont_max]+'  ', font_weekdays)
    text_width, text_height, text_ascent, text_descent = pdb.gimp_text_get_extents_fontname(weekdays[cont_max],\
                                                            int(size_font_weekdays), PIXELS, font_weekdays)
    height_layer_weekdays = text_height + text_ascent + text_descent

    sep_h = (heightCal-height_layer_weekdays)/5    # vertical separation between week row, not exact for grid
    size_font_number = size_font_cal*Size_font(sep_w, sep_h, '28  ', font_number)

    #5 WEEKDAY HEADER LAYER =============================================

    ##### variables defines in this section
    cont = 0        # reseetting the counter
    py = pcy + height_layer_weekdays/5     # y coodinate for the top of the text bounding box for header
    # centerX :
    # drawa_weekdays : 
    #####################################
    
    while(cont<7):
        centerX = int(dx+(sep_w - pdb.gimp_text_get_extents_fontname(weekdays[cont], int(size_font_weekdays),\
                                                                     PIXELS, font_weekdays)[0])/ 2.0)
        px = (cont%7)*sep_w + centerX   # center justify
        layer_txt = pdb.gimp_text_fontname(img, drawable, px, py, weekdays[cont], -1, True,\
                                           int(size_font_weekdays), PIXELS, font_weekdays)
        pdb.gimp_layer_set_lock_alpha(layer_txt, 255)
        if (first_weekday == 0 and cont==0) or (first_weekday == 1 and cont==6):
            pdb.gimp_context_set_background(color_festive)
        else:
            pdb.gimp_context_set_background(color_weekdays)

        pdb.gimp_edit_fill(layer_txt, BACKGROUND_FILL)
        pdb.gimp_floating_sel_to_layer(layer_txt)
        if cont > 0:
            pdb.gimp_image_merge_down(img, layer_txt, 0)
        cont = cont + 1

    drawa_weekdays = pdb.gimp_image_get_active_drawable(img)
    pdb.gimp_drawable_set_name( drawa_weekdays, _("weekday header"))

    #6 CREATE BACKGROUND WEEKDAYS (BANNER) ================================

    ##### variables defines in this section
    # layer_bg_weekdays : 
    day = datetime.date(year, month, 1)
    dayofmonth = datetime.date.weekday(day)     # integer variable, give the weekday of the first of that month
    #####################################
    
    layer_bg_weekdays = pdb.gimp_layer_new(img, widthCal , height_layer_weekdays, RGB_IMAGE,\
                                           _("weekday background"), 100, NORMAL_MODE)
    layer_bg_weekdays.add_alpha()
    img.add_layer(layer_bg_weekdays, -1)
    pdb.gimp_image_lower_layer(img, layer_bg_weekdays)

    pdb.gimp_context_set_background(bg_weekdays)
    layer_bg_weekdays.fill(BACKGROUND_FILL)
    
    # try to center horizontaly, with same width as the photo or the calendar grid
    layer_bg_weekdays.set_offsets(int(round(dx)), int(round(pcy)))

    # preparation for the grid layer
    if first_weekday == 0:  # False: week start on sunday, True: week start on monday
        dayofmonth = (dayofmonth + 1)%7

    nextmonth_year = year
    nextmonth_month = month+1
    if nextmonth_month > 12:
        nextmonth_year = year+1
        nextmonth_month = 1     

    ndaysofmonth = datetime.date(int(nextmonth_year), int(nextmonth_month), 1) - day  # number of days in that month, a datetime.date object
    daysInMonth = ndaysofmonth.days     # number of days in that month, an integer variable
    # nbDay6Row is the nb of day(s) at beginning  of 6e row of dates
    nbDay6Row = dayofmonth + daysInMonth - 35

    #7 GRID LAYER =========================================================
    
    ##### variables defines in this section
    hbswb = 3.5     # half brush stroke width for the external border in the grid layer below
    pgy = (pcy + height_layer_weekdays - 2*hbswb)   # position y of the top of the external border of grid layer
    # step_h : one cell vertical size from center to center grid line (half width of grid line = 1.5 px)
    # step_w : one cell horizontal size from center to center grid line
    #####################################
    
    pdb.gimp_progress_set_text(_("Calendar with image: sections 7-10 out of 12"))
    pdb.gimp_progress_pulse()
    # place a rectangle around the dates the same color as bg_weekdays
    layer_grid = pdb.gimp_layer_new(img, w, h, RGB_IMAGE, _("grid"), 100, NORMAL_MODE)
    layer_grid.add_alpha()
    img.add_layer(layer_grid, -1)   # second param. give position on layer stack
    pdb.gimp_image_lower_layer(img, layer_grid)
    pdb.gimp_edit_clear(layer_grid)

    pdb.gimp_context_swap_colors()
    # 2*hbswb is the width of the brush
    pdb.gimp_rect_select(img, dx+3, pgy+hbswb, 7*sep_w-2*hbswb, 5*sep_h+2*hbswb, 2, False, 0)
    pdb.gimp_context_set_brush('Circle (07)')
    bckg_grid_drawa = pdb.gimp_image_get_active_drawable(img)
    pdb.gimp_edit_stroke(bckg_grid_drawa)

    # Traces lines  (don't work? Solution is a layer the size of 'img')
    pdb.gimp_context_set_brush('Circle (03)')
    # divide space inside border of box, the inside 3 px of the above border are in this space.
    x1 = dx + 2*hbswb-1.5; x2 = x1 + widthCal - 4*hbswb + 3.0 # the 1.5 is there for the grid line width
    y1 = pcy + height_layer_weekdays - 1.5; y2 = 5*sep_h + y1 + 1.5
    step_w = (x2-x1)/7.0; step_h = (y2-y1)/5.0
    for i in range(1, 7):
        segment = [x1+i*(step_w), y1, x1+i*(step_w), y2]        # vertical lines
        pdb.gimp_pencil(bckg_grid_drawa, 4, segment)
        if i < 5:
            segment = [x1, y1+i*(step_h), x2, y1+i*(step_h)]    # horizontal lines
            pdb.gimp_pencil(bckg_grid_drawa, 4, segment)
    pdb.gimp_context_swap_colors()
    pdb.gimp_selection_none(img)

    # erase the empty part of the grid at the beginning and at the end of the month
    pdb.gimp_context_set_brush('Circle (05)')
    cont = 1
    while cont < dayofmonth:    # at the beginning
        segment = [x1+cont*(step_w), y1+3.5, x1+cont*(step_w), y1+step_h-4.0]
        pdb.gimp_eraser_default(bckg_grid_drawa, 4, segment)
        cont += 1
    cont = 6
    while cont > 7+nbDay6Row:   # at the end
        segment = [x1+cont*(step_w), y2-2.5, x1+cont*(step_w), y2-step_h+4.0]
        pdb.gimp_eraser_default(bckg_grid_drawa, 4, segment)
        cont -= 1

    #8 MONTH'S NAME LAYER =================================================

    ##### variables defines in this section
    size_font_month = rel_size_my*heightPho     # for the options: space_month < 3
    # layer_month : ID of the editable month text layer
    ofY = 0     # for added space at the top of the page
    #####################################
    
    if space_month > 2: # for option : add in calendar (with possibility of same font size for each month).
        cont_max = max_text_width(months, font_month)
        if space_month == 3:
            size_font_month = Size_font(2*step_w, step_h, months[cont_max]+'  '+str(year), font_month)
        else:
            size_font_month = Size_font(2*step_w, step_h, months[cont_max], font_month)
            
    layer_month = pdb.gimp_text_fontname(img, drawable, 0, 0, months[month], -1, True,
                                         int(size_font_month),  PIXELS, font_month)
    pdb.gimp_layer_set_lock_alpha(layer_month, 255)
    pdb.gimp_floating_sel_to_layer(layer_month)
    pdb.gimp_layer_set_opacity(layer_month, 40)

    # resize the page now that we have all the parts
    h = pcy + height_layer_weekdays +  5*sep_h + 2*hbswb + 1.7*layer_month.height
    x = w/15
    y = 0.4*layer_month.height
    if space_month == 0: # add bottom
        y = h - 1.4*layer_month.height
    if space_month == 2: # don't add, place on image
        h = pcy + height_layer_weekdays +  5*sep_h + 2*hbswb
    if space_month == 1: # add top
        ofY = 1.7*layer_month.height
    if space_month > 2: # don't add, place in calendar (empty cells)
        x = 0
        h = pcy + height_layer_weekdays +  5*sep_h + 2*hbswb
        y = h - 1.5*layer_month.height - 2*hbswb
        pdb.gimp_layer_set_opacity(layer_month, 100)

    pdb.gimp_image_resize(img, w, h, 0, ofY)
    
    layer_month.set_offsets(int(round(x)), int(round(y)))

    #9 YEAR LAYER =========================================================

    ##### variables defines in this section
    layer_year = pdb.gimp_text_fontname(img, drawable, 0, 0, str(int(year)), -1, True,
                        int(size_font_month),  PIXELS, font_month)  # ID of the editable year text layer
    # x et y are coord. for offsetting the year text in the layer
    #####################################

    pdb.gimp_layer_set_lock_alpha(layer_year, 255)
    pdb.gimp_floating_sel_to_layer(layer_year)
    pdb.gimp_layer_set_opacity(layer_year, 40)

    x = w - layer_year.width - (w/15)
    y = h - 1.4*layer_year.height
    if space_month == 1 or space_month == 2 or space_month == 4:
        y = 0.4*layer_month.height
    # place month & year in the empty grid if it's the user choice (the min. is 2 cells)
    if space_month == 3:
        x = pdb.gimp_text_get_extents_fontname(months[month]+'  ', int(size_font_month), PIXELS, font_month)[0]
        y = h - 1.5*layer_year.height - 2*hbswb
        pdb.gimp_layer_set_opacity(layer_year, 100)

    layer_year.set_offsets(int(round(x)), int(round(y)))

    if space_month == 3:        # complete sections 8 
        center_my = dx + step_w - pdb.gimp_text_get_extents_fontname(months[month]+'  '+str(year),
                                        int(size_font_month), PIXELS, font_month)[0]/2
        pdb.gimp_layer_set_linked(layer_year, True)        
        pdb.gimp_layer_set_linked(layer_month, True)
        if dayofmonth > -nbDay6Row:
            pdb.gimp_layer_translate(layer_month, center_my+2, -4*step_h-1)
        else:
            pdb.gimp_layer_translate(layer_year, 5*step_w+center_my, 1)
        pdb.gimp_layer_set_linked(layer_year, False)        
        pdb.gimp_layer_set_linked(layer_month, False)
    if space_month == 4:
        center_my = dx + step_w - pdb.gimp_text_get_extents_fontname(months[month],
                                    int(size_font_month), PIXELS, font_month)[0]/2
        if dayofmonth > -nbDay6Row:
            pdb.gimp_layer_translate(layer_month, center_my+2, -4*step_h-1)
        else:
            pdb.gimp_layer_translate(layer_month, 5*step_w+center_my, 1)

    #10 TABLE OF DATES ====================================================

    ##### variables defines in this section
    nbCol = 0  # column counter to implement a 'split level' on row 5 to absorb row 6.
    # offset_centerX :
    # offset_Y :
    # dpx : delta on position x (px : position of date along x)
    dpy = 0     # delta on position y (py)
    size_font = size_font_number    #
    cont = 0    # reset counter of the day of the month, here
    cell_date_shift = dayofmonth # the variable 'dayofmonth' will be change in the table construction
    try: holiday_list = eval("["+holidays_om+"]") # list of date with the same color as Sundays
    except: holiday_list = []
    #####################################

    # to center horizontaly and verticaly the 'table of dates'
    text_width, text_height = pdb.gimp_text_get_extents_fontname('10', int(size_font_number), PIXELS,
                                                                 font_number)[0:2]
    height_date = text_height   # need it later
    offset_centerX = dx + 5.5 +(step_w - text_width)/ 2.0
    offset_Y = pcy + ofY + height_layer_weekdays - 1.5 + (step_h-text_height)/2.0 
    # right justification for single digit date
    dpxR = pdb.gimp_text_get_extents_fontname('3', int(size_font_number), PIXELS, font_number)[0]
    # for the placement of double dates in one cell
    if nbDay6Row > 0:
        text_width, text_height = pdb.gimp_text_get_extents_fontname('24 /11', int(0.7*size_font_number),
                                                                     PIXELS, font_number)[0:2]
        # width_dl_date = text_width   # need it later
        dpx_s = dx+5+(step_w - text_width)/ 2.0 - offset_centerX
        dpy_s = pcy + ofY + height_layer_weekdays + (step_h/2-text_height) - offset_Y
    
    while(cont<daysInMonth):
        px = (dayofmonth%7)*step_w + offset_centerX
        py = (dayofmonth/7)*step_h + offset_Y
        if cont+1 < 10:
            date = str(cont+1)
            dpx = dpxR
        else:
            date = str(cont+1)
            dpx = 0
            if nbDay6Row > 0 and nbCol < 2*nbDay6Row and int(dayofmonth/7) > 3:
                if int(dayofmonth/7) == 4 and nbCol < nbDay6Row:
                    size_font = int(0.7*size_font_number)
                    date = str(cont+1) + " /"
                    dpx = dpx_s; dpy = dpy_s
                    nbCol += 1
                elif int(dayofmonth/7) == 5 and nbCol < 2*nbDay6Row:
                    size_font = int(0.7*size_font_number)
                    dpx = dpx_s + size_font*1.3; dpy = dpy_s + text_height - step_h
                    date = "/ " + str(cont+1)
                    nbCol += 1
                else:
                    dpx = 0 ; dpy = 0; size_font = size_font_number
        layer_txt = pdb.gimp_text_fontname(img, drawable, px+dpx, py+dpy, date, -1, True,
                                           int(size_font), PIXELS, font_number)
        pdb.gimp_layer_set_lock_alpha(layer_txt, 255)
        # Sundays and selected holidays have the 'color_festive', the rest have the 'color_number'
        if (first_weekday == 1 and dayofmonth%7==6) or (first_weekday == 0 and dayofmonth%7==0) \
           or cont+1 in holiday_list:
            pdb.gimp_context_set_background(color_festive)
        else:
            pdb.gimp_context_set_background(color_number)

        pdb.gimp_edit_fill(layer_txt, BACKGROUND_FILL)
        pdb.gimp_floating_sel_to_layer(layer_txt)
        if cont > 0: pdb.gimp_image_merge_down(img, layer_txt, 0)

        cont += 1
        dayofmonth += 1
    pdb.gimp_drawable_set_name(pdb.gimp_image_get_active_drawable(img), _("table of dates"))

    #11a HELPING WITH DATE COMMENTS (preparation) =========================

    ##### variables define in this section
    # row_nb : row number of the cell starting at 0
    # col_nb : column number of the cell starting at 1
    file_name = ""
    double_date_comm = []   # list of double date comments
    comm_w = step_w - 5.0 # max width of the text bounding box for comment
    comm_h = [] ; comm_h.append(0) # max height of the text bounding box for comment, index nb of date in cell
    comm_list = [] # comments as it appears in the following 'date comments dialog', for existence and auto sizing
    comm_line = [] # template: ['comment line', float(cx), float(cy), index=1 or 2, day], give what
    #and where to display, index is for one date cell or two
    # return_list : user entry of the comments, a function return tuple (tasks, font, color)
    mess = ""       # one message that depends on different situations
    #####################################

    pdb.gimp_progress_set_text(_("Calendar with image: sections 11-12 out of 12"))

    # space available in cell for comment (if we do not move the dates)
    comm_h.append((step_h -3.0 - height_date)/2.0 -2.0) # max height of the bounding box normal cell
    # max height of the bounding box for double day cell but max width is the same
    comm_h.append((step_h -3.0)/4.0 -2.0)
    
    max_size_comm = Size_line(comm_w, comm_h[1], "Ben's birthday", default_font)[0]
    if max_size_comm >= 6.0 and comment_op > 0 : # if it is worthwhile and OK give the option
        # adapt the messages to the situation
        char_nr = int(round(max_size_comm * 1.4))  # suppose a proportional rule: 'max_font_size * number_char_2obtain / acceptable_font_size' 
        if char_nr > 35: char_nr = 35
        mess = _("""    This is an interactive window to enter your comment's control line.
Short comment, the estimation is less than %d characters per text line for this calendar, base on the font 'Sans', to \
include in the calendar cells.\n""")% char_nr
        check_s11 = (lang, month, year)   # the new parameters for (checking) the 'parasite'
        
        # sources of comments: the parasite ...
        comments_pre = instance_paras.old_dic.get('s11')
        if len(comments_pre) != 0 and check_s11 == comments_pre[0]:
            comments_text = comments_pre[1]
            mess += _("The initial comment(s) in the 'List of control lines' frame, came from an attachment to the image.")
        # try reading comments from standard file done in section #1 to produce 'comment_file',          
        else:
            comments_text = comment_file[0]
            file_name = comment_file[2]
            if comments_text != "":
                mess += _("""The initial comment(s) in the 'List of control lines' frame, came from your yearly comment file.""")
            # 'holiday_list' begs for comment?
            elif holiday_list:
                for holiday in holiday_list:
                    comments_text += "['"+str(holiday)+_("', 'holiday']\n")
                mess += _("""The initial and incomplete comment(s) in the 'List of control lines' frame, came from your holiday list!""")

        # adapt the font messages to the choice made at the beginning dialog
        theRest_list = _("Possible font size for 'Sans' is about 6<font_size<%s PX, for a 14 characters line. ")%str(max_size_comm)
        if comment_op==2: theRest_list = [theRest_list, _("You are in MANUAL font size.")]
        else:  theRest_list = [theRest_list, _("You are in AUTO font size.")]
        theRest_list.append((comm_w, comm_h[1])) # width and height tuple (index 2 of 'theRest_list') of the bounding text box, for normal cell
        theRest_list.append(lang) # index 3, for text rendering language
        gimp.displays_flush()

        # user entry of the comments, return a tuple ( comment's tasks, font, color)
        return_list = commentBox(mess, char_nr, comments_text, cell_date_shift, daysInMonth, theRest_list)
        # permits bottom line and top line comment text for single date cell
        pdb.gimp_progress_pulse()
        if return_list != '': # The 'else:' is after line ~1126
            # parse the return values of commentBox()
            tasks_line = return_list[0].splitlines()
            font_comm_words = return_list[1].split()    # split on space to permit next 2 lines
            font_comm_name = ' '.join(font_comm_words[:len(font_comm_words)-1])
            size_comm_adj = int(font_comm_words[len(font_comm_words)-1])    # size of the font if needed
            # set color of the text for the comments
            pdb.gimp_context_set_foreground(return_list[2])

            mess = ""
            # construct the 'comm_line' list for displaying the comments
            for i in range(len(tasks_line)):
                task = eval(tasks_line[i])
                day_pt = int(task[0])
                row_nb = int((day_pt + cell_date_shift - 0.1)/7)
                col_nb = int(day_pt + cell_date_shift - row_nb*7)
                cx = x1 + step_w*(col_nb-1.0) + 1.5 
                # find rhe top left coord of the text bounding box for top, for both need both?
                cyt = y1 + ofY + step_h*(row_nb) + 2.5
                # find rhe top left coord of the text bounding box to be put at bottom of a normal date
                cyb = y1 + ofY + step_h*(row_nb+1) - comm_h[1] - 1.5
                comm_list.extend(task[1:])
                # complications with double date per cell for 'nbDay6Row > 0', put in an empty space at the top of the grid
                if nbDay6Row >= col_nb and row_nb >= 4:
                    # remove line break if there and -2.5 in cx
                    cx = x1 + step_w*(cell_date_shift-2.5) + 1.5 
                    cyt = y1 + ofY + comm_h[2]/10.0 +1.5; cyb = -1
                    #replace the break line by a space to make one comment (if 'place' at beginning or end, discard)
                    one_line = task[1]
                    if len(task) > 2 and one_line != '': one_line += ' ' + task[2]
                    if len(task) > 2 and one_line == '': one_line = task[2]
                    comment_t = str(day_pt) + ') '+ one_line    # text to put at the top of the grid
                    double_date_comm.append(day_pt)     # list of day for comment in double date cell
                    if row_nb > 4:
                        row_nb = 4  # this is the lower day
                    comm_line.append([comment_t, cx, cyt, 2, day_pt])
                else:   # normal cell
                    comment_t = task[1] # at the top
                    if len(task) > 2:
                        if len(comment_t) > 0:
                            comm_line.append([comment_t, cx, cyt, 1, day_pt])
                        comment_b = task[2] # at the bottom
                        if len(comment_b) > 0:
                            comm_line.append([comment_b, cx, cyb, 1, day_pt])
                    else:
                        comm_line.append([comment_t, cx, cyt, 1, day_pt])

            instance_paras.new_dic['s11'] = [check_s11, return_list[0]+"\n"]

    #11b HELPING WITH DATE COMMENTS (display)==============================

            double_date_comm.sort()
            # find the largest commentfirst_weekday line and adjust the font size to the space
            index_comm = max_text_width(comm_list, font_comm_name)
            if int(comment_op) != 2:   #  if comment_op = '1' use auto for font size
                size_comm_adj, constraint = Size_line(comm_w, comm_h[1], comm_list[index_comm], font_comm_name)
                
                mess_comm = _("was constrained in auto by text ")+constraint
                if size_comm_adj < 6: mess = _("WARNING IN COMMENTS:\n The comment line '%s' is too large for the \
space available!\n The 'date comments' layer will be skipped.\n A parasite \
called 'calendar-input' has been attached to the image.") %comm_list[index_comm]
            else:
                mess_comm = _("was chosen by you")
                if size_comm_adj > 1.3*max_size_comm : mess = _("WARNING IN COMMENTS: \nThe font size chosen by you is \
too big (%s PX). This added comment section of the plug-in will be \
skipped.\n A parasite called 'calendar-input' has been attached to the image.") %str(size_comm_adj)

            if mess:    
                # warning mess for font size
                gimp.message(mess)
                mess = ""
            else :      
                # no warning message, do it
                # for the info comments messages formatting
                if file_name != "":
                    if comment_file[1] < 5:
                        error_list = [_('path'), _('existence'), _('content'), _('suitability') ,  _('no_tags:')] 
                        mess = _("Problem with the standard yearly comments file at: '%r'. File %s error.\n")%(file_name, error_list[comment_file[1]])
                    else: mess = _("Found an OK standard yearly comments file at : '%r'.\n")%file_name
                mess_comm_dict = {'file': mess, 'widest line': comm_list[index_comm], 'font name': font_comm_name,
                                  'font size': str(size_comm_adj), 'font choice': mess_comm}
                gimp.message(_("INFO FOR COMMENTS:  \n %(file)s Your longest comment line was '%(widest \
line)s' and the font used was '%(font name)s'.\n The font size of %(font size)s PX, \
%(font choice)s.")%mess_comm_dict)
                # display the date comment in a new layer
                done_col = [];  cont = 0
                for item in comm_line:
                    comment_pt = item[0]
                    text_width, text_height = pdb.gimp_text_get_extents_fontname(comment_pt, size_comm_adj, 
                                                                                 PIXELS, font_comm_name)[0:2]
                    del_x = (comm_w - text_width)/2.0
                    del_y = (comm_h[item[3]] - text_height)/2.0
                    cy = item[2]
                    if item[3] == 2: # double date cell
                        comment_order = double_date_comm.index(item[4])+1    # the comment order
                        offset_scomm = (step_h - 3.0 - len(double_date_comm)*comm_h[2])/2.0
                        cy += (comment_order-1.0)*comm_h[2] + offset_scomm # to terminate the evaluation of cy
                        del_x = 0   # special case for double date, it is left justified
                        # remember the previous cell(s) column: row always 4 and col. is 1 or 2 (done_col = [1,2])
                        if len(done_col) != 2:
                            col_nb = int(item[4] + cell_date_shift)%7
                            if col_nb not in done_col:
                                done_col.append(col_nb)
                    layer_comment = pdb.gimp_text_fontname(img, drawable, item[1]+del_x, cy+del_y,\
                                                        comment_pt, -1, True, size_comm_adj,  PIXELS, font_comm_name)
                    pdb.gimp_layer_set_lock_alpha(layer_comment, 255)   # for coloring the non transparent parts only
                    pdb.gimp_edit_fill(layer_comment, FOREGROUND_FILL)
                    pdb.gimp_floating_sel_to_layer(layer_comment)
                    if cont > 0: layer_comment = pdb.gimp_image_merge_down(img, layer_comment, 0)
                    cont += 1
                pdb.gimp_drawable_set_name(pdb.gimp_image_get_active_drawable(img), _("date comments"))
                    
                for col in done_col:
                    # add a thin border, same color as comment around the double cell affected to draw attention
                    bx = x1 + step_w*(col-1.0) + 0.5
                    by = y1 + ofY + step_h*4.0 + 1.0
                    pdb.gimp_layer_resize_to_image_size(layer_comment)
                    pdb.gimp_rect_select(img, bx, by, step_w-1.5, step_h-0.5, 2, False, 0) # faster with line tracing?
                    pdb.gimp_context_set_brush('Circle (01)')
                    comment_drawa = pdb.gimp_image_get_active_drawable(img)
                    pdb.gimp_edit_stroke(comment_drawa)  # put line inside selection? 
                    pdb.gimp_selection_none(img)
                
        else: gimp.message(_("WARNING IN COMMENTS:\n No comment available or found!\n The 'date comments' \
layer will be skipped."))
    elif comment_op > 0: gimp.message(_("WARNING IN COMMENTS:\n Not enough available space for date comments!\n\
That option has been skipped."))
        
    # store the actual input data of the secondary dialogs in a persistent parasite
##    img.parasite_detach('calendar-input')   #if cale_parasit:
##    img.attach_new_parasite('calendar-input', 1, new_para_data)


    #12 PRINTING BORDERS & WRAPPING UP ====================================

    ##### variables define in this section
    asp_rat = float(w/h)    # aspect ratio (width/height) without printing borders
    deltaw = 0              # the width of the added border to one side
    print_pap = [asp_rat, float(4.0/6.0), float(8.0/10.0), float(8.5/11), float(8.5/14), float(11.0/17)]
    # 'mess_values_dict': the following dictionary is for helping in translation of the messages by 'mapping'
    paper_width = [4.0, 8.0, 8.5, 8.5, 11.0]
    mess_print = ''     # string for gimp.message() in this section
    #####################################

    pdb.gimp_progress_pulse()
    # add an extension like '_cale09_01' to the name for documentation and protection of the initial file
    #     (less likely to overwrite);  corrected the bug with extension similar to '.xcf.bz2' which should be use
    ye_mo_s = str(year)[2:4] + "_%02d" % month
    add_s = "_cale" + ye_mo_s
    im_name = im_pre_name = img.name
    im_file = img.filename
    if im_name != None and im_file != None:
        dotp = im_name.find('.')   # return -1 if a dot is not found
        if dotp == -1:
            dotp = len(im_name)
        ext = im_name[dotp:]        # if no dot: ext = ''
         # check if not already there
        if im_name[dotp-10:dotp] != add_s:
            if im_name[dotp-10:dotp-5] == "_cale":
                im_name = im_name[0:dotp-5]+ye_mo_s+ext
            else :
                im_name = im_name[0:dotp]+add_s+ext
            img.filename = im_file.replace(im_pre_name, im_name)

    # for the messages formatting
    mess_values_dict = {'aspect_ratio_before': asp_rat, 'aspect_ratio_after': print_pap[print_ratio], 'month_id': "'"+\
                ye_mo_s, 'suggest_rel_height_cal': 0., 'paper_width': str(paper_width[print_ratio-1]), 'new_image_width': ""}
    # make the added border the same color as the background
    pdb.gimp_image_set_active_layer(img, layer_bg)
    gimp.context_pop()
    pdb.gimp_layer_resize_to_image_size(layer_bg)
    
    if print_ratio != 0:
        gimp.message( _("INFO FOR PRINTING:\n The aspect ratio (width/height) of the page before added borders is : \
%(aspect_ratio_before)1.4f \n After, for the month %(month_id)s, it will be : %(aspect_ratio_after)1.4f")\
% mess_values_dict)
        if asp_rat > print_pap[print_ratio]:
            # add to the top or bottom (TODO? split the border 1/2 on separation photo-calendar)
            deltah = w /  print_pap[print_ratio] - h
            h += deltah
            if space_month == 1:
                img.resize(int(w), int(h), 0, int(deltah))                
            else:
                img.resize(int(w), int(h), 0, 0)
            cal_size_h_new = (heightCal + deltah) /  widthCal
            PhoCal_sep_new = (sep + deltah) / (heightPho*2.0)
            if cal_size_h_new <= 0.7 and deltah > 4: # and (space_month==0 or space_month==1):
                mess_values_dict.update({'suggest_rel_height_cal': cal_size_h_new})
                mess_values_dict.update({'suggest_rel_separation': PhoCal_sep_new})
                gimp.message( _("SUGGESTION FOR PRINTING:\n If you whish to avoid the added border at the top or \
bottom, start over with the relative height of the calendar box = %(suggest_rel_height_cal)1.3f\
 .\n If you whish to centre vertically the block between the separation photo-calendar and exterior \
border you will need a relative separation of %(suggest_rel_separation)1.3f") % mess_values_dict)
        if asp_rat < print_pap[print_ratio]:
            deltaw = (h *  print_pap[print_ratio] - w)/2    # add to each side
            w += 2*deltaw
            img.resize(int(w), int(h), int(deltaw), 0)
            cal_size_h_new = (heightCal + (w-2*deltaw)/print_pap[print_ratio] - h) /  widthCal
            if cal_size_h_new > 0.3 and deltaw > 3:
                mess_values_dict.update({'suggest_rel_height_cal': cal_size_h_new})
                gimp.message( _("SUGGESTION FOR PRINTING:\n If you wish to avoid the added border at the side, start\
 over with the relative height of the calendar box = %(suggest_rel_height_cal)1.3f ") % mess_values_dict)
        
        pdb.gimp_layer_resize_to_image_size(layer_bg)
        # verify if the image size is suitable for printing
        if (widthPho+2*deltaw)/200 < paper_width[print_ratio-1]:
            mess_values_dict.update({'new_image_width': str(int(paper_width[print_ratio-1]*200 - 2*deltaw))})
            mess_print = _("WARNING FOR PRINTING:\n The size of the photo seems too small for quality printing on \
%(paper_width)s wide paper.\nSuggestion : up-size or start over with a photo width of more than %(new_image_width)s\
 pixels, for more than 200 dpi.") % mess_values_dict
        gimp.message(mess_print or _("INFO:\n Successful completion of the plug-in. Before printing you need to decide \
for yourself if you would re-sized the image, maybe sharpening or/and adjust printing resolution. Changing \
the unit, at the bottom left of the display, to inch should help.\n A parasite called 'calendar-input' has been attached to the image.\
 \n Bye-bye now.")) 
    else:
        gimp.message( _("INFO:\n Successful completion of the plug-in.\n The aspect ratio (width/height) for the \
 month %(month_id)s is : %(aspect_ratio_before)1.4f\n A parasite called 'calendar-input' has been attached to the image.\
 \n  See you later.") % mess_values_dict)

     
    instance_paras.save_all()
    gimp.displays_flush()
    # agrupamos UNDO
    img.undo_group_end()

# función principal
if __name__ == '__main__':

    # llamada a función register
    register(
        "calendar",
        _("The background colour in GIMP will be the background of the calendar page. There are four main \
elements: page background, starting image, calendar box and month-year info. From: '%s'")%from_file,  # calendar-en.py
        _("Make a standard image plus month calendar. Produce eight additional layers: one for the \
background, five for the calendar box (one for comments is optional) and two for month-year."),
        "Javi Pacheco and R. Brizard",
        "Javi Pacheco",
        "2007",
        _("Calendar with image..."),
        "*",
        [ (PF_IMAGE, "img", "IMAGE:", 0),
          (PF_DRAWABLE, "drawable", "DRAWABLE:", 0),  # need those if you use the menu path at the end of 'register()'
          (PF_SPINNER, "PhoCal_sep", _("Separation, photo & calendar box, relative to the photo height: "),\
                 0.050, (0.0, 0.500, 0.001)),
          (PF_SPINNER, "cal_size_w", _("Width of the calendar box, relative to the photo width: "), 1.0, (0.20, 1.80, 0.01)),
          (PF_SPINNER, "cal_size_h", _("Height of the calendar box, relative to the box width: "), 0.50, (0.30, 0.70, 0.01)),
          (PF_OPTION, "first_weekday", _("Select the first weekday: "), 0, [_("Sunday"), _("Monday")]), # , _("At first of the month")
          (PF_OPTION, "weekday_header", _("Choice of weekday tags header:\n  (Default: first 3 char. of name) "), 1, [_("Default"),
                _("Default, ASCII in CAP."), _("Customize...")]),
          (PF_FONT, "font_weekdays", _("Font for weekday's tags: "), 'Sans Bold'),
          (PF_FONT, "font_number", _("Font for date:"), default_font),
          (PF_SPINNER, "size_font_cal", _("Relative size of the font for date, to the automatic one:"),\
                 0.8, (0.6, 1.3, 0.1)),
          (PF_SPINNER, "month", _("Month desired: "), 1, (1, 12, 1)),
          (PF_SPINNER, "year", _("Which year? "), 2011, (1900, 2999, 1)),
          (PF_STRING, "holidays_om", _("Date(s) of holiday to have the festive colour:\n Example: 8, 21"), ""),
          (PF_OPTION, "lang", _("Language on the calendar page: "), 0,\
                ["English", "Français", "Deutsch", "Italiano", "Castellano", "Русский", "Português"]),
          (PF_FONT, "font_month", _("Font for month & year text: "), 'Lucida Sans Bold'),
          (PF_OPTION, "space_month", _("Add space, on the page, for month & year:\n If not select 'Don't ...'"), 4,\
                [_("At the bottom"), _("At the top"), _("Don't, both on image"), _("Don't, both in calendar"), _("Don't, month in calendar")]),
          (PF_OPTION, "print_ratio", _("Target: printing on a common paper size (print later):"), 0, [_("Don't care"),\
                _("4x6 in"), _("8x10 in"), _("8.5x11 in"), _("8.5x14 in"), _("11x17 in")]),
          (PF_OPTION, "comment_op", _("Add date comments (if possible)?"), 0, [_("No"), _("Yes, auto font size..."), \
                _("Yes, manual font size...")])
        ],
        [],
        Calendar,
        domain=( "gimp20-python", locale_directory),
        menu=_("<Image>/Plugins-Python/Image/Additions"))     # end register()
    
    main()
