#!/usr/bin/env python
# -*- coding: utf-8 -*-

# GIMP plug-in 'centroid', version 0.1 (release 2012/10/02)
#
# Description:
#   places the light and shadow centroids on a layer above the photo,
#   put guides on the center and display results in Error console.

# Tested on GIMP-2.8
#
# Access to pixel from 'colorxhtml.py' by Manish Singh and Carol Spears
#
# Changes:
#   2012/10/05 corrected first bug ('stripe') after release
#   2012/10/12 added more robustness
#
# License: GPL v2
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# The GNU Public License is available at
# http://www.gnu.org/copyleft/gpl.datfile

import struct, os
import datetime, math
import gimp, gettext

try:
    from gimpfu import *
except ImportError:
    import sys
    print("Note: GIMP is needed, '%s' is a plug-in for it."%__file__)
    sys.exit(1)

# Internationalization 'i18n'
locale_directory = os.path.join(os.path.dirname(os.path.abspath(__file__)), \
                    'locale') # for users; administrator: gimp.locale_directory

gettext.install("centroid", locale_directory, unicode=True)

# to make it run for 2.6
version = gimp.version
start_minver = 6 # GIMP minor version for plug-in

def centroid(img, drawable):

    img.undo_group_start()
    gimp.context_push()

    # determine type of image: GRAY?
    pdb.gimp_selection_none(img)
    base_type = pdb.gimp_image_base_type(img)
    fname_orig = img.filename
    # name of the image
    if fname_orig:
        last_sep = fname_orig.rfind(os.sep)
        img_name = fname_orig[last_sep+1:]
    # if it's a non saved image
    else: img_name = _("Untitled")

    # make a new image?
    new_img = img.duplicate()
    drawable_new = new_img.merge_visible_layers(1)
    #pdb.gimp_layer_add_alpha(drawable_new)

    # If image is not type GRAY
    if base_type != 1:
        pdb.gimp_image_convert_grayscale(new_img)

    width = drawable_new.width
    height = drawable_new.height
    bpp = drawable_new.bpp
    centerx = width / 2.0
    centery = height / 2.0

    gimp.tile_cache_ntiles(width / gimp.tile_width() + 1)
    pr = drawable_new.get_pixel_rgn(0, 0, width, height, False, False)

    gimp.progress_init(_("Computing centroids"))

    # compute the shadow and light centroid for GRAY, remenber px values
    sum_momentl = [0.0, 0.0]    # [x, y] ending with 'l' for light & 's' is shadow
    sum_moments = [0.0, 0.0]
    sum_pixvaltl = 0.0
    sum_pixvalts = 0.0
    # save pixel values in a 'pix_lum' list of list []
    pix_lum = []
    for y in range(0, height):
        row = pr[0:width, y]
        coord_row = [0.5, y + 0.5]
        pix_lum_row = []

        sum_pixvall = 0.0
        sum_pixvals = 0.0
        sum_momentl_row = 0.0
        sum_moments_row = 0.0
        for pixel in RowIterator(row, bpp):
            #pixel is 4 tuple: (r, g, b, a); pixel for gray is a 2 tuple
            #if coord_row == [0.5, 0.5]: print("lenght of pixel is %d"%len(pixel))
            pixvall = pixel[0]
            pixvals = 255 - pixel[0]

            sum_pixvall += pixvall
            sum_pixvals += pixvals
            sum_momentl_row += coord_row[0] * pixvall
            sum_moments_row += coord_row[0] * pixvals

            pix_lum_row.append(pixel)
            coord_row[0] += 1.0
        # store the row of pixel value
        pix_lum.append(pix_lum_row)
        # accumulate the necessary sums
        sum_pixvaltl += sum_pixvall
        sum_pixvalts += sum_pixvals
        sum_momentl[0] += sum_momentl_row
        sum_momentl[1] += coord_row[1] * sum_pixvall
        sum_moments[0] += sum_moments_row
        sum_moments[1] += coord_row[1] * sum_pixvals

        gimp.progress_update(y / float(height))

    gimp.delete(drawable_new)

    # coordinates of the centroids
    msg_lbar = _("light centroid")
    msg_sbar = _("shadow centroid")
    msg_cen =  ""
    # possible problem, see if alpha channel has variable values?
    if len(pixel) == 2: var_bool = is_channel_variable(1, pix_lum, width, height)
    else: var_bool = False  #i.e.: gif file don't have an alpha channel
    if sum_pixvaltl != 0:
        coor_lbar = [sum_momentl[0]/sum_pixvaltl, sum_momentl[1]/sum_pixvaltl]
        # origin translation to image center
        deltal = [coor_lbar[0] - centerx, coor_lbar[1] - centery]
    else:
        msg_cen =  msg_lbar
    if sum_pixvalts != 0:
        coor_sbar = [sum_moments[0]/sum_pixvalts, sum_moments[1]/sum_pixvalts]
        deltas = [coor_sbar[0] - centerx, coor_sbar[1] - centery]
    else:
        msg_cen =  msg_sbar
    # compute the shadow_fraction: 'black_pc'
    if not msg_cen: black_pc = sum_pixvalts/(width * height * 255.0)
    
    # recompute centroids for a non-uniform alpha
    if sum_pixvaltl != 0 and sum_pixvalts != 0 and var_bool:
        coor_lbar, coor_sbar, black_pc = centroids_alpha(pix_lum, width, height)

    #####################################################################
    # stop the plug-in if one centroid is missing or seems unreliable
    if sum_pixvaltl == 0 or sum_pixvalts == 0 or coor_lbar == 'not OK':
        msg_value = _("Coordinates for the other centroid are x = %6.2f px , ")\
            +_("y = %6.2f px .\n%s")
        if var_bool: 
            response = _("Non-uniform alpha channel, so the above coordinates are unreliable.")
        # var_bool == True is 'Yes'
        else: 
            response = _("Uniform alpha channel or absent, so the above coordinates are reliable.")
        # moment of the one left for later adjustement (vignetting)?
        if sum_pixvaltl == 0 and sum_pixvalts == 0:
            msg = _("ERROR for %s: unable to compute both centroids, division by 0 .")\
                %img_name
        elif sum_pixvaltl == 0 and sum_pixvalts != 0:
            # find moment of shadow (act down) about center (sign important)
            msg_value =  msg_value%(coor_sbar[0], coor_sbar[1], response)
            msg = _("ERROR for %s: unable to compute %s")%(img_name, msg_cen)\
                +_(", division by 0 .\n%s\nPlug-in must exit!")%msg_value
        elif sum_pixvaltl != 0 and sum_pixvalts == 0:
            #mlac = -float(deltal[0]*sum_pixvaltl)
            #msg_value = _("Value for the center moment of the other is %6.5e px_dist*light.")%mlac
            msg_value =  msg_value%(coor_lbar[0], coor_lbar[1], response)
            msg = _("ERROR for %s: unable to compute %s")%(img_name, msg_cen)\
                +_(", division by 0 .\n%s\nPlug-in must exit!")%msg_value
        elif coor_lbar == 'not OK':
            msg = _("ERROR for %s: sorry this plug-in don't work")%img_name\
                +_(" for a nonuniform alpha channel like here.")
        gimp.message(msg)

        gimp.delete(new_img)
        gimp.context_pop()
        img.undo_group_end()
        return
    ######################################################################

    # make layer below half transparent, info on above layer more readable
    drawable.opacity = 50.0

    # compute an actual correction with the pixels values (black strip)
    stripe = 0
    if not var_bool:
        if abs(deltas[0]) > 1.6:
            total_mom = abs(deltas[0])*sum_pixvalts
            center = float(centerx-0.5)
            if deltas[0] < 0.0:
                # start with column at right
                column = width-1
                while total_mom > 0.0:
                    correct = (column-center)*sum([pix[column][0] for pix in \
                        pix_lum[:height]])
                    if correct >= 0.0:
                        total_mom -= correct
                        column -= 1
                    else:
                        # should not happen!
                        stripe = 'NoS'
                        break
                if stripe != 'NoS': stripe = width -1 - column
            if deltas[0] > 0.0:
                # start with column at left
                column = 0
                while total_mom > 0.0:
                    correct = (center-column)*sum([pix[column][0] for pix in \
                        pix_lum[:height]])
                    if correct >= 0.0:
                        total_mom -= correct
                        column += 1
                    else:
                        stripe = 'NoS'
                        break
                if stripe != 'NoS': stripe = column

    # choose colors FG = black, BG = white (red) for dot?
    pdb.gimp_context_set_default_colors()

    # put centroid dot on a layer above original
    layer = gimp.Layer(new_img, "shadow centroid", \
        width, height, GRAYA_IMAGE, 100, NORMAL_MODE)
    new_img.add_layer(layer, 0)
    diam = width / 75.0
    # black dot
    if version[1]  >  start_minver :
        pdb.gimp_image_select_ellipse(new_img, 0, coor_sbar[0]-diam/2.0, \
            coor_sbar[1]-diam/2.0, diam, diam)
    else:
        pdb.gimp_ellipse_select(new_img, coor_sbar[0]-diam/2.0, coor_sbar[1]\
            -diam/2.0, diam, diam, 0, TRUE, FALSE, 0)
    pdb.gimp_edit_bucket_fill(layer, 1, 0, 100, 0,  FALSE, 0, 0)
    pdb.gimp_selection_shrink(new_img, int(diam/6))
    pdb.gimp_edit_bucket_fill(layer, 0, 0, 100, 0,  FALSE, 0, 0)
    pdb.gimp_selection_none(new_img)
    # white dot
    if version[1]  >  start_minver :
        pdb.gimp_image_select_ellipse(new_img, 0, coor_lbar[0]-diam/2.0, \
            coor_lbar[1]-diam/2.0, diam, diam)
    else:
        pdb.gimp_ellipse_select(new_img, coor_lbar[0]-diam/2.0, coor_lbar[1]\
            -diam/2.0, diam, diam, 0, TRUE, FALSE, 0)
    pdb.gimp_edit_bucket_fill(layer, 0, 0, 100, 0,  FALSE, 0, 0)
    pdb.gimp_selection_shrink(new_img, int(diam/6))
    pdb.gimp_edit_bucket_fill(layer, 1, 0, 100, 0,  FALSE, 0, 0)
    pdb.gimp_selection_none(new_img)

    # label the centroid dots
    dx = coor_sbar[0] - coor_lbar[0]
    dy = coor_sbar[1] - coor_lbar[1]
    if abs(dy) > 1.6 * diam :
        # shadow centroid
        text_layer = pdb.gimp_text_fontname(new_img, new_img.active_drawable, \
         coor_sbar[0]+diam, coor_sbar[1]- 1.5*diam, msg_sbar+"\n  [%6.1f ; %6.1f]"\
         %(coor_sbar[0], coor_sbar[1]), 0, TRUE, float(diam+2.0), 0, 'Sans')

        #TODO: put a contrasting halo around text by stroking a text path
        #vectors = pdb.gimp_vectors_new_from_text_layer(image, layer)
        #make an halo_layer
        #pdb.gimp_edit_stroke_vectors(drawable, vectors)

        pdb.gimp_context_swap_colors()
        # light centroid
        text_layer = pdb.gimp_text_fontname(new_img, text_layer, coor_lbar[0]+diam,\
            coor_lbar[1]- 1.5*diam,  msg_lbar+"\n  [%6.1f ; %6.1f]"\
            %(coor_lbar[0], coor_lbar[1]), 0, TRUE, float(diam+2.0), 0, 'Sans')
        #TODO: put a contrasting halo around text here also
        pdb.gimp_floating_sel_anchor(text_layer)
    else: pdb.gimp_context_swap_colors()

    # add the stripe correction to the message and centroids layer
    if stripe != 'NoS':
        if var_bool: 
            correc = _("\nBlack stripe balance is not available for an a non-uniform alpha")
        elif abs(deltas[0]) < 1.6:
            if deltas[1] > 0.6:
                correc = _("\nIt's near shadow stable equilibrium.")
            elif abs(deltas[1]) < 0.6:
                correc = _("\nIt's near shadow indifferent equilibrium.")
            else:
                correc = _("\nIt's near shadow unstable equilibrium.")
        else:
            strip_txt = _(" %d px,\nshadow balance")%stripe
            correc = _("\nObtaining shadow equilibrium is equivalent")\
                +_(" to painting a black stripe of width %d px at edge.")%stripe
            if stripe-diam/3.0 > 0.0: width_strip = stripe-diam/3.0
            else: width_strip = 0.0
            if deltas[0] > 0:       # rect selection at the left, illustrate
                # new version: pdb.gimp_image_select_rectangle(image, operation,\
                #    x, y, width, height)
                if version[1]  >  start_minver :
                    pdb.gimp_image_select_rectangle(new_img, 0, 0.0, 0.0, stripe,\
                        height)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 40, 0, FALSE, 0, 0)
                    pdb.gimp_image_select_rectangle(new_img, 1, diam/6+0.5, \
                        diam/6+0.5, width_strip, height-diam/3.0)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 100, 0,  FALSE, 0, 0)
                    pdb.gimp_selection_none(new_img)
                    pdb.gimp_context_swap_colors()
                    text_layer = pdb.gimp_text_fontname(new_img, new_img.active_drawable,\
                        stripe+2, 20, '<- '+strip_txt, 0, TRUE, float(diam+2.0),\
                        0, 'Sans')
                else:
                    # old version: pdb.gimp_rect_select(image, x, y, width, height,\
                    #    operation, feather, feather_radius)
                    pdb.gimp_rect_select(new_img, 0.0, 0.0, stripe, height, 0,\
                        FALSE, 0.0)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 40, 0, FALSE, 0, 0)
                    pdb.gimp_rect_select(new_img, diam/6+0.5, diam/6+0.5, width_strip,\
                        height-diam/3.0, 1, FALSE, 0.0)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 100, 0,  FALSE, 0, 0)
                    pdb.gimp_selection_none(new_img)
                    pdb.gimp_context_swap_colors()
                    text_layer = pdb.gimp_text_fontname(new_img, new_img.active_drawable,\
                        stripe+2, 20, '<- '+strip_txt, 0, TRUE, float(diam+2.0),\
                        0, 'Sans')
            elif deltas[0] < 0:     # at the right
                if version[1]  >  start_minver :
                    pdb.gimp_image_select_rectangle(new_img, 0, width-stripe, 0.0,\
                        stripe, height)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 40, 0, FALSE, 0, 0)
                    pdb.gimp_image_select_rectangle(new_img, 1, width-stripe+diam/6+0.5,\
                        diam/6+0.5, width_strip, height-diam/3.0)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 100, 0,  FALSE, 0, 0)
                    pdb.gimp_selection_none(new_img)
                    pdb.gimp_context_swap_colors()
                    width_txt = pdb.gimp_text_get_extents_fontname(strip_txt+' ->',\
                        float(diam+2.0), 0, 'Sans')[0] + 2
                    text_layer = pdb.gimp_text_fontname(new_img, new_img.active_drawable,\
                        width-stripe-width_txt, 20, strip_txt+' ->', 0, TRUE,\
                        float(diam+2.0), 0, 'Sans')
                else:
                    pdb.gimp_rect_select(new_img, width-stripe, 0.0, stripe, height,\
                         stripe, 0, FALSE, 0.0)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 40, 0, FALSE, 0, 0)
                    pdb.gimp_rect_select(new_img,  width-stripe+diam/6+0.5, diam/6+0.5, \
                        width_strip, height-diam/3.0, 1, FALSE, 0.0)
                    pdb.gimp_edit_bucket_fill(layer, 1, 0, 100, 0,  FALSE, 0, 0)
                    pdb.gimp_selection_none(new_img)
                    pdb.gimp_context_swap_colors()
                    width_txt = pdb.gimp_text_get_extents_fontname(strip_txt+' ->',\
                        float(diam+2.0), 0, 'Sans')[0] + 2
                    text_layer = pdb.gimp_text_fontname(new_img, new_img.active_drawable,\
                        width-stripe-width_txt, 20, strip_txt+' ->', 0, TRUE,\
                        float(diam+2.0), 0, 'Sans')
            pdb.gimp_floating_sel_anchor(text_layer)
    else:
        correc = _("\nFound no solution for a balancing black strip!")

    # compute equation of the centroids line
    #if dx != 0.0:
        #slope = round(dy / dx, 3)
        #ordinate = coor_sbar[1] - slope * coor_sbar[0]
        ## seems to pass by center
    #else:   # vertical line: x = cte
        #slope = 'vertical'

    # collect the results
    name_parasit = img.parasite_find('centroids')
    centroids_info = _("INFO:\n At %s for image '%s'.\n Light centroid coordinates ")\
        %(datetime.datetime.strftime(datetime.datetime.now(), "%c"), img_name)\
        +_("are:\nx = %6.2f px , y = %6.2f px .\n")%(coor_lbar[0], coor_lbar[1])\
        +_(" Shadow centroid coordinates are:\nx = %6.2f px , y = %6.2f px ;\n")\
        %(coor_sbar[0], coor_sbar[1])+_("shadow_fraction = %5.2f .")%black_pc \
        +correc+_("\n\nPrevious record: \n")+str(name_parasit)
    gimp.message(centroids_info)
    #attach a parasite with date and above info or add to one.
    img.attach_new_parasite('centroids', 1, centroids_info)

    layer_copy = pdb.gimp_layer_new_from_drawable(layer, img)
    img.add_layer(layer_copy, 0)
    layer_copy.name = _("Centroids")
    gimp.delete(new_img)

    # and add guides
    if not var_bool:
        guide_v = img.add_vguide(int(centerx))
        guide_h = img.add_hguide(int(centery))
    # saving for batch one
    #fname_new = fname_orig[:fname_orig.rfind('.')]+'_centroid.xcf.bz2'
    #pdb.gimp_file_save(img, img.active_drawable, fname_new, fname_new)
    #pdb.gimp_xcf_save(0, img, drawable, fname_orig, fname_orig)

    gimp.displays_flush()
    gimp.context_pop()
    img.undo_group_end()


class RowIterator:
    def __init__(self, row, bpp):
        self.row = row
        self.bpp = bpp

        self.start = 0
        self.stop = bpp

        self.length = len(row)
        self.fmt = 'B' * bpp

    def __iter__(self):
        return iter(self.get_pixel, None)

    def get_pixel(self):
        if self.stop > self.length:
            return None

        pixel = struct.unpack(self.fmt, self.row[self.start:self.stop])
        self.start += self.bpp
        self.stop += self.bpp

        return pixel

def is_channel_variable(channel, pix_lum, width, height):
    """
     return true or false for the uniformity of value in that pixel channel
    """
    first_val = pix_lum[0][0][channel]
    for y in range(0, height):
        for x in range(0, width):
            if first_val != pix_lum[y][x][channel]: return True
    return False
    
def centroids_alpha(pix_lum, width, height):
    """
     recomputes the centroids in the case of a non-uniform alpha channel like
     some 'pgn' or GIMP image
    """

    # recomputes the shadow and light centroid
    sum_momentl = [0.0, 0.0]    # [x, y] ending with 'l' for light & 's' is shadow
    sum_moments = [0.0, 0.0]
    sum_pixvaltl = 0.0
    sum_pixvalts = 0.0
    for y in range(0, height):
        coord_row = [0.5, y + 0.5]
        sum_pixvall = 0.0
        sum_pixvals = 0.0
        sum_momentl_row = 0.0
        sum_moments_row = 0.0
        for x in range(0, width):
            # pix_lum[y][x][1]/255.0 transform the alpha values between 0 and 1.0
            opacity = pix_lum[y][x][1]/255.0
            pixvall = pix_lum[y][x][0]*opacity
            pixvals = 255*opacity - pixvall
            sum_pixvall += pixvall
            sum_pixvals += pixvals
            sum_momentl_row += coord_row[0] * pixvall
            sum_moments_row += coord_row[0] * pixvals
            coord_row[0] += 1.0
        sum_pixvaltl += sum_pixvall
        sum_pixvalts += sum_pixvals
        sum_momentl[0] += sum_momentl_row
        sum_momentl[1] += coord_row[1] * sum_pixvall
        sum_moments[0] += sum_moments_row
        sum_moments[1] += coord_row[1] * sum_pixvals

    msg_cen =  ''
    if sum_pixvaltl != 0:
        coor_lbar = [sum_momentl[0]/sum_pixvaltl, sum_momentl[1]/sum_pixvaltl]
    else:
        msg_cen =  'no'
    if sum_pixvalts != 0:
        coor_sbar = [sum_moments[0]/sum_pixvalts, sum_moments[1]/sum_pixvalts]
    else:
        msg_cen =  'no'
    if msg_cen: return ['not OK', 'not OK', 'not OK']
    black_pc = sum_pixvalts/(width * height * 255.0)
    return coor_lbar, coor_sbar, black_pc


################################################################################

register(
        "centroid",
        _("Aid in equilibrium composition, it gives the\nlight & shadow ")\
            +_("centroids of an image.\nFrom: ")+__file__,
        _("Converts to greyscale the image if not GRAY type, computes ")
            +_("and displays light & shadow centroids on the initial image."),
        "R. Brizard",
        "(c) R. Brizard",
        "2012",
        _("Centroids"),
        "*",
        [
            (PF_IMAGE, "img", "Input image", None),
            (PF_DRAWABLE, "drawable", "Input drawable", None)
        ],
        [],
        centroid,
        menu = "<Image>/Extensions/Plugins-Python/Composition",
        domain = ("centroid", locale_directory) )
main()
