#!/usr/bin/env python

# GIMP python plugin to interactively change brush size
# Copyright (C) 2012
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gimpfu import *
from gobject import timeout_add
import time

# windows api libraries (for cursor position)
from ctypes import*
user32 = windll.user32

# point structure definition
class POINT(Structure):
	_fields_ = [("x", c_ulong), ("y", c_ulong)]

# Gets mouse position (windows only)
def getMousePos():
	pt = POINT()
	windll.user32.GetCursorPos(byref(pt))
	return pt.x, pt.y

# Sets mouse position (windows only)
def setMousePos(x, y):
	windll.user32.SetCursorPos(x, y)

# clamp value to range
def clamp(v, minv, maxv):
	return max(min(maxv, v), minv)

SLEEP = 0.02 # how smoothly the brush resizes
TIMEOUT_VALUE = 500 # 10~ seconds
timeoutCounter = 0
startPointX = 0 # start point for resizing
startPointY = 0
startBrushSize = 0

userSpacing = 0 # stores the users spacing value so i can restore it later
START_KEY = 475 # script sets this value
END_KEY = 476 # script sets this to end current script
MAX_BRUSH_SIZE = 10000

# update procedure
def update():
	# check for timeout
	global timeoutCounter
	timeoutCounter += 1
	if timeoutCounter > TIMEOUT_VALUE:
		pdb.gimp_brushes_set_spacing(userSpacing) # set spacing back to user value
		return False
	
	# check for key to end script
	if int(pdb.gimp_brushes_get_spacing()) == END_KEY:
		pdb.gimp_brushes_set_spacing(userSpacing) # set spacing back to user value
		return False
	
	x, y = getMousePos()
	
	if x != startPointX:
		newBrushSize = clamp(pdb.gimp_context_get_brush_size() + (x - startPointX), 1, MAX_BRUSH_SIZE)
		pdb.gimp_context_set_brush_size(newBrushSize)
	
	# If cursor has moved
	if x != startPointX or y != startPointY:
		setMousePos(startPointX, startPointY) # set cursor back to start point
	
	return True

def dynamicBrushResize(img, drw):
	global userSpacing
	userSpacing = int(pdb.gimp_brushes_get_spacing())
	
	if userSpacing == START_KEY: # means script already running
		pdb.gimp_brushes_set_spacing(END_KEY) # key to end current running script
		return r # end this script too
	
	pdb.gimp_brushes_set_spacing(START_KEY)
	
	# store start point
	global startPointX, startPointY
	startPointX, startPointY = getMousePos()
	
	# store brush size
	global startBrushSize
	startBrushSize = pdb.gimp_context_get_brush_size()
	
	while(update()):
		time.sleep(SLEEP)

register(
		 "dynamicBrushResize",
		 "Tap the hotkey to start. Drag your mouse left to right to change brush size. Tap the hotkey again to end. Times out after 10~ seconds.",
		 "",
		 "",
		 "",
		 "2012",
		 "<Image>/Tools/Dynamic brush resize",
		 "*",
		 [],
		 [],
		 dynamicBrushResize)

main()
