/*
 * Copyright (C) 1999 Winston Chang
 *                    <winstonc@cs.wisc.edu>
 *                    <winston@stdout.org>
 * Copyright (C) 2010 Johannes Hanika
 *                    Michael Munzert
 *                    elsamuko
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "equalizer_eaw.h"

#define PLUG_IN_PROC    "elsamuko-eaw-sharpen"
#define PLUG_IN_BINARY  "elsamuko-eaw-sharpen"

#define N_(x) (x)
#define _(x) (x)

#define SCALE_WIDTH   120
#define ENTRY_WIDTH     5

/* Uncomment this line to get a rough estimate of how long the plug-in
 * takes to run.
 */

/*  #define TIMER  */

typedef struct
{
    gdouble  alpha;
    gint     maxband;
    gint     inband;
    //gint     threshold;
    gint     mode;
} EAWParams;

typedef struct
{
    gboolean  run;
} EAWInterface;

/* local function prototypes */
static inline float EAW_CLAMP( float x, float a, float b) {
    return x < a ? a : (x > b ? b : x);
};
static inline float EAW_MIN( float x, float y) {
    return x < y ? x : y;
};

static void      process (float *image,
                          int    width,
                          int    height,
                          float  alpha,
                          int    maxband,
                          int    inband,
                          int    mode);
static void      reset_channel (GtkWidget * w,
                                gpointer data);

static void      query (void);
static void      run   (const gchar      *name,
                        gint              nparams,
                        const GimpParam  *param,
                        gint             *nreturn_vals,
                        GimpParam       **return_vals);

static void      eaw_region      (GimpPixelRgn   *srcPTR,
                                  GimpPixelRgn   *dstPTR,
                                  const gint     bpp,
                                  const gint     x,
                                  const gint     y,
                                  const gint     width,
                                  const gint     height,
                                  const gboolean show_progress);

static void      eaw_sharpen      (GimpDrawable   *drawable);
static gboolean  eaw_dialog       (GimpDrawable   *drawable);
static void      preview_update   (GimpPreview    *preview);


/* create a few globals, set default values */
static EAWParams eaw_params =
{
    1.5, /* default alpha     */
    10,  /* default maxband   */
    5,   /* default inband    */
    //5,   /* default threshold */
    0    /* default mode      */
};

/* Setting PLUG_IN_INFO */
const GimpPlugInInfo PLUG_IN_INFO =
{
    NULL,  /* init_proc  */
    NULL,  /* quit_proc  */
    query, /* query_proc */
    run,   /* run_proc   */
};


MAIN ()

static void
query (void)
{
    static const GimpParamDef args[] =
    {
        { GIMP_PDB_INT32,    (gchar*)"run-mode",  (gchar*)"The run mode { RUN-INTERACTIVE (0), RUN-NONINTERACTIVE (1) }" },
        { GIMP_PDB_IMAGE,    (gchar*)"image",     (gchar*)"(unused)" },
        { GIMP_PDB_DRAWABLE, (gchar*)"drawable",  (gchar*)"Drawable to draw on" },
        { GIMP_PDB_FLOAT,    (gchar*)"alpha",     (gchar*)"Strength (0-4)" },
        { GIMP_PDB_INT32,    (gchar*)"maxband",   (gchar*)"Levels (1-20)" },
        { GIMP_PDB_INT32,    (gchar*)"inband",    (gchar*)"Output (1-20)" },
        //{ GIMP_PDB_INT32,    (gchar*)"threshold", (gchar*)"Threshold (0-100)" },
        { GIMP_PDB_INT32,    (gchar*)"mode",      (gchar*)"Mode, 0 Linear Amplification, 1 Local Contrast, 2 Output level" }
    };

    gimp_install_procedure (PLUG_IN_PROC,
                            N_("Edge avoiding wavelet sharpen."),
                            "Edge avoiding wavelet sharpen.",
                            "elsamuko <elsamuko@web.de>",
                            "elsamuko",
                            "2010",
                            N_("_EAW Sharpen..."),
                            "GRAY*, RGB*",
                            GIMP_PLUGIN,
                            G_N_ELEMENTS (args), 0,
                            args, NULL);

    gimp_plugin_menu_register (PLUG_IN_PROC, "<Image>/Filters/Enhance");
}

static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
    static GimpParam   values[1];
    GimpPDBStatusType  status = GIMP_PDB_SUCCESS;
    GimpDrawable      *drawable;
    GimpRunMode        run_mode;
#ifdef TIMER
    GTimer            *timer = g_timer_new ();
#endif

    run_mode = (GimpRunMode)param[0].data.d_int32;

    *return_vals  = values;
    *nreturn_vals = 1;

    values[0].type          = GIMP_PDB_STATUS;
    values[0].data.d_status = status;

    //INIT_I18N ();

    /*
     * Get drawable information...
     */
    drawable = gimp_drawable_get (param[2].data.d_drawable);

    switch (run_mode)
    {
    case GIMP_RUN_INTERACTIVE:
        gimp_get_data (PLUG_IN_PROC, &eaw_params);
        /* Reset default values show preview unmodified */

        /* initialize pixel regions and buffer */
        if (! eaw_dialog (drawable))
            return;

        break;

    case GIMP_RUN_NONINTERACTIVE:
        if (nparams != 7)
        {
            status = GIMP_PDB_CALLING_ERROR;
        }
        else
        {
            eaw_params.alpha     = param[3].data.d_float;
            eaw_params.maxband   = param[4].data.d_int32;
            eaw_params.inband    = param[5].data.d_int32;
            //eaw_params.threshold = param[6].data.d_int32;
            eaw_params.mode      = param[7].data.d_int32;

            /* make sure there are legal values */
            if ((eaw_params.alpha < 0.0)   || (eaw_params.alpha    > 4.0  ||
                 eaw_params.maxband < 0)   || (eaw_params.maxband   > 20  ||
                 eaw_params.inband < 0)    || (eaw_params.inband    > 20  ||
                 //eaw_params.threshold < 0) || (eaw_params.threshold > 100 ||
                 eaw_params.mode < 0)      || (eaw_params.mode      > 2   )){
                printf( "L%i: Wrong input \n", __LINE__);
                status = GIMP_PDB_CALLING_ERROR;
            }

        }
        break;

    case GIMP_RUN_WITH_LAST_VALS:
        gimp_get_data (PLUG_IN_PROC, &eaw_params);
        break;

    default:
        break;
    }

    if (status == GIMP_PDB_SUCCESS)
    {
        drawable = gimp_drawable_get (param[2].data.d_drawable);

        /* here we go */
        eaw_sharpen (drawable);

        gimp_displays_flush ();

        /* set data for next use of filter */
        if (run_mode == GIMP_RUN_INTERACTIVE)
            gimp_set_data (PLUG_IN_PROC, &eaw_params, sizeof (EAWParams));

        gimp_drawable_detach(drawable);
        values[0].data.d_status = status;
    }

#ifdef TIMER
    g_printerr ("%f seconds\n", g_timer_elapsed (timer, NULL));
    g_timer_destroy (timer);
#endif
}

static void
eaw_sharpen (GimpDrawable *drawable)
{
    GimpPixelRgn srcPR, destPR;
    gint         x1, y1, x2, y2;

    /* initialize pixel regions */
    gimp_pixel_rgn_init (&srcPR, drawable,
                         0, 0, drawable->width, drawable->height, FALSE, FALSE);
    gimp_pixel_rgn_init (&destPR, drawable,
                         0, 0, drawable->width, drawable->height, TRUE, TRUE);

    /* Get the input */
    gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);

    eaw_region (&srcPR, &destPR, drawable->bpp,
                x1, y1, x2-x1, y2-y1, TRUE);

    gimp_drawable_flush (drawable);
    gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
    gimp_drawable_update (drawable->drawable_id, x1, y1, x2-x1, y2-y1);
}

/* Copy the selected region to a float array, paste it into the EAW algorithm,
 * and copy it back to the selected region.
 */
static void
eaw_region (GimpPixelRgn  *srcPR,
            GimpPixelRgn  *destPR,
            const gint     bpp,
            const gint     x,
            const gint     y,
            const gint     width,
            const gint     height,
            const gboolean show_progress)
{
    printf( "L%i: EAW Sharpen begin\n", __LINE__);
    printf( "L%i: Channels: %i\n", __LINE__, bpp );
    printf( "L%i: x:%i, y:%i\n", __LINE__, x, y);
    printf( "L%i: w:%i, h:%i\n", __LINE__, width, height);

    guchar     *src;
    guchar     *dest;
    float      *image;
    int ret;
    gint error = FALSE;

    src   = g_new( guchar, bpp * width * height );
    dest  = g_new( guchar, bpp * width * height );
    image = g_new( float,  3   * width * height );

    gimp_pixel_rgn_get_rect( srcPR,
                             src,
                             x, y,
                             width, height );

    if (show_progress) gimp_progress_init (_("Sharpening..."));

    gint limit = height*width*bpp;
    printf("L%i: limit: %i\n", __LINE__, limit);

    if (bpp==1) {
        int k = 0;
        for (int i = 0; i < limit; i++ ) {
            image[k] = (float)src[i];
            k++;
            image[k] = (float)src[i];
            k++;
            image[k] = (float)src[i];
            k++;
        }
        printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, k);
    } else if (bpp==2) {
        int k = 0;
        for (int i = 0; i < limit; i++ ) {
            if (i%2 != 1) {
                image[k] = (float)src[i];
                k++;
                image[k] = (float)src[i];
                k++;
                image[k] = (float)src[i];
                k++;
            }
        }
        printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, k);
    } else if (bpp==3) {
        int i;
        for (i = 0; i < limit; i++ ) {
            image[i] = (float)src[i];
        }
        printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, i);
    } else if (bpp==4) {
        int k = 0;
        for (int i = 0; i < limit; i++ ) {
            if (i%4 != 3) {
                image[k] = (float)src[i];
                k++;
            }
        }
        printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, k);
    } else {
        printf("L%i: Error: Could not handle image. Exiting.\n", __LINE__);
        error = TRUE;
    }

    if (error == FALSE) {
        if (show_progress) gimp_progress_update (0.25);

        process (image, width, height,
                 eaw_params.alpha, eaw_params.maxband,
                 eaw_params.inband, eaw_params.mode);

        if (show_progress) gimp_progress_update (0.75);

        if (bpp==1) {
            int k = 0;
            for (int i = 0; i < limit; i++ ) {
                dest[i] = (int)EAW_CLAMP(image[k],0,255);
                k=k+3;
            }
            printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, k);
        } else if (bpp==2) {
            int k = 0;
            for (int i = 0; i < limit; i++ ) {
                if (i%2 != 1) {
                    dest[i] = (int)EAW_CLAMP(image[k],0,255);;
                    k=k+3;
                } else {
                    dest[i] = src[i];
                }
            }
            printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, k);
        } else if (bpp==3) {
            int i;
            for (i = 0; i < limit; i++ ) {
                // if (abs ((int)EAW_CLAMP(image[i],0,255)-src[i]) > eaw_params.threshold){
                dest[i] = (int)EAW_CLAMP(image[i],0,255);
                // }else{
                //     dest[i] = src[i];
                // }
            }
            printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, i);
        } else if (bpp==4) {
            int k = 0;
            for (int i = 0; i < limit; i++ ) {
                if (i%4 != 3) {
                    dest[i] = (int)EAW_CLAMP(image[k],0,255);
                    k++;
                } else {
                    dest[i] = src[i];
                }
            }
            printf("L%i: bpp: %i, k: %i\n", __LINE__, bpp, k);
        } else {
            printf("L%i: Error: Could not handle image. Exiting.\n", __LINE__, width, height);
            error = TRUE;
        }
    }

    if (error) {
        gimp_pixel_rgn_set_rect( destPR, src, x, y, width, height );
    } else { // this one should be default
        gimp_pixel_rgn_set_rect( destPR, dest, x, y, width, height );
    }

    if (show_progress) gimp_progress_update (1.0);

    //tidy up
    g_free (image);
    g_free (dest);
    g_free (src);
    printf( "L%i: EAW Sharpen finished\n\n", __LINE__);
}

static gboolean
eaw_dialog (GimpDrawable *drawable)
{
    GtkWidget *dialog;
    GtkWidget *main_vbox;
    GtkWidget *preview;
    GtkWidget *table;
    GtkObject *adj;
    gboolean   run;
    GtkWidget *frame;
    GtkWidget *hbox;
    GtkWidget *button, *button2, *button3;

    gimp_ui_init (PLUG_IN_BINARY, TRUE);

    dialog = gimp_dialog_new (_("EAW Sharpen"), PLUG_IN_BINARY,
                              NULL, (GtkDialogFlags)0,
                              gimp_standard_help_func, PLUG_IN_PROC,
                              GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                              GTK_STOCK_OK,     GTK_RESPONSE_OK,
                              NULL);

    gtk_dialog_set_alternative_button_order (GTK_DIALOG (dialog),
            GTK_RESPONSE_OK,
            GTK_RESPONSE_CANCEL,
            -1);

    gimp_window_set_transient (GTK_WINDOW (dialog));

    main_vbox = gtk_vbox_new (FALSE, 12);
    gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 12);
    gtk_container_add (GTK_CONTAINER (gtk_dialog_get_content_area (GTK_DIALOG (dialog))),
                       main_vbox);
    gtk_widget_show (main_vbox);

    preview = gimp_drawable_preview_new (drawable, NULL);
    gtk_box_pack_start (GTK_BOX (main_vbox), preview, TRUE, TRUE, 0);
    gtk_widget_show (preview);

    g_signal_connect (preview, "invalidated",
                      G_CALLBACK (preview_update),
                      NULL);

    table = gtk_table_new (3, 4, FALSE);
    gtk_table_set_col_spacings (GTK_TABLE (table), 6);
    gtk_table_set_row_spacings (GTK_TABLE (table), 6);
    gtk_box_pack_start (GTK_BOX (main_vbox), table, FALSE, FALSE, 0);
    gtk_widget_show (table);


    adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 0,
                                _("_Strength:"), SCALE_WIDTH, ENTRY_WIDTH,
                                eaw_params.alpha, 0.0, 4.0, 0.05, 0.25, 2,
                                TRUE, 0, 0,
                                NULL, NULL);
    g_signal_connect (adj, "value-changed",
                      G_CALLBACK (gimp_double_adjustment_update),
                      &eaw_params.alpha);
    g_signal_connect_swapped (adj, "value-changed",
                              G_CALLBACK (gimp_preview_invalidate),
                              preview);


    adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 1,
                                _("_Levels:"), SCALE_WIDTH, ENTRY_WIDTH,
                                eaw_params.maxband, 1, 20, 1, 5, 0,
                                TRUE, 0, 0,
                                NULL, NULL);
    g_signal_connect (adj, "value-changed",
                      G_CALLBACK (gimp_int_adjustment_update),
                      &eaw_params.maxband);
    g_signal_connect_swapped (adj, "value-changed",
                              G_CALLBACK (gimp_preview_invalidate),
                              preview);


    adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 2,
                                _("_Output:"), SCALE_WIDTH, ENTRY_WIDTH,
                                eaw_params.inband, 1, 20, 1, 5, 0,
                                TRUE, 0, 0,
                                NULL, NULL);
    g_signal_connect (adj, "value-changed",
                      G_CALLBACK (gimp_int_adjustment_update),
                      &eaw_params.inband);
    g_signal_connect_swapped (adj, "value-changed",
                              G_CALLBACK (gimp_preview_invalidate),
                              preview);

                              
//     adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 3,
//                                 _("_Threshold:"), SCALE_WIDTH, ENTRY_WIDTH,
//                                 eaw_params.threshold, 0, 100, 1, 5, 0,
//                                 TRUE, 0, 0,
//                                 NULL, NULL);
//     g_signal_connect (adj, "value-changed",
//                       G_CALLBACK (gimp_int_adjustment_update),
//                       &eaw_params.threshold);
//     g_signal_connect_swapped (adj, "value-changed",
//                               G_CALLBACK (gimp_preview_invalidate),
//                               preview);


    hbox = gtk_hbox_new (FALSE, 12);
    gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show (hbox);

    frame = gimp_int_radio_group_new (TRUE, _("Mode"),
                                      G_CALLBACK (gimp_radio_button_update),
                                      &eaw_params.mode, eaw_params.mode,
                                      _("_Linear amplification"), 0, &button,
                                      _("_Local contrast"), 1, &button2,
                                      _("_Output level"), 2, &button3,
                                      NULL);

    g_signal_connect_swapped (button, "toggled",
                              G_CALLBACK (gimp_preview_invalidate),
                              preview);
    g_signal_connect_swapped (button2, "toggled",
                              G_CALLBACK (gimp_preview_invalidate),
                              preview);
    g_signal_connect_swapped (button3, "toggled",
                              G_CALLBACK (gimp_preview_invalidate),
                              preview);

    gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);
    gtk_widget_show (frame);


    gtk_widget_show (dialog);
    run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);
    gtk_widget_destroy (dialog);

    return run;
}

static void
preview_update (GimpPreview *preview)
{
    GimpDrawable *drawable;
    gint          x1, x2;
    gint          y1, y2;
    gint          x, y;
    gint          width, height;
    gint          border;
    GimpPixelRgn  srcPR;
    GimpPixelRgn  destPR;

    drawable =
        gimp_drawable_preview_get_drawable (GIMP_DRAWABLE_PREVIEW (preview));

    gimp_pixel_rgn_init (&srcPR, drawable,
                         0, 0, drawable->width, drawable->height, FALSE, FALSE);
    gimp_pixel_rgn_init (&destPR, drawable,
                         0, 0, drawable->width, drawable->height, TRUE, TRUE);

    gimp_preview_get_position (preview, &x, &y);
    gimp_preview_get_size (preview, &width, &height);

    eaw_region (&srcPR, &destPR, drawable->bpp,
                x, y, width, height,
                FALSE);

    gimp_pixel_rgn_init (&destPR, drawable, x, y, width, height, FALSE, TRUE);
    gimp_drawable_preview_draw_region (GIMP_DRAWABLE_PREVIEW (preview), &destPR);
}

static void
process (float *image,
         int    width,
         int    height,
         float  alpha,
         int    maxband,
         int    inband,
         int    mode)
{
    /* PROCESSING*/
    // 1 pixel in this buffer represents 1.0/scale pixels in original image:
    const float l1 = 1.0f; // finest level
    float lm = 0;
    for (int k = EAW_MIN(width,height); k; k>>=1) lm++; // coarsest level
    lm = EAW_MIN(maxband, l1 + lm);
    // level 1 => full resolution

    int numl = 0;
    for (int k=EAW_MIN(width,height); k; k>>=1) numl++;
    const int numl_cap = EAW_MIN(maxband - l1 + 1.5, numl);
    // printf("level range in %d %d: %f %f, cap: %d\n", 1, maxband, l1, lm, numl_cap);

    // TODO: fixed alloc for data piece at capped resolution?
    float **tmp = (float **)malloc(sizeof(float *)*numl_cap);
    for (int k=1; k<numl_cap; k++)
    {
        const int wd = (int)(1 + (width>>(k-1)));
        const int ht = (int)(1 + (height>>(k-1)));
        tmp[k] = (float *)malloc(sizeof(float)*wd*ht);
    }

    for (int level=1; level<numl_cap; level++) dt_iop_equalizer_wtf(image, tmp, level, width, height);

    for (int l=1; l<numl_cap; l++)
    {
        const float lv = (lm-l1)*(l-1)/(float)(numl_cap-1) + l1; // appr level in real image.
        const float band = EAW_CLAMP((1.0 - lv / maxband), 0, 1.0);
        // printf("lv: %f; band: %f, step: %d\n", lv, band, 1<<l);

        // coefficients in range [0, 2], 1 being neutral.
        float coeff = 1;
        if (mode == 0) { // linear amplification
            if(alpha>1.0) coeff = 1 + fabs(1-alpha)/lv;
            else          coeff = 1 - fabs(1-alpha)/lv;
        } else if (mode == 1) { // suppress lowest level
            if      ((int)lv == 1) coeff = 1;
            else if ((int)lv == 2) coeff = (alpha>1?(fabs(1-alpha)/lv/2+1):(1-fabs(1-alpha)/lv/2));
            else                   coeff = (alpha>1?(fabs(1-alpha)/lv+1):(1-fabs(1-alpha)/lv));
        } else if (mode == 2) { // show only one level
            if ((int)lv == inband ) coeff = alpha;
            else coeff = 0;
        }

        for (int ch=0; ch<3; ch++)
        {
            const int step = 1<<l;
#if 1 // scale coefficients
            for (int j=0; j<height; j+=step)      for (int i=step/2; i<width; i+=step) image[3*width*j + 3*i + ch] *= coeff;
            for (int j=step/2; j<height; j+=step) for (int i=0; i<width; i+=step)      image[3*width*j + 3*i + ch] *= coeff;
            for (int j=step/2; j<height; j+=step) for (int i=step/2; i<width; i+=step) image[3*width*j + 3*i + ch] *= coeff*coeff;
#else // soft-thresholding (shrinkage)
#define wshrink (copysignf(fmaxf(0.0f, fabsf(image[3*width*j + 3*i + ch]) - (1.0-coeff)), image[3*width*j + 3*i + ch]))
            for (int j=0; j<height; j+=step)      for (int i=step/2; i<width; i+=step) image[3*width*j + 3*i + ch] = wshrink;
            for (int j=step/2; j<height; j+=step) for (int i=0; i<width; i+=step)      image[3*width*j + 3*i + ch] = wshrink;
            for (int j=step/2; j<height; j+=step) for (int i=step/2; i<width; i+=step) image[3*width*j + 3*i + ch] = wshrink;
#undef wshrink
#endif
        }
    }
    // printf("applied\n");
    for (int level=numl_cap-1; level>0; level--) dt_iop_equalizer_iwtf(image, tmp, level, width, height);

    for (int k=1; k<numl_cap; k++) free(tmp[k]);
    free(tmp);
}
