/*
 * Copyright (C) 1999 Winston Chang
 *                    <winstonc@cs.wisc.edu>
 *                    <winston@stdout.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

// OpenCV
#include "cv.h"
#include "highgui.h"

#define PLUG_IN_PROC    "elsamuko-facedetect-cv"
#define PLUG_IN_BINARY  "elsamuko-facedetect-cv"

#define N_(x) (x)
#define _(x) (x)

#define SCALE_WIDTH   120
#define ENTRY_WIDTH     5
#define SCALE           1

#define FACE_XML     "haarcascade_frontalface_alt.xml"
#define PLUG_IN_DIR  "plug-ins"

// static int DEBUG;
static CvMemStorage* storage = 0;
static CvHaarClassifierCascade* cascade = 0;

/* Uncomment this line to get a rough estimate of how long the plug-in
 * takes to run.
 */

/*  #define TIMER  */

typedef struct
{
    gdouble  radius;
    gdouble  amount;
    gint     threshold;
} FaceDetectParams;

typedef struct
{
    gboolean  run;
} FaceDetectInterface;

/* local function prototypes */
static inline gint coord( gint i, gint j, gint k, gint channels, gint width ) {
    return channels*( width*i + j ) + k;
};

// static CvSeq*    cv_detect( IplImage* img );
static CvSeq*    cv_detect( IplImage* img, CvMemStorage* storage, CvHaarClassifierCascade* cascade);
static void      cv_draw(IplImage* img, CvSeq* faces);

static void      query (void);
static void      run   (const gchar      *name,
                        gint              nparams,
                        const GimpParam  *param,
                        gint             *nreturn_vals,
                        GimpParam       **return_vals);

static CvSeq*    facedetect_region     (GimpPixelRgn   *srcPTR,
                                        GimpPixelRgn   *dstPTR,
                                        gint            bpp,
                                        gdouble         radius,
                                        gdouble         amount,
                                        gint            x,
                                        gint            y,
                                        gint            width,
                                        gint            height,
                                        gboolean        show_progress,
                                        gboolean        preview);

static void      facedetect        (GimpDrawable   *drawable,
                                    gdouble         radius,
                                    gdouble         amount);

static gboolean  facedetect_dialog (GimpDrawable   *drawable);
static void      preview_update      (GimpPreview    *preview);


/* create a few globals, set default values */
static FaceDetectParams facedetect_params =
{
    5.0, /* default radius    */
    0.5, /* default amount    */
    0    /* default threshold */
};

/* Setting PLUG_IN_INFO */
const GimpPlugInInfo PLUG_IN_INFO =
{
    NULL,  /* init_proc  */
    NULL,  /* quit_proc  */
    query, /* query_proc */
    run,   /* run_proc   */
};


MAIN ();

static void
query (void)
{
    static const GimpParamDef args[] =
    {
        { GIMP_PDB_INT32,    "run-mode",  "The run mode { RUN-INTERACTIVE (0), RUN-NONINTERACTIVE (1) }" },
        { GIMP_PDB_IMAGE,    "image",     "(unused)" },
        { GIMP_PDB_DRAWABLE, "drawable",  "Drawable to draw on" },
        { GIMP_PDB_FLOAT,    "radius",    "not used" },
        { GIMP_PDB_FLOAT,    "amount",    "not used" },
        { GIMP_PDB_INT32,    "threshold", "not used" }
    };

    gimp_install_procedure (PLUG_IN_PROC,
                            N_("Detecting faces."),
                            "Detecting faces.",
                            "elsamuko <elsamuko@web.de>",
                            "elsamuko",
                            "2010",
                            N_("_Face Detection CV..."),
                            "GRAY*, RGB*",
                            GIMP_PLUGIN,
                            G_N_ELEMENTS (args), 0,
                            args, NULL);

    gimp_plugin_menu_register (PLUG_IN_PROC, "<Image>/Filters/Generic");
}

static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
    static GimpParam   values[1];
    GimpPDBStatusType  status = GIMP_PDB_SUCCESS;
    GimpDrawable      *drawable;
    GimpRunMode        run_mode;
#ifdef TIMER
    GTimer            *timer = g_timer_new ();
#endif

    run_mode = param[0].data.d_int32;

    *return_vals  = values;
    *nreturn_vals = 1;

    values[0].type          = GIMP_PDB_STATUS;
    values[0].data.d_status = status;

    //INIT_I18N ();

    /*
     * Get drawable information...
     */
    drawable = gimp_drawable_get (param[2].data.d_drawable);
    gimp_tile_cache_ntiles (2 * MAX (drawable->width  / gimp_tile_width () + 1 ,
                                     drawable->height / gimp_tile_height () + 1));

    switch (run_mode)
    {
    case GIMP_RUN_INTERACTIVE:
        gimp_get_data (PLUG_IN_PROC, &facedetect_params);
        /* Reset default values show preview unmodified */

        /* initialize pixel regions and buffer */
        if (! facedetect_dialog (drawable))
            return;

        break;

    case GIMP_RUN_NONINTERACTIVE:
        if (nparams != 6)
        {
            status = GIMP_PDB_CALLING_ERROR;
        }
        else
        {
            facedetect_params.radius = param[3].data.d_float;
            facedetect_params.amount = param[4].data.d_float;
            facedetect_params.threshold = param[5].data.d_int32;

            /* make sure there are legal values */
            if ((facedetect_params.radius < 0.0) ||
                    (facedetect_params.amount < 0.0))
                status = GIMP_PDB_CALLING_ERROR;
        }
        break;

    case GIMP_RUN_WITH_LAST_VALS:
        gimp_get_data (PLUG_IN_PROC, &facedetect_params);
        break;

    default:
        break;
    }

    if (status == GIMP_PDB_SUCCESS)
    {
        drawable = gimp_drawable_get (param[2].data.d_drawable);

        /* here we go */
        facedetect (drawable, facedetect_params.radius, facedetect_params.amount);

        gimp_displays_flush ();

        /* set data for next use of filter */
        if (run_mode == GIMP_RUN_INTERACTIVE)
            gimp_set_data (PLUG_IN_PROC,
                           &facedetect_params, sizeof (FaceDetectParams));

        gimp_drawable_detach(drawable);
        values[0].data.d_status = status;
    }

#ifdef TIMER
    g_printerr ("%f seconds\n", g_timer_elapsed (timer, NULL));
    g_timer_destroy (timer);
#endif
}

static void
facedetect (GimpDrawable *drawable,
            gdouble       radius,
            gdouble       amount)
{
    GimpPixelRgn srcPR, destPR;
    gint         x1, y1, x2, y2;
    gint         width, height;
    int i;

    /* initialize pixel regions */
    gimp_pixel_rgn_init (&srcPR, drawable,
                         0, 0, drawable->width, drawable->height, FALSE, FALSE);
    gimp_pixel_rgn_init (&destPR, drawable,
                         0, 0, drawable->width, drawable->height, TRUE, TRUE);

    /* Get the input */
    gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);
    width = x2-x1;
    height = y2-y1;

    CvSeq* faces = facedetect_region (&srcPR, &destPR, drawable->bpp,
                                      radius, amount,
                                      x1, y1, width, height,
                                      TRUE, FALSE);

    for (i = 0; i < faces->total; i++ ){
        CvRect* r = (CvRect*)cvGetSeqElem( faces, i );
        gimp_rect_select( gimp_drawable_get_image (drawable->drawable_id),
                          r->x * SCALE,
                          r->y * SCALE,
                          r->width * SCALE,
                          r->height * SCALE,
                          0, 0, 0);
    }
                                      

    gimp_drawable_flush (drawable);
    gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
    //gimp_drawable_update (drawable->drawable_id, x1, y1, width, height);
}

/* Copy the GIMP pixel region to an OpenCV image,
 * perform the face detection and copy the image back
 */
static CvSeq*
facedetect_region (GimpPixelRgn *srcPR,
                   GimpPixelRgn *destPR,
                   gint          bpp,
                   gdouble       radius, /* Radius, AKA standard deviation */
                   gdouble       amount,
                   gint          x,
                   gint          y,
                   gint          width,
                   gint          height,
                   gboolean      show_progress,
                   gboolean      preview)
{
    printf( "\nL%i: **** Begin of facedetect: **** \n", __LINE__);
    if (show_progress)
        gimp_progress_init (_("Searching..."));

    gint  i,j;
    gint  value;
    CvSeq* faces = 0;
    const char* gimp_local_path = gimp_directory();
    char  plugin_path[PATH_MAX + 1];
    sprintf (plugin_path, "%s/%s/%s", gimp_local_path, PLUG_IN_DIR, FACE_XML);

    //Initialise memory
    guchar *src = g_new( guchar, bpp * width * height );
    guchar *dest = g_new( guchar, bpp * width * height );

    printf( "L%i: Channels: %i\n", __LINE__, bpp );
    printf( "L%i: x:%i, y:%i\n", __LINE__, x, y);
    printf( "L%i: w:%i, h:%i\n", __LINE__, width, height);

    //Save pixel regions in arrays
    gimp_pixel_rgn_get_rect( srcPR,
                             src,
                             x, y,
                             width, height );
    gimp_pixel_rgn_get_rect( destPR,
                             dest,
                             x, y,
                             width, height );

    //cvimage
    IplImage* cvImg = cvCreateImage(cvSize(width,height), IPL_DEPTH_8U, 1);
    if (show_progress) gimp_progress_update (0.1);

    // B/W
    if (bpp == 1 || bpp==2) {
        printf( "L%i: B/W: Get image values\n", __LINE__);
        for (i = 0; i < height; i++) { //rows
            for (j = 0; j < width; j++) { //columns
                ((uchar *)( cvImg->imageData + i* cvImg->widthStep))[j] = (gint)src[coord(i,j,0,bpp,width)];
            }
        }

        // Color
    } else {
        printf( "L%i: Color: Get image values\n", __LINE__);
        for (i = 0; i < height; i++) { //rows
            for (j = 0; j < width; j++) { //columns
                //Luminance: 0.2126R + 0.7152G + 0.0722B
                ((uchar *)( cvImg->imageData + i* cvImg->widthStep))[j] = 0.2126*(gint)src[coord(i,j,0,bpp,width)] +
                        0.7152*(gint)src[coord(i,j,1,bpp,width)] +
                        0.0722*(gint)src[coord(i,j,2,bpp,width)];
            }
        }
    }
    if (show_progress) gimp_progress_update (0.2);

    cascade = (CvHaarClassifierCascade*)cvLoad( plugin_path, 0, 0, 0 );
    if ( cascade ) {
        storage = cvCreateMemStorage(0);
        //CvSeq* faces = cv_detect( cvImg );
        //cv_detect( cvImg );
        faces = cv_detect( cvImg, storage, cascade);
        cv_draw( cvImg, faces);
        
    } else {
        printf("L%i: ERROR: Could not load classifier cascade\n", __LINE__);
    }
    if (show_progress) gimp_progress_update (0.8);

    // B/W
    if (bpp == 1 || bpp==2) {
        printf( "L%i: B/W: Set depth values\n", __LINE__);
        for (i = 0; i < height; i++) { //rows
            for (j = 0; j < width; j++) { //columns
                dest[coord(i,j,0,bpp,width)] = ((uchar*)(cvImg->imageData + i*cvImg->widthStep))[j];
                dest[coord(i,j,1,bpp,width)] = 255;
            }
        }

        // Color
    } else {
        printf( "L%i: Color: Set depth values\n", __LINE__);
        for (i = 0; i < height; i++) { //rows
            for (j = 0; j < width; j++) { //columns
                value = ((uchar*)(cvImg->imageData + i*cvImg->widthStep))[j];
                dest[coord(i,j,0,bpp,width)] = value;
                dest[coord(i,j,1,bpp,width)] = value;
                dest[coord(i,j,2,bpp,width)] = value;
                dest[coord(i,j,3,bpp,width)] = 255;
            }
        }
    }

    //save depthmap in array
    if(preview){
        gimp_pixel_rgn_set_rect( destPR,
                                 dest,
                                 x, y,
                                 width, height );
    }

    if (show_progress)
        gimp_progress_update (1.0);

    //cvimages
    cvReleaseImage(&cvImg);

    g_free (dest);
    g_free (src);
    printf( "L%i: **** End of facedetect:   **** \n", __LINE__);
    return faces;
}

// Function to detect any faces that is present in an image
static CvSeq*
cv_detect( IplImage* img, CvMemStorage* storage, CvHaarClassifierCascade* cascade) {
    cvClearMemStorage( storage );
    if ( cascade ){
        return cvHaarDetectObjects( img, cascade, storage,
                                     1.1, 3, CV_HAAR_DO_CANNY_PRUNING,
                                     cvSize(40, 40) );
    }
    return 0;
}

// draw boxes around these faces
static void
cv_draw(IplImage* img, CvSeq* faces) {
    CvPoint pt1, pt2;
    int i;
    for (i = 0; i < faces->total; i++ ){
        CvRect* r = (CvRect*)cvGetSeqElem( faces, i );
        pt1.x =  r->x            * SCALE;
        pt2.x = (r->x+r->width)  * SCALE;
        pt1.y =  r->y            * SCALE;
        pt2.y = (r->y+r->height) * SCALE;
        cvRectangle( img, pt1, pt2, CV_RGB(255,255,255), 3, 8, 0 );
    }
}

static gboolean
facedetect_dialog (GimpDrawable *drawable)
{
    GtkWidget *dialog;
    GtkWidget *main_vbox;
    GtkWidget *preview;
    GtkWidget *table;
    // GtkObject *adj;
    gboolean   run;

    gimp_ui_init (PLUG_IN_BINARY, TRUE);

    dialog = gimp_dialog_new (_("Face Detection CV"), PLUG_IN_BINARY,
                              NULL, 0,
                              gimp_standard_help_func, PLUG_IN_PROC,

                              GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                              GTK_STOCK_OK,     GTK_RESPONSE_OK,

                              NULL);

    gtk_dialog_set_alternative_button_order (GTK_DIALOG (dialog),
            GTK_RESPONSE_OK,
            GTK_RESPONSE_CANCEL,
            -1);

    gimp_window_set_transient (GTK_WINDOW (dialog));

    main_vbox = gtk_vbox_new (FALSE, 12);
    gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 12);
    gtk_container_add (GTK_CONTAINER (gtk_dialog_get_content_area (GTK_DIALOG (dialog))),
                       main_vbox);
    gtk_widget_show (main_vbox);

    preview = gimp_drawable_preview_new (drawable, NULL);
    gtk_box_pack_start (GTK_BOX (main_vbox), preview, TRUE, TRUE, 0);
    gtk_widget_show (preview);

    g_signal_connect (preview, "invalidated",
                      G_CALLBACK (preview_update),
                      NULL);

    table = gtk_table_new (3, 3, FALSE);
    gtk_table_set_col_spacings (GTK_TABLE (table), 6);
    gtk_table_set_row_spacings (GTK_TABLE (table), 6);
    gtk_box_pack_start (GTK_BOX (main_vbox), table, FALSE, FALSE, 0);
    gtk_widget_show (table);

//     adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 0,
//                                 _("_Radius:"), SCALE_WIDTH, ENTRY_WIDTH,
//                                 facedetect_params.radius, 0.1, 120.0, 0.1, 1.0, 1,
//                                 TRUE, 0, 0,
//                                 NULL, NULL);
// 
//     g_signal_connect (adj, "value-changed",
//                       G_CALLBACK (gimp_double_adjustment_update),
//                       &facedetect_params.radius);
//     g_signal_connect_swapped (adj, "value-changed",
//                               G_CALLBACK (gimp_preview_invalidate),
//                               preview);
// 
//     adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 1,
//                                 _("_Amount:"), SCALE_WIDTH, ENTRY_WIDTH,
//                                 facedetect_params.amount, 0.0, 10.0, 0.01, 0.1, 2,
//                                 TRUE, 0, 0,
//                                 NULL, NULL);
// 
//     g_signal_connect (adj, "value-changed",
//                       G_CALLBACK (gimp_double_adjustment_update),
//                       &facedetect_params.amount);
//     g_signal_connect_swapped (adj, "value-changed",
//                               G_CALLBACK (gimp_preview_invalidate),
//                               preview);
// 
//     adj = gimp_scale_entry_new (GTK_TABLE (table), 0, 2,
//                                 _("_Threshold:"), SCALE_WIDTH, ENTRY_WIDTH,
//                                 facedetect_params.threshold,
//                                 0.0, 255.0, 1.0, 10.0, 0,
//                                 TRUE, 0, 0,
//                                 NULL, NULL);
// 
//     g_signal_connect (adj, "value-changed",
//                       G_CALLBACK (gimp_int_adjustment_update),
//                       &facedetect_params.threshold);
//     g_signal_connect_swapped (adj, "value-changed",
//                               G_CALLBACK (gimp_preview_invalidate),
//                               preview);

    gtk_widget_show (dialog);

    run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);

    gtk_widget_destroy (dialog);

    return run;
}

static void
preview_update (GimpPreview *preview)
{
    GimpDrawable *drawable;
    gint          x, y;
    gint          width, height;
    gint          border;
    GimpPixelRgn  srcPR;
    GimpPixelRgn  destPR;

    drawable =
        gimp_drawable_preview_get_drawable (GIMP_DRAWABLE_PREVIEW (preview));

    gimp_pixel_rgn_init (&srcPR, drawable,
                         0, 0, drawable->width, drawable->height, FALSE, FALSE);
    gimp_pixel_rgn_init (&destPR, drawable,
                         0, 0, drawable->width, drawable->height, TRUE, TRUE);

    gimp_preview_get_position (preview, &x, &y);
    gimp_preview_get_size (preview, &width, &height);

    /* enlarge the region to avoid artefacts at the edges of the preview */
    border = 2.0 * facedetect_params.radius + 0.5;

    facedetect_region (&srcPR, &destPR, drawable->bpp,
                       facedetect_params.radius, facedetect_params.amount,
                       x, y, width, height,
                       FALSE, TRUE);

    gimp_pixel_rgn_init (&destPR, drawable, x, y, width, height, FALSE, TRUE);
    gimp_drawable_preview_draw_region (GIMP_DRAWABLE_PREVIEW (preview), &destPR);
}
