/* 
GIMP Plug-in Laso
Ondrej Fiala

curve.c

data structures for curve and points
*/

#include <stdio.h>
#include <libgimp/gimp.h>

#include "curve.h"

gdouble get_module(gdouble dx, gdouble dy)
{
	return sqrt(dx*dx + dy*dy);   
}

Point point_new (gdouble x, gdouble y)
{
	Point point;

	point.x = x;
	point.y = y;

	return point;
}

void point_clear (Point *point)
{
	point->x = 0;
	point->y = 0;
}

void point_set (Point *point, gdouble x, gdouble y)
{
	point->x = x;
	point->y = y;
}

void point_set_interpolate (Point *point, Point** vector_field, gdouble x, gdouble y, gint width, gint height)
{
	gdouble nx, ny;
	gint x1, x2, y1, y2;

	/* pro posun doprostred pixelu */
	x = x - 0.5;
	y = y - 0.5;

	/* border constraints */
	if(x < 0){x = 0;}
	if(x >= (gdouble)width){x = (gdouble)(width-1);}
	if(y < 0){y = 0;}
	if(y >= (gdouble)height){y = (gdouble)(height-1);}

	x1 = (gint)floor(x);
	x2 = (gint)floor(x+1);
	y1 = (gint)floor(y);
	y2 = (gint)floor(y+1);
	if(x2 >= width){x2 = (width-1);}
	if(y2 >= height){y2 = (height-1);}


	/* del = (x2 - x1)*(y2 - y1); je vzdy 1 (pokud neni chyba a podob) */

	nx = vector_field[x1][y1].x * (x2 - x) * (y2 - y) 
	   + vector_field[x2][y1].x * (x - x1) * (y2 - y)
	   + vector_field[x1][y2].x * (x2 - x) * (y - y1)
	   + vector_field[x2][y2].x * (x - x1) * (y - y1);
	ny = vector_field[x1][y1].y * (x2 - x) * (y2 - y)
	   + vector_field[x2][y1].y * (x - x1) * (y2 - y)
	   + vector_field[x1][y2].y * (x2 - x) * (y - y1)
	   + vector_field[x2][y2].y * (x - x1) * (y - y1);

	point->x = nx;
	point->y = ny;
}

void point_multiple (Point *point, gdouble number)
{
	point->x = number * point->x;
	point->y = number * point->y;
}

Point point_copy (Point *point)
{
	Point point2;

	point2.x = point->x;
	point2.y = point->y;

	return point2;
}

void point_print (Point *point) 
{
	printf(" %f, %f\n",point->x,point->y);
}


Curve curve_new ()
{
	Curve curve;
	Point *points;

	points = (Point*) g_malloc(CURVE_MAX * sizeof(Point));

	curve._max = CURVE_MAX;
	curve._number = 0;
	curve.points = points;

	return curve;
}

void curve_free (Curve *curve) 
{
	g_free(curve->points);
	curve->_number = 0;
}

void curve_clear (Curve *curve)
{
	curve->_number = 0;
}

gboolean curve_add (Curve *curve, Point *point)
{
	if (curve->_number < curve->_max )
	{
		curve->points[curve->_number].x = point->x;
		curve->points[curve->_number].y = point->y;

		curve->_number++;
		return TRUE;
	} else {
		return FALSE;	
	}
}

gboolean curve_add_xy (Curve *curve, gdouble x, gdouble y)
{
	if (curve->_number < curve->_max )
	{
		Point point;
		point = point_new(x,y);

		curve->points[curve->_number] = point;
		curve->_number++;

		return TRUE;
	} else {
		return FALSE;	
	}
}

gboolean curve_copy(Curve *new_curve,Curve *curve)
{
	gint i;

	new_curve->_number = 0;

	for(i=0;i < curve->_number;i++){
		new_curve->points[i].x = curve->points[i].x;
		new_curve->points[i].y = curve->points[i].y;
		new_curve->_number++;
	}

	return TRUE;
}

/* prvni bod je potreba presunout na posledni misto 
 gimp to vraci ve formatu CACCACCAC controlpoint/anchor	*/
Curve curve_from_gimp_points(gint number_points,gdouble *points)
{
	Curve curve;
	gint i;
	gint pair; 
	gint pair_max; 

	pair = number_points / 2; 
	curve = curve_new();
	if (pair > 0) {
		if(pair>CURVE_MAX){
			pair_max = (CURVE_MAX-5);
			/* ve have 5 points for interspacing created big hole */
		} else {
			pair_max = pair;	
		}
		for(i=1;i < pair_max;i++){
			curve_add_xy(&curve,points[2*i],points[2*i+1]);	
		}
		curve_add_xy(&curve,points[0],points[0+1]);
	}

	return curve;
}

/* posledni bod potreba presunout na prvni misto */
gdouble* curve_to_points (Curve *curve, gint *number_points)
{
	gint i,n;
	gdouble *new_points;
	n = curve->_number * 2;
	*number_points = n;
	new_points = (gdouble*) g_malloc(*number_points * sizeof(gdouble));

	if (*number_points > 0){
		new_points[0] = curve->points[curve->_number-1].x;
		new_points[0+1] = curve->points[curve->_number-1].y;
		for(i=0;i<(curve->_number-1);i++)
		{
			new_points[2*(i+1)] = curve->points[i].x;
			new_points[2*(i+1)+1] = curve->points[i].y;
		}
	}
	return new_points;
}

gboolean curve_check_bounds(Curve *curve, gint width, gint height)
{
	gint n,i;
	gboolean repair;

	n = curve->_number;
	repair = FALSE;

	for(i=0;i<n;i++){
		if(curve->points[i].x < 0){	curve->points[i].x = 0; repair = TRUE; }
		if(curve->points[i].x > width){	curve->points[i].x = width; repair = TRUE; }
		if(curve->points[i].y < 0){	curve->points[i].y = 0; repair = TRUE; }
		if(curve->points[i].y > height){ curve->points[i].y = height; repair = TRUE; }
	}
	return repair;
}

/* add points between too distant points on curve  */
gboolean curve_interspace_points(Curve *curve)
{
	gint n,i,j;
	gboolean repair;
	gdouble lenght,nx,ny;

	n = curve->_number;
	repair = FALSE;
	if(n>CURVE_MAX)
		return repair;

	for(i=0; i<n;i++)
	{
		lenght = get_module(curve->points[i%n].x - curve->points[(i+1)%n].x,
							curve->points[i%n].y - curve->points[(i+1)%n].y);
		if((lenght>CURVE_POINTS_DISTANCE_MAX)&&((n<CURVE_MAX))){

			nx = (curve->points[i%n].x + curve->points[(i+1)%n].x) / 2;
			ny = (curve->points[i%n].y + curve->points[(i+1)%n].y) / 2;

			/* potreba vlozit novy bod na krivku */
			curve->_number++;			
			n = curve->_number;
			i++;

			for (j = (n-1);j >i;j--){
				curve->points[j] = curve->points[j-1];
			}
			curve->points[i].x = nx;
			curve->points[i].y = ny;
			repair = TRUE;
		}
	}

	return repair;
}

/* remove close points on curve  */
gboolean curve_remove_points(Curve *curve)
{
	gint n,i,j;
	gboolean repair;
	gdouble lenght1,lenght2;

	n = curve->_number;
	repair = FALSE;

	for(i=0; i<n;i++)
	{
		lenght1 = get_module(curve->points[i%n].x - curve->points[(i+1)%n].x,
							 curve->points[i%n].y - curve->points[(i+1)%n].y);
		lenght2 = get_module(curve->points[i%n].x - curve->points[(i-1)%n].x,
							 curve->points[i%n].y - curve->points[(i-1)%n].y);
		if((lenght1<CURVE_POINTS_DISTANCE_MIN)&&(lenght2<CURVE_POINTS_DISTANCE_MIN)){

			/* potreba odstranit bod na krivce */
			for (j = i;j < (n-1);j++){
				curve->points[j%n] = curve->points[(j+1)%n];
			}
			curve->_number--;			
			n = curve->_number;

			repair = TRUE;
		}
	}

	return repair;
}

Curve curve_to_bezier (Curve *curve)
{
	gint i,i3,n;
	Curve new_curve;
	Point point;

	n = curve->_number;

	/* %n needed everywhere ! */

	new_curve = curve_new();
	point = point_new(0,0);

	i = 0;
	for(i=0; i < n ;i++)
	{
		i3 = i%3;
		i3 = 0;
		if(i3 == 0){
			point.x = (curve->points[i%n].x + 4 * curve->points[(i+1)%n].x + curve->points[(i+2)%n].x)/6;
			point.y = (curve->points[i%n].y + 4 * curve->points[(i+1)%n].y + curve->points[(i+2)%n].y)/6;
			curve_add(&new_curve, &point);
		}
		if((i3 == 0)||(i3 == 1)){
			point.x = (2 * curve->points[(i+1)%n].x + curve->points[(i+2)%n].x)/3;
			point.y = (2 * curve->points[(i+1)%n].y + curve->points[(i+2)%n].y)/3;
			curve_add(&new_curve, &point);
		}
		if((i3 == 0)||(i3 == 2)){
			point.x = (curve->points[(i+1)%n].x + 2 * curve->points[(i+2)%n].x)/3;
			point.y = (curve->points[(i+1)%n].y + 2 * curve->points[(i+2)%n].y)/3;
			curve_add(&new_curve, &point);
		}
		if(i3 == 10){
			point.x = (curve->points[(i+1)%n].x + 4 * curve->points[(i+2)%n].x + curve->points[(i+3)%n].x)/6;
			point.y = (curve->points[(i+1)%n].y + 4 * curve->points[(i+2)%n].y + curve->points[(i+3)%n].y)/6;
			curve_add(&new_curve, &point);
		}
	}

	/* pocet bodu musi byt delitelny 3 | n % 3 = 0 */
	/* add 2 points */
	if(new_curve._number%3 == 1){
		curve_add(&new_curve, &new_curve.points[new_curve._number-1]);
		curve_add(&new_curve, &new_curve.points[new_curve._number-1]);
	}
	/* add 1 point */
	if(new_curve._number%3 == 2){
		curve_add(&new_curve, &new_curve.points[new_curve._number-1]);
	}
	return new_curve;
}

void curve_min_count(Curve *curve, gdouble min_lenght, gint *num_stay){
	gint n,i;
	gdouble sum;
	n = curve->_number;
	*num_stay = 0;
	sum = 0;

	for(i=0;i<n;i++){
		if (get_module(curve->points[i].x, curve->points[i].y) <= min_lenght){
			(*num_stay)++;
			sum = sum + get_module(curve->points[i].x, curve->points[i].y); 
		}
	}
}

void curve_diff(Curve *curve1, Curve *curve2, gdouble *diff){
	gint n,i;
	n = MIN(curve1->_number, curve2->_number);

	*diff = 0;

	for(i=0;i<n;i++){
		*diff += get_module(curve1->points[i].x - curve2->points[i].x, curve1->points[i].y - curve2->points[i].y);
	}
}

void curve_multiple (Curve *curve, gdouble number){
	gint n,i;
	n = curve->_number;
	for(i=0;i<n;i++){
		curve->points[i].x = number * curve->points[i].x;
		curve->points[i].y = number * curve->points[i].y;
	}
}

void curve_sum_vector (Curve *curve, Curve *vector_curve){
	gint n,i;
	n = curve->_number;
	if(n > vector_curve->_number) n = vector_curve->_number;
	for(i=0;i<n;i++){
		curve->points[i].x = curve->points[i].x + vector_curve->points[i].x;
		curve->points[i].y = curve->points[i].y + vector_curve->points[i].y;
	}
}

void curve_divide_vector (Curve *curve, Curve *vector_curve){
	gint n,i;
	n = MIN(curve->_number,vector_curve->_number);

	for(i=0;i<n;i++){
		if(vector_curve->points[i].x != 0) 
			curve->points[i].x = curve->points[i].x / vector_curve->points[i].x;
		if(vector_curve->points[i].y != 0) 
			curve->points[i].y = curve->points[i].y / vector_curve->points[i].y;
	}
}

void curve_print (Curve *curve) {

	gint i;
	for(i=0;i<curve->_number;i++){
		printf(" %f, %f\n",curve->points[i].x,curve->points[i].y);
	}
}
