/* 
GIMP Plug-in Laso
Ondrej Fiala

curve.c

data structures for curve and points
*/

#include <libgimp/gimp.h>

#include "force.h"

/* macros for checking arrays bounds */
#define BW(value)  (value>(BOUNDS_WIDTH-1)?(BOUNDS_WIDTH-1):(value<0?0:value))
#define BH(value)  (value>(BOUNDS_HEIGHT-1)?(BOUNDS_HEIGHT-1):(value<0?0:value))


/* tensile force - tahova sila */
void tensile_force (Point *point, Curve *curve, gint index)
{
	gint n;
	Point *V_i_1, *V_i, *V_i1;

	if (index < 0){ index = 0; }
	if (index > (curve->_number-1)){ index = curve->_number-1; }

	n = curve->_number;
	V_i_1 = &curve->points[(index-1 + n) % n];
	V_i = &curve->points[(index + n) % n];
	V_i1 = &curve->points[(index+1 + n) % n];

	point->x = V_i_1->x + V_i1->x - 2*V_i->x;
	point->y = V_i_1->y + V_i1->y - 2*V_i->y;	

}

/* tensile_force() with normalization */
gdouble tensile_force_normalize(Curve *curve)
{
	gdouble dfv, dmax, dx, dy;
	gint n,i;

	dmax = -Inf;
	dfv = 0;
	n = curve->_number;

	for(i=1; i<=n;i++)
	{
		dx = curve->points[(i-1)%n].x + curve->points[(i+1)%n].x - 2*curve->points[i%n].x; 
		dy = curve->points[(i-1)%n].y + curve->points[(i+1)%n].y - 2*curve->points[i%n].y; 
		dfv = get_module(dx,dy);
		if(dfv > dmax) dmax = dfv;
	}

	for(i=0; i<n;i++)
	{
		curve->points[i].x = curve->points[i].x / dmax; 
		curve->points[i].y = curve->points[i].y / dmax; 
	}
	return dmax;
}

/* ohybova sila */
void flexure_force (Point *point, Curve *curve, gint index){
	gint n;
	Point *V_i_2, *V_i_1, *V_i, *V_i1, *V_i2;

	n = curve->_number;

	V_i_2 = &curve->points[(index-2 + n) % n];
	V_i_1 = &curve->points[(index-1 + n) % n];
	V_i = &curve->points[(index + n) % n];
	V_i1 = &curve->points[(index+1 + n) % n];
	V_i2  = &curve->points[(index+2 + n) % n];


	point->x = (V_i_2->x - 2*V_i_1->x + V_i->x ) - 2*( V_i_1->x - 2*V_i->x + V_i1->x ) + ( V_i->x - 2*V_i1->x + V_i2->x );
	point->y = (V_i_2->y - 2*V_i_1->y + V_i->y ) - 2*( V_i_1->y - 2*V_i->y + V_i1->y ) + ( V_i->y - 2*V_i1->y + V_i2->y );

}

/* OPTIMALIZATION FORCES*/
gint force_stopping_point(Point *point,gdouble height){
	gdouble dheight, stopping;
	if(height > 204){
		dheight = 255 - height; // 0..50 - vzdalenost od max vrcholu

		/* dlap = 0 => dinternal = 100 | dlap = 50 => dinternal = 1 */
		/* f(x) = -499/50x + 500 */

		stopping = -499/50 * dheight + 500;
		point->x = stopping;
		point->y = stopping;

		return 1;
	} else {
		point->x = 1;
		point->y = 1;
		return 0;
	}
}

gint force_stopping_point_normalized(Point *point,gdouble height){
	gdouble dheight, stopping;
	if(height > 0.6){
		dheight = 1 - height; // 0..0.3 - vzdalenost od max vrcholu

		/* dheight = 0 => stopping = 10 | dheight = 50 => stopping = 1 */
		/* f(x) = -9/50x + 10 */
		//stopping = -9/50 * dheight + 10;
		/* dheight = 0 => stopping = 100 | dheight = 0.2 => stopping = 1 */
		/* f(x) = -99/50x + 10 */

		//stopping = -99/0.2 * dheight + 100;
		
		stopping = -9/0.4 * dheight + 10;
		point->x = stopping;
		point->y = stopping;
		return 1;
	} else {
		point->x = 1;
		point->y = 1;
		return 0;
	}
}

/* IMAGE FORCES */
Point** image_force3x3(guint **laplace, GimpDrawable *drawable)
{
	gint width, height;
	gint i,j,max;
	gint dx,dy,dv;
	Point** vf;
	Point point,zero;

	width = drawable->width;
	height = drawable->height;

	zero = point_new(0,0);
	point = point_new(4,5);

	/* 2D u,v */
	vf = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		vf[i] = g_malloc(height * sizeof(Point));
	}
	
	for (i=0; i<width; i++){
		vf[i][0] = zero;
		vf[i][height-1] = zero;
	} 
	for (j=0; j<height; j++){
		vf[0][j] = zero;
		vf[width-1][j] = zero;
	} 

	for (j=1; j<(height-1); j++) {
		for (i=1; i<(width-1); i++) {
			max = laplace[i][j];
			dx = 0;
			dy = 0;

			/* poradi hledani podle vzdalenosti j-radka i-sloupec  */
			/* up right down left */
			if(max < (gint)laplace[i][j-1]){max = laplace[i][j-1]; dx = 0; dy = -1;}
			if(max < (gint)laplace[i+1][j]){max = laplace[i+1][j]; dx = +1; dy = 0;}
			if(max < (gint)laplace[i][j+1]){max = laplace[i][j+1]; dx = 0; dy = +1;}
			if(max < (gint)laplace[i-1][j]){max = laplace[i-1][j]; dx = -1; dy = 0;}

			/* diagonaly */
			if(max < (gint)laplace[i-1][j-1]){max = laplace[i-1][j-1]; dx = -1; dy = -1;}
			if(max < (gint)laplace[i+1][j-1]){max = laplace[i+1][j-1]; dx = +1; dy = -1;}
			if(max < (gint)laplace[i+1][j+1]){max = laplace[i+1][j+1]; dx = +1; dy = +1;}
			if(max < (gint)laplace[i-1][j+1]){max = laplace[i-1][j+1]; dx = -1; dy = +1;}

			dv = (max - laplace[i][j]); 

			/* prenasobeni vyskovym rozdilem */
			vf[i][j].x = (gdouble)dx * dv; 
			vf[i][j].y = (gdouble)dy * dv; 

		}
	}

	return vf;
}

/* counts  force field (max 5x5 matrix) */
Point** image_force5x5_double(gdouble **laplace, GimpDrawable *drawable)
{
	gint width, height;
	gint i,j;
	gdouble max,dv;
	gint dx,dy;
	Point** vf;

	gint BOUNDS_HEIGHT, BOUNDS_WIDTH;

	width = drawable->width;
	BOUNDS_WIDTH = width;
	height = drawable->height;
	BOUNDS_HEIGHT = height;

	/* 2D vf */
	vf = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		vf[i] = g_malloc(height * sizeof(Point));
	}

	for (j=0; j<height; j++) {
		for (i=0; i<width; i++) {
			max = laplace[i][j];
			dx = 0;
			dy = 0;

			/* poradi hledani podle vzdalenosti j-radka i-sloupec  */
			/* up right down left */
			if(max < laplace[BW(i)][BH(j-1)])   {max = laplace[BW(i)][BH(j-1)]; dx = 0;  dy = -1;}
			if(max < laplace[BW(i+1)][BH(j)])   {max = laplace[BW(i+1)][BH(j)]; dx = +1; dy = 0;}
			if(max < laplace[BW(i)][BH(j+1)])	  {max = laplace[BW(i)][BH(j+1)]; dx = 0;  dy = +1;}
			if(max < laplace[BW(i-1)][BH(j)])   {max = laplace[BW(i-1)][BH(j)]; dx = -1; dy = 0;}

			/* blizke diagonaly */
			if(max < laplace[BW(i-1)][BH(j-1)]) {max = laplace[BW(i-1)][BH(j-1)]; dx = -1; dy = -1;}
			if(max < laplace[BW(i+1)][BH(j-1)]) {max = laplace[BW(i+1)][BH(j-1)]; dx = +1; dy = -1;}
			if(max < laplace[BW(i-1)][BH(j+1)]) {max = laplace[BW(i-1)][BH(j+1)]; dx = -1; dy = +1;}
			if(max < laplace[BW(i+1)][BH(j+1)]) {max = laplace[BW(i+1)][BH(j+1)]; dx = +1; dy = +1;}

			/* vzdalenejsi kriz */
			if(max < laplace[BW(i)][BH(j-2)])   {max = laplace[BW(i)][BH(j-2)]; dx = 0;  dy = -2;}
			if(max < laplace[BW(i-2)][BH(j)])   {max = laplace[BW(i-2)][BH(j)]; dx = -2; dy = 0;}
			if(max < laplace[BW(i)][BH(j+2)])   {max = laplace[BW(i)][BH(j+2)]; dx = 0;  dy = +2;}
			if(max < laplace[BW(i+2)][BH(j)])   {max = laplace[BW(i+2)][BH(j)]; dx = +2; dy = 0;}

			/* vzdalenejsi ostatni */
			if(max < laplace[BW(i-1)][BH(j-2)]) {max = laplace[BW(i-1)][BH(j-2)]; dx = -1; dy = -2;}
			if(max < laplace[BW(i+1)][BH(j-2)]) {max = laplace[BW(i+1)][BH(j-2)]; dx = +1; dy = -2;}
			if(max < laplace[BW(i+2)][BH(j-1)]) {max = laplace[BW(i+2)][BH(j-1)]; dx = +2; dy = -1;}
			if(max < laplace[BW(i-2)][BH(j+1)]) {max = laplace[BW(i-2)][BH(j+1)]; dx = -2; dy = +1;}
			if(max < laplace[BW(i-1)][BH(j+2)]) {max = laplace[BW(i-1)][BH(j+2)]; dx = -1; dy = +2;}
			if(max < laplace[BW(i+1)][BH(j+2)]) {max = laplace[BW(i+1)][BH(j+2)]; dx = +1; dy = +2;}
			if(max < laplace[BW(i+2)][BH(j+1)]) {max = laplace[BW(i+2)][BH(j+1)]; dx = +2; dy = +1;}			
			if(max < laplace[BW(i-2)][BH(j-1)]) {max = laplace[BW(i-2)][BH(j-1)]; dx = -2; dy = -1;}

			/* vzdalenejsi diagonaly */
			if(max < laplace[BW(i-2)][BH(j-2)]) {max = laplace[BW(i-2)][BH(j-2)]; dx = -2; dy = -2;}
			if(max < laplace[BW(i+2)][BH(j-2)]) {max = laplace[BW(i+2)][BH(j-2)]; dx = +2; dy = -2;}
			if(max < laplace[BW(i-2)][BH(j+2)]) {max = laplace[BW(i-2)][BH(j+2)]; dx = -2; dy = +2;}
			if(max < laplace[BW(i+2)][BH(j+2)]) {max = laplace[BW(i+2)][BH(j+2)]; dx = +2; dy = +2;}

			dv = (max - laplace[i][j]);

			/* prenasobeni vyskovym rozdilem */
			vf[i][j].x = (gdouble)dx * dv; 
			vf[i][j].y = (gdouble)dy * dv; 
		}
	}

	return vf;
}


Point** image_force5x5(guint **laplace, GimpDrawable *drawable)
{
	gint width, height;
	gint i,j,max;
	gint dx,dy,dv;
	Point** vf;

	gint BOUNDS_HEIGHT, BOUNDS_WIDTH;

	width = drawable->width;
	BOUNDS_WIDTH = width;
	height = drawable->height;
	BOUNDS_HEIGHT = height;

	/* 2D vf */
	vf = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		vf[i] = g_malloc(height * sizeof(Point));
	}

	for (j=0; j<height; j++) {
		for (i=0; i<width; i++) {
			max = laplace[i][j];
			dx = 0;
			dy = 0;

			/* poradi hledani podle vzdalenosti j-radka i-sloupec  */
			/* up right down left */
			if(max < (gint)laplace[BW(i)][BH(j-1)])   {max = laplace[BW(i)][BH(j-1)]; dx = 0;  dy = -1;}
			if(max < (gint)laplace[BW(i+1)][BH(j)])   {max = laplace[BW(i+1)][BH(j)]; dx = +1; dy = 0;}
			if(max < (gint)laplace[BW(i)][BH(j+1)])	  {max = laplace[BW(i)][BH(j+1)]; dx = 0;  dy = +1;}
			if(max < (gint)laplace[BW(i-1)][BH(j)])   {max = laplace[BW(i-1)][BH(j)]; dx = -1; dy = 0;}

			/* blizke diagonaly */
			if(max < (gint)laplace[BW(i-1)][BH(j-1)]) {max = laplace[BW(i-1)][BH(j-1)]; dx = -1; dy = -1;}
			if(max < (gint)laplace[BW(i+1)][BH(j-1)]) {max = laplace[BW(i+1)][BH(j-1)]; dx = +1; dy = -1;}
			if(max < (gint)laplace[BW(i-1)][BH(j+1)]) {max = laplace[BW(i-1)][BH(j+1)]; dx = -1; dy = +1;}
			if(max < (gint)laplace[BW(i+1)][BH(j+1)]) {max = laplace[BW(i+1)][BH(j+1)]; dx = +1; dy = +1;}

			/* vzdalenejsi kriz */
			if(max < (gint)laplace[BW(i)][BH(j-2)])   {max = laplace[BW(i)][BH(j-2)]; dx = 0;  dy = -2;}
			if(max < (gint)laplace[BW(i-2)][BH(j)])   {max = laplace[BW(i-2)][BH(j)]; dx = -2; dy = 0;}
			if(max < (gint)laplace[BW(i)][BH(j+2)])   {max = laplace[BW(i)][BH(j+2)]; dx = 0;  dy = +2;}
			if(max < (gint)laplace[BW(i+2)][BH(j)])   {max = laplace[BW(i+2)][BH(j)]; dx = +2; dy = 0;}

			/* vzdalenejsi ostatni */
			if(max < (gint)laplace[BW(i-1)][BH(j-2)]) {max = laplace[BW(i-1)][BH(j-2)]; dx = -1; dy = -2;}
			if(max < (gint)laplace[BW(i+1)][BH(j-2)]) {max = laplace[BW(i+1)][BH(j-2)]; dx = +1; dy = -2;}
			if(max < (gint)laplace[BW(i+2)][BH(j-1)]) {max = laplace[BW(i+2)][BH(j-1)]; dx = +2; dy = -1;}
			if(max < (gint)laplace[BW(i-2)][BH(j+1)]) {max = laplace[BW(i-2)][BH(j+1)]; dx = -2; dy = +1;}
			if(max < (gint)laplace[BW(i-1)][BH(j+2)]) {max = laplace[BW(i-1)][BH(j+2)]; dx = -1; dy = +2;}
			if(max < (gint)laplace[BW(i+1)][BH(j+2)]) {max = laplace[BW(i+1)][BH(j+2)]; dx = +1; dy = +2;}
			if(max < (gint)laplace[BW(i+2)][BH(j+1)]) {max = laplace[BW(i+2)][BH(j+1)]; dx = +2; dy = +1;}			
			if(max < (gint)laplace[BW(i-2)][BH(j-1)]) {max = laplace[BW(i-2)][BH(j-1)]; dx = -2; dy = -1;}

			/* vzdalenejsi diagonaly */
			if(max < (gint)laplace[BW(i-2)][BH(j-2)]) {max = laplace[BW(i-2)][BH(j-2)]; dx = -2; dy = -2;}
			if(max < (gint)laplace[BW(i+2)][BH(j-2)]) {max = laplace[BW(i+2)][BH(j-2)]; dx = +2; dy = -2;}
			if(max < (gint)laplace[BW(i-2)][BH(j+2)]) {max = laplace[BW(i-2)][BH(j+2)]; dx = -2; dy = +2;}
			if(max < (gint)laplace[BW(i+2)][BH(j+2)]) {max = laplace[BW(i+2)][BH(j+2)]; dx = +2; dy = +2;}

			dv = (max - laplace[i][j]);

			/* prenasobeni vyskovym rozdilem */
			vf[i][j].x = (gdouble)dx * dv; 
			vf[i][j].y = (gdouble)dy * dv; 

		}
	}

	return vf;
}

gint force_normalize(Curve *curve, gdouble max)
{
	gint i;

	if (max == 0) return 0;

	for(i=0; i<curve->_number;i++)
	{
		curve->points[i].x = curve->points[i].x / max; 
		curve->points[i].y = curve->points[i].y / max; 
	}
	return 1;
}

gint force_normalize_2(Curve *curve, gdouble max)
{
	gint i;
	gdouble max2;

	if (max == 0) return 0;

	max2 = max * max;

	for(i=0; i<curve->_number;i++)
	{
		curve->points[i].x = (curve->points[i].x) * (curve->points[i].x) / max2; 
		curve->points[i].y = (curve->points[i].y) * (curve->points[i].y) / max2; 
	}
	return 1;
}

gint force_print(Curve *curve, Curve *vector_curve, gdouble multiple, GimpDrawable *drawable)
{
	gint n,i;
	gdouble end_points[4];
	gdouble r,g,b;
	gboolean set;

	GimpRGB color;
	set = TRUE;

	n = MIN(curve->_number,vector_curve->_number);

	r = g_random_double();
	g = g_random_double();
	b = g_random_double();
	gimp_rgb_set(&color,r,g,b);
	set = set && gimp_palette_set_foreground(&color);

	for(i=0;i<n;i++){
		end_points[0] = curve->points[i].x;
		end_points[1] = curve->points[i].y;
		end_points[2] = curve->points[i].x + multiple * vector_curve->points[i].x;
		end_points[3] = curve->points[i].y + multiple * vector_curve->points[i].y;
		gimp_pencil(drawable->drawable_id,4,end_points);
	}

	r = 0;
	g = 0;
	b = 0;
	gimp_rgb_set(&color,r,g,b);
	set = set && gimp_palette_set_foreground(&color);

	for(i=0;i<n;i++){
		end_points[0] = curve->points[i].x;
		end_points[1] = curve->points[i].y;
		end_points[2] = curve->points[i].x + multiple/3 * vector_curve->points[i].x;
		end_points[3] = curve->points[i].y + multiple/3 * vector_curve->points[i].y;
		gimp_pencil(drawable->drawable_id,4,end_points);
	}
	return set;
}