/* 
GIMP Plug-in Laso
Ondrej Fiala

image.c

extraction informations from image
*/


#include <stdio.h>

#include <libgimp/gimp.h>

/* translations strings */
#include "config.h"
#include "plugin-intl.h"

#include "image.h"

/* macros for checking arrays bounds */
#define BW(value)  (value>(BOUNDS_WIDTH-1)?(BOUNDS_WIDTH-1):(value<0?0:value))
#define BH(value)  (value>(BOUNDS_HEIGHT-1)?(BOUNDS_HEIGHT-1):(value<0?0:value))

guint** image_intensity_field(GimpDrawable *drawable)
{
	GimpPixelRgn src_pr;
	gint width, height;
	gint i,j;
	guint k;
	guint** in;
	guint bpp;

	guchar *row, *cr;

	width = drawable->width;
	height = drawable->height;

	/* remove alfa if exists */
	if (gimp_drawable_has_alpha(drawable->drawable_id)){
		gimp_layer_flatten(drawable->drawable_id);
	}

	bpp = drawable->bpp;

	/* corvention OK, or image was grayscale before */
	if(gimp_drawable_is_indexed(drawable->drawable_id)){
		return NULL;
	}

	/* image intensity field */
	in = g_malloc(width * sizeof(guint *));
	for(i = 0; i < width; i++){
		in[i] = g_malloc(height * sizeof(guint));
	}

	row = g_malloc(((width + 2) * bpp) * sizeof(guchar *));
	cr = row + 1;

	/* for different GIMP version
	gimp_tile_cache_ntiles (bpp * 2 * (drawable->width / gimp_tile_width () + 1)); */
	gimp_tile_cache_ntiles (2 * (width / gimp_tile_width () + 1));	
	gimp_pixel_rgn_init (&src_pr, drawable, 0, 0, width, height, FALSE, FALSE);


	for (j=0; j<height; j++) {
		gimp_pixel_rgn_get_row (&src_pr, cr, 0, j, width);
		for (i=0; i<width; i++) {
			in[i][j] = 0;
			/* average color intensity */
			for (k=0; k<bpp; k++) {
				in[i][j] = in[i][j] + cr[i * bpp + k];
			}
			in[i][j] = (in[i][j] / bpp);
		}
	}

	g_free(row);

	return in;
}

/* reads image intensity function */
gdouble** image_intensity_field_normalized(GimpDrawable *drawable)
{
	GimpPixelRgn src_pr;
	gint width, height;
	gint i,j;
	guint k;
	gdouble** in;
	gdouble min, max;
	guint bpp;

	guchar *row, *cr;

	min = Inf;
	max = -Inf;
	width = drawable->width;
	height = drawable->height;

	/* remove alfa if exists */
	if (gimp_drawable_has_alpha(drawable->drawable_id)){
		gimp_layer_flatten(drawable->drawable_id);
	}

	bpp = drawable->bpp;

	/* corvention OK, or image was grayscale before */
	if(gimp_drawable_is_indexed(drawable->drawable_id)){
		return NULL;
	}

	/* image intensity field */
	in = g_malloc(width * sizeof(gdouble *));
	for(i = 0; i < width; i++){
		in[i] = g_malloc(height * sizeof(gdouble));
	}
	row = g_malloc(((width + 2) * bpp) * sizeof(guchar *));
	cr = row + 1;

/* for different GIMP version
	gimp_tile_cache_ntiles (2 * (width / gimp_tile_width () + 1)); */
	gimp_tile_cache_ntiles ((width / gimp_tile_width () + 1));	

	gimp_pixel_rgn_init (&src_pr, drawable, 0, 0, width, height, FALSE, FALSE);

	for (j=0; j<height; j++) {
		gimp_pixel_rgn_get_row (&src_pr, cr, 0, j, width);
		for (i=0; i<width; i++) {
			in[i][j] = 0;
			/* average color intensity */
			for (k=0; k<bpp; k++) {
				in[i][j] = in[i][j] + cr[i * bpp + k];
			}
			in[i][j] = in[i][j] / bpp;
			min = MIN(min,in[i][j]);
			max = MAX(max,in[i][j]);
		}
	}

	/* normalization */
	for (j=0; j<height; j++) {
		for (i=0; i<width; i++) {
			in[i][j] = (in[i][j]-min)/(max-min); 
		}
	}
	g_free (row);

	return in;
}

/* compute image gradient field from image intensity function */
Point** image_gradient_field(gdouble** image_field_normalized, GimpDrawable *drawable)
{
	Point** gf;
	gdouble** ifn;
	gint width, height;
	gint i,j;

	width = drawable->width;
	height = drawable->height;

	/* rename - shortcut */
	ifn = image_field_normalized;

	/* image gradient field */
	gf = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		gf[i] = g_malloc(height * sizeof(Point));
	}

	/* corners */
	gf[0][0].x = gf[width-1][0].x = gf[0][height-1].x = gf[width-1][height-1].x = 0;
	gf[0][0].y = gf[width-1][0].y = gf[0][height-1].y = gf[width-1][height-1].y = 0;

	/* borders - rows */
	for(i=1; i<(width-1); i++){
		gf[i][0].x = (ifn[i+1][0] - ifn[i-1][0]) / 2;
		gf[i][height-1].x = (ifn[i+1][height-1] - ifn[i-1][height-1]) / 2;
		gf[i][0].y = gf[i][height-1].y = 0;  
	}
	/* borders - cols */
	for(j=1; j<(height-1); j++){
		gf[0][j].x = gf[width-1][j].x = 0;  
		gf[0][j].y = (ifn[0][j+1] - ifn[0][j-1]) / 2;
		gf[width-1][j].y = (ifn[width-1][j+1] - ifn[width-1][j-1]) / 2;
	}
	/* gradient */
	for (j=1; j<(height-1); j++) {
		for (i=1; i<(width-1); i++) {
			gf[i][j].x = (ifn[i+1][j] - ifn[i-1][j]) / 2;
			gf[i][j].y = (ifn[i][j+1] - ifn[i][j-1]) / 2;
		}
	}

	return gf;
}


Point** image_gvf(Point** image_gradient_normalized, gint iterations, gdouble mu, GimpDrawable *drawable)
{
	Point** gvf, **l, **c; // laplace
	gdouble **b;
	gint width, height;
	gint i,j,k;
	Point** ign;

	width = drawable->width;
	height = drawable->height;

	gimp_progress_init (_("Gradient Vector Field"));

	/* rename - shortcut */
	ign = image_gradient_normalized;

	/* image vector field */
	gvf = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		gvf[i] = g_malloc(height * sizeof(Point));
	}
	/* laplace field */
	l = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		l[i] = g_malloc(height * sizeof(Point));
	}
	/* field */
	c = g_malloc(width * sizeof(Point *));
	for(i = 0; i < width; i++){
		c[i] = g_malloc(height * sizeof(Point));
	}
	/* field */
	b = g_malloc(width * sizeof(gdouble *));
	for(i = 0; i < width; i++){
		b[i] = g_malloc(height * sizeof(gdouble));
	}

	/* INITIALIZATIONS */
	for (j=0; j<height; j++) {
		for (i=0; i<width; i++) {
			gvf[i][j] = ign[i][j];
			b[i][j] = gvf[i][j].x * gvf[i][j].x + gvf[i][j].y * gvf[i][j].y; 
			c[i][j].x = b[i][j] * gvf[i][j].x;
			c[i][j].y = b[i][j] * gvf[i][j].y;
		}
	}

	for(k=0;k<iterations;k++){

		/* Laplace operator */
		/* corners */	
		l[0][0].x = (gvf[0][1].x + gvf[1][0].x) / 2 - gvf[0][0].x;
		l[0][0].y = (gvf[0][1].y + gvf[1][0].y) / 2 - gvf[0][0].y;
		l[width-1][0].x = (gvf[width-2][0].x + gvf[width-1][1].x) / 2 - gvf[width-1][0].x;
		l[width-1][0].y = (gvf[width-2][0].y + gvf[width-1][1].y) / 2 - gvf[width-1][0].y;
		l[0][height-1].x = (gvf[1][height-1].x + gvf[0][height-2].x) / 2 - gvf[0][height-1].x;
		l[0][height-1].y = (gvf[1][height-1].y + gvf[0][height-2].y) / 2 - gvf[0][height-1].y;
		l[width-1][height-1].x = (gvf[width-2][height-1].x + gvf[width-1][height-2].x) / 2 - gvf[width-1][height-1].x;
		l[width-1][height-1].y = (gvf[width-2][height-1].y + gvf[width-1][height-2].y) / 2 - gvf[width-1][height-1].y;
		/* borders - rows */
		for(i=1; i<(width-1); i++){
			l[i][0].x = (2*gvf[i][1].x + gvf[i-1][0].x + gvf[i+1][0].x) / 4 - gvf[i][0].x;
			l[i][0].y = (2*gvf[i][1].y + gvf[i-1][0].y + gvf[i+1][0].y) / 4 - gvf[i][0].y;
			l[i][height-1].x = (2*gvf[i][height-2].x + gvf[i-1][height-1].x + gvf[i+1][height-1].x) / 4 - gvf[i][height-1].x;
			l[i][height-1].y = (2*gvf[i][height-2].y + gvf[i-1][height-1].y + gvf[i+1][height-1].y) / 4 - gvf[i][height-1].y;
		}
		/* borders - cols */
		for(j=1; j<(height-1); j++){
			l[0][j].x = (2*gvf[1][j].x + gvf[0][j-1].x + gvf[0][j+1].x) / 4 - gvf[0][j].x;
			l[0][j].y = (2*gvf[1][j].y + gvf[0][j-1].y + gvf[0][j+1].y) / 4 - gvf[0][j].y;
			l[width-1][j].x = (2*gvf[width-2][j].x + gvf[width-1][j-1].x + gvf[width-1][j+1].x) / 4 - gvf[width-1][j].x;
			l[width-1][j].y = (2*gvf[width-2][j].y + gvf[width-1][j-1].y + gvf[width-1][j+1].y) / 4 - gvf[width-1][j].y;
		}
		/* inner */
		for (j=1; j<(height-1); j++) {
			for (i=1; i<(width-1); i++) {
				l[i][j].x = (gvf[i-1][j].x + gvf[i+1][j].x + gvf[i][j-1].x + gvf[i][j+1].x) / 4 - gvf[i][j].x;
				l[i][j].y = (gvf[i-1][j].y + gvf[i+1][j].y + gvf[i][j-1].y + gvf[i][j+1].y) / 4 - gvf[i][j].y;
			}
		}

		/* GVF LOOP */
		for (j=0; j<height; j++) {
			for (i=0; i<width; i++) {
				gvf[i][j].x = (1-b[i][j]) * gvf[i][j].x + 4 * mu * l[i][j].x + c[i][j].x;
				gvf[i][j].y = (1-b[i][j]) * gvf[i][j].y + 4 * mu * l[i][j].y + c[i][j].y;
			}
		}


		if ((iterations % 10) == 0)
			gimp_progress_update ((gdouble) k / (gdouble) (iterations));
	}


/*
	for (j=0; j<height; j++) {
		for (i=0; i<width; i++) {
			printf("%0.2f\t",gvf[i][j].x);
		}
		printf("\n");
	}
	printf("\n");
*/

	g_free(l);
	g_free(c);
	g_free(b);

	return gvf;
}


void image_get_bound_points(gint *num_points, gdouble **points, GimpDrawable *drawable){
	gint i,j;
	gdouble width_1, height_1;

	width_1 = (gdouble)(drawable->width-1);
	height_1 = (gdouble)(drawable->height-1);

	*num_points = 24; // 4 * 3 points ; point(x,y)

	*points = g_malloc(*num_points * sizeof(gdouble));

	(*points)[1] = 5.0;

	i = 0;
	for(j=0;j<3;j++){
		(*points)[i*6 + j*2] = 0.0;
		(*points)[i*6 + j*2 + 1] = 0.0;
	}
	i++;
	for(j=0;j<3;j++){
		(*points)[i*6 + j*2] = width_1;
		(*points)[i*6 + j*2 + 1] = 0.0;
	}
	i++;
	for(j=0;j<3;j++){
		(*points)[i*6 + j*2] = width_1;
		(*points)[i*6 + j*2 + 1] = height_1;
	}
	i++;
	for(j=0;j<3;j++){
		(*points)[i*6 + j*2] = 0.0;
		(*points)[i*6 + j*2 + 1] = height_1;
	}

}

