/* 
GIMP Plug-in Laso
Ondrej Fiala

main.c

pouzit Plug-in template - Michael Natterer
*/

#include <string.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

/* translations strings */
#include "config.h"
#include "plugin-intl.h"

#include "main.h"

#include "interface.h"
#include "render.h"

#if DEBUGGING
#include <time.h>
#endif

/*  Constants  */

#define PROCEDURE_NAME   "laso-plug-in"
#define PROCEDURE_NAME_ADVANCED   "laso-plug-in-advanced"

#define DATA_KEY_VALS    "laso"
#define DATA_KEY_UI_VALS "laso_ui"

#define PARASITE_KEY     "laso-options"


/*  Local function prototypes  */

static void   query (void);
static void   run   (const gchar      *name,
					 gint              nparams,
					 const GimpParam  *param,
					 gint             *nreturn_vals,
					 GimpParam       **return_vals);


/*  Local variables  */

const PlugInVals default_vals =
{		// inicializovane hodnoty ovladacich prvku
	500,	// iterations
	1,		// image force (choose)
	40,		// gvf_iterations
	0.2,	// gvf_noise
	1.0,	// rounding weight
	0.7,
	2,		// image weight
	1,		// stopping
	1,		// iterations_force
	1,		// auto_iterations
	0,		// gauss blur
	5.0,	// blur radius
	0,		// export path
};

const PlugInImageVals default_image_vals =
{
	0
};

const PlugInDrawableVals default_drawable_vals =
{
	0
};

const PlugInUIVals default_ui_vals =
{
	TRUE
};

static PlugInVals         vals;
static PlugInImageVals    image_vals;
static PlugInDrawableVals drawable_vals;
static PlugInUIVals       ui_vals;


GimpPlugInInfo PLUG_IN_INFO =
{
	NULL,  /* init_proc  */
	NULL,  /* quit_proc  */
	query, /* query_proc */
	run,   /* run_proc   */
};

MAIN ()

static void
query (void)
{
	gchar *help_path;
	gchar *help_uri;

	static GimpParamDef args[] =
	{
		{ GIMP_PDB_INT32,    "run_mode",	"Interactive, non-interactive"	},
		{ GIMP_PDB_IMAGE,    "image",		"Input image"					},
		{ GIMP_PDB_DRAWABLE, "drawable",	"Input drawable" 				},
		{ GIMP_PDB_INT32,    "iterations",	"Iterations"					},
		{ GIMP_PDB_INT32,    "image_force",	"Image force"					},
		{ GIMP_PDB_INT32,    "gvf_iterations",	"GVF iterations"			},
		{ GIMP_PDB_FLOAT,    "gvf_noise",	"GVF noise parameter"			},
		{ GIMP_PDB_FLOAT,    "tension",		"Tension weight"				},
		{ GIMP_PDB_FLOAT,    "flexure",		"Flexure weight"				},
		{ GIMP_PDB_FLOAT,    "image_weight",	"Image force weight"		},
		{ GIMP_PDB_INT32,    "stopping",	"Edge-stopping force"			},
		{ GIMP_PDB_INT32,    "iterations_force",	"Iterations force"		},
		{ GIMP_PDB_INT32,    "auto_iterations",	"Auto iterations"			},
		{ GIMP_PDB_INT32,    "gaussian_blur",	"Gaussian blur"				},
		{ GIMP_PDB_FLOAT,    "blur_radius",	"Gaussian blur radius"			},
		{ GIMP_PDB_INT32,    "export_path",	"Export final path"				}
	};

	// help registration - (PLUGIN_NAME, LOCALEDIR)
	gimp_plugin_domain_register (GETTEXT_PACKAGE, LOCALEDIR);


	help_path = g_build_filename (gimp_data_directory(), DATADIR, "help", NULL);
	help_uri = g_filename_to_uri (help_path, NULL, NULL);
	g_free (help_path);

	gimp_plugin_help_register ("http://laso.strevlik.net/help",
		help_uri);

	gimp_install_procedure (PROCEDURE_NAME,
		"Laso Selection Tool",
		"Select object at image",
		"Ondrej Fiala <ofiala@mujmail.cz>",
		"Ondrej Fiala <ofiala@mujmail.cz>",
		PLUGIN_VERSION,
		N_("Laso plugin"),  // nazev polozky v menu
		"RGB*, GRAY*, INDEXED*",
		GIMP_PLUGIN,
		G_N_ELEMENTS (args), 0,
		args, NULL);

	gimp_install_procedure (PROCEDURE_NAME_ADVANCED,
		"Laso Selection Tool",
		"Select object at image (with advanced menu)",
		"Ondrej Fiala <ofiala@mujmail.cz>",
		"Ondrej Fiala <ofiala@mujmail.cz>",
		PLUGIN_VERSION,
		N_("Laso plugin"),  // nazev polozky v menu
		"RGB*, GRAY*, INDEXED*",
		GIMP_PLUGIN,
		G_N_ELEMENTS (args), 0,
		args, NULL);

	gimp_plugin_menu_register (PROCEDURE_NAME, "<Image>/Select");

	gimp_plugin_menu_register (PROCEDURE_NAME_ADVANCED, "<Image>/Tools/Selection Tools");
	// neukoncovat adresu lomtkem. vnitrni cesta GIMPU by pak obsahovala dve

	gimp_plugin_icon_register (PROCEDURE_NAME, GIMP_ICON_TYPE_STOCK_ID,
		(const guint8 *) GIMP_STOCK_TOOL_FREE_SELECT);

	gimp_plugin_icon_register (PROCEDURE_NAME_ADVANCED, GIMP_ICON_TYPE_STOCK_ID,
		(const guint8 *) GIMP_STOCK_TOOL_FREE_SELECT);

}

static void
run (const gchar      *name,
	 gint              n_params,
	 const GimpParam  *param,
	 gint             *nreturn_vals,
	 GimpParam       **return_vals)
{
	static GimpParam   values[1];
	GimpDrawable      *drawable;
	gint32             image_ID;
	GimpRunMode        run_mode;
	GimpPDBStatusType  status = GIMP_PDB_SUCCESS;
	gboolean		   dialog_return, run_advanced;

	*nreturn_vals = 1;
	*return_vals  = values;

	/*  Initialize i18n support  */
	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
	textdomain (GETTEXT_PACKAGE);

	run_mode = param[0].data.d_int32;
	image_ID = param[1].data.d_int32;
	if (image_ID < 0) {
		return;
    }

	run_advanced = (strcmp (name, PROCEDURE_NAME_ADVANCED) == 0);
	drawable = gimp_drawable_get (param[2].data.d_drawable);

	/*  Initialize with default values  */
	vals          = default_vals;
	image_vals    = default_image_vals;
	drawable_vals = default_drawable_vals;
	ui_vals       = default_ui_vals;

	if ((strcmp (name, PROCEDURE_NAME)== 0)||run_advanced)
	{
		switch (run_mode)
		{
		case GIMP_RUN_NONINTERACTIVE:
			if (n_params != 16)
			{
				status = GIMP_PDB_CALLING_ERROR;
			}
			else
			{
				vals.iterations		= param[3].data.d_int32;
				vals.image_force	= param[4].data.d_int32;
				vals.gvf_iterations	= param[5].data.d_int32;
				vals.gvf_noise		= param[6].data.d_float;
				vals.flexure		= param[7].data.d_float;
				vals.tension		= param[8].data.d_float;
				vals.image_weight	= param[9].data.d_float;
				vals.stopping		= param[10].data.d_int32;
				vals.iterations_force	= param[11].data.d_int32;
				vals.auto_iterations	= param[12].data.d_int32;
				vals.gaussian_blur	= param[13].data.d_int32;
				vals.blur_radius	= param[14].data.d_float;
				vals.export_path	= param[15].data.d_int32;
			}
			break;

		case GIMP_RUN_INTERACTIVE:
			/*  Possibly retrieve data  */
			gimp_get_data (DATA_KEY_VALS,    &vals);
			gimp_get_data (DATA_KEY_UI_VALS, &ui_vals);
			if(run_advanced){
				if (! dialog (image_ID, drawable,
					&vals, &image_vals, &drawable_vals, &ui_vals))
				{
					status = GIMP_PDB_CANCEL;
				}
			} else {
				status = GIMP_PDB_SUCCESS;
			}
			break;

		case GIMP_RUN_WITH_LAST_VALS:
			/*  Possibly retrieve data  */
			gimp_get_data (DATA_KEY_VALS, &vals);

			break;

		default:
			break;
		}
	}
	else
	{
		status = GIMP_PDB_CALLING_ERROR;
	}

	if (status == GIMP_PDB_SUCCESS)
	{	
		/* depends on GIMP version - left for future
		render (image_ID, drawable, &vals, &image_vals, &drawable_vals, &ui_vals);
		*/
		render (image_ID, drawable, &vals, &image_vals, &drawable_vals);

		if (run_mode != GIMP_RUN_NONINTERACTIVE)
			gimp_displays_flush ();

		if (run_mode == GIMP_RUN_INTERACTIVE)
		{
			gimp_set_data (DATA_KEY_VALS,    &vals,    sizeof (vals));
			gimp_set_data (DATA_KEY_UI_VALS, &ui_vals, sizeof (ui_vals));
		}

		gimp_drawable_detach (drawable);
	}

	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = status;
}
