/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include <gtk/gtk.h>

#include <libgimp/gimp.h>

#include "main.h"
#include "render.h"
#include "ring.h"

#include "plugin-intl.h"


static void interpolate(GimpDrawable *drawable, GimpDrawable *sel, GimpDrawable *mask, PlugInVals *opts, gint image_id);
inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres);
static void slopes(GimpDrawable *drawable, GimpDrawable *mask,
                   gint left, gint top, gint right, gint bottom,
                   gboolean halfsel, gboolean leastsq,
                   gint window, gfloat *kernel, gfloat **pdx, gfloat **pdy);
inline static void gray_pixel(GimpImageType type, guchar *pixel);
//inline static void find_center(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel, guchar *pixel,
//                               gfloat *pcenterx, gfloat *outcy, gfloat *outcz);
inline static void old_slope(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel, guchar *center,
                             gfloat *dxpixel, gfloat *dypixel);
//inline static void old_slope2(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel,
//                              gfloat *pcenterx, gfloat *pcentery, gfloat *centerz, gfloat *dxpixel, gfloat *dypixel);
inline static void least_squares_slope(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel,
                                       //gfloat *pcenterx, gfloat *pcentery, gfloat *centerz,
                                       guchar *center, gfloat *dxpixel, gfloat *dypixel);
inline static void flat_slope(GimpImageType type, gfloat *dxpixel, gfloat *dypixel);
static gfloat *make_kernel(InterpolateKernelType type, gint window, gfloat s);
inline static gfloat gaussian(gfloat x, gfloat s);


void render(gint32 image_ID, GimpDrawable *drawable, PlugInVals *vals) {
	gimp_image_undo_group_start(image_ID);


	/*
	 * get input mask (first channel found, or selection)
	 */
	gint mask_id = 0;
	GimpDrawable *mask = 0;

	if (vals->usesel) {
		gimp_selection_invert(image_ID);
		mask_id = gimp_selection_save(image_ID);
		gimp_selection_invert(image_ID);
		mask = gimp_drawable_get(mask_id);
	} else {
		mask = gimp_drawable_get(vals->chanid);
	}


	/*
	 * save selection to channel, so we can modify it
	 */
	gint sel_id = gimp_selection_save(image_ID);
	GimpDrawable *sel = gimp_drawable_get(sel_id);


	/*
	 * interpolate
	 */
	interpolate(drawable, sel, mask, vals, image_ID);


	/*
	 * no more need for temporary selection channel
	 */
	gimp_drawable_detach(sel);
	if (vals->changesel) {
		gimp_selection_load(sel_id);
	}
	gimp_image_remove_channel(image_ID, sel_id);


	/*
	 * remove input mask channel if it is our temporary
	 */
	gimp_drawable_detach(mask);
	if (mask_id) {
		gimp_image_remove_channel(image_ID, mask_id);
	}


	gimp_image_undo_group_end(image_ID);
}


static void interpolate(GimpDrawable *drawable, GimpDrawable *sel, GimpDrawable *mask, PlugInVals *opts, gint image_id) {
	gint x,y;
	gint deltay, deltax;

	gint window = opts->window;
	gboolean wslopes = opts->slopes;


	gint left,top,right,bottom,width,height;
	gimp_drawable_mask_bounds(drawable->drawable_id, &left, &top, &right, &bottom);
	width = right-left;
	height = bottom-top;

	GimpImageType drawable_type = gimp_drawable_type(drawable->drawable_id);
	gint channels = gimp_drawable_bpp(drawable->drawable_id);
	gint mchannels = gimp_drawable_bpp(mask->drawable_id);

	GimpDrawable *outdrawable, *outslopesdrawable;
	gboolean newlayer = opts->newlayer;
	if (newlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Interpolated"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_NORMAL_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outdrawable = gimp_drawable_get(layer_id);
	} else {
		outdrawable = drawable;
	}

	GimpImageType outdrawable_type = gimp_drawable_type(outdrawable->drawable_id);
	gint outchannels = gimp_drawable_bpp(outdrawable->drawable_id);
	gint outslopeschannels;

	gboolean linnewlayer = opts->linnewlayer;
	if (linnewlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Slopes Offset"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_GRAIN_MERGE_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outslopesdrawable = gimp_drawable_get(layer_id);
		outslopeschannels = gimp_drawable_bpp(outslopesdrawable->drawable_id);
	} else {
		outslopesdrawable = 0;
		outslopeschannels = outchannels;
	}


	gint inleft = MAX(left-window,0);
	gint intop = MAX(top-window,0);
	gint inwidth = MIN(right+window,drawable->width)-inleft;
	gint inheight = MIN(bottom+window,drawable->height)-intop;


	/*
	 * calculate slopes of sample areas
	 */
	gfloat *dx=0, *dy=0;
	if (wslopes) {
		gfloat *linkernel = make_kernel(opts->linkernel,opts->linwindow,opts->linsigma);
		slopes(drawable, mask,
		       inleft, intop, inleft+inwidth, intop+inheight,
		       opts->halfsel, opts->leastsq,
		       opts->linwindow, linkernel, &dx, &dy);
		g_free(linkernel);
	}

	gint tw = gimp_tile_width();
	gimp_tile_cache_ntiles(2*(drawable->width+tw-1)/tw);


	/*
	 * make kernel layer (brush?)
	 */
	gint kernel_width = (1+2*window);
	gfloat *kernel = make_kernel(opts->kernel,window,opts->sigma);


	/*
	 * get input pixel regions; fill rings
	 */
	GimpPixelRgn in;
	gimp_pixel_rgn_init(&in, drawable, inleft, intop, inwidth, inheight, FALSE, FALSE);

	GimpPixelRgn min;
	gimp_pixel_rgn_init(&min, mask, inleft, intop, inwidth, inheight, FALSE, FALSE);

	gint in_ring = window;
	gint inrow_bytes = channels*inwidth;
	guchar *inrows = g_new(guchar, kernel_width*inrow_bytes);
	gint inmrow_bytes = mchannels*inwidth;
	guchar *inmrows = g_new(guchar, kernel_width*inmrow_bytes);
	for (y = MAX(top-window,0) ;
	     y <= MIN(top+window,drawable->height-1) ; ++y) {
		gimp_pixel_rgn_get_row(&in,
			inrows + (in_ring+y-top)*inrow_bytes,
			inleft, y, inwidth
		);
		gimp_pixel_rgn_get_row(&min,
			inmrows + (in_ring+y-top)*inmrow_bytes,
			inleft, y, inwidth
		);
	}

	GimpPixelRgn ins;
	gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
	guchar *insrow = g_new(guchar, mchannels*width);


	/*
	 * get output pixel regions
	 */
	GimpPixelRgn out;
	if (newlayer) {
		gimp_pixel_rgn_init(&out, outdrawable, 0, 0, width, height, TRUE, TRUE);
	} else {
		gimp_pixel_rgn_init(&out, outdrawable, left, top, width, height, TRUE, TRUE);
	}
	guchar *outrow = g_new(guchar, outchannels*width);

	GimpPixelRgn outslopes;
	if (linnewlayer) {
		gimp_pixel_rgn_init(&outslopes, outslopesdrawable, 0, 0, width, height, TRUE, TRUE);
	}
	guchar *outslopesrow = g_new(guchar, outslopeschannels*width);

	GimpPixelRgn outs;
	if (opts->changesel) {
		gimp_pixel_rgn_init(&outs, sel, left, top, width, height, TRUE, TRUE);
	}
	guchar *outsrow = g_new(guchar, mchannels*width);


	/*
	 * do the interpolation
	 */
	gimp_progress_init(_("Interpolating..."));
	for (y = top ; y < bottom ; ++y) {
		guchar *inmrow = inmrows + ( (in_ring%kernel_width)*inmrow_bytes );
		gimp_pixel_rgn_get_row(&ins, insrow, left, y, width);

		for (x = left ; x < right ; ++x) {
			guchar *inmpixel = inmrow + (x-inleft)*mchannels;
			guchar *inspixel = insrow + (x-left)*mchannels;
			guchar *outpixel = outrow + (x-left)*outchannels;
			guchar *outslopespixel = outslopesrow + (x-left)*outslopeschannels;
			guchar *outspixel = outsrow + (x-left)*mchannels;  // opts->changesel?

			gfloat sum[4];
			sum[0] = 0.0;
			sum[1] = 0.0;
			sum[2] = 0.0;
			sum[3] = 0.0;
			gfloat dsum[4];
			dsum[0] = 0.0;
			dsum[1] = 0.0;
			dsum[2] = 0.0;
			dsum[3] = 0.0;
			gfloat ws = 0.0;

			gint deltay_min = MAX(-window,-y);
			gint deltay_max = MIN(window,drawable->height-1-y);
			for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
				guchar *inrow2 = inrows + ( ((in_ring+deltay)%kernel_width)*inrow_bytes );
				guchar *inmrow2 = inmrows + ( ((in_ring+deltay)%kernel_width)*inmrow_bytes );
				gfloat *dxrow = 0;
				gfloat *dyrow = 0;

				gint deltax_min = MAX(-window,-x);
				gint deltax_max = MIN(window,drawable->width-1-x);

				if (wslopes) {
					dxrow = dx + channels*inwidth*(y+deltay-intop);
					dyrow = dy + channels*inwidth*(y+deltay-intop);
				}

				for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
					guchar *inpixel2 = inrow2 + (x+deltax-inleft)*channels;
					guchar *inmpixel2 = inmrow2 + (x+deltax-inleft)*mchannels;

					gfloat w = kernel[kernel_width*(window+deltay) + window+deltax];

					if ( ! ( opts->halfsel ? (inmpixel2[0]>0) : (inmpixel2[0]==255) ) ) {
						continue;
					}

					if (w<0.0) {
						/*
						 * -1 means infinity. this input pixel has infinite weight
						 * (center pixel in 'inverse' kernels)
						 */
						ws = opts->highthres;
						switch (drawable_type) {
						case GIMP_RGBA_IMAGE:
							sum[3] = opts->highthres*inpixel2[3];
						case GIMP_RGB_IMAGE:
							sum[2] = opts->highthres*inpixel2[2];
						case GIMP_GRAYA_IMAGE:
							sum[1] = opts->highthres*inpixel2[1];
						case GIMP_GRAY_IMAGE: default:
							sum[0] = opts->highthres*inpixel2[0];
							break;
						}
						dsum[0] = 0.0;
						dsum[1] = 0.0;
						dsum[2] = 0.0;
						dsum[3] = 0.0;
						goto breakout;

					} else {
						/*
						 * ordinary input pixel
						 */
						if (opts->halfsel)
							w *= inmpixel2[0];

						switch (drawable_type) {
						case GIMP_RGBA_IMAGE:
							sum[3] += inpixel2[3]*w;
						case GIMP_RGB_IMAGE:
							sum[2] += inpixel2[2]*w;
						case GIMP_GRAYA_IMAGE:
							sum[1] += inpixel2[1]*w;
						case GIMP_GRAY_IMAGE: default:
							sum[0] += inpixel2[0]*w;
							break;
						}

						if (wslopes) {
							gfloat *dxpixel = dxrow + (x+deltax-inleft)*channels;
							gfloat *dypixel = dyrow + (x+deltax-inleft)*channels;

							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:
								dsum[3] += ( - dxpixel[3]*deltax - dypixel[3]*deltay)*w;
							case GIMP_RGB_IMAGE:
								dsum[2] += ( - dxpixel[2]*deltax - dypixel[2]*deltay)*w;
							case GIMP_GRAYA_IMAGE:
								dsum[1] += ( - dxpixel[1]*deltax - dypixel[1]*deltay)*w;
							case GIMP_GRAY_IMAGE: default:
								dsum[0] += ( - dxpixel[0]*deltax - dypixel[0]*deltay)*w;
								break;
							}
						}

						ws += w;
					}
				}
			}

			breakout:

			/*
			 * now we can calculate the average, to set the output pixel
			 */
			if (wslopes) {
				if (linnewlayer) {

					if (ws>0.0) {
						switch (drawable_type) {
						case GIMP_GRAY_IMAGE: default:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = 255;
							break;
						case GIMP_GRAYA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							break;
						case GIMP_RGB_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/ws);
							outslopespixel[3] = 255;
							break;
						case GIMP_RGBA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/ws);
							outslopespixel[3] = CLAMP0255(128+dsum[3]/ws);
							break;
						}
					} else {
						gray_pixel(outdrawable_type, outslopespixel);
					}

				} else {
					sum[0] += dsum[0];
					sum[1] += dsum[1];
					sum[2] += dsum[2];
					sum[3] += dsum[3];
				}
			}

			if (ws>0.0) {
				switch (drawable_type) {
				case GIMP_GRAY_IMAGE: default:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					if (newlayer) {
						outpixel[1] = 255;
					}
					break;
				case GIMP_GRAYA_IMAGE:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					outpixel[1] = CLAMP0255(sum[1]/ws);
					break;
				case GIMP_RGB_IMAGE:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					outpixel[1] = CLAMP0255(sum[1]/ws);
					outpixel[2] = CLAMP0255(sum[2]/ws);
					if (newlayer) {
						outpixel[3] = 255;
					}
					break;
				case GIMP_RGBA_IMAGE:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					outpixel[1] = CLAMP0255(sum[1]/ws);
					outpixel[2] = CLAMP0255(sum[2]/ws);
					outpixel[3] = CLAMP0255(sum[3]/ws);
					break;
				}
			} else {
				/*
				 * there were no input pixels in range, so set pixel to gray
				 */
				gray_pixel(outdrawable_type, outpixel);
			}

			if (opts->changesel) {
				// input mask is multiplied with pixels without being divided by 255, so divide here
				if (opts->halfsel) ws /= 255.0;

				gfloat outsel = wsum2sel(ws, opts->lowthres, opts->highthres, opts->logthres) * inspixel[0];
				if (opts->multmask) outsel *= inmpixel[0]/255.0;
				outspixel[0] = outsel;
			}
		}

		if (newlayer) {
			gimp_pixel_rgn_set_row(&out,outrow,0,y-top,right-left);
		} else {
			gimp_pixel_rgn_set_row(&out,outrow,left,y,right-left);
		}
		if (linnewlayer) {
			gimp_pixel_rgn_set_row(&outslopes,outslopesrow,0,y-top,right-left);
		}
		if (opts->changesel) {
			gimp_pixel_rgn_set_row(&outs,outsrow,left,y,right-left);
		}


		/*
		 * rotate input row buffers, to move to the next row
		 */
		++in_ring;
		if ((y+1)+window < drawable->height) {
			gimp_pixel_rgn_get_row(&in,
				inrows + ((in_ring+window)%kernel_width)*inrow_bytes,
				inleft, (y+1)+window, inwidth
			);
			gimp_pixel_rgn_get_row(&min,
				inmrows + ((in_ring+window)%kernel_width)*inmrow_bytes,
				inleft, (y+1)+window, inwidth
			);
		}

		if (!(y%4)) {
			gimp_progress_update(1.0*(y-top+1)/(bottom-top));
		}
	}
	gimp_progress_end();


	/*
	 * done with the pixel regions, kernel and slopes
	 */
	g_free(outsrow);
	g_free(outslopesrow);
	g_free(outrow);

	g_free(inmrows);
	g_free(inrows);

	g_free(kernel);

	g_free(dx);
	g_free(dy);


	/*
	 * we can just flush here
	 */
	gimp_drawable_flush(outdrawable);
	gimp_drawable_merge_shadow(outdrawable->drawable_id, TRUE);
	if (newlayer) {
		gimp_drawable_update(outdrawable->drawable_id, 0, 0, width, height);
		gimp_drawable_detach(outdrawable);
	} else {
		gimp_drawable_update(outdrawable->drawable_id, left, top, width, height);
	}

	if (linnewlayer) {
		gimp_drawable_flush(outslopesdrawable);
		gimp_drawable_merge_shadow(outslopesdrawable->drawable_id, TRUE);
		gimp_drawable_update(outslopesdrawable->drawable_id, 0, 0, width, height);
		gimp_drawable_detach(outslopesdrawable);
	}


	/*
	 * only remove the selection if we are modifying it
	 */
	if (opts->changesel) {
		gint tmp_sel_id = gimp_selection_save(image_id);
		gimp_selection_none(image_id);

		gimp_drawable_flush(sel);
		gimp_drawable_merge_shadow(sel->drawable_id, TRUE);
		gimp_drawable_update(sel->drawable_id, left, top, width, height);

		gimp_selection_load(tmp_sel_id);
		gimp_image_remove_channel(image_id, tmp_sel_id);
	}
}


inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres) {
	if (logthres) w = log(w);

	gfloat ww;
	if (w<=lt) {
		ww = 0.0;
	} else if (w>=ht) {
		ww = 1.0;
	} else {
		ww = (w-lt)/(ht-lt);
	}
	return 1.0-ww;
}


static void slopes(GimpDrawable *drawable, GimpDrawable *mask,
                   gint left, gint top, gint right, gint bottom,
                   gboolean halfsel, gboolean leastsq,
                   gint window, gfloat *kernel, gfloat **pdx, gfloat **pdy) {
	gint width = right-left;
	gint height = bottom-top;

	Ring ring;
	ring_init(&ring, drawable, left, top, right, bottom, window);

	Ring mring;
	ring_init(&mring, mask, left, top, right, bottom, window);

	gfloat *dx = *pdx = g_new(gfloat, width*height*ring.bpp);
	gfloat *dy = *pdy = g_new(gfloat, width*height*ring.bpp);

	gimp_progress_init(_("Calculating slopes..."));
	for (; ring_loop(&ring) && ring_loop(&mring) ; ring_rotate(&ring), ring_rotate(&mring)) {
		guchar *row = ring_row(&ring,ring.y);
		guchar *mrow = ring_row(&mring,mring.y);
		gfloat *dxrow = dx + ring.bpp*width*(ring.y-top);
		gfloat *dyrow = dy + ring.bpp*width*(ring.y-top);

		gint x;
		for (x = left ; x < right ; ++x) {
			guchar *pixel = row + ( (x-left)*ring.bpp );  // center pixel
			guchar *mpixel = mrow + ( (x-left)*mring.bpp );
			gfloat *dxpixel = dxrow + ring.bpp*(x-left);
			gfloat *dypixel = dyrow + ring.bpp*(x-left);

			if (halfsel ? !mpixel[0] : mpixel[0]<255) {
				flat_slope(ring.type, dxpixel, dypixel);
			} else if (leastsq) {
				//gfloat centerx;
				//gfloat centery;
				//gfloat centerz[4];
				//find_center(&ring, &mring, x, kernel, window, halfsel, pixel, &centerx, &centery, centerz);
				least_squares_slope(&ring, &mring, x, kernel, window, halfsel, pixel, dxpixel, dypixel);
			} else {
				old_slope(&ring, &mring, x, kernel, window, halfsel, pixel, dxpixel, dypixel);
			}
		}

		if (!(ring.y%4)) {
			gimp_progress_update(1.0*(ring.y-top+1)/(bottom-top));
		}
	}
	gimp_progress_end();
	ring_fini(&ring);
}


inline static void gray_pixel(GimpImageType type, guchar *pixel) {
	switch (type) {
	case GIMP_RGBA_IMAGE:
		pixel[3] = 128;
	case GIMP_RGB_IMAGE:
		pixel[2] = 128;
	case GIMP_GRAYA_IMAGE:
		pixel[1] = 128;
	case GIMP_GRAY_IMAGE: default:
		pixel[0] = 128;
		break;
	}
}


#if 0
inline static void find_center(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel, guchar *pixel,
                               gfloat *pcenterx, gfloat *pcentery, gfloat *centerz) {
	gint y = ring->y;
	gfloat wsum = 0.0;
	gfloat xsum = 0.0;
	gfloat ysum = 0.0;
	gfloat zsum[4];
	zsum[0] = 0.0;
	zsum[1] = 0.0;
	zsum[2] = 0.0;
	zsum[3] = 0.0;

	gint y2_min = MAX(0,y-window);
	gint y2_max = MIN(y+window,ring->dr->height-1);
	gint y2;
	for (y2 = y2_min ; y2 <= y2_max ; ++y2) {
		guchar *row = ring_row(ring,y2);
		guchar *mrow = ring_row(mring,y2);

		gint x2_min = MAX(0,x-window);
		gint x2_max = MIN(x+window,ring->dr->width-1);
		gint x2;
		for (x2 = x2_min ; x2 <= x2_max ; ++x2) {
			guchar *mpixel = mrow + ( (x2-ring->left)*mring->bpp );
			if (halfsel ? mpixel[0]>0 : mpixel[0]==255) {
				gfloat w = kernel[(1+2*window)*(window+y2-y) + (window+x2-x)];
				if (halfsel) {
					w *= mpixel[0];
				}
				if (w>0) {
					wsum += w;
					xsum += w*(x2-x);
					ysum += w*(y2-y);

					guchar *pixel = row + ( (x2-ring->left)*ring->bpp );
					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						zsum[3] = w*pixel[3];
					case GIMP_RGB_IMAGE:
						zsum[2] = w*pixel[2];
					case GIMP_GRAYA_IMAGE:
						zsum[1] = w*pixel[1];
					case GIMP_GRAY_IMAGE: default:
						zsum[0] = w*pixel[0];
						break;
					}
				}
			}
		}
	}

	if (wsum>0) {
		*pcenterx = xsum/wsum;
		*pcentery = ysum/wsum;
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			centerz[3] = zsum[3]/wsum;
		case GIMP_RGB_IMAGE:
			centerz[2] = zsum[2]/wsum;
		case GIMP_GRAYA_IMAGE:
			centerz[1] = zsum[1]/wsum;
		case GIMP_GRAY_IMAGE: default:
			centerz[0] = zsum[0]/wsum;
			break;
		}
	} else {
		*pcenterx = 0;
		*pcentery = 0;
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			centerz[3] = pixel[3];
		case GIMP_RGB_IMAGE:
			centerz[2] = pixel[2];
		case GIMP_GRAYA_IMAGE:
			centerz[1] = pixel[1];
		case GIMP_GRAY_IMAGE: default:
			centerz[0] = pixel[0];
			break;
		}
	}
}
#endif


inline static void old_slope(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel, guchar *center,
                             gfloat *dxpixel, gfloat *dypixel) {
	gint y = ring->y;

	gfloat wsum = 0.0;
	gfloat dxsum[4];
	dxsum[0] = 0.0;
	dxsum[1] = 0.0;
	dxsum[2] = 0.0;
	dxsum[3] = 0.0;
	gfloat dysum[4];
	dysum[0] = 0.0;
	dysum[1] = 0.0;
	dysum[2] = 0.0;
	dysum[3] = 0.0;

	gint y2_min = MAX(0,y-window);
	gint y2_max = MIN(y+window,ring->dr->height-1);
	gint y2;
	for (y2 = y2_min ; y2 <= y2_max ; ++y2) {
		guchar *row = ring_row(ring,y2);
		guchar *mrow = ring_row(mring,y2);

		gint x2_min = MAX(0,x-window);
		gint x2_max = MIN(x+window,ring->dr->width-1);
		gint x2;
		for (x2 = x2_min ; x2 <= x2_max ; ++x2) {
			guchar *mpixel = mrow + ( (x2-ring->left)*mring->bpp );
			gfloat deltax = x2-x;
			gfloat deltay = y2-y;
			gfloat distsquared = (deltax*deltax + deltay*deltay);
			if ( distsquared>0.0 && (halfsel ? mpixel[0]>0 : mpixel[0]==255) ) {
				gfloat w = kernel[(1+2*window)*(window+y2-y) + (window+x2-x)];
				if (halfsel) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x2-ring->left)*ring->bpp );
					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						dxsum[3] += w*(pixel[3]-center[3])*deltax/distsquared;
						dysum[3] += w*(pixel[3]-center[3])*deltay/distsquared;
					case GIMP_RGB_IMAGE:
						dxsum[2] += w*(pixel[2]-center[2])*deltax/distsquared;
						dysum[2] += w*(pixel[2]-center[2])*deltay/distsquared;
					case GIMP_GRAYA_IMAGE:
						dxsum[1] += w*(pixel[1]-center[1])*deltax/distsquared;
						dysum[1] += w*(pixel[1]-center[1])*deltay/distsquared;
					case GIMP_GRAY_IMAGE: default:
						dxsum[0] += w*(pixel[0]-center[0])*deltax/distsquared;
						dysum[0] += w*(pixel[0]-center[0])*deltay/distsquared;
						break;
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = 2*dxsum[3]/wsum;
			dypixel[3] = 2*dysum[3]/wsum;
		case GIMP_RGB_IMAGE:
			dxpixel[2] = 2*dxsum[2]/wsum;
			dypixel[2] = 2*dysum[2]/wsum;
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = 2*dxsum[1]/wsum;
			dypixel[1] = 2*dysum[1]/wsum;
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = 2*dxsum[0]/wsum;
			dypixel[0] = 2*dysum[0]/wsum;
			break;
		}
	} else {
		flat_slope(ring->type, dxpixel, dypixel);
	}
}


#if 0
inline static void old_slope2(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel,
                             gfloat *pcenterx, gfloat *pcentery, gfloat *centerz, gfloat *dxpixel, gfloat *dypixel) {
	gint y = ring->y;

	gfloat wsum = 0.0;
	gfloat dxsum[4];
	dxsum[0] = 0.0;
	dxsum[1] = 0.0;
	dxsum[2] = 0.0;
	dxsum[3] = 0.0;
	gfloat dysum[4];
	dysum[0] = 0.0;
	dysum[1] = 0.0;
	dysum[2] = 0.0;
	dysum[3] = 0.0;

	gint y2_min = MAX(0,y-window);
	gint y2_max = MIN(y+window,ring->dr->height-1);
	gint y2;
	for (y2 = y2_min ; y2 <= y2_max ; ++y2) {
		guchar *row = ring_row(ring,y2);
		guchar *mrow = ring_row(mring,y2);

		gint x2_min = MAX(0,x-window);
		gint x2_max = MIN(x+window,ring->dr->width-1);
		gint x2;
		for (x2 = x2_min ; x2 <= x2_max ; ++x2) {
			guchar *mpixel = mrow + ( (x2-ring->left)*mring->bpp );
			gfloat deltax = x2-x-*pcenterx;
			gfloat deltay = y2-y-*pcentery;
			gfloat distsquared = (deltax*deltax + deltay*deltay);
			if ( distsquared>0.0 && (halfsel ? mpixel[0]>0 : mpixel[0]==255) ) {
				gfloat w = kernel[(1+2*window)*(window+y2-y) + (window+x2-x)];
				if (halfsel) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x2-ring->left)*ring->bpp );
					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						dxsum[3] += w*(pixel[3]-centerz[3])*deltax/distsquared;
						dysum[3] += w*(pixel[3]-centerz[3])*deltay/distsquared;
					case GIMP_RGB_IMAGE:
						dxsum[2] += w*(pixel[2]-centerz[2])*deltax/distsquared;
						dysum[2] += w*(pixel[2]-centerz[2])*deltay/distsquared;
					case GIMP_GRAYA_IMAGE:
						dxsum[1] += w*(pixel[1]-centerz[1])*deltax/distsquared;
						dysum[1] += w*(pixel[1]-centerz[1])*deltay/distsquared;
					case GIMP_GRAY_IMAGE: default:
						dxsum[0] += w*(pixel[0]-centerz[0])*deltax/distsquared;
						dysum[0] += w*(pixel[0]-centerz[0])*deltay/distsquared;
						break;
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = 2*dxsum[3]/wsum;
			dypixel[3] = 2*dysum[3]/wsum;
		case GIMP_RGB_IMAGE:
			dxpixel[2] = 2*dxsum[2]/wsum;
			dypixel[2] = 2*dysum[2]/wsum;
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = 2*dxsum[1]/wsum;
			dypixel[1] = 2*dysum[1]/wsum;
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = 2*dxsum[0]/wsum;
			dypixel[0] = 2*dysum[0]/wsum;
			break;
		}
	} else {
		flat_slope(ring->type, dxpixel, dypixel);
	}
}
#endif


inline static void least_squares_slope(Ring *ring, Ring *mring, gint x, gfloat *kernel, gint window, gboolean halfsel,
                                       //gfloat *pcenterx, gfloat *pcentery, gfloat *centerz,
                                       guchar *center, gfloat *dxpixel, gfloat *dypixel) {
	gint y = ring->y;

	gfloat ws = 0.0;

	gfloat m[2][2];
	m[0][0] = 0.0;
	m[0][1] = 0.0;
	m[1][0] = 0.0;
	m[1][1] = 0.0;

	gfloat v[4][2];
	v[0][0] = 0.0;
	v[0][1] = 0.0;
	v[1][0] = 0.0;
	v[1][1] = 0.0;
	v[2][0] = 0.0;
	v[2][1] = 0.0;
	v[3][0] = 0.0;
	v[3][1] = 0.0;

	gint y2_min = MAX(0,y-window);
	gint y2_max = MIN(y+window,ring->dr->height-1);
	gint y2;
	for (y2 = y2_min ; y2 <= y2_max ; ++y2) {
		guchar *row = ring_row(ring,y2);
		guchar *mrow = ring_row(mring,y2);

		gint x2_min = MAX(0,x-window);
		gint x2_max = MIN(x+window,ring->dr->width-1);
		gint x2;
		for (x2 = x2_min ; x2 <= x2_max ; ++x2) {
			guchar *mpixel = mrow + ( (x2-ring->left)*mring->bpp );
			gfloat deltax = x2-x;
			gfloat deltay = y2-y;
			if ( halfsel ? mpixel[0]>0 : mpixel[0]==255 ) {
				gfloat w = kernel[(1+2*window)*(window+y2-y) + (window+x2-x)];
				if (halfsel) {
					w *= mpixel[0];
				}
				if (w>0) {
					ws += w;

					guchar *pixel = row + ( (x2-ring->left)*ring->bpp );

					m[0][0] += deltax*deltax*w;
					m[0][1] += deltax*deltay*w;
					m[1][0] += deltay*deltax*w;
					m[1][1] += deltay*deltay*w;

					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						v[3][0] += deltax*(pixel[3]-center[3])*w;
						v[3][1] += deltay*(pixel[3]-center[3])*w;
					case GIMP_RGB_IMAGE:
						v[2][0] += deltax*(pixel[2]-center[2])*w;
						v[2][1] += deltay*(pixel[2]-center[2])*w;
					case GIMP_GRAYA_IMAGE:
						v[1][0] += deltax*(pixel[1]-center[1])*w;
						v[1][1] += deltay*(pixel[1]-center[1])*w;
					case GIMP_GRAY_IMAGE: default:
						v[0][0] += deltax*(pixel[0]-center[0])*w;
						v[0][1] += deltay*(pixel[0]-center[0])*w;
						break;
					}
				}
			}
		}
	}

	gfloat det;

	if (ws!=0) {
		m[0][0] /= ws;
		m[0][1] /= ws;
		m[1][0] /= ws;
		m[1][1] /= ws;
	
		v[0][0] /= ws;
		v[0][1] /= ws;
		v[1][0] /= ws;
		v[1][1] /= ws;
		v[2][0] /= ws;
		v[2][1] /= ws;
		v[3][0] /= ws;
		v[3][1] /= ws;

		det = m[0][0]*m[1][1] - m[1][0]*m[0][1];
	}

	if (ws!=0 && det!=0) {
		gfloat invm[2][2];
		invm[0][0] =  m[1][1]/det;
		invm[0][1] = -m[0][1]/det;
		invm[1][0] = -m[1][0]/det;
		invm[1][1] =  m[0][0]/det;

		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = invm[0][0]*v[3][0] + invm[0][1]*v[3][1];
			dypixel[3] = invm[1][0]*v[3][0] + invm[1][1]*v[3][1];
		case GIMP_RGB_IMAGE:
			dxpixel[2] = invm[0][0]*v[2][0] + invm[0][1]*v[2][1];
			dypixel[2] = invm[1][0]*v[2][0] + invm[1][1]*v[2][1];
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = invm[0][0]*v[1][0] + invm[0][1]*v[1][1];
			dypixel[1] = invm[1][0]*v[1][0] + invm[1][1]*v[1][1];
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = invm[0][0]*v[0][0] + invm[0][1]*v[0][1];
			dypixel[0] = invm[1][0]*v[0][0] + invm[1][1]*v[0][1];
			break;
		}
	} else {
		/*
		 * only one point, or colinear points
		 */
		flat_slope(ring->type, dxpixel, dypixel);
	}
}


inline static void flat_slope(GimpImageType type, gfloat *dxpixel, gfloat *dypixel) {
	switch (type) {
	case GIMP_RGBA_IMAGE:
		dxpixel[3] = 0.0;
		dypixel[3] = 0.0;
	case GIMP_RGB_IMAGE:
		dxpixel[2] = 0.0;
		dypixel[2] = 0.0;
	case GIMP_GRAYA_IMAGE:
		dxpixel[1] = 0.0;
		dypixel[1] = 0.0;
	case GIMP_GRAY_IMAGE: default:
		dxpixel[0] = 0.0;
		dypixel[0] = 0.0;
		break;
	}
}


static gfloat *make_kernel(InterpolateKernelType type, gint window, gfloat s) {
	gint kernel_width = (1+2*window);
	gfloat *kernel = g_new(gfloat, kernel_width*kernel_width);
	gint y;
	for (y = -window ; y <= window ; ++y) {
		gfloat *frow = kernel + kernel_width*(window+y);
		gint x;
		for (x = -window ; x <= window ; ++x) {
			gfloat d = sqrt(x*x+y*y);
			switch (type) {
			case KERNEL_CONSTANT:
				frow[window+x] = 1.0;
				break;
			case KERNEL_LINEAR:
				frow[window+x] = MAX(0.0,1.0-d/s);
				break;
			case KERNEL_SQUARE:
				frow[window+x] = MAX(0.0,1.0-(d/s)*(d/s));
				break;
			case KERNEL_INVLINEAR:
				frow[window+x] = (d>0.0 ? 1.0/d : -1.0);
				break;
			case KERNEL_INVSQUARE:
				frow[window+x] = (d>0.0 ? 1.0/(d*d) : -1.0);
				break;
			case KERNEL_EXPONENTIAL:
				frow[window+x] = exp(-d/s);
				break;
			case KERNEL_GAUSSIAN:
				frow[window+x] = gaussian(d,s);
				break;
			case KERNEL_INVNTH:
				frow[window+x] = (d>0.0 ? pow(d,-s) : -1.0);
				break;
			}
		}
	}
	return kernel;
}


inline static gfloat gaussian(gfloat x, gfloat s) {
	return exp(-0.5*(x/s)*(x/s))/(s*sqrt(2*M_PI));
}
