/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include "main.h"
#include "interface.h"
#include "render.h"
#include "plugin-intl.h"

#include <string.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/*  Constants  */

#define PROCEDURE_NAME   "gimp_plugin_interpolate"

#define DATA_KEY_VALS    "plug_in_interpolate"
#define DATA_KEY_UI_VALS "plug_in_interpolate_ui"

#define PARASITE_KEY     "plug-in-interpolate-options"


/*  Local function prototypes  */

static void query(void);
static void run(const gchar *name, gint nparams, const GimpParam *param, gint *nreturn_vals, GimpParam **return_vals);


/*  Local variables  */

const PlugInVals default_vals = {
	-1,  /* chanid */
	TRUE,  /* usesel */
	TRUE,  /* halfsel */

	64,  /* window */
	3.0,  /* sigma */
	KERNEL_INVNTH,  /* kernel */
	FALSE,  /* newlayer */

	TRUE,  /* slopes */
	TRUE,  /* leastsq */
	2,  /* linwindow */
	3.0,  /* linsigma */
	KERNEL_SQUARE,  /* linkernel */
	FALSE,  /* linnewlayer */

	FALSE,  /* changesel */
	0.0,  /* lowthres */
	0.01,  /* highthres */
	FALSE,  /* logthres */
	FALSE,  /* multmask */

	FALSE,  /* all_white_input */
};

const PlugInUIVals default_ui_vals = {
	TRUE,  /* preview */
};

static PlugInVals vals;
static PlugInUIVals ui_vals;


GimpPlugInInfo PLUG_IN_INFO = {
	NULL,  /* init_proc  */
	NULL,  /* quit_proc  */
	query, /* query_proc */
	run,   /* run_proc   */
};

MAIN ()

static void query(void) {
	gchar *help_path;
	gchar *help_uri;

	static GimpParamDef args[] = {
		{ GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
		{ GIMP_PDB_IMAGE, "image", "Input image" },
		{ GIMP_PDB_DRAWABLE, "drawable", "Input drawable" },
		{ GIMP_PDB_CHANNEL, "chanid", "Input mask" },
		{ GIMP_PDB_INT8, "usesel", "Subtract selection from input mask" },
		{ GIMP_PDB_INT8, "halfsel", "Read pixels with input mask<255" },
		{ GIMP_PDB_INT32, "window", "Size of window, around each pixel, for taking weighted average" },
		{ GIMP_PDB_FLOAT, "sigma", "Parameter for kernel" },
		{ GIMP_PDB_INT32, "kernel", "Type of kernel" },
		{ GIMP_PDB_INT8, "newlayer", "New layer for interpolation result" },

		{ GIMP_PDB_INT8, "slopes", "Do slopes calculation" },
		{ GIMP_PDB_INT8, "leastsq", "Least squares estimation for slopes" },
		{ GIMP_PDB_INT32, "linwindow", "Window for slopes estimation" },
		{ GIMP_PDB_FLOAT, "linsigma", "Parameter for kernel for slopes estimation" },
		{ GIMP_PDB_INT32, "linkernel", "Type of kernel for slopes estimation" },
		{ GIMP_PDB_INT8, "linnewlayer", "Create new grain merge layer for slopes" },

		{ GIMP_PDB_INT8, "changesel", "Modify selection" },
		{ GIMP_PDB_FLOAT, "lowthres", "Weight sum that gives completely selected pixels" },
		{ GIMP_PDB_FLOAT, "highthres", "Weight sum that gives completely unselected pixels" },
		{ GIMP_PDB_INT8, "logthres", "Log scale new selection" },
		{ GIMP_PDB_INT8, "multmask", "Multiply new selection with mask" },

		{ GIMP_PDB_INT8, "all_white_input", "Use all-white input mask, instead of selected channel" },
	};

	gimp_plugin_domain_register(PLUGIN_NAME, LOCALEDIR);

	help_path = g_build_filename(DATADIR, "help", NULL);
	help_uri = g_filename_to_uri(help_path, NULL, NULL);
	g_free(help_path);

	/* help is in tooltips */
	/*gimp_plugin_help_register
	    ("http://developer.gimp.org/plug-in-template/help", help_uri);*/

	gimp_install_procedure(PROCEDURE_NAME,
			       "Simple interpolatin",
			       "Does simple interpolation using weighted averages (\"guesses\" pixel values from surroundings)",
			       "stamit@stamit.gr",
			       "stamit@stamit.gr",
			       "2010",
			       N_("_Simple Interpolation..."),
			       "RGB*, GRAY*",
			       GIMP_PLUGIN,
			       G_N_ELEMENTS(args), 0, args, NULL);

	gimp_plugin_menu_register(PROCEDURE_NAME, "<Image>/Filters/Generic/");
}

static void run(const gchar *name,
		gint n_params,
		const GimpParam *param,
		gint *nreturn_vals, GimpParam **return_vals)
{
	static GimpParam values[1];
	GimpDrawable *drawable;
	gint32 image_ID;
	GimpRunMode run_mode;
	GimpPDBStatusType status = GIMP_PDB_SUCCESS;

	*nreturn_vals = 1;
	*return_vals = values;

	/*  Initialize i18n support  */
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
#endif
	textdomain(GETTEXT_PACKAGE);

	run_mode = param[0].data.d_int32;
	image_ID = param[1].data.d_int32;
	drawable = gimp_drawable_get(param[2].data.d_drawable);

	/*  Initialize with default values  */
	vals = default_vals;
	ui_vals = default_ui_vals;

	if (strcmp(name, PROCEDURE_NAME) != 0) {
		status = GIMP_PDB_CALLING_ERROR;

	} else switch (run_mode) {

	case GIMP_RUN_NONINTERACTIVE:
		if (n_params != 8) {
			status = GIMP_PDB_CALLING_ERROR;
		} else {
			vals.chanid = param[3].data.d_channel;
			vals.usesel = param[4].data.d_int8;
			vals.halfsel = param[5].data.d_int8;
			vals.window = param[6].data.d_int32;
			vals.sigma = param[7].data.d_float;
			vals.kernel = param[8].data.d_int32;
			vals.newlayer = param[9].data.d_int8;

			vals.slopes = param[10].data.d_int8;
			vals.leastsq = param[11].data.d_int8;
			vals.linwindow = param[12].data.d_int32;
			vals.linsigma = param[13].data.d_float;
			vals.linkernel = param[14].data.d_int32;
			vals.linnewlayer = param[15].data.d_int8;

			vals.changesel = param[16].data.d_int8;
			vals.lowthres = param[17].data.d_float;
			vals.highthres = param[18].data.d_float;
			vals.logthres = param[19].data.d_int8;
			vals.multmask = param[20].data.d_int8;

			vals.all_white_input = param[21].data.d_int8;

			/*if (vals.random_seed) vals.seed = g_random_int ();*/
		}
		break;

	case GIMP_RUN_INTERACTIVE:
		/*  Possibly retrieve data  */
		gimp_get_data(DATA_KEY_VALS, &vals);
		gimp_get_data(DATA_KEY_UI_VALS, &ui_vals);

		if (!dialog(image_ID, drawable, &vals, &ui_vals)) {
			status = GIMP_PDB_CANCEL;
		}
		break;

	case GIMP_RUN_WITH_LAST_VALS:
		/*  Possibly retrieve data  */
		gimp_get_data(DATA_KEY_VALS, &vals);

		/*if (vals.random_seed) vals.seed = g_random_int ();*/
		break;

	default:
		break;
	}

	if (status == GIMP_PDB_SUCCESS) {
		render(drawable, &vals, 0);

		if (run_mode != GIMP_RUN_NONINTERACTIVE)
			gimp_displays_flush();

		if (run_mode == GIMP_RUN_INTERACTIVE) {
			gimp_set_data(DATA_KEY_VALS, &vals, sizeof(vals));
			gimp_set_data(DATA_KEY_UI_VALS, &ui_vals,
				      sizeof(ui_vals));
		}

		gimp_drawable_detach(drawable);
	}

	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = status;
}
