/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include <string.h>
#include <gtk/gtk.h>
#include <libgimp/gimp.h>

#include "main.h"
#include "render.h"
#include "ring.h"

#include "plugin-intl.h"

#define SKIP_UNSELECTED 1


typedef struct InterpolateKernel {
	gfloat *values;
	gint width, height;
} InterpolateKernel;
static void kernel_init(InterpolateKernel *kernel, InterpolateKernelType type, gint window, gfloat s);
static void kernel_fini(InterpolateKernel *kernel);
inline static gfloat kernel_value(InterpolateKernel *kernel, gint deltax, gint deltay) {
	return kernel->values[kernel->width*(kernel->height/2+deltay) + (kernel->width/2+deltax)];
}


static void interpolate(GimpDrawable *drawable, PlugInVals *opts, gint image_id, GimpPreview *preview);
inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres);
static void slopes(GimpDrawable *drawable, GimpDrawable *mask,
                   gint left, gint top, gint right, gint bottom,
                   InterpolateKernel *kernel, gboolean halfsel, gboolean leastsq,
                   gfloat **pdx, gfloat **pdy, GimpPreview *preview);
inline static void old_slope(Ring *ring, Ring *mring, gint x,
                             InterpolateKernel *kernel, gboolean halfsel,
                             guchar *center, gfloat *dxpixel, gfloat *dypixel);
inline static void least_squares_slope(Ring *ring, Ring *mring, gint x,
                                       InterpolateKernel *kernel, gboolean halfsel,
                                       guchar *center, gfloat *dxpixel, gfloat *dypixel);
inline static void flat_slope(GimpImageType type, gfloat *dxpixel, gfloat *dypixel);


inline static gfloat gaussian(gfloat x, gfloat s);
inline static void gray_pixel(GimpImageType type, guchar *pixel);


void invoke_render(InterpolateInvocation *inv, GimpPreview *preview) {
	return render(inv->drawable, inv->vals, preview);
}

void render(GimpDrawable *drawable, PlugInVals *vals, GimpPreview *preview) {
	gint32 image_id = gimp_drawable_get_image(drawable->drawable_id);

	if (preview) {
		gimp_image_undo_freeze(image_id);
	} else {
		gimp_image_undo_group_start(image_id);
	}

	interpolate(drawable, vals, image_id, preview);

	if (preview) {
		gimp_image_undo_thaw(image_id);
	} else {
		gimp_image_undo_group_end(image_id);
	}
}


static void interpolate(GimpDrawable *drawable, PlugInVals *opts, gint image_id, GimpPreview *preview) {
	GimpImageType drawable_type = gimp_drawable_type(drawable->drawable_id);
	gint channels = gimp_drawable_bpp(drawable->drawable_id);

	gboolean changesel = opts->changesel && !preview;
	gboolean newlayer = opts->newlayer && !preview;
	gboolean linnewlayer = opts->linnewlayer && !preview;

	gboolean wslopes = opts->slopes;


	gint left,top,right,bottom,width,height;
	if (preview) {
		gimp_preview_get_position(preview, &left, &top);
		gimp_preview_get_size(preview, &width, &height);
		right = left + width;
		bottom = top + height;
	} else {
		gimp_drawable_mask_bounds(drawable->drawable_id, &left, &top, &right, &bottom);
		width = right-left;
		height = bottom-top;
	}


	/*
	 * make input mask
	 */
	gint tmp_mask_id = -1;
	GimpDrawable *mask = 0;

	if (!opts->all_white_input && gimp_drawable_is_valid(opts->chanid)) {
		if (opts->usesel) {
			tmp_mask_id = gimp_channel_copy(opts->chanid);
			gimp_image_add_channel(image_id, tmp_mask_id, 0);
			mask = gimp_drawable_get(tmp_mask_id);
		} else {
			mask = gimp_drawable_get(opts->chanid);
		}
	}
	
	if (!mask) {
		GimpRGB rgb;
		gimp_rgb_set(&rgb, 0, 0, 0);
		tmp_mask_id = gimp_channel_new(
			image_id, "Input mask temporary",
			gimp_image_width(image_id),
			gimp_image_height(image_id),
			0.5, &rgb
		);
		gimp_drawable_fill(tmp_mask_id, GIMP_WHITE_FILL);
		gimp_image_add_channel(image_id, tmp_mask_id, 0);

		mask = gimp_drawable_get(tmp_mask_id);
	}

	if (opts->usesel) {
		GimpPixelRgn outm;
		gimp_pixel_rgn_init(&outm, mask, left, top, width, height, TRUE, TRUE);
		guchar *outmrow = g_new(guchar, mask->bpp*width);
		memset(outmrow, 0, mask->bpp*width);
		gint y;
		for (y = 0 ; y < height ; ++y) {
			gimp_pixel_rgn_set_row(&outm,outmrow,0,y,width);
		}
		gimp_drawable_flush(mask);
		gimp_drawable_merge_shadow(tmp_mask_id, TRUE);
	}

	gint mchannels = gimp_drawable_bpp(mask->drawable_id);


	/*
	 * save selection to channel, so we can modify it (or to skip unselected pixels)
	 */
	gint sel_id = -1;
	GimpDrawable *sel = 0;
	#ifdef SKIP_UNSELECTED
		sel_id = gimp_selection_save(image_id);
		sel = gimp_drawable_get(sel_id);
	#else
		if (changesel) {
			sel_id = gimp_selection_save(image_id);
			sel = gimp_drawable_get(sel_id);
		}
	#endif


	/*
	 * make the new layers, if we are making new layers
	 */
	GimpDrawable *outdrawable, *outslopesdrawable;
	if (newlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Interpolated"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_NORMAL_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outdrawable = gimp_drawable_get(layer_id);
	} else {
		outdrawable = drawable;
	}

	GimpImageType outdrawable_type = gimp_drawable_type(outdrawable->drawable_id);
	gint outchannels = gimp_drawable_bpp(outdrawable->drawable_id);
	gint outslopeschannels;

	if (linnewlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Slopes Offset"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_GRAIN_MERGE_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outslopesdrawable = gimp_drawable_get(layer_id);
		outslopeschannels = gimp_drawable_bpp(outslopesdrawable->drawable_id);
	} else {
		outslopesdrawable = 0;
		outslopeschannels = outchannels;
	}


	/*
	 * make kernel
	 */
	InterpolateKernel kernel;
	kernel_init(&kernel, opts->kernel, opts->window, opts->sigma);


	/*
	 * calculate slopes of sample areas (this is the first pass)
	 */
	gint inleft = MAX(left-kernel.width/2,0);
	gint intop = MAX(top-kernel.height/2,0);
	gint inwidth = MIN(right+kernel.width/2,drawable->width)-inleft;
	gint inheight = MIN(bottom+kernel.height/2,drawable->height)-intop;
	gfloat *dx=0, *dy=0;
	if (wslopes) {
		InterpolateKernel linkernel;
		kernel_init(&linkernel, opts->linkernel, opts->linwindow, opts->linsigma);
		slopes(drawable, mask,
		       inleft, intop, inleft+inwidth, intop+inheight,
		       &linkernel, opts->halfsel, opts->leastsq,
		       &dx, &dy, preview);
		kernel_fini(&linkernel);
	}

	gint tw = gimp_tile_width();
	gimp_tile_cache_ntiles(2*(drawable->width+tw-1)/tw);


	/*
	 * get input/output pixel regions for interpolation pass
	 */
	Ring in; ring_init(&in, drawable, left, top, right, bottom, kernel.width/2, kernel.height/2);
	Ring inm; ring_init(&inm, mask, left, top, right, bottom, kernel.width/2, kernel.height/2);

	GimpPixelRgn out;
	if (newlayer) {
		gimp_pixel_rgn_init(&out, outdrawable, 0, 0, width, height, TRUE, TRUE);
	} else {
		gimp_pixel_rgn_init(&out, outdrawable, left, top, width, height, !preview, TRUE);
	}
	guchar *outrow = g_new(guchar, outchannels*width);

	GimpPixelRgn outslopes;
	guchar *outslopesrow = 0;
	if (linnewlayer) {
		gimp_pixel_rgn_init(&outslopes, outslopesdrawable, 0, 0, width, height, TRUE, TRUE);
		outslopesrow = g_new(guchar, outslopeschannels*width);
	}


	/*
	 * get selection, to skip unselected pixels
	 */
	GimpPixelRgn ins;
	guchar *insrow = 0;
	#ifdef SKIP_UNSELECTED
		gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
		insrow = g_new(guchar, mchannels*width);
	#else
		if (changesel) {
			gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
			insrow = g_new(guchar, mchannels*width);
		}
	#endif

	GimpPixelRgn outs;
	guchar *outsrow = 0;
	if (changesel) {
		gimp_pixel_rgn_init(&outs, sel, left, top, width, height, TRUE, TRUE);
		outsrow = g_new(guchar, mchannels*width);
	}


	/*
	 * do the interpolation
	 */
	gint x;
	gint deltay, deltax;
	if (!preview) gimp_progress_init(_("Interpolating..."));
	for (; ring_loop(&in) && ring_loop(&inm) ; ring_rotate(&in), ring_rotate(&inm)) {
		guchar *inmrow = 0;
		if (changesel) {
			inmrow = ring_row(&inm);
		}

		#ifdef SKIP_UNSELECTED
			gimp_pixel_rgn_get_row(&ins, insrow, left, in.y, width);
		#else
			if (changesel) {
				gimp_pixel_rgn_get_row(&ins, insrow, left, in.y, width);
			}
		#endif

		for (x = left ; x < right ; ++x) {
			guchar *inspixel = 0;
			#ifdef SKIP_UNSELECTED
				inspixel = insrow + (x-left)*mchannels;
			#else
				if (changesel) {
					inspixel = insrow + (x-left)*mchannels;
				}
			#endif
			guchar *outpixel = outrow + (x-left)*outchannels;
			guchar *outslopespixel = 0;
			if (linnewlayer) {
				outslopespixel = outslopesrow + (x-left)*outslopeschannels;
			}

			gfloat sum[4];
			sum[0] = 0.0;
			sum[1] = 0.0;
			sum[2] = 0.0;
			sum[3] = 0.0;
			gfloat dsum[4];
			dsum[0] = 0.0;
			dsum[1] = 0.0;
			dsum[2] = 0.0;
			dsum[3] = 0.0;
			gfloat ws = 0.0;

			gint deltay_min = ring_min_deltay(&in);
			gint deltay_max = ring_max_deltay(&in);
			#ifdef SKIP_UNSELECTED
			if (inspixel[0])
			#endif
			for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
				guchar *inrow2 = ring_row_delta(&in, deltay);
				guchar *inmrow2 = ring_row_delta(&inm, deltay);

				gfloat *dxrow = 0;
				gfloat *dyrow = 0;
				if (wslopes) {
					dxrow = dx + channels*inwidth*(in.y+deltay-intop);
					dyrow = dy + channels*inwidth*(in.y+deltay-intop);
				}

				gint deltax_min = ring_min_deltax(&in, x);
				gint deltax_max = ring_max_deltax(&in, x);
				for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
					guchar *inpixel2 = inrow2 + (x+deltax-left)*channels;
					guchar *inmpixel2 = inmrow2 + (x+deltax-left)*mchannels;

					gfloat w = kernel_value(&kernel, deltax, deltay);

					if ( ! ( opts->halfsel ? (inmpixel2[0]>0) : (inmpixel2[0]==255) ) ) {
						continue;
					}

					if (w>=0.0) {
						/*
						 * ordinary input pixel
						 */
						if (opts->halfsel)
							w *= inmpixel2[0];

						switch (drawable_type) {
						case GIMP_RGBA_IMAGE:
							sum[3] += inpixel2[3]*w;
						case GIMP_RGB_IMAGE:
							sum[2] += inpixel2[2]*w;
						case GIMP_GRAYA_IMAGE:
							sum[1] += inpixel2[1]*w;
						case GIMP_GRAY_IMAGE: default:
							sum[0] += inpixel2[0]*w;
							break;
						}

						if (wslopes) {
							gfloat *dxpixel = dxrow + (x+deltax-inleft)*channels;
							gfloat *dypixel = dyrow + (x+deltax-inleft)*channels;

							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:
								dsum[3] += ( - dxpixel[3]*deltax - dypixel[3]*deltay)*w;
							case GIMP_RGB_IMAGE:
								dsum[2] += ( - dxpixel[2]*deltax - dypixel[2]*deltay)*w;
							case GIMP_GRAYA_IMAGE:
								dsum[1] += ( - dxpixel[1]*deltax - dypixel[1]*deltay)*w;
							case GIMP_GRAY_IMAGE: default:
								dsum[0] += ( - dxpixel[0]*deltax - dypixel[0]*deltay)*w;
								break;
							}
						}

						ws += w;
					} else {
						/*
						 * -1 means infinity. this input pixel has infinite weight
						 * (center pixel in 'inverse' kernels)
						 */
						ws = opts->highthres;
						switch (drawable_type) {
						case GIMP_RGBA_IMAGE:
							sum[3] = opts->highthres*inpixel2[3];
						case GIMP_RGB_IMAGE:
							sum[2] = opts->highthres*inpixel2[2];
						case GIMP_GRAYA_IMAGE:
							sum[1] = opts->highthres*inpixel2[1];
						case GIMP_GRAY_IMAGE: default:
							sum[0] = opts->highthres*inpixel2[0];
							break;
						}
						dsum[0] = 0.0;
						dsum[1] = 0.0;
						dsum[2] = 0.0;
						dsum[3] = 0.0;
						goto breakout;
					}
				}
			}

			breakout:

			/*
			 * now we can calculate the average, to set the output pixel
			 */
			if (wslopes) {
				if (linnewlayer) {
					if (ws>0.0) {
						switch (drawable_type) {
						case GIMP_GRAY_IMAGE: default:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = 255;
							break;
						case GIMP_GRAYA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							break;
						case GIMP_RGB_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/ws);
							outslopespixel[3] = 255;
							break;
						case GIMP_RGBA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/ws);
							outslopespixel[3] = CLAMP0255(128+dsum[3]/ws);
							break;
						}
					} else {
						gray_pixel(outdrawable_type, outslopespixel);
					}

				} else {
					sum[0] += dsum[0];
					sum[1] += dsum[1];
					sum[2] += dsum[2];
					sum[3] += dsum[3];
				}
			}

			if (ws>0.0) {
				switch (drawable_type) {
				case GIMP_GRAY_IMAGE: default:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					if (newlayer) {
						outpixel[1] = 255;
					}
					break;
				case GIMP_GRAYA_IMAGE:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					outpixel[1] = CLAMP0255(sum[1]/ws);
					break;
				case GIMP_RGB_IMAGE:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					outpixel[1] = CLAMP0255(sum[1]/ws);
					outpixel[2] = CLAMP0255(sum[2]/ws);
					if (newlayer) {
						outpixel[3] = 255;
					}
					break;
				case GIMP_RGBA_IMAGE:
					outpixel[0] = CLAMP0255(sum[0]/ws);
					outpixel[1] = CLAMP0255(sum[1]/ws);
					outpixel[2] = CLAMP0255(sum[2]/ws);
					outpixel[3] = CLAMP0255(sum[3]/ws);
					break;
				}
			} else {
				/*
				 * there were no input pixels in range, so set pixel to gray
				 */
				gray_pixel(outdrawable_type, outpixel);
			}

			if (changesel) {
				guchar *inmpixel = inmrow + (x-left)*mchannels;
				guchar *outspixel = outsrow + (x-left)*mchannels;

				/*
				 * input mask is multiplied with pixels without being divided by 255, so divide here
				 */
				if (opts->halfsel) ws /= 255.0;

				gfloat outsel = wsum2sel(ws, opts->lowthres, opts->highthres, opts->logthres) * inspixel[0];
				if (opts->multmask) outsel *= inmpixel[0]/255.0;
				outspixel[0] = outsel;
			}
		}

		if (newlayer) {
			gimp_pixel_rgn_set_row(&out,outrow,0,in.y-top,right-left);
		} else {
			gimp_pixel_rgn_set_row(&out,outrow,left,in.y,right-left);
		}
		if (linnewlayer) {
			gimp_pixel_rgn_set_row(&outslopes,outslopesrow,0,in.y-top,right-left);
		}
		if (changesel) {
			gimp_pixel_rgn_set_row(&outs,outsrow,left,in.y,right-left);
		}

		if (!preview && !(in.y%4)) {
			gimp_progress_update(1.0*(in.y-top+1)/(bottom-top));
		}
	}
	if (!preview) gimp_progress_end();


	/*
	 * done with the pixel regions, kernel and slopes
	 */
	if (changesel) {
		g_free(outsrow);
	}
	if (linnewlayer) {
		g_free(outslopesrow);
	}
	g_free(outrow);

	ring_fini(&inm);
	ring_fini(&in);

	kernel_fini(&kernel);

	g_free(dx);
	g_free(dy);


	/*
	 * the plumbing code
	 */
	if (preview) {
		gimp_drawable_preview_draw_region(GIMP_DRAWABLE_PREVIEW(preview), &out);
	} else {
		gimp_drawable_flush(outdrawable);
		gimp_drawable_merge_shadow(outdrawable->drawable_id, TRUE);
		if (newlayer) {
			gimp_drawable_update(outdrawable->drawable_id, 0, 0, width, height);
			gimp_drawable_detach(outdrawable);
		} else {
			gimp_drawable_update(outdrawable->drawable_id, left, top, width, height);
		}

		if (linnewlayer) {
			gimp_drawable_flush(outslopesdrawable);
			gimp_drawable_merge_shadow(outslopesdrawable->drawable_id, TRUE);
			gimp_drawable_update(outslopesdrawable->drawable_id, 0, 0, width, height);
			gimp_drawable_detach(outslopesdrawable);
		}
	}


	/*
	 * only remove the selection if we are modifying it
	 */
	if (changesel) {
		gint tmp_sel_id = gimp_selection_save(image_id);
		gimp_selection_none(image_id);

		gimp_drawable_flush(sel);
		gimp_drawable_merge_shadow(sel->drawable_id, TRUE);
		gimp_drawable_update(sel->drawable_id, left, top, width, height);

		gimp_selection_load(tmp_sel_id);
		gimp_image_remove_channel(image_id, tmp_sel_id);
	}


	/*
	 * no more need for temporary selection channel
	 */
	if (changesel) {
		gimp_selection_load(sel_id);
		#ifndef SKIP_UNSELECTED
			gimp_drawable_detach(sel);
			gimp_image_remove_channel(image_id, sel_id);
		#endif
	}
	#ifdef SKIP_UNSELECTED
		gimp_drawable_detach(sel);
		gimp_image_remove_channel(image_id, sel_id);
	#endif


	/*
	 * remove input mask channel if it is our temporary
	 */
	gimp_drawable_detach(mask);
	if (tmp_mask_id != -1) {
		gimp_image_remove_channel(image_id, tmp_mask_id);
	}
}


inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres) {
	if (logthres) w = log(w);

	gfloat ww;
	if (w<=lt) {
		ww = 0.0;
	} else if (w>=ht) {
		ww = 1.0;
	} else {
		ww = (w-lt)/(ht-lt);
	}
	return 1.0-ww;
}


static void slopes(GimpDrawable *drawable, GimpDrawable *mask,
                   gint left, gint top, gint right, gint bottom,
                   InterpolateKernel *kernel, gboolean halfsel, gboolean leastsq,
                   gfloat **pdx, gfloat **pdy, GimpPreview *preview) {
	gint width = right-left;
	gint height = bottom-top;

	Ring ring; ring_init(&ring, drawable, left, top, right, bottom, kernel->width/2, kernel->height/2);
	Ring mring; ring_init(&mring, mask, left, top, right, bottom, kernel->width/2, kernel->height/2);
	gfloat *dx = *pdx = g_new(gfloat, width*height*ring.bpp);
	gfloat *dy = *pdy = g_new(gfloat, width*height*ring.bpp);

	if (!preview) gimp_progress_init(_("Calculating slopes..."));
	for (; ring_loop(&ring) && ring_loop(&mring) ; ring_rotate(&ring), ring_rotate(&mring)) {
		guchar *row = ring_row(&ring);
		guchar *mrow = ring_row(&mring);
		gfloat *dxrow = dx + ring.bpp*width*(ring.y-top);
		gfloat *dyrow = dy + ring.bpp*width*(ring.y-top);

		gint x;
		for (x = left ; x < right ; ++x) {
			guchar *pixel = row + ( (x-left)*ring.bpp );  /* center pixel */
			guchar *mpixel = mrow + ( (x-left)*mring.bpp );
			gfloat *dxpixel = dxrow + ring.bpp*(x-left);
			gfloat *dypixel = dyrow + ring.bpp*(x-left);

			/*
			 * calculate average of nearby pixels
			 */
			if (halfsel ? !mpixel[0] : mpixel[0]<255) {
				flat_slope(ring.type, dxpixel, dypixel);
			} else if (leastsq) {
				least_squares_slope(&ring, &mring, x, kernel, halfsel, pixel, dxpixel, dypixel);
			} else {
				old_slope(&ring, &mring, x, kernel, halfsel, pixel, dxpixel, dypixel);
			}
		}

		if (!preview && !(ring.y%4)) {
			gimp_progress_update(1.0*(ring.y-top+1)/(bottom-top));
		}
	}
	if (!preview) gimp_progress_end();
	ring_fini(&ring);
}


inline static void old_slope(Ring *ring, Ring *mring, gint x,
                             InterpolateKernel *kernel, gboolean halfsel,
                             guchar *center, gfloat *dxpixel, gfloat *dypixel) {
	gfloat wsum = 0.0;
	gfloat dxsum[4];
	dxsum[0] = 0.0;
	dxsum[1] = 0.0;
	dxsum[2] = 0.0;
	dxsum[3] = 0.0;
	gfloat dysum[4];
	dysum[0] = 0.0;
	dysum[1] = 0.0;
	dysum[2] = 0.0;
	dysum[3] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			gfloat distsquared = (deltax*deltax + deltay*deltay);
			if ( distsquared>0.0 && (halfsel ? mpixel[0]>0 : mpixel[0]==255) ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );
					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						dxsum[3] += w*(pixel[3]-center[3])*deltax/distsquared;
						dysum[3] += w*(pixel[3]-center[3])*deltay/distsquared;
					case GIMP_RGB_IMAGE:
						dxsum[2] += w*(pixel[2]-center[2])*deltax/distsquared;
						dysum[2] += w*(pixel[2]-center[2])*deltay/distsquared;
					case GIMP_GRAYA_IMAGE:
						dxsum[1] += w*(pixel[1]-center[1])*deltax/distsquared;
						dysum[1] += w*(pixel[1]-center[1])*deltay/distsquared;
					case GIMP_GRAY_IMAGE: default:
						dxsum[0] += w*(pixel[0]-center[0])*deltax/distsquared;
						dysum[0] += w*(pixel[0]-center[0])*deltay/distsquared;
						break;
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = 2*dxsum[3]/wsum;
			dypixel[3] = 2*dysum[3]/wsum;
		case GIMP_RGB_IMAGE:
			dxpixel[2] = 2*dxsum[2]/wsum;
			dypixel[2] = 2*dysum[2]/wsum;
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = 2*dxsum[1]/wsum;
			dypixel[1] = 2*dysum[1]/wsum;
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = 2*dxsum[0]/wsum;
			dypixel[0] = 2*dysum[0]/wsum;
			break;
		}
	} else {
		flat_slope(ring->type, dxpixel, dypixel);
	}
}


inline static void least_squares_slope(Ring *ring, Ring *mring, gint x,
                                       InterpolateKernel *kernel, gboolean halfsel,
                                       guchar *center, gfloat *dxpixel, gfloat *dypixel) {
	gfloat ws = 0.0;

	gfloat m[2][2];
	m[0][0] = 0.0;
	m[0][1] = 0.0;
	m[1][0] = 0.0;
	m[1][1] = 0.0;

	gfloat v[4][2];
	v[0][0] = 0.0;
	v[0][1] = 0.0;
	v[1][0] = 0.0;
	v[1][1] = 0.0;
	v[2][0] = 0.0;
	v[2][1] = 0.0;
	v[3][0] = 0.0;
	v[3][1] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			if ( halfsel ? mpixel[0]>0 : mpixel[0]==255 ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel) {
					w *= mpixel[0];
				}
				if (w>0) {
					ws += w;

					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );

					m[0][0] += deltax*deltax*w;
					m[0][1] += deltax*deltay*w;
					m[1][0] += deltay*deltax*w;
					m[1][1] += deltay*deltay*w;

					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						v[3][0] += deltax*(pixel[3]-center[3])*w;
						v[3][1] += deltay*(pixel[3]-center[3])*w;
					case GIMP_RGB_IMAGE:
						v[2][0] += deltax*(pixel[2]-center[2])*w;
						v[2][1] += deltay*(pixel[2]-center[2])*w;
					case GIMP_GRAYA_IMAGE:
						v[1][0] += deltax*(pixel[1]-center[1])*w;
						v[1][1] += deltay*(pixel[1]-center[1])*w;
					case GIMP_GRAY_IMAGE: default:
						v[0][0] += deltax*(pixel[0]-center[0])*w;
						v[0][1] += deltay*(pixel[0]-center[0])*w;
						break;
					}
				}
			}
		}
	}

	gfloat det;

	if (ws!=0) {
		m[0][0] /= ws;
		m[0][1] /= ws;
		m[1][0] /= ws;
		m[1][1] /= ws;
	
		v[0][0] /= ws;
		v[0][1] /= ws;
		v[1][0] /= ws;
		v[1][1] /= ws;
		v[2][0] /= ws;
		v[2][1] /= ws;
		v[3][0] /= ws;
		v[3][1] /= ws;

		det = m[0][0]*m[1][1] - m[1][0]*m[0][1];
	}

	if (ws!=0 && det!=0) {
		gfloat invm[2][2];
		invm[0][0] =  m[1][1]/det;
		invm[0][1] = -m[0][1]/det;
		invm[1][0] = -m[1][0]/det;
		invm[1][1] =  m[0][0]/det;

		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = invm[0][0]*v[3][0] + invm[0][1]*v[3][1];
			dypixel[3] = invm[1][0]*v[3][0] + invm[1][1]*v[3][1];
		case GIMP_RGB_IMAGE:
			dxpixel[2] = invm[0][0]*v[2][0] + invm[0][1]*v[2][1];
			dypixel[2] = invm[1][0]*v[2][0] + invm[1][1]*v[2][1];
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = invm[0][0]*v[1][0] + invm[0][1]*v[1][1];
			dypixel[1] = invm[1][0]*v[1][0] + invm[1][1]*v[1][1];
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = invm[0][0]*v[0][0] + invm[0][1]*v[0][1];
			dypixel[0] = invm[1][0]*v[0][0] + invm[1][1]*v[0][1];
			break;
		}
	} else {
		/*
		 * only one point, or colinear points
		 */
		flat_slope(ring->type, dxpixel, dypixel);
	}
}


inline static void flat_slope(GimpImageType type, gfloat *dxpixel, gfloat *dypixel) {
	switch (type) {
	case GIMP_RGBA_IMAGE:
		dxpixel[3] = 0.0;
		dypixel[3] = 0.0;
	case GIMP_RGB_IMAGE:
		dxpixel[2] = 0.0;
		dypixel[2] = 0.0;
	case GIMP_GRAYA_IMAGE:
		dxpixel[1] = 0.0;
		dypixel[1] = 0.0;
	case GIMP_GRAY_IMAGE: default:
		dxpixel[0] = 0.0;
		dypixel[0] = 0.0;
		break;
	}
}


static void kernel_init(InterpolateKernel *kernel, InterpolateKernelType type, gint window, gfloat s) {
	gint bwidth=0, bheight=0;
	gint mask_bpp=0, num_mask_bytes=0;
	guint8 *mask_bytes=0;
	gchar *brush_name = gimp_context_get_brush();
	gint color_bpp=0, num_color_bytes=0;
	guint8 *color_bytes=0;
	if (type==KERNEL_BRUSH) {
		gimp_brush_get_pixels(brush_name, &bwidth, &bheight,
		                      &mask_bpp, &num_mask_bytes, &mask_bytes,
		                      &color_bpp, &num_color_bytes, &color_bytes);
		kernel->width = (bwidth/2)*2 + 1;
		kernel->height = (bheight/2)*2 + 1;
	} else {
		kernel->width = kernel->height = 1 + 2*window;
	}

	kernel->values = g_new(gfloat, kernel->width*kernel->height);

	gint y;
	for (y = 0 ; y < kernel->height ; ++y) {
		gint deltay = y - kernel->height/2;

		gfloat *krow = kernel->values + kernel->width*y;

		gint x;
		for (x = 0 ; x < kernel->width ; ++x) {
			gint deltax = x - kernel->width/2;

			gfloat d = sqrt(deltax*deltax+deltay*deltay);

			switch (type) {
			case KERNEL_CONSTANT:
				krow[x] = 1.0;
				break;
			case KERNEL_LINEAR:
				krow[x] = MAX(0.0,1.0-d/s);
				break;
			case KERNEL_SQUARE:
				krow[x] = MAX(0.0,1.0-(d/s)*(d/s));
				break;
			case KERNEL_INVLINEAR:
				krow[x] = (d>0.0 ? 1.0/d : -1.0);
				break;
			case KERNEL_INVSQUARE:
				krow[x] = (d>0.0 ? 1.0/(d*d) : -1.0);
				break;
			case KERNEL_EXPONENTIAL:
				krow[x] = exp(-d/s);
				break;
			case KERNEL_GAUSSIAN:
				krow[x] = gaussian(d,s);
				break;
			case KERNEL_INVNTH:
				krow[x] = (d>0.0 ? pow(d,-s) : -1.0);
				break;
			case KERNEL_BRUSH:
				if (x < bwidth && y < bheight) {
					krow[x] = mask_bytes[mask_bpp*(bwidth*y + x)]/255.0;
				} else {
					krow[x] = 0.0;
				}
				break;
			}
		}
	}
}
static void kernel_fini(InterpolateKernel *kernel) {
	g_free(kernel->values);
}


inline static gfloat gaussian(gfloat x, gfloat s) {
	return exp(-0.5*(x/s)*(x/s))/(s*sqrt(2*M_PI));
}


inline static void gray_pixel(GimpImageType type, guchar *pixel) {
	switch (type) {
	case GIMP_RGBA_IMAGE:
		pixel[3] = 128;
	case GIMP_RGB_IMAGE:
		pixel[2] = 128;
	case GIMP_GRAYA_IMAGE:
		pixel[1] = 128;
	case GIMP_GRAY_IMAGE: default:
		pixel[0] = 128;
		break;
	}
}


