/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include <string.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "main.h"
#include "interface.h"
#include "render.h"

#include "plugin-intl.h"

static void gtk_combo_box_get_active_ptr(GtkComboBox *combo_box, gint *p) {
	*p = gtk_combo_box_get_active(combo_box);
}

static void check_and_make_visible(GtkToggleButton *toggle, GtkWidget *wid) {
	if (gtk_toggle_button_get_active(toggle)) {
		gtk_widget_show(wid);
	} else {
		gtk_widget_hide(wid);
	}
}

static void adj_set_upper_minus_one(GtkAdjustment *adj, GtkAdjustment *adjtoset) {
	gint upper = gtk_adjustment_get_value(adj)-1;
	gtk_adjustment_set_upper(adjtoset, upper);
	if (gtk_adjustment_get_value(adjtoset) > upper) {
		gtk_adjustment_set_value(adjtoset, upper);
	}
}

static void combo_add_kernel_options(GtkComboBox *combo) {
	gtk_combo_box_append_text(combo, _("Constant"));
	gtk_combo_box_append_text(combo, _("Linear"));
	gtk_combo_box_append_text(combo, _("Square"));
	gtk_combo_box_append_text(combo, _("Inverse Linear"));
	gtk_combo_box_append_text(combo, _("Inverse Square"));
	gtk_combo_box_append_text(combo, _("Exponential"));
	gtk_combo_box_append_text(combo, _("Gaussian"));
	gtk_combo_box_append_text(combo, _("Inverse to the nth"));
	gtk_combo_box_append_text(combo, _("Brush"));
	gtk_combo_box_append_text(combo, _("Brush, flipped"));
	gtk_combo_box_append_text(combo, _("Bilinear"));
	gtk_combo_box_append_text(combo, _("Bi-nth"));
}

static void kernel_combo_window_hbox_visibility(GtkComboBox *kernel_combo, GtkWidget *wid) {
	switch (gtk_combo_box_get_active(GTK_COMBO_BOX(kernel_combo))) {
	case KERNEL_BRUSH:
	case KERNEL_BRUSH_FLIPPED:
		gtk_widget_hide(wid);
		break;
	default:
		gtk_widget_show(wid);
		break;
	}
}
static void kernel_combo_sigma_visibility(GtkComboBox *kernel_combo, GtkWidget *wid) {
	switch (gtk_combo_box_get_active(GTK_COMBO_BOX(kernel_combo))) {
	case KERNEL_CONSTANT:
	case KERNEL_BRUSH:
	case KERNEL_BRUSH_FLIPPED:
	case KERNEL_BILINEAR:
		gtk_widget_hide(wid);
		break;
	default:
		gtk_widget_show(wid);
		break;
	}
}
static void kernel_combo_norm_visibility(GtkComboBox *kernel_combo, GtkWidget *wid) {
	switch (gtk_combo_box_get_active(GTK_COMBO_BOX(kernel_combo))) {
	case KERNEL_CONSTANT:
	case KERNEL_BRUSH:
	case KERNEL_BRUSH_FLIPPED:
	case KERNEL_BILINEAR:
	case KERNEL_BINTH:
		gtk_widget_hide(wid);
		break;
	default:
		gtk_widget_show(wid);
		break;
	}
}


/*  Public functions  */

gboolean dialog(gint32 image_ID, GimpDrawable * drawable,
		PlugInVals * vals, PlugInUIVals * ui_vals)
{
	PlugInVals new_vals;
	PlugInUIVals new_ui_vals;
	memcpy(&new_vals, vals, sizeof(new_vals));
	memcpy(&new_ui_vals, ui_vals, sizeof(new_ui_vals));


	gimp_ui_init(PLUGIN_NAME, TRUE);


	GtkWidget *dlg = gimp_dialog_new(
		_("Interpolation"), PLUGIN_NAME,
		NULL, 0,
		gimp_standard_help_func,
		"plug-in-interpolate", GTK_STOCK_CANCEL,
		GTK_RESPONSE_CANCEL, GTK_STOCK_OK,
		GTK_RESPONSE_OK, NULL
	);


	GtkWidget *main_hbox = gtk_hbox_new(FALSE, 12);
	gtk_widget_show(main_hbox);
	gtk_container_set_border_width(GTK_CONTAINER(main_hbox), 12);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), main_hbox);


	/*
	 * preview
	 */
	InterpolateInvocation inv;
	inv.drawable = drawable;
	inv.vals = &new_vals;
	GtkWidget *preview = gimp_drawable_preview_new(drawable, &new_ui_vals.preview);
	gtk_box_pack_start(GTK_BOX(main_hbox), preview, TRUE, TRUE, 0);
	gtk_widget_show(preview);
	g_signal_connect_swapped(preview, "invalidated", G_CALLBACK(invoke_render), &inv);
	invoke_render(&inv, GIMP_PREVIEW(preview));

	GtkWidget *main_vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(main_vbox);
	gtk_box_pack_start(GTK_BOX(main_hbox), main_vbox, FALSE, FALSE, 0);


	/*
	 * chanid
	 */
	GtkWidget *chanid_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(chanid_hbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), chanid_hbox, FALSE, FALSE, 0);

	GtkWidget *chanid_label = gtk_label_new_with_mnemonic(_("_Input mask: "));
	gtk_widget_show(chanid_label);
	gtk_box_pack_start(GTK_BOX(chanid_hbox), chanid_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(chanid_label), GTK_JUSTIFY_RIGHT);

	GtkWidget *chanid_combo = gimp_channel_combo_box_new(NULL, NULL);
	gtk_widget_show(chanid_combo);
	gimp_int_combo_box_set_active(GIMP_INT_COMBO_BOX(chanid_combo), new_vals.chanid);
	gtk_box_pack_start(GTK_BOX(chanid_hbox), chanid_combo, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(chanid_combo, _(
		"Which channel to use as the input mask."
	));

	g_signal_connect_swapped(chanid_combo, "changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(chanid_combo, "changed", G_CALLBACK(gimp_int_combo_box_get_active), &new_vals.chanid);
	gimp_int_combo_box_get_active(GIMP_INT_COMBO_BOX(chanid_combo), &new_vals.chanid);


	/*
	 * all_white_input
	 */
	GtkWidget *all_white_input_check = gtk_check_button_new_with_mnemonic(_("All-white input mask"));
	gtk_widget_show(all_white_input_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(all_white_input_check), new_vals.all_white_input);
	gtk_box_pack_start(GTK_BOX(main_vbox), all_white_input_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(all_white_input_check, _(
		"Start with a white input mask. "
		"This is used by default if no mask channel exists. "
		"If you turn off \"subtract selection\" and \"calculate slopes\", you can turn this filter into a simple blur filter. "
		"Using an all-white input mask with an inverse kernel (such as \"inverse to the nth\") will leave the image unchanged "
		"because the center pixel will always be used, with infinite weight. "
	));

	g_signal_connect_swapped(all_white_input_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(all_white_input_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.all_white_input);


	/*
	 * usesel
	 */
	GtkWidget *usesel_hbox = gtk_hbox_new(FALSE, 10);
	gtk_widget_show(usesel_hbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), usesel_hbox, FALSE, FALSE, 0);

	GtkWidget *usesel_check = gtk_check_button_new_with_mnemonic(_("Subtract selection"));
	gtk_widget_show(usesel_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(usesel_check), new_vals.usesel);
	gtk_box_pack_start(GTK_BOX(usesel_hbox), usesel_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(usesel_check, _(
		"Subtract the selection from the input mask. "
		"The original mask is not modified; only a temporary is created. "
		"With an all-white input mask, this will give the inverse of the selection. "
		"In other cases, this is useful so that the entire selected area is painted on. "
		"This way, it's easier to make input masks using the lasso tool."
	));

	g_signal_connect_swapped(usesel_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(usesel_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.usesel);


	/*
	 * halfsel
	 */
	GtkWidget *halfsel_label = gtk_label_new_with_mnemonic(_("Read threshold: "));
	gtk_widget_show(halfsel_label);
	gtk_box_pack_start(GTK_BOX(usesel_hbox), halfsel_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(halfsel_label), GTK_JUSTIFY_RIGHT);

	GtkObject *halfsel_adj = gtk_adjustment_new(new_vals.halfsel, 1, 255, 1, 0, 0);
	GtkWidget *halfsel_spin = gtk_spin_button_new(GTK_ADJUSTMENT(halfsel_adj), 1, 0);
	gtk_widget_show(halfsel_spin);
	gtk_box_pack_start(GTK_BOX(usesel_hbox), halfsel_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(halfsel_spin), TRUE);
	gtk_widget_set_tooltip_text(halfsel_spin, _(
		"Only pixels with at least this much input mask intensity will be read."
	));

	g_signal_connect_swapped(halfsel_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(halfsel_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.halfsel);


	/*
	 * kernel
	 */
	GtkWidget *kernel_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(kernel_hbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), kernel_hbox, FALSE, FALSE, 0);

	GtkWidget *kernel_label = gtk_label_new_with_mnemonic(_("_Kernel: "));
	gtk_widget_show(kernel_label);
	gtk_box_pack_start(GTK_BOX(kernel_hbox), kernel_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(kernel_label), GTK_JUSTIFY_RIGHT);

	GtkWidget *kernel_combo = gtk_combo_box_new_text();
	gtk_widget_show(kernel_combo);
	combo_add_kernel_options(GTK_COMBO_BOX(kernel_combo));
	gtk_combo_box_set_active(GTK_COMBO_BOX(kernel_combo), (int) new_vals.kernel);
	gtk_box_pack_start(GTK_BOX(kernel_hbox), kernel_combo, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(kernel_combo, _(
		"The kernel has the weights for the averaging of the nearby pixels. "
		"The weights are functions of the distance D from the center pixel. "
		"Constant generates a white square of width Window+1+Window. "
		"Linear is 1-D/s (cone with base 's', then a flat 0). "
		"Square is 1-(D/s)^2 (again up to distance 's'). "
		"Inverse Linear is 1/D . Inverse Square is 1/(D^2) . "
		"Exponential is e^(-D) and the result is similar to smudging from the edges of the mask. "
		"Gaussian is the standard Gaussian kernel with standard deviation 'S'. "
		"Inverse to the nth is 1/(D^s)."
	));

	g_signal_connect_swapped(kernel_combo, "changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(kernel_combo, "changed", G_CALLBACK(gtk_combo_box_get_active_ptr), &new_vals.kernel);


	/*
	 * window
	 */
	GtkWidget *window_hbox = gtk_hbox_new(FALSE, 0);
	kernel_combo_window_hbox_visibility(GTK_COMBO_BOX(kernel_combo), window_hbox);
	g_signal_connect(kernel_combo, "changed", G_CALLBACK(kernel_combo_window_hbox_visibility), window_hbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), window_hbox, FALSE, FALSE, 0);

	GtkWidget *window_label = gtk_label_new_with_mnemonic(_("_Window: "));
	gtk_widget_show(window_label);
	gtk_box_pack_start(GTK_BOX(window_hbox), window_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(window_label), GTK_JUSTIFY_RIGHT);

	GtkObject *window_adj = gtk_adjustment_new(new_vals.window, 0, 1024, 1, 0, 0);
	GtkWidget *window_spin = gtk_spin_button_new(GTK_ADJUSTMENT(window_adj), 1, 0);
	gtk_widget_show(window_spin);
	gtk_box_pack_start(GTK_BOX(window_hbox), window_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(window_spin), TRUE);
	gtk_widget_set_tooltip_text(window_spin, _(
		"Number of nearby pixels to sample for each output pixel, in order to calculate a weighted average. "
		"1 means a 3x3 rectangle, 2 means a 5x5 rectangle, and so on. "
		"Only pixels with mask values equal to 255 are read. "
		"If there is no input mask channel, the inverse of the selection is used."
	));

	g_signal_connect_swapped(window_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(window_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.window);


	/*
	 * sigma
	 */
	GtkWidget *sigma_label = gtk_label_new_with_mnemonic(_(" S: "));
	kernel_combo_sigma_visibility(GTK_COMBO_BOX(kernel_combo), sigma_label);
	g_signal_connect(kernel_combo, "changed", G_CALLBACK(kernel_combo_sigma_visibility), sigma_label);
	gtk_box_pack_start(GTK_BOX(window_hbox), sigma_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(sigma_label), GTK_JUSTIFY_RIGHT);

	GtkObject *sigma_adj = gtk_adjustment_new(new_vals.sigma, 0.0, 1024, 0.5, 0, 0);
	GtkWidget *sigma_spin = gtk_spin_button_new(GTK_ADJUSTMENT(sigma_adj), 1, 2);
	kernel_combo_sigma_visibility(GTK_COMBO_BOX(kernel_combo), sigma_spin);
	g_signal_connect(kernel_combo, "changed", G_CALLBACK(kernel_combo_sigma_visibility), sigma_spin);
	gtk_box_pack_start(GTK_BOX(window_hbox), sigma_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(sigma_spin), TRUE);
	gtk_widget_set_tooltip_text(sigma_spin, _(
		"If kernel is Linear or Square, S is the radius after which the weights are zero. "
		"For Exponential it is a scale factor. For Gaussian it is the standard deviation."
		"For \"Inverse to the nth\" it is the exponent. "
		"Generally, the smaller the S the 'sharper' the result."
	));

	g_signal_connect_swapped(sigma_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(sigma_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.sigma);


	/*
	 * norm
	 */
	GtkWidget *norm_label = gtk_label_new_with_mnemonic(_(" Norm: "));
	kernel_combo_norm_visibility(GTK_COMBO_BOX(kernel_combo), norm_label);
	g_signal_connect(kernel_combo, "changed", G_CALLBACK(kernel_combo_norm_visibility), norm_label);
	gtk_box_pack_start(GTK_BOX(window_hbox), norm_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(norm_label), GTK_JUSTIFY_RIGHT);

	GtkObject *norm_adj = gtk_adjustment_new(new_vals.norm, -1.0, 1024, 0.5, 0, 0);
	GtkWidget *norm_spin = gtk_spin_button_new(GTK_ADJUSTMENT(norm_adj), 1, 2);
	kernel_combo_norm_visibility(GTK_COMBO_BOX(kernel_combo), norm_spin);
	g_signal_connect(kernel_combo, "changed", G_CALLBACK(kernel_combo_norm_visibility), norm_spin);
	gtk_box_pack_start(GTK_BOX(window_hbox), norm_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(norm_spin), TRUE);
	gtk_widget_set_tooltip_text(norm_spin, _(
		"Distances are calculated using the formula (|x|^N + |y|^N)^(1/N) . "
		"Norm<0 means maximum norm."
	));

	g_signal_connect_swapped(norm_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(norm_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.norm);


	/*
	 * grid_x, grid_y, offs_x, offs_y
	 */
	GtkWidget *grid_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(grid_hbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), grid_hbox, FALSE, FALSE, 0);

	GtkWidget *grid_label = gtk_label_new_with_mnemonic(_("_Grid: "));
	gtk_widget_show(grid_label);
	gtk_box_pack_start(GTK_BOX(grid_hbox), grid_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(grid_label), GTK_JUSTIFY_RIGHT);

	GtkObject *grid_x_adj = gtk_adjustment_new(new_vals.grid_x, 1.0, 65536.0, 1.0, 0, 0);
	GtkWidget *grid_x_spin = gtk_spin_button_new(GTK_ADJUSTMENT(grid_x_adj), 1, 0);
	gtk_widget_show(grid_x_spin);
	gtk_box_pack_start(GTK_BOX(grid_hbox), grid_x_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(grid_x_spin), TRUE);
	gtk_widget_set_tooltip_text(grid_x_spin, _(
		"Horizontal sample grid spacing. "
		"Planes are calculated only for points of that grid. "
		"This is a speed/quality tradeoff."
	));
	g_signal_connect_swapped(grid_x_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(grid_x_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.grid_x);

	GtkWidget *grid_label2 = gtk_label_new_with_mnemonic(_("x"));
	gtk_widget_show(grid_label2);
	gtk_box_pack_start(GTK_BOX(grid_hbox), grid_label2, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(grid_label2), GTK_JUSTIFY_RIGHT);

	GtkObject *grid_y_adj = gtk_adjustment_new(new_vals.grid_y, 1.0, 65536.0, 1.0, 0, 0);
	GtkWidget *grid_y_spin = gtk_spin_button_new(GTK_ADJUSTMENT(grid_y_adj), 1, 0);
	gtk_widget_show(grid_y_spin);
	gtk_box_pack_start(GTK_BOX(grid_hbox), grid_y_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(grid_y_spin), TRUE);
	gtk_widget_set_tooltip_text(grid_y_spin, _(
		"Vertical sample grid spacing. "
	));
	g_signal_connect_swapped(grid_y_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(grid_y_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.grid_y);

	GtkWidget *grid_label3 = gtk_label_new_with_mnemonic(_(" _Offset: "));
	gtk_widget_show(grid_label3);
	gtk_box_pack_start(GTK_BOX(grid_hbox), grid_label3, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(grid_label3), GTK_JUSTIFY_RIGHT);

	GtkObject *offs_x_adj = gtk_adjustment_new(MIN(new_vals.offs_x,new_vals.grid_x-1), 0.0, new_vals.grid_x-1, 1.0, 0, 0);
	GtkWidget *offs_x_spin = gtk_spin_button_new(GTK_ADJUSTMENT(offs_x_adj), 1, 0);
	gtk_widget_show(offs_x_spin);
	gtk_box_pack_start(GTK_BOX(grid_hbox), offs_x_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(offs_x_spin), TRUE);
	gtk_widget_set_tooltip_text(offs_x_spin, _(
		"Distance of first column of grid pixels from left image edge."
	));
	g_signal_connect_swapped(offs_x_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(offs_x_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.offs_x);
	g_signal_connect(grid_x_adj, "value_changed", G_CALLBACK(adj_set_upper_minus_one), offs_x_adj);

	GtkWidget *grid_label4 = gtk_label_new_with_mnemonic(_(","));
	gtk_widget_show(grid_label4);
	gtk_box_pack_start(GTK_BOX(grid_hbox), grid_label4, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(grid_label4), GTK_JUSTIFY_RIGHT);

	GtkObject *offs_y_adj = gtk_adjustment_new(MIN(new_vals.offs_y,new_vals.grid_y-1), 0.0, new_vals.grid_x-1, 1.0, 0, 0);
	GtkWidget *offs_y_spin = gtk_spin_button_new(GTK_ADJUSTMENT(offs_y_adj), 1, 0);
	gtk_widget_show(offs_y_spin);
	gtk_box_pack_start(GTK_BOX(grid_hbox), offs_y_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(offs_y_spin), TRUE);
	gtk_widget_set_tooltip_text(offs_y_spin, _(
		"Distance of first row of grid pixels from top image edge."
	));
	g_signal_connect_swapped(offs_y_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(offs_y_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.offs_y);
	g_signal_connect(grid_y_adj, "value_changed", G_CALLBACK(adj_set_upper_minus_one), offs_y_adj);


	/*
	 * newlayer
	 */
	GtkWidget *newlayer_check = gtk_check_button_new_with_mnemonic(_("New layer"));
	gtk_widget_show(newlayer_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(newlayer_check), new_vals.newlayer);
	gtk_box_pack_start(GTK_BOX(main_vbox), newlayer_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(newlayer_check, _(
		"Make a new layer for the interpolation result. "
		"The new layer's alpha channel will be the selection. "
	));

	/*g_signal_connect_swapped(newlayer_adj, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(newlayer_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.newlayer);


	/*
	 * slopes
	 */
	GtkWidget *slopes_check = gtk_check_button_new_with_mnemonic(_("Calculate slopes"));
	gtk_widget_show(slopes_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(slopes_check), new_vals.slopes);
	gtk_box_pack_start(GTK_BOX(main_vbox), slopes_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(slopes_check, _(
		"Whether to calculate a plane for each input pixel. "
		"This is how much the light intensity near that point increases or decreases, horizontally and vertically. "
		"This changes the way that the pixel affects nearby interpolated pixels. "
	));

	g_signal_connect_swapped(slopes_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(slopes_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.slopes);


	/*
	 * leastsq
	 */
	GtkWidget *slopes_vbox = gtk_vbox_new(FALSE, 0);
	if (new_vals.slopes) gtk_widget_show(slopes_vbox);
	g_signal_connect(slopes_check, "toggled", G_CALLBACK(check_and_make_visible), slopes_vbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), slopes_vbox, FALSE, FALSE, 0);

	GtkWidget *method_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(method_hbox);
	gtk_box_pack_start(GTK_BOX(slopes_vbox), method_hbox, FALSE, FALSE, 0);

	GtkWidget *method_label = gtk_label_new_with_mnemonic(_("Method: "));
	gtk_widget_show(method_label);
	gtk_box_pack_start(GTK_BOX(method_hbox), method_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(method_label), GTK_JUSTIFY_RIGHT);

	GtkWidget *method_combo = gtk_combo_box_new_text();
	gtk_widget_show(method_combo);
	gtk_combo_box_append_text(GTK_COMBO_BOX(method_combo), _("Old from center pixel"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(method_combo), _("Least squares from center"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(method_combo), _("Flat averages"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(method_combo), _("Least squares"));
	gtk_combo_box_set_active(GTK_COMBO_BOX(method_combo), (int) new_vals.method);
	gtk_box_pack_start(GTK_BOX(method_hbox), method_combo, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(method_combo, _(
		"The method to use for plane fitting. Old methods retain the center pixel."
	));

	g_signal_connect_swapped(method_combo, "changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(method_combo, "changed", G_CALLBACK(gtk_combo_box_get_active_ptr), &new_vals.method);


	/*
	 * linkernel
	 */
	GtkWidget *linkernel_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(linkernel_hbox);
	gtk_box_pack_start(GTK_BOX(slopes_vbox), linkernel_hbox, FALSE, FALSE, 0);

	GtkWidget *linkernel_label = gtk_label_new_with_mnemonic(_("_Kernel: "));
	gtk_widget_show(linkernel_label);
	gtk_box_pack_start(GTK_BOX(linkernel_hbox), linkernel_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(linkernel_label), GTK_JUSTIFY_RIGHT);

	GtkWidget *linkernel_combo = gtk_combo_box_new_text();
	gtk_widget_show(linkernel_combo);
	combo_add_kernel_options(GTK_COMBO_BOX(linkernel_combo));
	gtk_combo_box_set_active(GTK_COMBO_BOX(linkernel_combo), (int) new_vals.linkernel);
	gtk_box_pack_start(GTK_BOX(linkernel_hbox), linkernel_combo, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(linkernel_combo, _(
		"The type of kernel, as above."
	));

	g_signal_connect_swapped(linkernel_combo, "changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(linkernel_combo, "changed", G_CALLBACK(gtk_combo_box_get_active_ptr), &new_vals.linkernel);


	/*
	 * linwindow
	 */
	GtkWidget *linwindow_hbox = gtk_hbox_new(FALSE, 0);
	kernel_combo_window_hbox_visibility(GTK_COMBO_BOX(linkernel_combo), linwindow_hbox);
	g_signal_connect(linkernel_combo, "changed", G_CALLBACK(kernel_combo_window_hbox_visibility), linwindow_hbox);
	gtk_box_pack_start(GTK_BOX(slopes_vbox), linwindow_hbox, FALSE, FALSE, 0);

	GtkWidget *linwindow_label = gtk_label_new_with_mnemonic(_("_Window: "));
	gtk_widget_show(linwindow_label);
	gtk_box_pack_start(GTK_BOX(linwindow_hbox), linwindow_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(linwindow_label), GTK_JUSTIFY_RIGHT);

	GtkObject *linwindow_adj = gtk_adjustment_new(new_vals.linwindow, 0, 1024, 1, 0, 0);
	GtkWidget *linwindow_spin = gtk_spin_button_new(GTK_ADJUSTMENT(linwindow_adj), 1, 0);
	gtk_widget_show(linwindow_spin);
	gtk_box_pack_start(GTK_BOX(linwindow_hbox), linwindow_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(linwindow_spin), TRUE);
	gtk_widget_set_tooltip_text(linwindow_spin, _(
		"The size of the window to use for calculating the horizontal and vertical angles, as above."
		"If you increase the grid spacing, it is a good idea to increase this too."
	));

	g_signal_connect_swapped(linwindow_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(linwindow_adj, "value_changed", G_CALLBACK(gimp_int_adjustment_update), &new_vals.linwindow);


	/*
	 * linsigma
	 */
	GtkWidget *linsigma_label = gtk_label_new_with_mnemonic(_(" S: "));
	kernel_combo_sigma_visibility(GTK_COMBO_BOX(linkernel_combo), linsigma_label);
	g_signal_connect(linkernel_combo, "changed", G_CALLBACK(kernel_combo_sigma_visibility), linsigma_label);
	gtk_box_pack_start(GTK_BOX(linwindow_hbox), linsigma_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(linsigma_label), GTK_JUSTIFY_RIGHT);

	GtkObject *linsigma_adj = gtk_adjustment_new(new_vals.linsigma, 0.0, 1024, 0.5, 0, 0);
	GtkWidget *linsigma_spin = gtk_spin_button_new(GTK_ADJUSTMENT(linsigma_adj), 1, 2);
	kernel_combo_sigma_visibility(GTK_COMBO_BOX(linkernel_combo), linsigma_spin);
	g_signal_connect(linkernel_combo, "changed", G_CALLBACK(kernel_combo_sigma_visibility), linsigma_spin);
	gtk_box_pack_start(GTK_BOX(linwindow_hbox), linsigma_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(linsigma_spin), TRUE);
	gtk_widget_set_tooltip_text(linsigma_spin, _(
		"The parameter for the kernel, as above."
	));

	g_signal_connect_swapped(linsigma_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(linsigma_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.linsigma);


	/*
	 * linnorm
	 */
	GtkWidget *linnorm_label = gtk_label_new_with_mnemonic(_(" Norm: "));
	kernel_combo_norm_visibility(GTK_COMBO_BOX(linkernel_combo), linnorm_label);
	g_signal_connect(linkernel_combo, "changed", G_CALLBACK(kernel_combo_norm_visibility), linnorm_label);
	gtk_box_pack_start(GTK_BOX(linwindow_hbox), linnorm_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(linnorm_label), GTK_JUSTIFY_RIGHT);

	GtkObject *linnorm_adj = gtk_adjustment_new(new_vals.linnorm, -1.0, 1024, 0.5, 0, 0);
	GtkWidget *linnorm_spin = gtk_spin_button_new(GTK_ADJUSTMENT(linnorm_adj), 1, 2);
	kernel_combo_norm_visibility(GTK_COMBO_BOX(linkernel_combo), linnorm_spin);
	g_signal_connect(linkernel_combo, "changed", G_CALLBACK(kernel_combo_norm_visibility), linnorm_spin);
	gtk_box_pack_start(GTK_BOX(linwindow_hbox), linnorm_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(linnorm_spin), TRUE);
	gtk_widget_set_tooltip_text(linnorm_spin, _(
		"The Norm parameter, as above."
	));

	g_signal_connect_swapped(linnorm_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(linnorm_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.linnorm);


	/*
	 * linnewlayer
	 */
	GtkWidget *linnewlayer_check = gtk_check_button_new_with_mnemonic(_("New slopes layer"));
	gtk_widget_show(linnewlayer_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(linnewlayer_check), new_vals.linnewlayer);
	gtk_box_pack_start(GTK_BOX(slopes_vbox), linnewlayer_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(linnewlayer_check, _(
		"Make a new \"grain merge\" layer for the difference between using the calculated slopes and using flat slopes. "
		"The base layer will use the flat slopes."
	));

	/*g_signal_connect_swapped(linnewlayer_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(linnewlayer_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.linnewlayer);


	/*
	 * gamma_compensation
	 */
	GtkWidget *gamma_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(gamma_hbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), gamma_hbox, FALSE, FALSE, 0);

	GtkWidget *gamma_compensation_check = gtk_check_button_new_with_mnemonic(_("Gamma-compressed image"));
	gtk_widget_show(gamma_compensation_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gamma_compensation_check), new_vals.gamma_compensation);
	gtk_box_pack_start(GTK_BOX(gamma_hbox), gamma_compensation_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(gamma_compensation_check, _(
		"Assume image is gamma-compressed (artificially lightened, for display on CRT screens, like most pictures in circulation). "
		"This will map the pixels into linear space before working with them, and the result back into gamma-compressed space. "
		"Floating-point numbers are always used, so there is no loss in bit-depth. "
		"Turn this off if your image is already in linear space."
	));

	g_signal_connect_swapped(gamma_compensation_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(gamma_compensation_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.gamma_compensation);


	/*
	 * gamma
	 */
	GtkObject *gamma_adj = gtk_adjustment_new(new_vals.gamma, 0.05, 1073741824.0, 0.05, 0, 0);
	GtkWidget *gamma_spin = gtk_spin_button_new(GTK_ADJUSTMENT(gamma_adj), 1, 3);
	gtk_widget_show(gamma_spin);
	gtk_box_pack_start(GTK_BOX(gamma_hbox), gamma_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(gamma_spin), TRUE);
	gtk_widget_set_tooltip_text(gamma_spin, _(
		"The decoding gamma value."
	));

	g_signal_connect_swapped(gamma_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);
	g_signal_connect(gamma_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.gamma);


	/*
	 * changesel
	 */
	GtkWidget *changesel_check = gtk_check_button_new_with_mnemonic(_("Modify selection"));
	gtk_widget_show(changesel_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(changesel_check), new_vals.changesel);
	gtk_box_pack_start(GTK_BOX(main_vbox), changesel_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(changesel_check, _(
		"Set new selection according to weight sums. "
		"This will shrink the selection so you can apply the filter again."
	));

	/*g_signal_connect_swapped(changesel_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(changesel_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.changesel);


	/*
	 * lowthres
	 */
	GtkWidget *changesel_vbox = gtk_vbox_new(FALSE, 0);
	if (new_vals.changesel) gtk_widget_show(changesel_vbox);
	g_signal_connect(changesel_check, "toggled", G_CALLBACK(check_and_make_visible), changesel_vbox);
	gtk_box_pack_start(GTK_BOX(main_vbox), changesel_vbox, FALSE, FALSE, 0);

	GtkWidget *lowthres_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(lowthres_hbox);
	gtk_box_pack_start(GTK_BOX(changesel_vbox), lowthres_hbox, FALSE, FALSE, 0);

	GtkWidget *lowthres_label = gtk_label_new_with_mnemonic(_("_Low threshold: "));
	gtk_widget_show(lowthres_label);
	gtk_box_pack_start(GTK_BOX(lowthres_hbox), lowthres_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(lowthres_label), GTK_JUSTIFY_RIGHT);

	GtkObject *lowthres_adj = gtk_adjustment_new(new_vals.lowthres, 0.0, 1073741824.0, 0.01, 0, 0);
	GtkWidget *lowthres_spin = gtk_spin_button_new(GTK_ADJUSTMENT(lowthres_adj), 0.05, 7);
	gtk_widget_show(lowthres_spin);
	gtk_box_pack_start(GTK_BOX(lowthres_hbox), lowthres_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(lowthres_spin), TRUE);
	gtk_widget_set_tooltip_text(lowthres_spin, _(
		"If the selection is modified, this is the weight-sum value that will give completely selected pixels. "
		"This must be smaller than \"High threshold\". "
		"Smaller values here and in \"High threshold\" shrink the selection more."
	));

	/*g_signal_connect_swapped(lowthres_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(lowthres_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.lowthres);


	/*
	 * highthres
	 */
	GtkWidget *highthres_hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(highthres_hbox);
	gtk_box_pack_start(GTK_BOX(changesel_vbox), highthres_hbox, FALSE, FALSE, 0);

	GtkWidget *highthres_label = gtk_label_new_with_mnemonic(_("_High threshold: "));
	gtk_widget_show(highthres_label);
	gtk_box_pack_start(GTK_BOX(highthres_hbox), highthres_label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(highthres_label), GTK_JUSTIFY_RIGHT);

	GtkObject *highthres_adj = gtk_adjustment_new(new_vals.highthres, 0.0, 1073741824.0, 0.01, 0, 0);
	GtkWidget *highthres_spin = gtk_spin_button_new(GTK_ADJUSTMENT(highthres_adj), 0.05, 7);
	gtk_widget_show(highthres_spin);
	gtk_box_pack_start(GTK_BOX(highthres_hbox), highthres_spin, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(highthres_spin), TRUE);
	gtk_widget_set_tooltip_text(highthres_spin, _(
		"If the selection is modified, this is the weight-sum value that will give completely unselected pixels. "
		"This must be larger than \"Low threshold\". "
		"Smaller values here and in \"Low threshold\" shrink the selection more."
	));

	/*g_signal_connect_swapped(highthres_adj, "value_changed", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(highthres_adj, "value_changed", G_CALLBACK(gimp_float_adjustment_update), &new_vals.highthres);


	/*
	 * logthres
	 */
	GtkWidget *logthres_check = gtk_check_button_new_with_mnemonic(_("Log scale new selection"));
	gtk_widget_show(logthres_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(logthres_check), new_vals.logthres);
	gtk_box_pack_start(GTK_BOX(changesel_vbox), logthres_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(logthres_check, _(
		"Use with Exponential and Gaussian kernels. "
		"By default there is a linear relation between the weight-sums and the new selection."
	));

	/*g_signal_connect_swapped(logthres_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(logthres_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.logthres);


	/*
	 * multmask
	 */
	GtkWidget *multmask_check = gtk_check_button_new_with_mnemonic(_("Multiply with input mask"));
	gtk_widget_show(multmask_check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(multmask_check), new_vals.multmask);
	gtk_box_pack_start(GTK_BOX(changesel_vbox), multmask_check, FALSE, FALSE, 0);
	gtk_widget_set_tooltip_text(multmask_check, _(
		"Whether to multiply new selection with input mask. "
		"New selection is always multiplied with old selection."
	));

	/*g_signal_connect_swapped(multmask_check, "toggled", G_CALLBACK(gimp_preview_invalidate), preview);*/
	g_signal_connect(multmask_check, "toggled", G_CALLBACK(gimp_toggle_button_update), &new_vals.multmask);


	gtk_widget_show(dlg);

	gboolean run = (gimp_dialog_run(GIMP_DIALOG(dlg)) == GTK_RESPONSE_OK);

	if (run) {
		gimp_int_combo_box_get_active(GIMP_INT_COMBO_BOX(chanid_combo), &new_vals.chanid);
		new_vals.usesel = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(usesel_check));
		new_vals.all_white_input = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(all_white_input_check));
		new_vals.halfsel = gtk_adjustment_get_value(GTK_ADJUSTMENT(halfsel_adj));

		new_vals.window = gtk_adjustment_get_value(GTK_ADJUSTMENT(window_adj));
		new_vals.sigma = gtk_adjustment_get_value(GTK_ADJUSTMENT(sigma_adj));
		new_vals.kernel = gtk_combo_box_get_active(GTK_COMBO_BOX(kernel_combo));
		new_vals.newlayer = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(newlayer_check));

		new_vals.multmask = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(multmask_check));

		new_vals.slopes = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(slopes_check));
		new_vals.method = gtk_combo_box_get_active(GTK_COMBO_BOX(method_combo));
		new_vals.linwindow = gtk_adjustment_get_value(GTK_ADJUSTMENT(linwindow_adj));
		new_vals.linsigma = gtk_adjustment_get_value(GTK_ADJUSTMENT(linsigma_adj));
		new_vals.linkernel = gtk_combo_box_get_active(GTK_COMBO_BOX(linkernel_combo));
		new_vals.linnewlayer = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(linnewlayer_check));

		new_vals.changesel = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(changesel_check));
		new_vals.lowthres = gtk_adjustment_get_value(GTK_ADJUSTMENT(lowthres_adj));
		new_vals.highthres = gtk_adjustment_get_value(GTK_ADJUSTMENT(highthres_adj));
		new_vals.logthres = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(logthres_check));

		memcpy(vals, &new_vals, sizeof(new_vals));
		memcpy(ui_vals, &new_ui_vals, sizeof(new_vals));
	}

	gtk_widget_destroy(dlg);

	return run;
}
