/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libgimp/gimp.h>

#include "main.h"
#include "render.h"
#include "ring.h"

#include "plugin-intl.h"

#define SKIP_UNSELECTED 1


typedef struct InterpolateKernel {
	gfloat *values;
	gint width, height;
} InterpolateKernel;
static void kernel_init(InterpolateKernel *kernel, InterpolateKernelType type, gint window, gfloat s, gfloat norm);
static void kernel_fini(InterpolateKernel *kernel);
inline static gfloat kernel_value(InterpolateKernel *kernel, gint deltax, gint deltay) {
	return kernel->values[kernel->width*(kernel->height/2+deltay) + (kernel->width/2+deltax)];
}

static void interpolate(GimpDrawable *drawable, PlugInVals *opts, gint image_id, GimpPreview *preview);
inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres);
static void slopes_grid(GimpDrawable *drawable, GimpDrawable *mask,
                        gint left, gint top, gint right, gint bottom,
                        InterpolateKernel *kernel, PlugInVals *opts,
                        gfloat *g,
                        gfloat **pz, gfloat **pdx, gfloat **pdy, GimpPreview *preview);
inline static void flat_avg_slope(Ring *ring, Ring *mring, gint x,
                                  InterpolateKernel *kernel, gboolean halfsel,
                                  guchar *center, gfloat *g,
                                  gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel);
inline static void old_slope(Ring *ring, Ring *mring, gint x,
                             InterpolateKernel *kernel, gint halfsel,
                             guchar *center, gfloat *g,
                             gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel);
inline static void least_squares_slope(Ring *ring, Ring *mring, gint x,
                                       InterpolateKernel *kernel, gint halfsel,
                                       guchar *center, gfloat *g,
                                       gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel);
inline static void least_squares3_slope(Ring *ring, Ring *mring, gint x,
                                        InterpolateKernel *kernel, gint halfsel,
                                        guchar *center, gfloat *g,
                                        gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel);
inline static void flat_slope(GimpImageType type, guchar *center, gfloat *g,
                              gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel);
inline static void float_pixel(GimpImageType type, guchar *pixel, gfloat *g, gfloat *fpixel);

static gfloat *gamma_new(gfloat g);
static void gamma_free(gfloat *gamma);

inline static gboolean gauss_solve(gfloat *m, gint rows, gint cols);
inline static void swap_row(gfloat *a, gfloat *b, gint count);

inline static gfloat gaussian(gfloat x, gfloat s);
inline static void gray_pixel(GimpImageType type, guchar *pixel);


/******************************************************************************/


void invoke_render(InterpolateInvocation *inv, GimpPreview *preview) {
	return render(inv->drawable, inv->vals, preview);
}

void render(GimpDrawable *drawable, PlugInVals *vals, GimpPreview *preview) {
	gint32 image_id = gimp_drawable_get_image(drawable->drawable_id);

	/* to restore the previously active layer */
	gint32 active_channel = gimp_image_get_active_channel(image_id);

	if (preview) {
		gimp_image_undo_freeze(image_id);
	} else {
		gimp_image_undo_group_start(image_id);
	}

	interpolate(drawable, vals, image_id, preview);

	if (preview) {
		gimp_image_undo_thaw(image_id);
	} else {
		gimp_image_undo_group_end(image_id);
	}

	if (active_channel == -1) {
		gimp_image_unset_active_channel(image_id);
	} else {
		gimp_image_set_active_channel(image_id, active_channel);
	}
}


static void interpolate(GimpDrawable *drawable, PlugInVals *opts, gint image_id, GimpPreview *preview) {
	GimpImageType drawable_type = gimp_drawable_type(drawable->drawable_id);
	gint channels = gimp_drawable_bpp(drawable->drawable_id);

	gboolean changesel = opts->changesel && !preview;
	gboolean newlayer = opts->newlayer && !preview;
	gboolean linnewlayer = opts->linnewlayer && !preview;

	gboolean wslopes = opts->slopes;
	gint halfsel = opts->halfsel;

	gfloat lowthres = opts->lowthres;
	gfloat highthres = opts->highthres;
	gboolean logthres = opts->logthres;

	gint left,top,right,bottom,width,height;
	if (preview) {
		gimp_preview_get_position(preview, &left, &top);
		gimp_preview_get_size(preview, &width, &height);
		right = left + width;
		bottom = top + height;
	} else {
		gimp_drawable_mask_bounds(drawable->drawable_id, &left, &top, &right, &bottom);
		width = right-left;
		height = bottom-top;
	}


	/*
	 * make gamma curve
	 */
	gfloat inv_gamma = 1.0/opts->gamma;
	gfloat *g = 0;
	if (opts->gamma_compensation) {
		g = gamma_new(opts->gamma);
	}


	/*
	 * make white input mask (or copy existing), and subtract selection
	 */
	gint tmp_mask_id = -1;
	GimpDrawable *mask = 0;

	if (!opts->all_white_input && gimp_drawable_is_valid(opts->chanid)) {
		if (opts->usesel) {
			tmp_mask_id = gimp_channel_copy(opts->chanid);
			gimp_image_add_channel(image_id, tmp_mask_id, 0);
			mask = gimp_drawable_get(tmp_mask_id);
		} else {
			mask = gimp_drawable_get(opts->chanid);
		}
	}
	
	if (!mask) {
		GimpRGB rgb;
		gimp_rgb_set(&rgb, 0, 0, 0);
		tmp_mask_id = gimp_channel_new(
			image_id, "Input mask temporary",
			gimp_image_width(image_id),
			gimp_image_height(image_id),
			0.5, &rgb
		);
		gimp_drawable_fill(tmp_mask_id, GIMP_WHITE_FILL);
		gimp_image_add_channel(image_id, tmp_mask_id, 0);

		mask = gimp_drawable_get(tmp_mask_id);
	}

	if (opts->usesel) {
		GimpPixelRgn outm;
		gimp_pixel_rgn_init(&outm, mask, left, top, width, height, TRUE, TRUE);
		guchar *outmrow = g_new(guchar, mask->bpp*width);
		memset(outmrow, 0, mask->bpp*width);
		gint y;
		for (y = 0 ; y < height ; ++y) {
			gimp_pixel_rgn_set_row(&outm,outmrow,0,y,width);
		}
		gimp_drawable_flush(mask);
		gimp_drawable_merge_shadow(tmp_mask_id, TRUE);
	}

	gint mchannels = gimp_drawable_bpp(mask->drawable_id);


	/*
	 * save selection to channel, so we can modify it (or to skip unselected pixels)
	 */
	gint sel_id = -1;
	GimpDrawable *sel = 0;
	#ifdef SKIP_UNSELECTED
		sel_id = gimp_selection_save(image_id);
		sel = gimp_drawable_get(sel_id);
	#else
		if (changesel) {
			sel_id = gimp_selection_save(image_id);
			sel = gimp_drawable_get(sel_id);
		}
	#endif


	/*
	 * make the new layers, if we are making new layers
	 */
	GimpDrawable *outdrawable, *outslopesdrawable;
	if (newlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Interpolated"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_NORMAL_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outdrawable = gimp_drawable_get(layer_id);
	} else {
		outdrawable = drawable;
	}

	GimpImageType outdrawable_type = gimp_drawable_type(outdrawable->drawable_id);
	gint outchannels = gimp_drawable_bpp(outdrawable->drawable_id);
	gint outslopeschannels;

	if (linnewlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Slopes Offset"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_GRAIN_MERGE_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outslopesdrawable = gimp_drawable_get(layer_id);
		outslopeschannels = gimp_drawable_bpp(outslopesdrawable->drawable_id);
	} else {
		outslopesdrawable = 0;
		outslopeschannels = outchannels;
	}


	/*
	 * make kernel
	 */
	InterpolateKernel kernel;
	kernel_init(&kernel, opts->kernel, opts->window, opts->sigma, opts->norm);


	/*
	 * calculate slopes of sample areas (this is the first pass)
	 *
	 * when the grid is 1x1, this gives two inwidth*inheight*bpp tables
	 * with the horizontal and vertical slopes for each pixel
	 *
	 * otherwise the size depends on the grid pixels that are enclosed
	 */
	gint inleft = MAX(left-kernel.width/2,0);
	gint intop = MAX(top-kernel.height/2,0);
	gint inwidth = MIN(right+kernel.width/2,drawable->width)-inleft;
	gint inheight = MIN(bottom+kernel.height/2,drawable->height)-intop;

	gint gleft = (inleft + opts->grid_x - 1 - opts->offs_x)/opts->grid_x;
	gint gtop = (intop + opts->grid_y - 1 - opts->offs_y)/opts->grid_y;
	gint gright = (inleft + inwidth + opts->grid_x - 1 - opts->offs_x)/opts->grid_x;
	gint gbottom = (intop + inheight + opts->grid_y - 1 - opts->offs_y)/opts->grid_y;
	gint gwidth = gright-gleft;
	gint gheight = gbottom-gtop;

	gfloat *z=0, *dx=0, *dy=0;
	if (wslopes && gwidth>0 && gheight>0) {
		InterpolateKernel linkernel;
		kernel_init(&linkernel, opts->linkernel, opts->linwindow, opts->linsigma, opts->linnorm);
		slopes_grid(drawable, mask,
		            gleft, gtop, gright, gbottom,
		            &linkernel, opts,
		            g,
		            &z, &dx, &dy, preview);
		kernel_fini(&linkernel);
	}

	gint tw = gimp_tile_width();
	gimp_tile_cache_ntiles(4*(drawable->width+tw-1)/tw);


	/*
	 * get input/output pixel regions for interpolation pass
	 */
	Ring in; ring_init(&in, drawable, left, top, right, bottom, kernel.width/2, kernel.height/2);
	Ring inm; ring_init(&inm, mask, left, top, right, bottom, kernel.width/2, kernel.height/2);

	GimpPixelRgn out;
	if (newlayer) {
		gimp_pixel_rgn_init(&out, outdrawable, 0, 0, width, height, TRUE, TRUE);
	} else {
		gimp_pixel_rgn_init(&out, outdrawable, left, top, width, height, !preview, TRUE);
	}
	guchar *outrow = g_new(guchar, outchannels*width);

	GimpPixelRgn outslopes;
	guchar *outslopesrow = 0;
	if (linnewlayer) {
		gimp_pixel_rgn_init(&outslopes, outslopesdrawable, 0, 0, width, height, TRUE, TRUE);
		outslopesrow = g_new(guchar, outslopeschannels*width);
	}


	/*
	 * get selection, to skip unselected pixels (and, optionally, to change it)
	 */
	GimpPixelRgn ins;
	guchar *insrow = 0;
	#ifdef SKIP_UNSELECTED
		gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
		insrow = g_new(guchar, mchannels*width);
	#else
		if (changesel) {
			gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
			insrow = g_new(guchar, mchannels*width);
		}
	#endif

	GimpPixelRgn outs;
	guchar *outsrow = 0;
	if (changesel) {
		gimp_pixel_rgn_init(&outs, sel, left, top, width, height, TRUE, TRUE);
		outsrow = g_new(guchar, mchannels*width);
	}


	/*
	 * do the interpolation
	 */
	if (!preview) gimp_progress_init(_("Interpolating..."));
	for (; ring_loop(&in) && ring_loop(&inm) ; ring_rotate(&in), ring_rotate(&inm)) {
		guchar *inmrow = 0;
		if (changesel) {
			inmrow = ring_row(&inm);
		}

		#ifdef SKIP_UNSELECTED
			gimp_pixel_rgn_get_row(&ins, insrow, left, in.y, width);
		#else
			if (changesel) {
				gimp_pixel_rgn_get_row(&ins, insrow, left, in.y, width);
			}
		#endif

		gint deltay_min = ring_min_deltay(&in);
		gint deltay_max = ring_max_deltay(&in);
		deltay_min = (in.y + deltay_min + opts->grid_y - opts->offs_y - 1)/opts->grid_y*opts->grid_y + opts->offs_y - in.y;
		deltay_max = ((in.y + deltay_max + opts->grid_y - opts->offs_y)/opts->grid_y-1)*opts->grid_y + opts->offs_y - in.y;

		gint x;
		for (x = left ; x < right ; ++x) {
			guchar *inspixel = 0;
			#ifdef SKIP_UNSELECTED
				inspixel = insrow + (x-left)*mchannels;
			#else
				if (changesel) {
					inspixel = insrow + (x-left)*mchannels;
				}
			#endif
			guchar *outpixel = outrow + (x-left)*outchannels;
			guchar *outslopespixel = 0;
			if (linnewlayer) {
				outslopespixel = outslopesrow + (x-left)*outslopeschannels;
			}

			gfloat sum[4];
			sum[0] = 0.0;
			sum[1] = 0.0;
			sum[2] = 0.0;
			sum[3] = 0.0;
			gfloat dsum[4];
			dsum[0] = 0.0;
			dsum[1] = 0.0;
			dsum[2] = 0.0;
			dsum[3] = 0.0;
			gfloat ws = 0.0;

			gint deltax_min = ring_min_deltax(&in, x);
			gint deltax_max = ring_max_deltax(&in, x);
			deltax_min = (x + deltax_min + opts->grid_x - opts->offs_x - 1)/opts->grid_x*opts->grid_x + opts->offs_x - x;
			deltax_max = ((x + deltax_max + opts->grid_x - opts->offs_x)/opts->grid_x-1)*opts->grid_x + opts->offs_x - x;

			gint deltay;
			#ifdef SKIP_UNSELECTED
			if (inspixel[0])
			#endif
			for (deltay = deltay_min ; deltay <= deltay_max ; deltay += opts->grid_y) {
				guchar *inrow2 = 0;
				guchar *inmrow2 = ring_row_delta(&inm, deltay);

				gfloat *zrow = 0;
				gfloat *dxrow = 0;
				gfloat *dyrow = 0;
				if (wslopes) {
					zrow = z + channels*gwidth*((in.y+deltay)/opts->grid_y - gtop);
					dxrow = dx + channels*gwidth*((in.y+deltay)/opts->grid_y - gtop);
					dyrow = dy + channels*gwidth*((in.y+deltay)/opts->grid_y - gtop);
				} else {
					inrow2 = ring_row_delta(&in, deltay);
				}

				gint deltax;
				for (deltax = deltax_min ; deltax <= deltax_max ; deltax += opts->grid_x) {
					guchar *inmpixel2 = inmrow2 + (x+deltax-left)*mchannels;
					if (inmpixel2[0]<halfsel)
						continue;

					gfloat w = kernel_value(&kernel, deltax, deltay);
					if (w >= 0.0) {
						/*
						 * ordinary input pixel
						 */
						if (halfsel<255)
							w *= inmpixel2[0];
					} else {
						/*
						 * -1 means infinity. this input pixel has infinite weight
						 * (center pixel in 'inverse' kernels)
						 */
						ws = 0.0;
						sum[0] = 0.0;
						sum[1] = 0.0;
						sum[2] = 0.0;
						sum[3] = 0.0;
						dsum[0] = 0.0;
						dsum[1] = 0.0;
						dsum[2] = 0.0;
						dsum[3] = 0.0;
						w = -highthres;
					}


					if (wslopes) {
						/* no need for gamma correction here, the slopes are already in linear space */
						gfloat *zpixel = zrow + ((x+deltax)/opts->grid_x-gleft)*channels;
						gfloat *dxpixel = dxrow + ((x+deltax)/opts->grid_x-gleft)*channels;
						gfloat *dypixel = dyrow + ((x+deltax)/opts->grid_x-gleft)*channels;

						switch (drawable_type) {
						case GIMP_RGBA_IMAGE:
							sum[3] += w*zpixel[3];
							dsum[3] += w*( - dxpixel[3]*deltax - dypixel[3]*deltay);
						case GIMP_RGB_IMAGE:
							sum[2] += w*zpixel[2];
							dsum[2] += w*( - dxpixel[2]*deltax - dypixel[2]*deltay);
						case GIMP_GRAYA_IMAGE:
							sum[1] += w*zpixel[1];
							dsum[1] += w*( - dxpixel[1]*deltax - dypixel[1]*deltay);
						case GIMP_GRAY_IMAGE: default:
							sum[0] += w*zpixel[0];
							dsum[0] += w*( - dxpixel[0]*deltax - dypixel[0]*deltay);
							break;
						}
					} else {
						guchar *inpixel2 = inrow2 + (x+deltax-left)*channels;

						if (g) {  /* gamma-compressed image */
							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:
								sum[3] += w*inpixel2[3];
								sum[2] += w*g[inpixel2[2]];
								sum[1] += w*g[inpixel2[1]];
								sum[0] += w*g[inpixel2[0]];
								break;
							case GIMP_RGB_IMAGE:
								sum[2] += w*g[inpixel2[2]];
								sum[1] += w*g[inpixel2[1]];
								sum[0] += w*g[inpixel2[0]];
								break;
							case GIMP_GRAYA_IMAGE:
								sum[1] += w*inpixel2[1];
								sum[0] += w*g[inpixel2[0]];
								break;
							case GIMP_GRAY_IMAGE: default:
								sum[0] += w*g[inpixel2[0]];
								break;
							}
						} else {  /* linear image, or gamma correction disabled */
							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:  sum[3] += w*inpixel2[3];
							case GIMP_RGB_IMAGE:   sum[2] += w*inpixel2[2];
							case GIMP_GRAYA_IMAGE: sum[1] += w*inpixel2[1];
								default:
							case GIMP_GRAY_IMAGE:  sum[0] += w*inpixel2[0]; break;
							}
						}
					}

					ws += w;

					if (w < 0.0)
						goto breakout;
				}
			}

			breakout:

			/*
			 * now we can calculate the average, to set the output pixel
			 */
			if (wslopes) {
				if (linnewlayer) {
					switch (drawable_type) {
					case GIMP_GRAY_IMAGE: default:
						break;
					case GIMP_GRAYA_IMAGE:
						sum[1] += dsum[1];  /* alpha of slopes layer becomes equal to selection */
						break;
					case GIMP_RGB_IMAGE:
						break;
					case GIMP_RGBA_IMAGE:
						sum[3] += dsum[3];  /* alpha of slopes layer becomes equal to selection */
						break;
					}

				} else {
					sum[0] += dsum[0];
					sum[1] += dsum[1];
					sum[2] += dsum[2];
					sum[3] += dsum[3];
				}
			}

			if (ws) {
				if (g) {
					switch (drawable_type) {
					case GIMP_GRAY_IMAGE: default:
						outpixel[0] = CLAMP0255(pow(sum[0]/ws/256.0, inv_gamma)*256.0);
						if (newlayer) {
							outpixel[1] = 255;
						}
						break;
					case GIMP_GRAYA_IMAGE:
						outpixel[0] = CLAMP0255(pow(sum[0]/ws/256.0, inv_gamma)*256.0);
						outpixel[1] = CLAMP0255(sum[1]/ws);
						break;
					case GIMP_RGB_IMAGE:
						outpixel[0] = CLAMP0255(pow(sum[0]/ws/256.0, inv_gamma)*256.0);
						outpixel[1] = CLAMP0255(pow(sum[1]/ws/256.0, inv_gamma)*256.0);
						outpixel[2] = CLAMP0255(pow(sum[2]/ws/256.0, inv_gamma)*256.0);
						if (newlayer) {
							outpixel[3] = 255;
						}
						break;
					case GIMP_RGBA_IMAGE:
						outpixel[0] = CLAMP0255(pow(sum[0]/ws/256.0, inv_gamma)*256.0);
						outpixel[1] = CLAMP0255(pow(sum[1]/ws/256.0, inv_gamma)*256.0);
						outpixel[2] = CLAMP0255(pow(sum[2]/ws/256.0, inv_gamma)*256.0);
						outpixel[3] = CLAMP0255(sum[3]/ws);
						break;
					}
				} else {
					switch (drawable_type) {
					case GIMP_GRAY_IMAGE: default:
						outpixel[0] = CLAMP0255(sum[0]/ws);
						if (newlayer) {
							outpixel[1] = 255;
						}
						break;
					case GIMP_GRAYA_IMAGE:
						outpixel[0] = CLAMP0255(sum[0]/ws);
						outpixel[1] = CLAMP0255(sum[1]/ws);
						break;
					case GIMP_RGB_IMAGE:
						outpixel[0] = CLAMP0255(sum[0]/ws);
						outpixel[1] = CLAMP0255(sum[1]/ws);
						outpixel[2] = CLAMP0255(sum[2]/ws);
						if (newlayer) {
							outpixel[3] = 255;
						}
						break;
					case GIMP_RGBA_IMAGE:
						outpixel[0] = CLAMP0255(sum[0]/ws);
						outpixel[1] = CLAMP0255(sum[1]/ws);
						outpixel[2] = CLAMP0255(sum[2]/ws);
						outpixel[3] = CLAMP0255(sum[3]/ws);
						break;
					}
				}
			} else {
				/*
				 * there were no input pixels in range, so set pixel to gray
				 */
				gray_pixel(outdrawable_type, outpixel);
			}

			if (wslopes && linnewlayer) {
				if (ws) {
					if (g) {
						/* making a separate slopes layer when you have a gamma-compressed image is not recommended */
						switch (drawable_type) {
						case GIMP_GRAY_IMAGE: default:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/ws/256.0, inv_gamma) - pow(sum[0]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[1] = 255;
							break;
						case GIMP_GRAYA_IMAGE:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/ws/256.0, inv_gamma) - pow(sum[0]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[1] = CLAMP0255(sum[1]/ws);
							break;
						case GIMP_RGB_IMAGE:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/ws/256.0, inv_gamma) - pow(sum[0]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[1] = CLAMP0255(128 + (pow((sum[1]+dsum[1])/ws/256.0, inv_gamma) - pow(sum[1]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[2] = CLAMP0255(128 + (pow((sum[2]+dsum[2])/ws/256.0, inv_gamma) - pow(sum[2]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[3] = 255;
							break;
						case GIMP_RGBA_IMAGE:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/ws/256.0, inv_gamma) - pow(sum[0]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[1] = CLAMP0255(128 + (pow((sum[1]+dsum[1])/ws/256.0, inv_gamma) - pow(sum[1]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[2] = CLAMP0255(128 + (pow((sum[2]+dsum[2])/ws/256.0, inv_gamma) - pow(sum[2]/ws/256.0, inv_gamma))*256.0);
							outslopespixel[3] = CLAMP0255(sum[3]/ws);
							break;
						}
					} else {
						switch (drawable_type) {
						case GIMP_GRAY_IMAGE: default:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = 255;
							break;
						case GIMP_GRAYA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(sum[1]/ws);
							break;
						case GIMP_RGB_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/ws);
							outslopespixel[3] = 255;
							break;
						case GIMP_RGBA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/ws);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/ws);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/ws);
							outslopespixel[3] = CLAMP0255(sum[3]/ws);
							break;
						}
					}
				} else {
					gray_pixel(outdrawable_type, outslopespixel);
				}
			}

			if (changesel) {
				guchar *inmpixel = inmrow + (x-left)*mchannels;
				guchar *outspixel = outsrow + (x-left)*mchannels;

				/*
				 * input mask is multiplied with pixels without being divided by 255, so divide here
				 */
				if (halfsel<255)
					ws /= 255.0;

				gfloat outsel = wsum2sel(ws, lowthres, highthres, logthres) * inspixel[0];
				if (opts->multmask) outsel *= inmpixel[0]/255.0;
				outspixel[0] = outsel;
			}
		}

		if (newlayer) {
			gimp_pixel_rgn_set_row(&out,outrow,0,in.y-top,right-left);
		} else {
			gimp_pixel_rgn_set_row(&out,outrow,left,in.y,right-left);
		}
		if (linnewlayer) {
			gimp_pixel_rgn_set_row(&outslopes,outslopesrow,0,in.y-top,right-left);
		}
		if (changesel) {
			gimp_pixel_rgn_set_row(&outs,outsrow,left,in.y,right-left);
		}

		if (!preview && !(in.y%4)) {
			gimp_progress_update(1.0*(in.y-top+1)/(bottom-top));
		}
	}
	if (!preview) gimp_progress_end();


	/*
	 * done with the pixel regions, kernel and slopes
	 */
	if (changesel) {
		g_free(outsrow);
	}
	if (linnewlayer) {
		g_free(outslopesrow);
	}
	g_free(outrow);

	ring_fini(&inm);
	ring_fini(&in);

	kernel_fini(&kernel);

	if (dx) g_free(dx);
	if (dy) g_free(dy);


	/*
	 * the plumbing code
	 */
	if (preview) {
		gimp_drawable_preview_draw_region(GIMP_DRAWABLE_PREVIEW(preview), &out);
	} else {
		gimp_drawable_flush(outdrawable);
		gimp_drawable_merge_shadow(outdrawable->drawable_id, TRUE);
		if (newlayer) {
			gimp_drawable_update(outdrawable->drawable_id, 0, 0, width, height);
			gimp_drawable_detach(outdrawable);
		} else {
			gimp_drawable_update(outdrawable->drawable_id, left, top, width, height);
		}

		if (linnewlayer) {
			gimp_drawable_flush(outslopesdrawable);
			gimp_drawable_merge_shadow(outslopesdrawable->drawable_id, TRUE);
			gimp_drawable_update(outslopesdrawable->drawable_id, 0, 0, width, height);
			gimp_drawable_detach(outslopesdrawable);
		}
	}


	/*
	 * only remove the selection if we are modifying it
	 */
	if (changesel) {
		gint tmp_sel_id = gimp_selection_save(image_id);
		gimp_selection_none(image_id);

		gimp_drawable_flush(sel);
		gimp_drawable_merge_shadow(sel->drawable_id, TRUE);
		gimp_drawable_update(sel->drawable_id, left, top, width, height);

		gimp_selection_load(tmp_sel_id);
		gimp_image_remove_channel(image_id, tmp_sel_id);
	}


	/*
	 * no more need for temporary selection channel
	 */
	if (changesel) {
		gimp_selection_load(sel_id);
		#ifndef SKIP_UNSELECTED
			gimp_drawable_detach(sel);
			gimp_image_remove_channel(image_id, sel_id);
		#endif
	}
	#ifdef SKIP_UNSELECTED
		gimp_drawable_detach(sel);
		gimp_image_remove_channel(image_id, sel_id);
	#endif


	/*
	 * remove input mask channel if it is our temporary
	 */
	gimp_drawable_detach(mask);
	if (tmp_mask_id != -1) {
		gimp_image_remove_channel(image_id, tmp_mask_id);
	}


	/*
	 * free gamma curve
	 */
	if (g) {
		gamma_free(g);
	}
}


inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres) {
	if (logthres) w = log(w);

	gfloat ww;
	if (w>=ht || w<0) {
		ww = 1.0;
	} else if (w<=lt) {
		ww = 0.0;
	} else {
		ww = (w-lt)/(ht-lt);
	}
	return 1.0-ww;
}


static void slopes_grid(GimpDrawable *drawable, GimpDrawable *mask,
                        gint gleft, gint gtop, gint gright, gint gbottom,
                        InterpolateKernel *kernel, PlugInVals *opts,
                        gfloat *g,
                        gfloat **pz, gfloat **pdx, gfloat **pdy, GimpPreview *preview) {
	const gint halfsel = opts->halfsel;
	const gboolean method = opts->method;

	gint gwidth = gright-gleft;
	gint gheight = gbottom-gtop;

	if (gwidth<=0 || gheight<=0) {
		*pz = 0;
		*pdx = 0;
		*pdy = 0;
		return;
	}

	gint left = gleft*opts->grid_x + opts->offs_x;
	gint top = gtop*opts->grid_y + opts->offs_y;
	gint right = gright*opts->grid_x + opts->offs_x - opts->grid_x + 1;
	gint bottom = gbottom*opts->grid_y + opts->offs_y - opts->grid_y + 1;

	Ring ring; ring_init(&ring, drawable, left, top, right, bottom, kernel->width/2, kernel->height/2);
	Ring mring; ring_init(&mring, mask, left, top, right, bottom, kernel->width/2, kernel->height/2);
	gfloat *z = *pz = g_new(gfloat, gheight*gwidth*ring.bpp);
	gfloat *dx = *pdx = g_new(gfloat, gheight*gwidth*ring.bpp);
	gfloat *dy = *pdy = g_new(gfloat, gheight*gwidth*ring.bpp);

	if (!preview) gimp_progress_init(_("Calculating slopes..."));
	gint y;
	for (y = 0 ; y < gheight ; ++y) {
		guchar *row = ring_row(&ring);
		guchar *mrow = ring_row(&mring);
		gfloat *zrow = z + y*gwidth*ring.bpp;
		gfloat *dxrow = dx + y*gwidth*ring.bpp;
		gfloat *dyrow = dy + y*gwidth*ring.bpp;

		gint x;
		for (x = 0 ; x < gwidth ; ++x) {
			guchar *pixel = row + x*opts->grid_x*ring.bpp;  /* center pixel */
			guchar *mpixel = mrow + x*opts->grid_x*mring.bpp;
			gfloat *zpixel = zrow + x*ring.bpp;
			gfloat *dxpixel = dxrow + x*ring.bpp;
			gfloat *dypixel = dyrow + x*ring.bpp;

			/*
			 * calculate slope of nearby pixels
			 */
			if (mpixel[0]<halfsel) {
				flat_slope(ring.type, pixel, g, zpixel, dxpixel, dypixel);
			} else {
				switch (method) {
				case METHOD_FLAT_AVERAGES:
					flat_avg_slope(&ring, &mring, left + x*opts->grid_x, kernel, halfsel, pixel, g, zpixel, dxpixel, dypixel);
					break;
				case METHOD_OLD_CENTER: default:
					old_slope(&ring, &mring, left + x*opts->grid_x, kernel, halfsel, pixel, g, zpixel, dxpixel, dypixel);
					break;
				case METHOD_LEASTSQ_CENTER:
					least_squares_slope(&ring, &mring, left + x*opts->grid_x, kernel, halfsel, pixel, g, zpixel, dxpixel, dypixel);
					break;
				case METHOD_LEASTSQ:
					least_squares3_slope(&ring, &mring, left + x*opts->grid_x, kernel, halfsel, pixel, g, zpixel, dxpixel, dypixel);
					break;
				}
			}
		}

		if (!preview && !(ring.y%4)) {
			gimp_progress_update(1.0*(y+1)/gheight);
		}

		/*
		 * go down grid_y lines
		 */
		gint i;
		for (i = 0 ; i < opts->grid_y ; ++i) {
			ring_rotate(&ring); ring_rotate(&mring);
		}
	}
	if (!preview) gimp_progress_end();
	ring_fini(&ring);
}


inline static void flat_avg_slope(Ring *ring, Ring *mring, gint x,
                                  InterpolateKernel *kernel, gboolean halfsel,
                                  guchar *center, gfloat *g,
                                  gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel) {
	float_pixel(ring->type, center, g, zpixel);

	gfloat wsum = 0.0;
	gfloat zsum[4];
	zsum[0] = 0.0;
	zsum[1] = 0.0;
	zsum[2] = 0.0;
	zsum[3] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );

			if ( mpixel[0] >= halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							zsum[3] += w*g[pixel[3]];
						case GIMP_RGB_IMAGE:
							zsum[2] += w*g[pixel[2]];
						case GIMP_GRAYA_IMAGE:
							zsum[1] += w*g[pixel[1]];
						case GIMP_GRAY_IMAGE: default:
							zsum[0] += w*g[pixel[0]];
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							zsum[3] += w*pixel[3];
						case GIMP_RGB_IMAGE:
							zsum[2] += w*pixel[2];
						case GIMP_GRAYA_IMAGE:
							zsum[1] += w*pixel[1];
						case GIMP_GRAY_IMAGE: default:
							zsum[0] += w*pixel[0];
							break;
						}
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3] = zsum[3]/wsum;
			dxpixel[3] = 0.0;
			dypixel[3] = 0.0;
		case GIMP_RGB_IMAGE:
			zpixel[2] = zsum[2]/wsum;
			dxpixel[2] = 0.0;
			dypixel[2] = 0.0;
		case GIMP_GRAYA_IMAGE:
			zpixel[1] = zsum[1]/wsum;
			dxpixel[1] = 0.0;
			dypixel[1] = 0.0;
		case GIMP_GRAY_IMAGE: default:
			zpixel[0] = zsum[0]/wsum;
			dxpixel[0] = 0.0;
			dypixel[0] = 0.0;
			break;
		}
	} else {
		flat_slope(ring->type, center, g, zpixel, dxpixel, dypixel);
	}
}


inline static void old_slope(Ring *ring, Ring *mring, gint x,
                             InterpolateKernel *kernel, gboolean halfsel,
                             guchar *center, gfloat *g,
                             gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel) {
	float_pixel(ring->type, center, g, zpixel);

	gfloat wsum = 0.0;
	gfloat dxsum[4];
	dxsum[0] = 0.0;
	dxsum[1] = 0.0;
	dxsum[2] = 0.0;
	dxsum[3] = 0.0;
	gfloat dysum[4];
	dysum[0] = 0.0;
	dysum[1] = 0.0;
	dysum[2] = 0.0;
	dysum[3] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			gfloat distsquared = (deltax*deltax + deltay*deltay);
			if ( distsquared>0.0 && mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );
					gint dpixel[4];

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = g[pixel[3]] - zpixel[3];
							dxsum[3] += w*dpixel[3]*deltax/distsquared;
							dysum[3] += w*dpixel[3]*deltay/distsquared;
						case GIMP_RGB_IMAGE:
							dpixel[2] = g[pixel[2]] - zpixel[2];
							dxsum[2] += w*dpixel[2]*deltax/distsquared;
							dysum[2] += w*dpixel[2]*deltay/distsquared;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = g[pixel[1]] - zpixel[1];
							dxsum[1] += w*dpixel[1]*deltax/distsquared;
							dysum[1] += w*dpixel[1]*deltay/distsquared;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = g[pixel[0]] - zpixel[0];
							dxsum[0] += w*dpixel[0]*deltax/distsquared;
							dysum[0] += w*dpixel[0]*deltay/distsquared;
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = pixel[3] - center[3];
							dxsum[3] += w*dpixel[3]*deltax/distsquared;
							dysum[3] += w*dpixel[3]*deltay/distsquared;
						case GIMP_RGB_IMAGE:
							dpixel[2] = pixel[2] - center[2];
							dxsum[2] += w*dpixel[2]*deltax/distsquared;
							dysum[2] += w*dpixel[2]*deltay/distsquared;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = pixel[1] - center[1];
							dxsum[1] += w*dpixel[1]*deltax/distsquared;
							dysum[1] += w*dpixel[1]*deltay/distsquared;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = pixel[0] - center[0];
							dxsum[0] += w*dpixel[0]*deltax/distsquared;
							dysum[0] += w*dpixel[0]*deltay/distsquared;
							break;
						}
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = 2*dxsum[3]/wsum;
			dypixel[3] = 2*dysum[3]/wsum;
		case GIMP_RGB_IMAGE:
			dxpixel[2] = 2*dxsum[2]/wsum;
			dypixel[2] = 2*dysum[2]/wsum;
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = 2*dxsum[1]/wsum;
			dypixel[1] = 2*dysum[1]/wsum;
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = 2*dxsum[0]/wsum;
			dypixel[0] = 2*dysum[0]/wsum;
			break;
		}
	} else {
		flat_slope(ring->type, center, g, zpixel, dxpixel, dypixel);
	}
}


inline static void least_squares_slope(Ring *ring, Ring *mring, gint x,
                                       InterpolateKernel *kernel, gint halfsel,
                                       guchar *center, gfloat *g,
                                       gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel) {
	float_pixel(ring->type, center, g, zpixel);

	gfloat ws = 0.0;

	gfloat m[2][2];
	m[0][0] = 0.0;
	m[0][1] = 0.0;
	m[1][0] = 0.0;
	m[1][1] = 0.0;

	gfloat v[4][2];
	v[0][0] = 0.0;
	v[0][1] = 0.0;
	v[1][0] = 0.0;
	v[1][1] = 0.0;
	v[2][0] = 0.0;
	v[2][1] = 0.0;
	v[3][0] = 0.0;
	v[3][1] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			if ( mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					ws += w;

					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );
					gint dpixel[4];

					m[0][0] += deltax*deltax*w;
					m[0][1] += deltax*deltay*w;
					m[1][0] += deltay*deltax*w;
					m[1][1] += deltay*deltay*w;

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = g[pixel[3]] - zpixel[3];
							v[3][0] += deltax*dpixel[3]*w;
							v[3][1] += deltay*dpixel[3]*w;
						case GIMP_RGB_IMAGE:
							dpixel[2] = g[pixel[2]] - zpixel[2];
							v[2][0] += deltax*dpixel[2]*w;
							v[2][1] += deltay*dpixel[2]*w;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = g[pixel[1]] - zpixel[1];
							v[1][0] += deltax*dpixel[1]*w;
							v[1][1] += deltay*dpixel[1]*w;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = g[pixel[0]] - zpixel[0];
							v[0][0] += deltax*dpixel[0]*w;
							v[0][1] += deltay*dpixel[0]*w;
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = pixel[3] - center[3];
							v[3][0] += deltax*dpixel[3]*w;
							v[3][1] += deltay*dpixel[3]*w;
						case GIMP_RGB_IMAGE:
							dpixel[2] = pixel[2] - center[2];
							v[2][0] += deltax*dpixel[2]*w;
							v[2][1] += deltay*dpixel[2]*w;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = pixel[1] - center[1];
							v[1][0] += deltax*dpixel[1]*w;
							v[1][1] += deltay*dpixel[1]*w;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = pixel[0] - center[0];
							v[0][0] += deltax*dpixel[0]*w;
							v[0][1] += deltay*dpixel[0]*w;
							break;
						}
					}
				}
			}
		}
	}

	gfloat det = 0.0;

	if (ws!=0.0) {
		m[0][0] /= ws;
		m[0][1] /= ws;
		m[1][0] /= ws;
		m[1][1] /= ws;
	
		v[0][0] /= ws;
		v[0][1] /= ws;
		v[1][0] /= ws;
		v[1][1] /= ws;
		v[2][0] /= ws;
		v[2][1] /= ws;
		v[3][0] /= ws;
		v[3][1] /= ws;

		det = m[0][0]*m[1][1] - m[1][0]*m[0][1];
	}

	if (ws!=0.0 && det!=0.0) {
		gfloat invm[2][2];
		invm[0][0] =  m[1][1]/det;
		invm[0][1] = -m[0][1]/det;
		invm[1][0] = -m[1][0]/det;
		invm[1][1] =  m[0][0]/det;

		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = invm[0][0]*v[3][0] + invm[0][1]*v[3][1];
			dypixel[3] = invm[1][0]*v[3][0] + invm[1][1]*v[3][1];
		case GIMP_RGB_IMAGE:
			dxpixel[2] = invm[0][0]*v[2][0] + invm[0][1]*v[2][1];
			dypixel[2] = invm[1][0]*v[2][0] + invm[1][1]*v[2][1];
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = invm[0][0]*v[1][0] + invm[0][1]*v[1][1];
			dypixel[1] = invm[1][0]*v[1][0] + invm[1][1]*v[1][1];
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = invm[0][0]*v[0][0] + invm[0][1]*v[0][1];
			dypixel[0] = invm[1][0]*v[0][0] + invm[1][1]*v[0][1];
			break;
		}
	} else {
		/*
		 * only one point, or colinear points
		 */
		flat_slope(ring->type, center, g, zpixel, dxpixel, dypixel);
	}
}


inline static void least_squares3_slope(Ring *ring, Ring *mring, gint x,
                                        InterpolateKernel *kernel, gint halfsel,
                                        guchar *center, gfloat *g,
                                        gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel) {
	gint i;

	gfloat ws = 0.0;

	gfloat m[3][7];
	for (i=0;i<7;++i) {
		m[0][i] = 0.0;
		m[1][i] = 0.0;
		m[2][i] = 0.0;
	}

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			if ( mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					ws += w;

					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );

					m[0][0] += w;
					m[0][1] += deltax*w;
					m[0][2] += deltay*w;
					m[1][0] += deltax*w;
					m[1][1] += deltax*deltax*w;
					m[1][2] += deltax*deltay*w;
					m[2][0] += deltay*w;
					m[2][1] += deltay*deltax*w;
					m[2][2] += deltay*deltay*w;

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							m[0][6] += g[pixel[3]]*w;
							m[1][6] += deltax*g[pixel[3]]*w;
							m[2][6] += deltay*g[pixel[3]]*w;
						case GIMP_RGB_IMAGE:
							m[0][5] += g[pixel[2]]*w;
							m[1][5] += deltax*g[pixel[2]]*w;
							m[2][5] += deltay*g[pixel[2]]*w;
						case GIMP_GRAYA_IMAGE:
							m[0][4] += g[pixel[1]]*w;
							m[1][4] += deltax*g[pixel[1]]*w;
							m[2][4] += deltay*g[pixel[1]]*w;
						case GIMP_GRAY_IMAGE: default:
							m[0][3] += g[pixel[0]]*w;
							m[1][3] += deltax*g[pixel[0]]*w;
							m[2][3] += deltay*g[pixel[0]]*w;
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							m[0][6] += pixel[3]*w;
							m[1][6] += deltax*pixel[3]*w;
							m[2][6] += deltay*pixel[3]*w;
						case GIMP_RGB_IMAGE:
							m[0][5] += pixel[2]*w;
							m[1][5] += deltax*pixel[2]*w;
							m[2][5] += deltay*pixel[2]*w;
						case GIMP_GRAYA_IMAGE:
							m[0][4] += pixel[1]*w;
							m[1][4] += deltax*pixel[1]*w;
							m[2][4] += deltay*pixel[1]*w;
						case GIMP_GRAY_IMAGE: default:
							m[0][3] += pixel[0]*w;
							m[1][3] += deltax*pixel[0]*w;
							m[2][3] += deltay*pixel[0]*w;
							break;
						}
					}
				}
			}
		}
	}

	if (ws!=0.0) {
		for (i=0;i<7;++i) {
			m[0][i] /= ws;
			m[1][i] /= ws;
			m[2][i] /= ws;
		}
	}


	if (ws!=0.0 && gauss_solve(*m,3,7)) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			 zpixel[3] = m[0][6];
			dxpixel[3] = m[1][6];
			dypixel[3] = m[2][6];
		case GIMP_RGB_IMAGE:
			 zpixel[2] = m[0][5];
			dxpixel[2] = m[1][5];
			dypixel[2] = m[2][5];
		case GIMP_GRAYA_IMAGE:
			 zpixel[1] = m[0][4];
			dxpixel[1] = m[1][4];
			dypixel[1] = m[2][4];
		case GIMP_GRAY_IMAGE: default:
			 zpixel[0] = m[0][3];
			dxpixel[0] = m[1][3];
			dypixel[0] = m[2][3];
			break;
		}
	} else {
		/*
		 * only one point, or colinear points
		 */
		flat_slope(ring->type, center, g, zpixel, dxpixel, dypixel);
	}
}


inline static void flat_slope(GimpImageType type, guchar *center, gfloat *g, gfloat *zpixel, gfloat *dxpixel, gfloat *dypixel) {
	float_pixel(type, center, g, zpixel);

	if (g) {
		switch (type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = 0.0;
			dypixel[3] = 0.0;
		case GIMP_RGB_IMAGE:
			dxpixel[2] = 0.0;
			dypixel[2] = 0.0;
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = 0.0;
			dypixel[1] = 0.0;
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = 0.0;
			dypixel[0] = 0.0;
			break;
		}
	} else {
		switch (type) {
		case GIMP_RGBA_IMAGE:
			dxpixel[3] = 0.0;
			dypixel[3] = 0.0;
		case GIMP_RGB_IMAGE:
			dxpixel[2] = 0.0;
			dypixel[2] = 0.0;
		case GIMP_GRAYA_IMAGE:
			dxpixel[1] = 0.0;
			dypixel[1] = 0.0;
		case GIMP_GRAY_IMAGE: default:
			dxpixel[0] = 0.0;
			dypixel[0] = 0.0;
			break;
		}
	}
}

inline static void float_pixel(GimpImageType type, guchar *pixel, gfloat *g, gfloat *zpixel) {
	if (g) {
		switch (type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3] = pixel[3];
			zpixel[2] = g[pixel[2]];
			zpixel[1] = g[pixel[1]];
			zpixel[0] = g[pixel[0]];
			break;
		case GIMP_RGB_IMAGE:
			zpixel[2] = g[pixel[2]];
			zpixel[1] = g[pixel[1]];
			zpixel[0] = g[pixel[0]];
			break;
		case GIMP_GRAYA_IMAGE:
			zpixel[1] = pixel[1];
			zpixel[0] = g[pixel[0]];
			break;
		case GIMP_GRAY_IMAGE: default:
			zpixel[0] = g[pixel[0]];
			break;
		}
	} else {
		switch (type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3] = pixel[3];
		case GIMP_RGB_IMAGE:
			zpixel[2] = pixel[2];
		case GIMP_GRAYA_IMAGE:
			zpixel[1] = pixel[1];
		case GIMP_GRAY_IMAGE: default:
			zpixel[0] = pixel[0];
			break;
		}
	}
}


static void kernel_init(InterpolateKernel *kernel, InterpolateKernelType type, gint window, gfloat s, gfloat norm) {
	gint bwidth=0, bheight=0;
	gint mask_bpp=0, num_mask_bytes=0;
	guint8 *mask_bytes=0;
	gchar *brush_name = gimp_context_get_brush();
	gint color_bpp=0, num_color_bytes=0;
	guint8 *color_bytes=0;
	if (type==KERNEL_BRUSH || type==KERNEL_BRUSH_FLIPPED) {
		gimp_brush_get_pixels(brush_name, &bwidth, &bheight,
		                      &mask_bpp, &num_mask_bytes, &mask_bytes,
		                      &color_bpp, &num_color_bytes, &color_bytes);
		kernel->width = (bwidth/2)*2 + 1;
		kernel->height = (bheight/2)*2 + 1;
	} else {
		kernel->width = kernel->height = 1 + 2*window;
	}

	kernel->values = g_new(gfloat, kernel->width*kernel->height);

	gint y;
	for (y = 0 ; y < kernel->height ; ++y) {
		gint deltay = y - kernel->height/2;

		gfloat *krow = kernel->values + kernel->width*y;

		gint x;
		for (x = 0 ; x < kernel->width ; ++x) {
			gint deltax = x - kernel->width/2;

			gfloat d;
			if (norm>0.0) {
				d = pow(pow(abs(deltax),norm) + pow(abs(deltay),norm), 1/norm);
			} else if (norm<0.0) {
				d = MAX(abs(deltax),abs(deltay));
			} else {
				if (deltax==0.0) {
					d = abs(deltay);
				} else if (deltay==0.0) {
					d = abs(deltax);
				} else {
					krow[x] = 0.0;
					continue;
				}
			}

			switch (type) {
			case KERNEL_CONSTANT:
				krow[x] = 1.0;
				break;
			case KERNEL_LINEAR:
				krow[x] = MAX(0.0,1.0-d/s);
				break;
			case KERNEL_SQUARE:
				krow[x] = MAX(0.0,1.0-(d/s)*(d/s));
				break;
			case KERNEL_INVLINEAR:
				krow[x] = (d>0.0 ? 1.0/d : -1.0);
				break;
			case KERNEL_INVSQUARE:
				krow[x] = (d>0.0 ? 1.0/(d*d) : -1.0);
				break;
			case KERNEL_EXPONENTIAL:
				krow[x] = exp(-d/s);
				break;
			case KERNEL_GAUSSIAN:
				krow[x] = gaussian(d,s);
				break;
			case KERNEL_INVNTH:
				krow[x] = (d>0.0 ? pow(d,-s) : -1.0);
				break;
			case KERNEL_BRUSH:
				if (x < bwidth && y < bheight) {
					krow[x] = mask_bytes[mask_bpp*(bwidth*y + x)]/255.0;
				} else {
					krow[x] = 0.0;
				}
				break;
			case KERNEL_BRUSH_FLIPPED:
				if (x < bwidth && y < bheight) {
					krow[x] = mask_bytes[mask_bpp*(bwidth*(bheight-y-1) + (bwidth-x-1))]/255.0;
				} else {
					krow[x] = 0.0;
				}
				break;
			case KERNEL_BILINEAR:
				krow[x] = MAX(0,(kernel->width/2 - abs(deltax))) * MAX(0,(kernel->height/2 - abs(deltay)));
				break;
			case KERNEL_BINTH:
				krow[x] = MAX( 0, pow(1.0 - (gfloat)abs(deltax)/(kernel->width/2),s) )
				        * MAX( 0, pow(1.0 - (gfloat)abs(deltay)/(kernel->height/2),s) );
				break;
			}
		}
	}
}
static void kernel_fini(InterpolateKernel *kernel) {
	g_free(kernel->values);
}


static gfloat *gamma_new(gfloat g) {
	gfloat *gamma = g_new(gfloat, 256);
	gint i;
	for (i = 0 ; i < 256 ; ++i) {
		gamma[i] = 256.0*pow(i/256.0, g);
	}
	return gamma;
}
static void gamma_free(gfloat *gamma) {
	g_free(gamma);
}

inline static gboolean gauss_solve(gfloat *m, gint rows, gint cols) {
	gint i, j, k;
	for (i = 0 ; i < rows ; ++i) {
		if ( m[i*cols + i] == 0.0 ) {
			for (j = i ; j < rows ; ++j) {
				if ( m[j*cols + i] != 0.0 ) {
					swap_row(m + i*cols + i, m + j*cols + i, cols - i);
					break;
				}
			}
			if (j == rows) {
				return FALSE;
			}
		}

		for (j = i+1 ; j < cols ; ++j)
			m[i*cols + j] /= m[i*cols + i];
		m[i*cols + i] = 1;
		for (j = 0 ; j < i ; ++j)
			m[i*cols + j] = 0.0;

		for (j = i+1 ; j < rows ; ++j) {
			for (k = i+1 ; k < cols ; ++k)
				m[j*cols + k] -= m[j*cols + i]*m[i*cols + k];
			m[j*cols + i] = 0.0;
		}
	}
	for (i = rows-1 ; i >= 0 ; --i) {
		for (j = i-1 ; j >=0 ; --j) {
			for (k = i+1 ; k < cols ; ++k)
				m[j*cols + k] -= m[j*cols + i]*m[i*cols + k];
			m[j*cols + i] = 0.0;
		}
	}

	return TRUE;
}
inline static void swap_row(gfloat *a, gfloat *b, gint count) {
	gint i;
	for (i=0;i<count;++i) {
		gfloat c = a[i];
		a[i] = b[i];
		b[i] = c;
	}
}

inline static gfloat gaussian(gfloat x, gfloat s) {
	return exp(-0.5*(x/s)*(x/s))/(s*sqrt(2*M_PI));
}


inline static void gray_pixel(GimpImageType type, guchar *pixel) {
	switch (type) {
	case GIMP_RGBA_IMAGE:
		pixel[3] = 128;
	case GIMP_RGB_IMAGE:
		pixel[2] = 128;
	case GIMP_GRAYA_IMAGE:
		pixel[1] = 128;
	case GIMP_GRAY_IMAGE: default:
		pixel[0] = 128;
		break;
	}
}


