/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include "main.h"
#include "interface.h"
#include "render.h"
#include "plugin-intl.h"

#include <string.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/*  Constants  */

#define PROCEDURE_NAME   "gimp_plugin_interpolate"

#define DATA_KEY_VALS    "plug_in_interpolate"
#define DATA_KEY_UI_VALS "plug_in_interpolate_ui"

#define PARASITE_KEY     "plug-in-interpolate-options"


/*  Local function prototypes  */

static void query(void);
static void run(const gchar *name, gint nparams, const GimpParam *param, gint *nreturn_vals, GimpParam **return_vals);


/*  Local variables  */

static GimpParamDef script_args[] = {
	{ GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
	{ GIMP_PDB_IMAGE, "image", "Input image" },
	{ GIMP_PDB_DRAWABLE, "drawable", "Input drawable" },
	{ GIMP_PDB_CHANNEL, "chanid", "Input mask" },
	{ GIMP_PDB_INT8, "usesel", "Subtract selection from input mask" },
	{ GIMP_PDB_INT32, "halfsel", "Read only pixels with input mask >= halfsel" },

	{ GIMP_PDB_INT32, "window", "Horizontal (or both horizontal and vertical) size of window, around each pixel, for taking weighted average" },
	{ GIMP_PDB_FLOAT, "sigma", "Parameter for kernel" },
	{ GIMP_PDB_INT32, "kernel", "Type of kernel" },
	{ GIMP_PDB_INT8, "newlayer", "New layer for interpolation result" },

	{ GIMP_PDB_INT8, "slopes", "Do slopes calculation" },
	{ GIMP_PDB_INT32, "method", "Plane-fitting method" },
	{ GIMP_PDB_INT32, "linwindow", "Window for slopes estimation" },
	{ GIMP_PDB_FLOAT, "linsigma", "Parameter for kernel for slopes estimation" },
	{ GIMP_PDB_INT32, "linkernel", "Type of kernel for slopes estimation" },
	{ GIMP_PDB_INT8, "linnewlayer", "Create new grain merge layer for slopes" },

	{ GIMP_PDB_INT8, "changesel", "Modify selection" },
	{ GIMP_PDB_FLOAT, "lowthres", "Weight sum that gives completely selected pixels" },
	{ GIMP_PDB_FLOAT, "highthres", "Weight sum that gives completely unselected pixels" },
	{ GIMP_PDB_INT8, "logthres", "Log scale new selection" },
	{ GIMP_PDB_INT8, "multmask", "Multiply new selection with mask" },

	{ GIMP_PDB_INT8, "all_white_input", "Use all-white input mask, instead of selected channel" },

	{ GIMP_PDB_INT8, "gamma_compensation", "Compensate for gamma-compressed images" },
	{ GIMP_PDB_FLOAT, "gamma", "Decoding gamma" },
	{ GIMP_PDB_INT32, "grid", "Horizontal (or both horizontal and vertical) sample grid spacing." },
	{ GIMP_PDB_INT32, "grid_y", "Vertical sample grid spacing." },
	{ GIMP_PDB_INT32, "offs", "Horizontal (or both horizontal and vertical) sample grid offset." },
	{ GIMP_PDB_INT32, "offs_y", "Vertical sample grid offset." },
	{ GIMP_PDB_FLOAT, "norm", "Norm for distance computation (<0 means maximum distance)." },
	{ GIMP_PDB_FLOAT, "linnorm", "Norm for distance computation  (<0 means maximum distance)." },

	{ GIMP_PDB_FLOAT, "slopesmult", "Artificial 'steepening' or 'flattening' of the slopes surface." },

	{ GIMP_PDB_INT32, "window_y", "Vertical size of window, around each pixel, for taking weighted average" },
	{ GIMP_PDB_INT32, "linwindow_y", "Vertical size of window for slopes estimation" },

	{ GIMP_PDB_FLOAT, "kernel_cap", "Kernel cap distance." },
	{ GIMP_PDB_FLOAT, "kernel_base", "Kernel base distance." },
	{ GIMP_PDB_FLOAT, "linkernel_cap", "Slopes kernel cap distance." },
	{ GIMP_PDB_FLOAT, "linkernel_base", "Slopes kernel base distance." },
	{ GIMP_PDB_INT8, "slopesmask", "Use separate slopes mask?" },
	{ GIMP_PDB_INT32, "slopeschanid", "Slopes mask ID" },
	{ GIMP_PDB_INT8, "slopesmaskwhite", "Use separate, white slopes mask?" },
	{ GIMP_PDB_INT8, "slopessubsel", "Subtract selection from slopes mask?" },
	{ GIMP_PDB_INT32, "linhalfsel", "For calculating slopes, read only pixels with slopes-mask value >= halfsel" },
	{ GIMP_PDB_INT8, "slopesweights", "Assign a weight to each slopes sample" },
};

const PlugInVals default_vals = {
	-1,  /* chanid */
	TRUE,  /* usesel */
	255,  /* halfsel */

	96,  /* window */
	3.0,  /* sigma */
	KERNEL_INVNTH,  /* kernel */
	FALSE,  /* newlayer */

	TRUE,  /* slopes */
	METHOD_LEASTSQ,  /* method */
	2,  /* linwindow */
	3.0,  /* linsigma */
	KERNEL_CONSTANT,  /* linkernel */
	FALSE,  /* linnewlayer */

	FALSE,  /* changesel */
	0.0,  /* lowthres */
	0.01,  /* highthres */
	FALSE,  /* logthres */
	FALSE,  /* multmask */

	FALSE,  /* all_white_input */

	TRUE,  /* gamma_compensation */
	2.5,  /* gamma */
	2,  /* grid */
	2,  /* grid_y */
	0,  /* offs */
	0,  /* offs_y */
	2.0,  /* norm */
	2.0,  /* linnorm */

	1.0,  /* slopesmult */

	96,  /* window_y */
	2,  /* linwindow_y */

	0.0,  /* kernel_cap */
	0.0,  /* kernel_base */
	0.0,  /* linkernel_cap */
	0.0,  /* linkernel_base */
	FALSE,  /* slopesmask */
	-1,  /* slopeschanid */
	FALSE,  /* slopesmaskwhite */
	TRUE,  /* slopessubsel */
	255,  /* linhalfsel */
	FALSE,  /* slopesweights */
};

const PlugInUIVals default_ui_vals = {
	TRUE,  /* preview */
};

static PlugInVals vals;
static PlugInUIVals ui_vals;


GimpPlugInInfo PLUG_IN_INFO = {
	NULL,  /* init_proc  */
	NULL,  /* quit_proc  */
	query, /* query_proc */
	run,   /* run_proc   */
};

MAIN ()

static void query(void) {
	gchar *help_path;
	gchar *help_uri;

	gimp_plugin_domain_register(PLUGIN_NAME, LOCALEDIR);

	help_path = g_build_filename(DATADIR, "help", NULL);
	help_uri = g_filename_to_uri(help_path, NULL, NULL);
	g_free(help_path);

	/* help is in tooltips */
	/*gimp_plugin_help_register
	    ("http://developer.gimp.org/plug-in-template/help", help_uri);*/

	gimp_install_procedure(PROCEDURE_NAME,
			       "Interpolation",
			       "Does interpolation using planes and weighted averages",
			       "stamit@stamit.gr",
			       "stamit@stamit.gr",
			       "2010",
			       N_("_Interpolate..."),
			       "RGB*, GRAY*",
			       GIMP_PLUGIN,
			       G_N_ELEMENTS(script_args), 0, script_args, NULL);

	gimp_plugin_menu_register(PROCEDURE_NAME, "<Image>/Filters/Generic/");
}

static void run(const gchar *name,
		gint n_params,
		const GimpParam *param,
		gint *nreturn_vals, GimpParam **return_vals)
{
	static GimpParam values[1];
	GimpDrawable *drawable;
	gint32 image_ID;
	GimpRunMode run_mode;
	GimpPDBStatusType status = GIMP_PDB_SUCCESS;

	*nreturn_vals = 1;
	*return_vals = values;

	/*  Initialize i18n support  */
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
#endif
	textdomain(GETTEXT_PACKAGE);

	run_mode = param[0].data.d_int32;
	image_ID = param[1].data.d_int32;
	drawable = gimp_drawable_get(param[2].data.d_drawable);

	/*  Initialize with default values  */
	vals = default_vals;
	ui_vals = default_ui_vals;

	if (strcmp(name, PROCEDURE_NAME) != 0) {
		status = GIMP_PDB_CALLING_ERROR;

	} else switch (run_mode) {

	case GIMP_RUN_NONINTERACTIVE:
		if (n_params>=4) vals.chanid = param[3].data.d_channel;
		if (n_params>=5) vals.usesel = param[4].data.d_int8;
		if (n_params>=6) vals.halfsel = param[5].data.d_int32;
		if (n_params>=7) vals.window = param[6].data.d_int32;
		if (n_params>=8) vals.sigma = param[7].data.d_float;
		if (n_params>=9) vals.kernel = param[8].data.d_int32;
		if (n_params>=10) vals.newlayer = param[9].data.d_int8;

		if (n_params>=11) vals.slopes = param[10].data.d_int8;
		if (n_params>=12) vals.method = param[11].data.d_int32;
		if (n_params>=13) vals.linwindow = param[12].data.d_int32;
		if (n_params>=14) vals.linsigma = param[13].data.d_float;
		if (n_params>=15) vals.linkernel = param[14].data.d_int32;
		if (n_params>=16) vals.linnewlayer = param[15].data.d_int8;

		if (n_params>=17) vals.changesel = param[16].data.d_int8;
		if (n_params>=18) vals.lowthres = param[17].data.d_float;
		if (n_params>=19) vals.highthres = param[18].data.d_float;
		if (n_params>=20) vals.logthres = param[19].data.d_int8;
		if (n_params>=21) vals.multmask = param[20].data.d_int8;

		if (n_params>=22) vals.all_white_input = param[21].data.d_int8;

		if (n_params>=23) vals.gamma_compensation = param[22].data.d_int8;
		if (n_params>=24) vals.gamma = param[23].data.d_float;
		if (n_params>=25) vals.grid = param[24].data.d_int32;
		if (n_params>=26) vals.grid_y = param[25].data.d_int32;
		if (n_params>=27) vals.norm = param[26].data.d_float;
		if (n_params>=28) vals.linnorm = param[27].data.d_float;

		if (n_params>=29) vals.slopesmult = param[28].data.d_float;
		if (n_params>=30) vals.window_y = param[29].data.d_int32;
		if (n_params>=31) vals.linwindow_y = param[30].data.d_int32;

		if (n_params>=32) vals.kernel_cap = param[31].data.d_float;
		if (n_params>=33) vals.kernel_base = param[32].data.d_float;
		if (n_params>=34) vals.linkernel_cap = param[33].data.d_float;
		if (n_params>=35) vals.linkernel_base = param[34].data.d_float;

		if (n_params>=36) vals.slopesmask = param[35].data.d_int8;
		if (n_params>=37) vals.slopeschanid = param[36].data.d_int32;
		if (n_params>=38) vals.slopesmaskwhite = param[37].data.d_int8;
		if (n_params>=39) vals.slopessubsel = param[38].data.d_int8;
		if (n_params>=40) vals.linhalfsel = param[39].data.d_int32;
		if (n_params>=41) vals.slopesweights = param[40].data.d_int8;

		/*if (vals.random_seed) vals.seed = g_random_int ();*/
		break;

	case GIMP_RUN_INTERACTIVE:
		/*  Possibly retrieve data  */
		gimp_get_data(DATA_KEY_VALS, &vals);
		gimp_get_data(DATA_KEY_UI_VALS, &ui_vals);

		if (!dialog(image_ID, drawable, &vals, &ui_vals)) {
			status = GIMP_PDB_CANCEL;
		}
		break;

	case GIMP_RUN_WITH_LAST_VALS:
		/*  Possibly retrieve data  */
		gimp_get_data(DATA_KEY_VALS, &vals);

		/*if (vals.random_seed) vals.seed = g_random_int ();*/
		break;

	default:
		break;
	}

	if (status == GIMP_PDB_SUCCESS) {
		render(drawable, &vals, 0);

		if (run_mode != GIMP_RUN_NONINTERACTIVE)
			gimp_displays_flush();

		if (run_mode == GIMP_RUN_INTERACTIVE) {
			gimp_set_data(DATA_KEY_VALS, &vals, sizeof(vals));
			gimp_set_data(DATA_KEY_UI_VALS, &ui_vals,
				      sizeof(ui_vals));
		}

		gimp_drawable_detach(drawable);
	}

	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = status;
}
