/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed
 * ("conveyed") under the terms of version 3 of the GNU General Public License
 * (no other version, see the file "COPYING" for details).  THERE IS NO IMPLIED
 * WARRANTY FOR THIS PROGRAM.  USE AT YOUR OWN RISK.  For inquiries, email the
 * author at stamit@stamit.gr .  You may not remove this notice.
 */
#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libgimp/gimp.h>

#include "main.h"
#include "render.h"
#include "ring.h"

#include "plugin-intl.h"

#define SKIP_UNSELECTED 1
#define ROUNDING 0.5
#define FAKE_INFINITY 1099511627776.0   /* not very nice, but much fewer special cases in the code */


/*
 * common functions
 */
inline static gfloat gaussian(gfloat x, gfloat s);
inline static gfloat sinc(gfloat x);
inline static gfloat sinc_lanczos(gfloat x, gfloat a);


/*
 * generate a gamma ramp
 */
static gfloat *gamma_new(gfloat g);
static void gamma_free(gfloat *gamma);


/*
 * Gaussian elimination
 */
inline static void swap_row(gfloat *a, gfloat *b, gint count);
inline static gboolean gauss_solve(gfloat *m, gint rows, gint cols);


/*
 * calculate series of constant, linear and non-linear components (1, x, y,
 * x*x, x*y, y*y, x*x*x, x*x*y, x*y*y, y*y*y, and so on; fpc in total) for each
 * kernel pixel
 */
static gfloat *nonlinear_new(gint width, gint height, gint fpc, gboolean gimp_progress);
inline static void nonlinear_free(gfloat *nl);


/*
 * weights of nearby pixels
 */
typedef struct InterpolateKernel {
	gfloat *values;
	gint width, height;
} InterpolateKernel;
static void kernel_init(InterpolateKernel *kernel, InterpolateKernelType type,
                        gint window_x, gint window_y, gfloat s, gfloat norm,
                        gfloat cap, gfloat base, gboolean gimp_progress);
static gfloat kernel_dist(gint deltax, gint deltay, gfloat norm);
static gfloat kernel_calc(InterpolateKernelType type, gfloat d, gfloat s);
static void kernel_fini(InterpolateKernel *kernel);
inline static gfloat kernel_value(InterpolateKernel *kernel, gint deltax, gint deltay) {
	return kernel->values[kernel->width*(kernel->height/2+deltay) + (kernel->width/2+deltax)];
}


/*
 * slopes-calculating methods
 */
inline static void pixel2float(GimpImageType type, guchar *pixel, gfloat *g, gfloat *fpixel, gint fpc);
inline static void flat_slope(GimpImageType type, guchar *center, gfloat *g, gfloat *zpixel, gint fpc);

inline static gfloat old_slope(Ring *ring, Ring *mring, gint x,
                               InterpolateKernel *kernel, gint halfsel,
                               guchar *center, gfloat *g,
                               gfloat *zpixel);
inline static gfloat least_squares_slope(Ring *ring, Ring *mring, gint x,
                                         InterpolateKernel *kernel, gint halfsel,
                                         guchar *center, gfloat *g,
                                         gfloat *zpixel);
inline static gfloat flat_avg_slope(Ring *ring, Ring *mring, gint x,
                                    InterpolateKernel *kernel, gboolean halfsel,
                                    guchar *center, gfloat *g,
                                    gfloat *zpixel);
inline static gfloat least_squares3_slope(Ring *ring, Ring *mring, gint x,
                                          InterpolateKernel *kernel, gint halfsel,
                                          guchar *center, gfloat *g,
                                          gfloat *zpixel);
inline static gfloat quadratic_slope(Ring *ring, Ring *mring, gint x,
                                     InterpolateKernel *kernel, gint halfsel,
                                     guchar *center, gfloat *g,
                                     gfloat *zpixel, gfloat *nl);


/*
 * main interpolation code
 */
static void slopes_grid(GimpDrawable *drawable, GimpDrawable *mask, GimpDrawable *slopesmask,
                        gint gleft, gint gtop, gint gright, gint gbottom,
                        InterpolateKernel *kernel,
                        gfloat *nl, gint fpc,
                        gint grid_x, gint grid_y,
                        gint offs_x, gint offs_y,
                        gfloat slopesmult,
                        gint halfsel, gint linhalfsel,
                        InterpolateMethod method,
                        gfloat *g,
                        GimpPreview *preview, gboolean progress,
                        gfloat *z, gfloat *w);

inline static void unknown_pixel(GimpImageType type, guchar *pixel);
inline static GimpDrawable *make_white_channel(gint image_id, gint *tmp_mask_id, char *name);
inline static void subtract_selection(GimpDrawable *mask, gint left, gint top, gint width, gint height);
inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres);
static void interpolate(GimpDrawable *drawable, PlugInVals *opts, gint image_id, GimpPreview *preview);



/******************************************************************************/


inline static gfloat gaussian(gfloat x, gfloat s) {
	return exp(-0.5*(x/s)*(x/s))/(s*sqrt(2*M_PI));
}
inline static gfloat sinc(gfloat x) {
	if (x==0.0) {
		return 1.0;
	} else {
		return sin(M_PI*x)/(M_PI*x);
	}
}
inline static gfloat sinc_lanczos(gfloat x, gfloat a) {
	if (x==0.0) {
		return 1.0;
	} else if (-a<x && x<a) {
		return ( a*sin(M_PI*x)*sin(M_PI*x/a) ) / ( M_PI*M_PI*x*x );
	} else {
		return 0;
	}
}


static gfloat *gamma_new(gfloat g) {
	gfloat *gamma = g_new(gfloat, 256);
	gint i;
	for (i = 0 ; i < 256 ; ++i) {
		gamma[i] = 256.0*pow(i/256.0, g);
	}
	return gamma;
}
static void gamma_free(gfloat *gamma) {
	g_free(gamma);
}


inline static void swap_row(gfloat *a, gfloat *b, gint count) {
	gint i;
	for (i=0;i<count;++i) {
		gfloat c = a[i];
		a[i] = b[i];
		b[i] = c;
	}
}
inline static gboolean gauss_solve(gfloat *m, gint rows, gint cols) {
	gint i, j, k;
	for (i = 0 ; i < rows ; ++i) {
		gint maxi = i;
		for (j = i+1 ; j < rows ; ++j) {
			if ( fabs(m[j*cols + i]) > fabs(m[maxi*cols + i]) ) {
				maxi = j;
			}
		}
		if ( m[maxi*cols + i] == 0.0 ) {
			return FALSE;
		}
		if (i != maxi) {
			swap_row(m+i*cols+i, m+maxi*cols+i, cols-i);
		}

		for (j = i+1 ; j < cols ; ++j)
			m[i*cols + j] /= m[i*cols + i];
		m[i*cols + i] = 1;
		for (j = 0 ; j < i ; ++j)
			m[i*cols + j] = 0.0;

		for (j = i+1 ; j < rows ; ++j) {
			for (k = i+1 ; k < cols ; ++k)
				m[j*cols + k] -= m[j*cols + i]*m[i*cols + k];
			m[j*cols + i] = 0.0;
		}
	}
	for (i = rows-1 ; i >= 0 ; --i) {
		for (j = i-1 ; j >=0 ; --j) {
			for (k = i+1 ; k < cols ; ++k)
				m[j*cols + k] -= m[j*cols + i]*m[i*cols + k];
			m[j*cols + i] = 0.0;
		}
	}

	return TRUE;
}


static gfloat *nonlinear_new(gint width, gint height, gint fpc, gboolean gimp_progress) {
	gfloat *nl = g_new(gfloat, fpc*width*height);

	if (gimp_progress) gimp_progress_init(_("Calculating coefficients..."));

	gint y, x, i, j, n;
	for (y = 0 ; y < height ; ++y) {
		gint deltay = y - height/2;
		for (x = 0 ; x < width ; ++x) {
			gint deltax = x - width/2;
			gfloat *pix = nl + fpc*(y*width + x);

			n = 0;
			for (i = 0 ; n<fpc ; ++i) {
				for (j = 0 ; j<=i && n<fpc ; ++j, ++n) {
					pix[i*(i+1)/2 + j] = pow(deltay, j)*pow(deltax, i-j);
				}
			}
		}

		if (gimp_progress && !(y%4)) gimp_progress_update(1.0*(y+1)/height);
	}
	if (gimp_progress) gimp_progress_end();

	return nl;
}
inline static void nonlinear_free(gfloat *nl) {
	g_free(nl);
}


static void kernel_init(InterpolateKernel *kernel, InterpolateKernelType type,
                        gint window_x, gint window_y, gfloat s, gfloat norm,
                        gfloat cap, gfloat base, gboolean gimp_progress) {
	if (window_y<0) window_y = window_x;

	gint bwidth=0, bheight=0;
	gint mask_bpp=0, num_mask_bytes=0;
	guint8 *mask_bytes=0;
	gchar *brush_name = gimp_context_get_brush();
	gint color_bpp=0, num_color_bytes=0;
	guint8 *color_bytes=0;
	if (type==KERNEL_BRUSH || type==KERNEL_BRUSH_FLIPPED) {
		gimp_brush_get_pixels(brush_name, &bwidth, &bheight,
		                      &mask_bpp, &num_mask_bytes, &mask_bytes,
		                      &color_bpp, &num_color_bytes, &color_bytes);
		kernel->width = (bwidth/2)*2 + 1;
		kernel->height = (bheight/2)*2 + 1;
	} else {
		kernel->width = 1 + 2*window_x;
		kernel->height = 1 + 2*window_y;
	}

	kernel->values = g_new(gfloat, kernel->width*kernel->height);

	gfloat basez = (base>0.0 ? kernel_calc(type, base, s) : 0.0);

	if (gimp_progress) gimp_progress_init(_("Creating kernel..."));
	gint y;
	for (y = 0 ; y < kernel->height ; ++y) {
		gint deltay = y - kernel->height/2;

		gfloat *krow = kernel->values + kernel->width*y;

		gint x;
		for (x = 0 ; x < kernel->width ; ++x) {
			gint deltax = x - kernel->width/2;

			gfloat d = kernel_dist(deltax, deltay, norm);
			if (d<0.0) { krow[x] = 0.0; continue; }

			switch (type) {
			case KERNEL_CONSTANT:
				krow[x] = 1.0;
				break;
			case KERNEL_BRUSH:
				if (x < bwidth && y < bheight) {
					krow[x] = mask_bytes[mask_bpp*(bwidth*y + x)]/255.0;
				} else {
					krow[x] = 0.0;
				}
				break;
			case KERNEL_BRUSH_FLIPPED:
				if (x < bwidth && y < bheight) {
					krow[x] = mask_bytes[mask_bpp*(bwidth*(bheight-y-1) + (bwidth-x-1))]/255.0;
				} else {
					krow[x] = 0.0;
				}
				break;
			case KERNEL_BILINEAR:
				krow[x] = MAX(0,(window_x - abs(deltax))) * MAX(0,(window_y - abs(deltay)));
				break;
			case KERNEL_BINTH:
				krow[x] = MAX( 0, pow(1.0 - (gfloat)abs(deltax)/window_x,s) )
				        * MAX( 0, pow(1.0 - (gfloat)abs(deltay)/window_y,s) );
				break;
			case KERNEL_SINC:
				krow[x] = (sinc(deltax/s))
				        * (sinc(deltay/s));
				break;
			case KERNEL_SINC_LANCZOS:
				krow[x] = (sinc_lanczos(deltax/s,window_x))
				        * (sinc_lanczos(deltay/s,window_y));
				break;
			default:
				if (cap>0.0 && d<cap) {
					d = cap;
				}
				if (base>0.0 && d>base) {
					krow[x] = 0.0;
					continue;
				}
				krow[x] = kernel_calc(type, d, s) - basez;
				break;
			}
		}
		if (gimp_progress && !(y%4)) gimp_progress_update(1.0*(y+1)/kernel->height);
	}
	if (gimp_progress) gimp_progress_end();
}

/*
 * -1 means infinity
 */
static gfloat kernel_dist(gint deltax, gint deltay, gfloat norm) {
	if (norm>0.0) {
		return pow(pow(abs(deltax),norm) + pow(abs(deltay),norm), 1/norm);
	} else if (norm<0.0) {
		return MAX(abs(deltax),abs(deltay));
	} else {
		if (deltax==0.0) {
			return abs(deltay);
		} else if (deltay==0.0) {
			return abs(deltax);
		} else {
			return -1.0;
		}
	}
}

static gfloat kernel_calc(InterpolateKernelType type, gfloat d, gfloat s) {
	switch (type) {
	case KERNEL_LINEAR:
		return MAX(0.0,1.0-d/s);
	case KERNEL_SQUARE:
		return MAX(0.0,1.0-(d/s)*(d/s));
	case KERNEL_INVLINEAR:
		return (d>0.0 ? 1.0/d : FAKE_INFINITY);
	case KERNEL_INVSQUARE:
		return (d>0.0 ? 1.0/(d*d) : FAKE_INFINITY);
	case KERNEL_EXPONENTIAL:
		return exp(-d/s);
	case KERNEL_GAUSSIAN:
		return gaussian(d,s);
	case KERNEL_INVNTH:
		return (d>0.0 ? pow(d,-s) : FAKE_INFINITY);
	case KERNEL_SINC_R:
		return sinc(d/s);
	default:
		return 0.0;
	}
}

static void kernel_fini(InterpolateKernel *kernel) {
	g_free(kernel->values);
}


inline static void pixel2float(GimpImageType type, guchar *pixel, gfloat *g, gfloat *zpixel, gint fpc) {
	if (g) {
		switch (type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3*fpc] = pixel[3];
		case GIMP_RGB_IMAGE:
			zpixel[2*fpc] = g[pixel[2]];
			zpixel[1*fpc] = g[pixel[1]];
			zpixel[0*fpc] = g[pixel[0]];
			break;
		case GIMP_GRAYA_IMAGE:
			zpixel[1*fpc] = pixel[1];
		case GIMP_GRAY_IMAGE: default:
			zpixel[0*fpc] = g[pixel[0]];
			break;
		}
	} else {
		switch (type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3*fpc] = pixel[3];
		case GIMP_RGB_IMAGE:
			zpixel[2*fpc] = pixel[2];
		case GIMP_GRAYA_IMAGE:
			zpixel[1*fpc] = pixel[1];
		case GIMP_GRAY_IMAGE: default:
			zpixel[0*fpc] = pixel[0];
			break;
		}
	}
}

inline static void flat_slope(GimpImageType type, guchar *center, gfloat *g, gfloat *zpixel, gint fpc) {
	pixel2float(type, center, g, zpixel, fpc);

	switch (type) {
	case GIMP_RGBA_IMAGE:
		zpixel[3*fpc + 1] = 0.0;
		zpixel[3*fpc + 2] = 0.0;
	case GIMP_RGB_IMAGE:
		zpixel[2*fpc + 1] = 0.0;
		zpixel[2*fpc + 2] = 0.0;
	case GIMP_GRAYA_IMAGE:
		zpixel[1*fpc + 1] = 0.0;
		zpixel[1*fpc + 2] = 0.0;
	case GIMP_GRAY_IMAGE: default:
		zpixel[0*fpc + 1] = 0.0;
		zpixel[0*fpc + 2] = 0.0;
		break;
	}
}


inline static gfloat old_slope(Ring *ring, Ring *mring, gint x,
                               InterpolateKernel *kernel, gint halfsel,
                               guchar *center, gfloat *g,
                               gfloat *zpixel) {
	const gint fpc = 3;

	gfloat centerz[4];
	centerz[0] = centerz[1] = centerz[2] = centerz[3] = 0.0;
	pixel2float(ring->type, center, g, centerz, 1);  /* gamma-corrected values are used in the loop */

	gfloat wsum = 0.0;
	gfloat dxsum[4];
	dxsum[0] = 0.0;
	dxsum[1] = 0.0;
	dxsum[2] = 0.0;
	dxsum[3] = 0.0;
	gfloat dysum[4];
	dysum[0] = 0.0;
	dysum[1] = 0.0;
	dysum[2] = 0.0;
	dysum[3] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			gfloat distsquared = (deltax*deltax + deltay*deltay);
			if ( distsquared>0.0 && mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );
					gint dpixel[4];

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = g[pixel[3]] - centerz[3];
							dxsum[3] += w*dpixel[3]*deltax/distsquared;
							dysum[3] += w*dpixel[3]*deltay/distsquared;
						case GIMP_RGB_IMAGE:
							dpixel[2] = g[pixel[2]] - centerz[2];
							dxsum[2] += w*dpixel[2]*deltax/distsquared;
							dysum[2] += w*dpixel[2]*deltay/distsquared;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = g[pixel[1]] - centerz[1];
							dxsum[1] += w*dpixel[1]*deltax/distsquared;
							dysum[1] += w*dpixel[1]*deltay/distsquared;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = g[pixel[0]] - centerz[0];
							dxsum[0] += w*dpixel[0]*deltax/distsquared;
							dysum[0] += w*dpixel[0]*deltay/distsquared;
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = pixel[3] - center[3];
							dxsum[3] += w*dpixel[3]*deltax/distsquared;
							dysum[3] += w*dpixel[3]*deltay/distsquared;
						case GIMP_RGB_IMAGE:
							dpixel[2] = pixel[2] - center[2];
							dxsum[2] += w*dpixel[2]*deltax/distsquared;
							dysum[2] += w*dpixel[2]*deltay/distsquared;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = pixel[1] - center[1];
							dxsum[1] += w*dpixel[1]*deltax/distsquared;
							dysum[1] += w*dpixel[1]*deltay/distsquared;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = pixel[0] - center[0];
							dxsum[0] += w*dpixel[0]*deltax/distsquared;
							dysum[0] += w*dpixel[0]*deltay/distsquared;
							break;
						}
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3*fpc + 0] = centerz[3];
			zpixel[3*fpc + 1] = 2*dxsum[3]/wsum;
			zpixel[3*fpc + 2] = 2*dysum[3]/wsum;
		case GIMP_RGB_IMAGE:
			zpixel[2*fpc + 0] = centerz[2];
			zpixel[2*fpc + 1] = 2*dxsum[2]/wsum;
			zpixel[2*fpc + 2] = 2*dysum[2]/wsum;
		case GIMP_GRAYA_IMAGE:
			zpixel[1*fpc + 0] = centerz[1];
			zpixel[1*fpc + 1] = 2*dxsum[1]/wsum;
			zpixel[1*fpc + 2] = 2*dysum[1]/wsum;
		case GIMP_GRAY_IMAGE: default:
			zpixel[0*fpc + 0] = centerz[0];
			zpixel[0*fpc + 1] = 2*dxsum[0]/wsum;
			zpixel[0*fpc + 2] = 2*dysum[0]/wsum;
			break;
		}
	} else {
		flat_slope(ring->type, center, g, zpixel, fpc);
	}

	return wsum;
}

inline static gfloat least_squares_slope(Ring *ring, Ring *mring, gint x,
                                         InterpolateKernel *kernel, gint halfsel,
                                         guchar *center, gfloat *g,
                                         gfloat *zpixel) {
	const gint fpc=3;

	gfloat centerz[4];
	centerz[0] = centerz[1] = centerz[2] = centerz[3] = 0.0;
	pixel2float(ring->type, center, g, centerz, 1);  /* gamma-corrected values are used in the loop */

	gfloat wsum = 0.0;

	gfloat m[2][2];
	m[0][0] = 0.0;
	m[0][1] = 0.0;
	m[1][0] = 0.0;
	m[1][1] = 0.0;

	gfloat v[4][2];
	v[0][0] = 0.0;
	v[0][1] = 0.0;
	v[1][0] = 0.0;
	v[1][1] = 0.0;
	v[2][0] = 0.0;
	v[2][1] = 0.0;
	v[3][0] = 0.0;
	v[3][1] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			if ( mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					wsum += w;

					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );
					gint dpixel[4];

					m[0][0] += deltax*deltax*w;
					m[0][1] += deltax*deltay*w;
					m[1][0] += deltay*deltax*w;
					m[1][1] += deltay*deltay*w;

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = g[pixel[3]] - centerz[3];
							v[3][0] += deltax*dpixel[3]*w;
							v[3][1] += deltay*dpixel[3]*w;
						case GIMP_RGB_IMAGE:
							dpixel[2] = g[pixel[2]] - centerz[2];
							v[2][0] += deltax*dpixel[2]*w;
							v[2][1] += deltay*dpixel[2]*w;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = g[pixel[1]] - centerz[1];
							v[1][0] += deltax*dpixel[1]*w;
							v[1][1] += deltay*dpixel[1]*w;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = g[pixel[0]] - centerz[0];
							v[0][0] += deltax*dpixel[0]*w;
							v[0][1] += deltay*dpixel[0]*w;
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							dpixel[3] = pixel[3] - center[3];
							v[3][0] += deltax*dpixel[3]*w;
							v[3][1] += deltay*dpixel[3]*w;
						case GIMP_RGB_IMAGE:
							dpixel[2] = pixel[2] - center[2];
							v[2][0] += deltax*dpixel[2]*w;
							v[2][1] += deltay*dpixel[2]*w;
						case GIMP_GRAYA_IMAGE:
							dpixel[1] = pixel[1] - center[1];
							v[1][0] += deltax*dpixel[1]*w;
							v[1][1] += deltay*dpixel[1]*w;
						case GIMP_GRAY_IMAGE: default:
							dpixel[0] = pixel[0] - center[0];
							v[0][0] += deltax*dpixel[0]*w;
							v[0][1] += deltay*dpixel[0]*w;
							break;
						}
					}
				}
			}
		}
	}

	gfloat det = 0.0;

	if (wsum!=0.0) {
		det = m[0][0]*m[1][1] - m[1][0]*m[0][1];
	}

	if (wsum!=0.0 && det!=0.0) {
		gfloat invm[2][2];
		invm[0][0] =  m[1][1]/det;
		invm[0][1] = -m[0][1]/det;
		invm[1][0] = -m[1][0]/det;
		invm[1][1] =  m[0][0]/det;

		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3*fpc + 0] = centerz[3];
			zpixel[3*fpc + 1] = invm[0][0]*v[3][0] + invm[0][1]*v[3][1];
			zpixel[3*fpc + 2] = invm[1][0]*v[3][0] + invm[1][1]*v[3][1];
		case GIMP_RGB_IMAGE:
			zpixel[2*fpc + 0] = centerz[2];
			zpixel[2*fpc + 1] = invm[0][0]*v[2][0] + invm[0][1]*v[2][1];
			zpixel[2*fpc + 2] = invm[1][0]*v[2][0] + invm[1][1]*v[2][1];
		case GIMP_GRAYA_IMAGE:
			zpixel[1*fpc + 0] = centerz[1];
			zpixel[1*fpc + 1] = invm[0][0]*v[1][0] + invm[0][1]*v[1][1];
			zpixel[1*fpc + 2] = invm[1][0]*v[1][0] + invm[1][1]*v[1][1];
		case GIMP_GRAY_IMAGE: default:
			zpixel[0*fpc + 0] = centerz[0];
			zpixel[0*fpc + 1] = invm[0][0]*v[0][0] + invm[0][1]*v[0][1];
			zpixel[0*fpc + 2] = invm[1][0]*v[0][0] + invm[1][1]*v[0][1];
			break;
		}
		return wsum;
	} else {
		/*
		 * only one point, or colinear points
		 */
		flat_slope(ring->type, center, g, zpixel, fpc);
		return 0.0;
	}
}

inline static gfloat flat_avg_slope(Ring *ring, Ring *mring, gint x,
                                    InterpolateKernel *kernel, gboolean halfsel,
                                    guchar *center, gfloat *g,
                                    gfloat *zpixel) {
        const gint fpc=3;  /* must be at least 3 */

	gfloat wsum = 0.0;
	gfloat zsum[4];
	zsum[0] = zsum[1] = zsum[2] = zsum[3] = 0.0;

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );

			if ( mpixel[0] >= halfsel ) {

				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							zsum[3] += w*g[pixel[3]];
						case GIMP_RGB_IMAGE:
							zsum[2] += w*g[pixel[2]];
						case GIMP_GRAYA_IMAGE:
							zsum[1] += w*g[pixel[1]];
						case GIMP_GRAY_IMAGE: default:
							zsum[0] += w*g[pixel[0]];
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							zsum[3] += w*pixel[3];
						case GIMP_RGB_IMAGE:
							zsum[2] += w*pixel[2];
						case GIMP_GRAYA_IMAGE:
							zsum[1] += w*pixel[1];
						case GIMP_GRAY_IMAGE: default:
							zsum[0] += w*pixel[0];
							break;
						}
					}
					wsum += w;
				}
			}
		}
	}

	if (wsum>0.0) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3*fpc] = zsum[3]/wsum;
			zpixel[3*fpc+1] = 0.0;
			zpixel[3*fpc+2] = 0.0;
		case GIMP_RGB_IMAGE:
			zpixel[2*fpc] = zsum[2]/wsum;
			zpixel[2*fpc+1] = 0.0;
			zpixel[2*fpc+2] = 0.0;
		case GIMP_GRAYA_IMAGE:
			zpixel[1*fpc] = zsum[1]/wsum;
			zpixel[1*fpc+1] = 0.0;
			zpixel[1*fpc+2] = 0.0;
		case GIMP_GRAY_IMAGE: default:
			zpixel[0*fpc] = zsum[0]/wsum;
			zpixel[0*fpc+1] = 0.0;
			zpixel[0*fpc+2] = 0.0;
			break;
		}
	} else {
		flat_slope(ring->type, center, g, zpixel, fpc);
	}

	return wsum;
}

inline static gfloat least_squares3_slope(Ring *ring, Ring *mring, gint x,
                                          InterpolateKernel *kernel, gint halfsel,
                                          guchar *center, gfloat *g,
                                          gfloat *zpixel) {
	const gint fpc=3;

	gint i;

	gfloat wsum = 0.0;

	gfloat m[3][7];
	for (i=0;i<7;++i) {
		m[0][i] = 0.0;
		m[1][i] = 0.0;
		m[2][i] = 0.0;
	}

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + ( (x+deltax-ring->left)*mring->bpp );
			if ( mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					wsum += w;

					guchar *pixel = row + ( (x+deltax-ring->left)*ring->bpp );

					m[0][0] += w;
					m[0][1] += deltax*w;
					m[0][2] += deltay*w;
					m[1][0] += deltax*w;
					m[1][1] += deltax*deltax*w;
					m[1][2] += deltax*deltay*w;
					m[2][0] += deltay*w;
					m[2][1] += deltay*deltax*w;
					m[2][2] += deltay*deltay*w;

					if (g) {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							m[0][6] += g[pixel[3]]*w;
							m[1][6] += deltax*g[pixel[3]]*w;
							m[2][6] += deltay*g[pixel[3]]*w;
						case GIMP_RGB_IMAGE:
							m[0][5] += g[pixel[2]]*w;
							m[1][5] += deltax*g[pixel[2]]*w;
							m[2][5] += deltay*g[pixel[2]]*w;
						case GIMP_GRAYA_IMAGE:
							m[0][4] += g[pixel[1]]*w;
							m[1][4] += deltax*g[pixel[1]]*w;
							m[2][4] += deltay*g[pixel[1]]*w;
						case GIMP_GRAY_IMAGE: default:
							m[0][3] += g[pixel[0]]*w;
							m[1][3] += deltax*g[pixel[0]]*w;
							m[2][3] += deltay*g[pixel[0]]*w;
							break;
						}
					} else {
						switch (ring->type) {
						case GIMP_RGBA_IMAGE:
							m[0][6] += pixel[3]*w;
							m[1][6] += deltax*pixel[3]*w;
							m[2][6] += deltay*pixel[3]*w;
						case GIMP_RGB_IMAGE:
							m[0][5] += pixel[2]*w;
							m[1][5] += deltax*pixel[2]*w;
							m[2][5] += deltay*pixel[2]*w;
						case GIMP_GRAYA_IMAGE:
							m[0][4] += pixel[1]*w;
							m[1][4] += deltax*pixel[1]*w;
							m[2][4] += deltay*pixel[1]*w;
						case GIMP_GRAY_IMAGE: default:
							m[0][3] += pixel[0]*w;
							m[1][3] += deltax*pixel[0]*w;
							m[2][3] += deltay*pixel[0]*w;
							break;
						}
					}
				}
			}
		}
	}

	if (wsum!=0.0 && gauss_solve(*m,3,7)) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			zpixel[3*fpc] = m[0][6];
			zpixel[3*fpc+1] = m[1][6];
			zpixel[3*fpc+2] = m[2][6];
		case GIMP_RGB_IMAGE:
			zpixel[2*fpc] = m[0][5];
			zpixel[2*fpc+1] = m[1][5];
			zpixel[2*fpc+2] = m[2][5];
		case GIMP_GRAYA_IMAGE:
			zpixel[1*fpc] = m[0][4];
			zpixel[1*fpc+1] = m[1][4];
			zpixel[1*fpc+2] = m[2][4];
		case GIMP_GRAY_IMAGE: default:
			zpixel[0*fpc] = m[0][3];
			zpixel[0*fpc+1] = m[1][3];
			zpixel[0*fpc+2] = m[2][3];
			break;
		}
		return wsum;
	} else {
		/*
		 * less than three, or colinear points
		 */
		flat_slope(ring->type, center, g, zpixel, fpc);
		return 0.0;
	}
}

inline static gfloat quadratic_slope(Ring *ring, Ring *mring, gint x,
                                     InterpolateKernel *kernel, gint halfsel,
                                     guchar *center, gfloat *g,
                                     gfloat *zpixel, gfloat *nl) {
        const int fpc = 6;
	gint i, j;

	gfloat wsum = 0.0;

	gfloat m[6][10];
	for (i=0;i<6;++i) {
		for (j=0;j<10;++j) {
			m[i][j] = 0.0;
		}
	}

	gint deltay_min = ring_min_deltay(ring);
	gint deltay_max = ring_max_deltay(ring);
	gint deltay;

	gint deltax_min = ring_min_deltax(ring, x);
	gint deltax_max = ring_max_deltax(ring, x);
	gint deltax;

	for (deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
		guchar *row = ring_row_delta(ring, deltay);
		guchar *mrow = ring_row_delta(mring, deltay);
		gfloat *nlrow = nl + (kernel->height/2+deltay)*fpc*kernel->width;

		for (deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
			guchar *mpixel = mrow + (x+deltax-ring->left)*mring->bpp;
			if ( mpixel[0]>=halfsel ) {
				gfloat w = kernel_value(kernel, deltax, deltay);
				if (halfsel<255) {
					w *= mpixel[0];
				}
				if (w>0) {
					wsum += w;

					guchar *pixel = row + (x+deltax-ring->left)*ring->bpp;
					gfloat *nlpixel = nlrow + (kernel->width/2+deltax)*fpc;

					for (i = 0 ; i < 6 ; ++i) {
						for (j = 0 ; j < 6 ; ++j) {
							m[i][j] += w*nlpixel[i]*nlpixel[j];
						}
					}

					gfloat pixelz[4];
					pixelz[0] = pixelz[1] = pixelz[2] = pixelz[3] = 0.0;
					pixel2float(ring->type, pixel, g, pixelz, 1);

					switch (ring->type) {
					case GIMP_RGBA_IMAGE:
						for (i = 0 ; i < 6 ; ++i)
							m[i][6+3] += w*nlpixel[i]*pixelz[3];
					case GIMP_RGB_IMAGE:
						for (i = 0 ; i < 6 ; ++i)
							m[i][6+2] += w*nlpixel[i]*pixelz[2];
					case GIMP_GRAYA_IMAGE:
						for (i = 0 ; i < 6 ; ++i)
							m[i][6+1] += w*nlpixel[i]*pixelz[1];
					case GIMP_GRAY_IMAGE: default:
						for (i = 0 ; i < 6 ; ++i)
							m[i][6+0] += w*nlpixel[i]*pixelz[0];
						break;
					}
				}
			}
		}
	}

	if (wsum!=0.0 && gauss_solve(*m,6,10)) {
		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			for (i = 0 ; i < 6 ; ++i)
				zpixel[3*fpc + i] = m[i][6+3];
		case GIMP_RGB_IMAGE:
			for (i = 0 ; i < 6 ; ++i)
				zpixel[2*fpc + i] = m[i][6+2];
		case GIMP_GRAYA_IMAGE:
			for (i = 0 ; i < 6 ; ++i)
				zpixel[1*fpc + i] = m[i][6+1];
		case GIMP_GRAY_IMAGE: default:
			for (i = 0 ; i < 6 ; ++i)
				zpixel[0*fpc + i] = m[i][6+0];
			break;
		}
		return wsum;
	} else {
		/*
		 * too few points, or colinear points
		 */
		pixel2float(ring->type, center, g, zpixel, fpc);

		switch (ring->type) {
		case GIMP_RGBA_IMAGE:
			for (i = 1 ; i < 6 ; ++i)
				zpixel[3*fpc + i] = 0.0;
		case GIMP_RGB_IMAGE:
			for (i = 1 ; i < 6 ; ++i)
				zpixel[2*fpc + i] = 0.0;
		case GIMP_GRAYA_IMAGE:
			for (i = 1 ; i < 6 ; ++i)
				zpixel[1*fpc + i] = 0.0;
		case GIMP_GRAY_IMAGE: default:
			for (i = 1 ; i < 6 ; ++i)
				zpixel[0*fpc + i] = 0.0;
			break;
		}
		return 0.0;
	}
}


static void slopes_grid(GimpDrawable *drawable, GimpDrawable *mask, GimpDrawable *slopesmask,
                        gint gleft, gint gtop, gint gright, gint gbottom,
                        InterpolateKernel *kernel,
                        gfloat *nl, gint fpc,
                        gint grid_x, gint grid_y,
                        gint offs_x, gint offs_y,
                        gfloat slopesmult,
                        gint halfsel, gint linhalfsel,
                        InterpolateMethod method,
                        gfloat *g, GimpPreview *preview, gboolean progress,
                        gfloat *z, gfloat *ws) {
	gint gwidth = gright-gleft;
	gint gheight = gbottom-gtop;

	gint left = gleft*grid_x + offs_x;
	gint top = gtop*grid_y + offs_y;
	gint right = gright*grid_x + offs_x - grid_x + 1;
	gint bottom = gbottom*grid_y + offs_y - grid_y + 1;

	Ring ring; ring_init(&ring, drawable, left, top, right, bottom, kernel->width/2, kernel->height/2);
	Ring mring; ring_init(&mring, mask, left, top, right, bottom, kernel->width/2, kernel->height/2);
	Ring smring; ring_init(&smring, slopesmask, left, top, right, bottom, kernel->width/2, kernel->height/2);

	if (!preview && progress) gimp_progress_init(_("Calculating slopes..."));
	gint y;
	for (y = 0 ; y < gheight ; ++y) {
		guchar *row = ring_row(&ring);
		guchar *mrow = ring_row(&mring);
		gfloat *zrow = z + y*fpc*ring.bpp*gwidth;
		gfloat *wrow = 0; if (ws) wrow = ws + y*gwidth;

		gint x;
		for (x = 0 ; x < gwidth ; ++x) {
			guchar *pixel = row + x*grid_x*ring.bpp;  /* center pixel */
			guchar *mpixel = mrow + x*grid_x*mring.bpp;
			gfloat *zpixel = zrow + fpc*ring.bpp*x;

			gfloat w = 0.0;

			/*
			 * calculate slope of nearby pixels
			 */
			if (mpixel[0]<halfsel) {
				flat_slope(ring.type, pixel, g, zpixel, fpc);
			} else {
				gint absx = left + x*grid_x;

				switch (method) {
				case METHOD_OLD_CENTER: default:
					w = old_slope(&ring, &smring, absx, kernel, linhalfsel, pixel, g, zpixel);
					break;
				case METHOD_LEASTSQ_CENTER:
					w = least_squares_slope(&ring, &smring, absx, kernel, linhalfsel, pixel, g, zpixel);
					break;
				case METHOD_FLAT_AVERAGES:
					w = flat_avg_slope(&ring, &smring, absx, kernel, linhalfsel, pixel, g, zpixel);
					break;
				case METHOD_LEASTSQ:
					w = least_squares3_slope(&ring, &smring, absx, kernel, linhalfsel, pixel, g, zpixel);
					break;
				case METHOD_QUADRATIC:
					w = quadratic_slope(&ring, &smring, absx, kernel, linhalfsel, pixel, g, zpixel, nl);
					break;
				}

				if (slopesmult != 1.0) {
					gint i;
					switch (ring.type) {
					case GIMP_RGBA_IMAGE:
						for (i = 1 ; i < fpc ; ++i)
							zpixel[3*fpc + i] *= slopesmult;
					case GIMP_RGB_IMAGE:
						for (i = 1 ; i < fpc ; ++i)
							zpixel[2*fpc + i] *= slopesmult;
					case GIMP_GRAYA_IMAGE:
						for (i = 1 ; i < fpc ; ++i)
							zpixel[1*fpc + i] *= slopesmult;
					case GIMP_GRAY_IMAGE: default:
						for (i = 1 ; i < fpc ; ++i)
							zpixel[0*fpc + i] *= slopesmult;
						break;
					}
				}
			}

			if (ws) {
				gint i;
				switch (ring.type) {
				case GIMP_RGBA_IMAGE:
					for (i = 0 ; i < fpc ; ++i)
						zpixel[3*fpc + i] *= w;
				case GIMP_RGB_IMAGE:
					for (i = 0 ; i < fpc ; ++i)
						zpixel[2*fpc + i] *= w;
				case GIMP_GRAYA_IMAGE:
					for (i = 0 ; i < fpc ; ++i)
						zpixel[1*fpc + i] *= w;
				case GIMP_GRAY_IMAGE: default:
					for (i = 0 ; i < fpc ; ++i)
						zpixel[0*fpc + i] *= w;
					break;
				}
				wrow[x] = w;
			}
		}

		if (!preview && progress && !(y%4)) {
			gimp_progress_update(1.0*(y+1)/gheight);
		}

		/*
		 * go down grid_y lines
		 */
		gint i;
		for (i = 0 ; i < grid_y ; ++i) {
			ring_rotate(&ring); ring_rotate(&mring); ring_rotate(&smring);
		}
	}
	if (!preview && progress) gimp_progress_end();
	ring_fini(&ring);
}


inline static void unknown_pixel(GimpImageType type, guchar *pixel) {
	switch (type) {
	case GIMP_RGBA_IMAGE:
		pixel[3] = 0;
		pixel[2] = 128;
		pixel[1] = 128;
		pixel[0] = 128;
		break;
	case GIMP_RGB_IMAGE:
		pixel[2] = 128;
		pixel[1] = 128;
		pixel[0] = 128;
		break;
	case GIMP_GRAYA_IMAGE:
		pixel[1] = 0;
		pixel[0] = 128;
		break;
	case GIMP_GRAY_IMAGE: default:
		pixel[0] = 128;
		break;
	}
}

inline static GimpDrawable *make_white_channel(gint image_id, gint *tmp_mask_id, char *name) {
	GimpRGB rgb;
	gimp_rgb_set(&rgb, 0.0, 0.0, 0.0);
	*tmp_mask_id = gimp_channel_new(
		image_id, name,
		gimp_image_width(image_id),
		gimp_image_height(image_id),
		0.5, &rgb
	);
	gimp_drawable_fill(*tmp_mask_id, GIMP_WHITE_FILL);
	gimp_image_add_channel(image_id, *tmp_mask_id, 0);

	return gimp_drawable_get(*tmp_mask_id);
}

inline static void subtract_selection(GimpDrawable *mask, gint left, gint top, gint width, gint height) {
	GimpPixelRgn outm;
	gimp_pixel_rgn_init(&outm, mask, left, top, width, height, TRUE, TRUE);
	guchar *outmrow = g_new(guchar, mask->bpp*width);
	memset(outmrow, 0, mask->bpp*width);
	gint y;
	for (y = 0 ; y < height ; ++y) {
		gimp_pixel_rgn_set_row(&outm,outmrow,0,y,width);
	}
	g_free(outmrow);
	gimp_drawable_flush(mask);
	gimp_drawable_merge_shadow(mask->drawable_id, TRUE);
}

inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres) {
	if (logthres) w = log(w);

	gfloat ww;
	if (w>=ht || w<0) {
		ww = 1.0;
	} else if (w<=lt) {
		ww = 0.0;
	} else {
		ww = (w-lt)/(ht-lt);
	}
	return 1.0-ww;
}

static void interpolate(GimpDrawable *drawable, PlugInVals *opts, gint image_id, GimpPreview *preview) {
	if (!preview && !opts->progress) gimp_progress_init(_("Interpolating..."));

	GimpImageType drawable_type = gimp_drawable_type(drawable->drawable_id);
	gint channels = gimp_drawable_bpp(drawable->drawable_id);

	gint grid_x = opts->grid;
	gint grid_y = (opts->grid_y>0 ? opts->grid_y : opts->grid);

	gboolean changesel = opts->changesel && !preview;
	gboolean newlayer = opts->newlayer && !preview;
	gboolean linnewlayer = opts->linnewlayer && !preview;

	gint halfsel = opts->halfsel;

	gfloat lowthres = opts->lowthres;
	gfloat highthres = opts->highthres;
	gboolean logthres = opts->logthres;

	gint left,top,right,bottom,width,height;
	if (preview) {
		gimp_preview_get_position(preview, &left, &top);
		gimp_preview_get_size(preview, &width, &height);
		right = left + width;
		bottom = top + height;
	} else {
		gimp_drawable_mask_bounds(drawable->drawable_id, &left, &top, &right, &bottom);
		width = right-left;
		height = bottom-top;
	}


	/*
	 * make gamma curve
	 */
	gfloat inv_gamma = 1.0/opts->gamma;
	gfloat *g = 0;
	if (opts->gamma_compensation) {
		g = gamma_new(opts->gamma);
	}


	/*
	 * make white input mask (or copy existing), and optionally subtract selection
	 */
	gint tmp_mask_id = -1;
	GimpDrawable *mask = 0;
	if (!preview && opts->progress) gimp_progress_init(_("Creating mask..."));

	if (!opts->all_white_input && gimp_drawable_is_valid(opts->chanid) && gimp_drawable_get_image(opts->chanid)==image_id) {
		if (opts->usesel) {
			tmp_mask_id = gimp_channel_copy(opts->chanid);
			gimp_image_add_channel(image_id, tmp_mask_id, 0);
			mask = gimp_drawable_get(tmp_mask_id);
		} else {
			mask = gimp_drawable_get(opts->chanid);
		}
	}
	
	if (!mask) {
		mask = make_white_channel(image_id, &tmp_mask_id, "Input mask");
	}

	if (opts->usesel) {
		subtract_selection(mask, left, top, width, height);
	}

	if (!preview && opts->progress) gimp_progress_end();

	gint mchannels = gimp_drawable_bpp(mask->drawable_id);


	/*
	 * save selection to channel, so we can modify it (or to skip unselected pixels)
	 */
	gint sel_id = -1;
	GimpDrawable *sel = 0;
	#ifdef SKIP_UNSELECTED
		sel_id = gimp_selection_save(image_id);
		sel = gimp_drawable_get(sel_id);
	#else
		if (changesel) {
			sel_id = gimp_selection_save(image_id);
			sel = gimp_drawable_get(sel_id);
		}
	#endif


	/*
	 * make the new layers, if we are making new layers
	 */
	GimpDrawable *outdrawable, *outslopesdrawable;
	if (newlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Interpolated"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_NORMAL_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outdrawable = gimp_drawable_get(layer_id);
	} else {
		outdrawable = drawable;
	}

	GimpImageType outdrawable_type = gimp_drawable_type(outdrawable->drawable_id);
	gint outchannels = gimp_drawable_bpp(outdrawable->drawable_id);
	gint outslopeschannels;

	if (linnewlayer) {
		gint layer_id = gimp_layer_new(
			image_id,
			_("Slopes Offset"),
			width,
			height,
			gimp_drawable_type_with_alpha(drawable->drawable_id),
			100.0,
			GIMP_GRAIN_MERGE_MODE
		);
		gimp_image_add_layer(image_id, layer_id, -1);
		gimp_layer_translate(layer_id, left, top);

		outslopesdrawable = gimp_drawable_get(layer_id);
		outslopeschannels = gimp_drawable_bpp(outslopesdrawable->drawable_id);
	} else {
		outslopesdrawable = 0;
		outslopeschannels = outchannels;
	}


	/*
	 * make kernel
	 */
	InterpolateKernel kernel;
	kernel_init(&kernel, opts->kernel, opts->window, opts->window_y, opts->sigma, opts->norm, opts->kernel_cap, opts->kernel_base, !preview && opts->progress);
	gint fpc = 0;  /* floats per channel */
	switch (opts->method) {
	case METHOD_OLD_CENTER: fpc = 3; break;
	case METHOD_LEASTSQ_CENTER: fpc = 3; break;
	case METHOD_FLAT_AVERAGES: fpc = 3; break;  /* must be at least 3 */
	case METHOD_LEASTSQ: fpc = 3; break;
	case METHOD_QUADRATIC: fpc = 6; break;
	}
	gfloat *nl = nonlinear_new(kernel.width, kernel.height, fpc, !preview && opts->progress);


	/*
	 * calculate slopes of sample areas (this is the first pass)
	 *
	 * when the grid is 1x1, this gives a inwidth*inheight*bpp*fpc table
	 * otherwise the size depends on the grid pixels that are enclosed
	 */
	gint inleft = MAX(left-kernel.width/2,0);
	gint intop = MAX(top-kernel.height/2,0);
	gint inwidth = MIN(right+kernel.width/2,drawable->width)-inleft;
	gint inheight = MIN(bottom+kernel.height/2,drawable->height)-intop;

	gint gleft = (inleft + grid_x - 1 - opts->offs)/grid_x;
	gint gtop = (intop + grid_y - 1 - opts->offs_y)/grid_y;
	gint gright = (inleft + inwidth + grid_x - 1 - opts->offs)/grid_x;
	gint gbottom = (intop + inheight + grid_y - 1 - opts->offs_y)/grid_y;
	gint gwidth = gright-gleft;
	gint gheight = gbottom-gtop;

	gfloat *z=0, *ws=0;
	if (opts->slopes && gwidth>0 && gheight>0) {
		InterpolateKernel linkernel;
		kernel_init(&linkernel, opts->linkernel, opts->linwindow, opts->linwindow_y, opts->linsigma, opts->linnorm, opts->linkernel_cap, opts->linkernel_base, !preview && opts->progress);

		gfloat *linnl = nonlinear_new(linkernel.width, linkernel.height, fpc, !preview && opts->progress);

		if (!preview && opts->progress) gimp_progress_init(_("Creating mask..."));

		gint tmp_slopesmask_id = -1;
		GimpDrawable *slopesmask = mask;
		if (opts->slopesmask) {
			if (!opts->slopesmaskwhite && gimp_drawable_is_valid(opts->slopeschanid) && gimp_drawable_get_image(opts->slopeschanid)==image_id) {
				if (opts->slopessubsel) {
					tmp_slopesmask_id = gimp_channel_copy(opts->slopeschanid);
					gimp_image_add_channel(image_id, tmp_slopesmask_id, 0);
					slopesmask = gimp_drawable_get(tmp_slopesmask_id);
				} else {
					slopesmask = gimp_drawable_get(opts->slopeschanid);
				}
			} else {
				slopesmask = make_white_channel(image_id, &tmp_slopesmask_id, "Slopes input mask");
			}
			if (slopesmask!=mask && opts->slopessubsel) {
				subtract_selection(slopesmask, left, top, width, height);
			}
		}

		if (!preview && opts->progress) gimp_progress_end();


		z = g_new(gfloat, fpc*channels*gwidth*gheight);
		if (opts->slopesweights) {
			ws = g_new(gfloat, gwidth*gheight);
		}
		slopes_grid(drawable, mask, slopesmask,
		            gleft, gtop, gright, gbottom,
		            &linkernel,
		            linnl, fpc,
		            grid_x, grid_y, opts->offs, opts->offs_y,
		            opts->slopesmult,
		            opts->halfsel, opts->linhalfsel,
		            opts->method,
		            g, preview, opts->progress,
		            z, ws);

		if (slopesmask!=mask) {
			gimp_drawable_detach(slopesmask);
			if (tmp_slopesmask_id != -1) {
				gimp_image_remove_channel(image_id, tmp_slopesmask_id);
			}
		}

		if (linnl) nonlinear_free(linnl);

		kernel_fini(&linkernel);
	}

	gint tw = gimp_tile_width();
	gimp_tile_cache_ntiles(4*(drawable->width+tw-1)/tw);


	/*
	 * get input/output pixel regions for interpolation pass
	 */
	Ring in; ring_init(&in, drawable, left, top, right, bottom, kernel.width/2, kernel.height/2);
	Ring inm; ring_init(&inm, mask, left, top, right, bottom, kernel.width/2, kernel.height/2);

	GimpPixelRgn out;
	if (newlayer) {
		gimp_pixel_rgn_init(&out, outdrawable, 0, 0, width, height, TRUE, TRUE);
	} else {
		gimp_pixel_rgn_init(&out, outdrawable, left, top, width, height, !preview, TRUE);
	}
	guchar *outrow = g_new(guchar, outchannels*width);

	GimpPixelRgn outslopes;
	guchar *outslopesrow = 0;
	if (linnewlayer) {
		gimp_pixel_rgn_init(&outslopes, outslopesdrawable, 0, 0, width, height, TRUE, TRUE);
		outslopesrow = g_new(guchar, outslopeschannels*width);
	}


	/*
	 * get selection, to skip unselected pixels (and, optionally, to change it)
	 */
	GimpPixelRgn ins;
	guchar *insrow = 0;
	#ifdef SKIP_UNSELECTED
		gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
		insrow = g_new(guchar, mchannels*width);
	#else
		if (changesel) {
			gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
			insrow = g_new(guchar, mchannels*width);
		}
	#endif

	GimpPixelRgn outs;
	guchar *outsrow = 0;
	if (changesel) {
		gimp_pixel_rgn_init(&outs, sel, left, top, width, height, TRUE, TRUE);
		outsrow = g_new(guchar, mchannels*width);
	}


	/*
	 * do the interpolation
	 */
	if (!preview && opts->progress) gimp_progress_init(_("Interpolating..."));
	for (; ring_loop(&in) && ring_loop(&inm) ; ring_rotate(&in), ring_rotate(&inm)) {
		guchar *inmrow = 0;
		if (changesel) {
			inmrow = ring_row(&inm);
		}

		#ifdef SKIP_UNSELECTED
			gimp_pixel_rgn_get_row(&ins, insrow, left, in.y, width);
		#else
			if (changesel) {
				gimp_pixel_rgn_get_row(&ins, insrow, left, in.y, width);
			}
		#endif

		gint deltay_min = ring_min_deltay(&in);
		gint deltay_max = ring_max_deltay(&in);
		deltay_min = (in.y + deltay_min + grid_y - opts->offs_y - 1)/grid_y*grid_y + opts->offs_y - in.y;
		deltay_max = ((in.y + deltay_max + grid_y - opts->offs_y)/grid_y-1)*grid_y + opts->offs_y - in.y;

		gint x;
		for (x = left ; x < right ; ++x) {
			guchar *inspixel = 0;
			#ifdef SKIP_UNSELECTED
				inspixel = insrow + (x-left)*mchannels;
			#else
				if (changesel) {
					inspixel = insrow + (x-left)*mchannels;
				}
			#endif
			guchar *outpixel = outrow + (x-left)*outchannels;
			guchar *outslopespixel = 0;
			if (linnewlayer) {
				outslopespixel = outslopesrow + (x-left)*outslopeschannels;
			}

			/*
			 * calculate the average of what the nearby sample points predict for this pixel
			 */
			gfloat sum[4];
			sum[0] = sum[1] = sum[2] = sum[3] = 0.0;
			gfloat dsum[4];
			dsum[0] = dsum[1] = dsum[2] = dsum[3] = 0.0;
			gfloat wsum = 0.0;

			gint deltax_min = ring_min_deltax(&in, x);
			gint deltax_max = ring_max_deltax(&in, x);
			deltax_min = (x + deltax_min + grid_x - opts->offs - 1)/grid_x*grid_x + opts->offs - x;
			deltax_max = ((x + deltax_max + grid_x - opts->offs)/grid_x-1)*grid_x + opts->offs - x;

			gint deltay;
			#ifdef SKIP_UNSELECTED
			if (inspixel[0])
			#endif
			for (deltay = deltay_min ; deltay <= deltay_max ; deltay += grid_y) {
				guchar *inrow2 = 0;
				guchar *inmrow2 = ring_row_delta(&inm, deltay);

				gfloat *zrow = 0;
				gfloat *wrow = 0;
				if (z) {
					zrow = z + fpc*channels*gwidth*((in.y+deltay)/grid_y - gtop);
					if (ws) wrow = ws + gwidth*((in.y+deltay)/grid_y - gtop);
				} else {
					inrow2 = ring_row_delta(&in, deltay);
				}
				gfloat *nlrow = nl + (kernel.height/2-deltay)*fpc*kernel.width;

				gint deltax;
				for (deltax = deltax_min ; deltax <= deltax_max ; deltax += grid_x) {
					guchar *inmpixel2 = inmrow2 + (x+deltax-left)*mchannels;
					if (inmpixel2[0]<halfsel)
						continue;

					guchar *inpixel2 = inrow2 + (x+deltax-left)*channels;

					gfloat w = kernel_value(&kernel, deltax, deltay);
					if (halfsel<255)
						w *= inmpixel2[0];

					if (z) {
						/* no need for gamma correction here, the slopes are already in linear space */
						gfloat *zpixel = zrow + fpc*channels*((x+deltax)/grid_x-gleft);

						switch (drawable_type) {
						case GIMP_RGBA_IMAGE:
							sum[3] += w*zpixel[3*fpc];
							dsum[3] -= w*(zpixel[3*fpc+1]*deltax + zpixel[3*fpc+2]*deltay);
						case GIMP_RGB_IMAGE:
							sum[2] += w*zpixel[2*fpc];
							dsum[2] -= w*(zpixel[2*fpc+1]*deltax + zpixel[2*fpc+2]*deltay);
						case GIMP_GRAYA_IMAGE:
							sum[1] += w*zpixel[1*fpc];
							dsum[1] -= w*(zpixel[1*fpc+1]*deltax + zpixel[1*fpc+2]*deltay);
						case GIMP_GRAY_IMAGE: default:
							sum[0] += w*zpixel[0*fpc];
							dsum[0] -= w*(zpixel[0*fpc+1]*deltax + zpixel[0*fpc+2]*deltay);
							break;
						}

						if (fpc>3) {
							gfloat *nlpixel = nlrow + (kernel.width/2-deltax)*fpc;

							gint i;
							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:
								for (i = 3 ; i < fpc ; ++i) {
									dsum[3] += w*zpixel[3*fpc+i]*nlpixel[i];
									dsum[2] += w*zpixel[2*fpc+i]*nlpixel[i];
									dsum[1] += w*zpixel[1*fpc+i]*nlpixel[i];
									dsum[0] += w*zpixel[0*fpc+i]*nlpixel[i];
								}
								break;
							case GIMP_RGB_IMAGE:
								for (i = 3 ; i < fpc ; ++i) {
									dsum[2] += w*zpixel[2*fpc+i]*nlpixel[i];
									dsum[1] += w*zpixel[1*fpc+i]*nlpixel[i];
									dsum[0] += w*zpixel[0*fpc+i]*nlpixel[i];
								}
								break;
							case GIMP_GRAYA_IMAGE:
								for (i = 3 ; i < fpc ; ++i) {
									dsum[1] += w*zpixel[1*fpc+i]*nlpixel[i];
									dsum[0] += w*zpixel[0*fpc+i]*nlpixel[i];
								}
								break;
							case GIMP_GRAY_IMAGE: default:
								for (i = 3 ; i < fpc ; ++i) {
									dsum[0] += w*zpixel[0*fpc+i]*nlpixel[i];
								}
								break;
							}
						}

						/*
						 * weight is output_kernel*input_mask*slopes_weight
						 * slopes_weight is sum of slopes_kernel*slopes_mask
						 */
						if (ws) {
							gfloat *wpixel = wrow + ((x+deltax)/grid_x-gleft);
							w *= wpixel[0];
						}

					} else {
						if (g) {  /* gamma-compressed image */
							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:
								sum[3] += w*inpixel2[3];
								sum[2] += w*g[inpixel2[2]];
								sum[1] += w*g[inpixel2[1]];
								sum[0] += w*g[inpixel2[0]];
								break;
							case GIMP_RGB_IMAGE:
								sum[2] += w*g[inpixel2[2]];
								sum[1] += w*g[inpixel2[1]];
								sum[0] += w*g[inpixel2[0]];
								break;
							case GIMP_GRAYA_IMAGE:
								sum[1] += w*inpixel2[1];
								sum[0] += w*g[inpixel2[0]];
								break;
							case GIMP_GRAY_IMAGE: default:
								sum[0] += w*g[inpixel2[0]];
								break;
							}
						} else {  /* linear image, or gamma correction disabled */
							switch (drawable_type) {
							case GIMP_RGBA_IMAGE:  sum[3] += w*inpixel2[3];
							case GIMP_RGB_IMAGE:   sum[2] += w*inpixel2[2];
							case GIMP_GRAYA_IMAGE: sum[1] += w*inpixel2[1];
							default:
							case GIMP_GRAY_IMAGE:  sum[0] += w*inpixel2[0]; break;
							}
						}
					}

					wsum += w;
				}
			}


			/*
			 * now we can calculate the average, to set the output pixel
			 */
			if (z) {
				if (linnewlayer) {
					switch (drawable_type) {
					case GIMP_GRAY_IMAGE: default:
						break;
					case GIMP_GRAYA_IMAGE:
						sum[1] += dsum[1];  /* alpha of slopes layer becomes equal to selection */
						break;
					case GIMP_RGB_IMAGE:
						break;
					case GIMP_RGBA_IMAGE:
						sum[3] += dsum[3];  /* alpha of slopes layer becomes equal to selection */
						break;
					}

				} else {
					sum[0] += dsum[0];
					sum[1] += dsum[1];
					sum[2] += dsum[2];
					sum[3] += dsum[3];
				}
			}

			if (wsum) {
				if (g) {
					switch (drawable_type) {
					case GIMP_GRAY_IMAGE: default:
						outpixel[0] = CLAMP0255(pow(sum[0]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						if (newlayer) outpixel[1] = 255;
						break;
					case GIMP_GRAYA_IMAGE:
						outpixel[0] = CLAMP0255(pow(sum[0]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						outpixel[1] = CLAMP0255(sum[1]/wsum + ROUNDING);
						break;
					case GIMP_RGB_IMAGE:
						outpixel[0] = CLAMP0255(pow(sum[0]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						outpixel[1] = CLAMP0255(pow(sum[1]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						outpixel[2] = CLAMP0255(pow(sum[2]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						if (newlayer) outpixel[3] = 255;
						break;
					case GIMP_RGBA_IMAGE:
						outpixel[0] = CLAMP0255(pow(sum[0]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						outpixel[1] = CLAMP0255(pow(sum[1]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						outpixel[2] = CLAMP0255(pow(sum[2]/wsum/256.0, inv_gamma)*256.0 + ROUNDING);
						outpixel[3] = CLAMP0255(sum[3]/wsum + ROUNDING);
						break;
					}
				} else {
					switch (drawable_type) {
					case GIMP_GRAY_IMAGE: default:
						outpixel[0] = CLAMP0255(sum[0]/wsum + ROUNDING);
						if (newlayer) outpixel[1] = 255;
						break;
					case GIMP_GRAYA_IMAGE:
						outpixel[0] = CLAMP0255(sum[0]/wsum + ROUNDING);
						outpixel[1] = CLAMP0255(sum[1]/wsum + ROUNDING);
						break;
					case GIMP_RGB_IMAGE:
						outpixel[0] = CLAMP0255(sum[0]/wsum + ROUNDING);
						outpixel[1] = CLAMP0255(sum[1]/wsum + ROUNDING);
						outpixel[2] = CLAMP0255(sum[2]/wsum + ROUNDING);
						if (newlayer) outpixel[3] = 255;
						break;
					case GIMP_RGBA_IMAGE:
						outpixel[0] = CLAMP0255(sum[0]/wsum + ROUNDING);
						outpixel[1] = CLAMP0255(sum[1]/wsum + ROUNDING);
						outpixel[2] = CLAMP0255(sum[2]/wsum + ROUNDING);
						outpixel[3] = CLAMP0255(sum[3]/wsum + ROUNDING);
						break;
					}
				}
			} else {
				unknown_pixel(outdrawable_type, outpixel);
			}

			if (z && linnewlayer) {
				if (wsum) {
					if (g) {
						/* making a separate slopes layer when you have a gamma-compressed image is not recommended */
						switch (drawable_type) {
						case GIMP_GRAY_IMAGE: default:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/wsum/256.0, inv_gamma) - pow(sum[0]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[1] = 255;
							break;
						case GIMP_GRAYA_IMAGE:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/wsum/256.0, inv_gamma) - pow(sum[0]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[1] = CLAMP0255(sum[1]/wsum + ROUNDING);
							break;
						case GIMP_RGB_IMAGE:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/wsum/256.0, inv_gamma) - pow(sum[0]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[1] = CLAMP0255(128 + (pow((sum[1]+dsum[1])/wsum/256.0, inv_gamma) - pow(sum[1]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[2] = CLAMP0255(128 + (pow((sum[2]+dsum[2])/wsum/256.0, inv_gamma) - pow(sum[2]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[3] = 255;
							break;
						case GIMP_RGBA_IMAGE:
							outslopespixel[0] = CLAMP0255(128 + (pow((sum[0]+dsum[0])/wsum/256.0, inv_gamma) - pow(sum[0]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[1] = CLAMP0255(128 + (pow((sum[1]+dsum[1])/wsum/256.0, inv_gamma) - pow(sum[1]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[2] = CLAMP0255(128 + (pow((sum[2]+dsum[2])/wsum/256.0, inv_gamma) - pow(sum[2]/wsum/256.0, inv_gamma))*256.0 + ROUNDING);
							outslopespixel[3] = CLAMP0255(sum[3]/wsum + ROUNDING);
							break;
						}
					} else {
						switch (drawable_type) {
						case GIMP_GRAY_IMAGE: default:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/wsum + ROUNDING);
							outslopespixel[1] = 255;
							break;
						case GIMP_GRAYA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/wsum + ROUNDING);
							outslopespixel[1] = CLAMP0255(sum[1]/wsum + ROUNDING);
							break;
						case GIMP_RGB_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/wsum + ROUNDING);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/wsum + ROUNDING);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/wsum + ROUNDING);
							outslopespixel[3] = 255;
							break;
						case GIMP_RGBA_IMAGE:
							outslopespixel[0] = CLAMP0255(128+dsum[0]/wsum + ROUNDING);
							outslopespixel[1] = CLAMP0255(128+dsum[1]/wsum + ROUNDING);
							outslopespixel[2] = CLAMP0255(128+dsum[2]/wsum + ROUNDING);
							outslopespixel[3] = CLAMP0255(sum[3]/wsum + ROUNDING);
							break;
						}
					}
				} else {
					unknown_pixel(outdrawable_type, outslopespixel);
				}
			}

			if (changesel) {
				guchar *inmpixel = inmrow + (x-left)*mchannels;
				guchar *outspixel = outsrow + (x-left)*mchannels;

				/*
				 * input mask is multiplied with pixels without being divided by 255, so divide here
				 */
				if (halfsel<255)
					wsum /= 255.0;

				gfloat outsel = wsum2sel(wsum, lowthres, highthres, logthres) * inspixel[0];
				if (opts->multmask) outsel *= inmpixel[0]/255.0;
				outspixel[0] = outsel + ROUNDING;
			}
		}

		if (newlayer) {
			gimp_pixel_rgn_set_row(&out,outrow,0,in.y-top,right-left);
		} else {
			gimp_pixel_rgn_set_row(&out,outrow,left,in.y,right-left);
		}
		if (linnewlayer) {
			gimp_pixel_rgn_set_row(&outslopes,outslopesrow,0,in.y-top,right-left);
		}
		if (changesel) {
			gimp_pixel_rgn_set_row(&outs,outsrow,left,in.y,right-left);
		}

		if (!preview && opts->progress && !(in.y%4)) {
			gimp_progress_update(1.0*(in.y-top+1)/(bottom-top));
		}
	}
	if (!preview && opts->progress) gimp_progress_end();


	/*
	 * done with the pixel regions, kernel and slopes
	 */
	if (insrow) g_free(insrow);
	if (outsrow) g_free(outsrow);
	if (outslopesrow) g_free(outslopesrow);
	g_free(outrow);

	ring_fini(&inm);
	ring_fini(&in);

	if (nl) nonlinear_free(nl);
	kernel_fini(&kernel);

	if (z) g_free(z);


	/*
	 * the plumbing code
	 */
	if (preview) {
		gimp_drawable_preview_draw_region(GIMP_DRAWABLE_PREVIEW(preview), &out);
	} else {
		gimp_drawable_flush(outdrawable);
		gimp_drawable_merge_shadow(outdrawable->drawable_id, TRUE);
		if (newlayer) {
			gimp_drawable_update(outdrawable->drawable_id, 0, 0, width, height);
			gimp_drawable_detach(outdrawable);
		} else {
			gimp_drawable_update(outdrawable->drawable_id, left, top, width, height);
		}

		if (linnewlayer) {
			gimp_drawable_flush(outslopesdrawable);
			gimp_drawable_merge_shadow(outslopesdrawable->drawable_id, TRUE);
			gimp_drawable_update(outslopesdrawable->drawable_id, 0, 0, width, height);
			gimp_drawable_detach(outslopesdrawable);
		}
	}


	/*
	 * only remove the selection if we are modifying it
	 */
	if (changesel) {
		gint tmp_sel_id = gimp_selection_save(image_id);
		gimp_selection_none(image_id);

		gimp_drawable_flush(sel);
		gimp_drawable_merge_shadow(sel->drawable_id, TRUE);
		gimp_drawable_update(sel->drawable_id, left, top, width, height);

		gimp_selection_load(tmp_sel_id);
		gimp_image_remove_channel(image_id, tmp_sel_id);
	}


	/*
	 * no more need for temporary selection channel
	 */
	if (changesel) {
		gimp_selection_load(sel_id);
		#ifndef SKIP_UNSELECTED
			gimp_drawable_detach(sel);
			gimp_image_remove_channel(image_id, sel_id);
		#endif
	}
	#ifdef SKIP_UNSELECTED
		gimp_drawable_detach(sel);
		gimp_image_remove_channel(image_id, sel_id);
	#endif


	/*
	 * remove input mask channel if it is our temporary
	 */
	gimp_drawable_detach(mask);
	if (tmp_mask_id != -1) {
		gimp_image_remove_channel(image_id, tmp_mask_id);
	}


	/*
	 * free gamma curve
	 */
	if (g) {
		gamma_free(g);
	}

	if (!preview && !opts->progress) gimp_progress_end();
}

void render(GimpDrawable *drawable, PlugInVals *vals, GimpPreview *preview) {
	gint32 image_id = gimp_drawable_get_image(drawable->drawable_id);

	/* to restore the previously active layer */
	gint32 active_channel = gimp_image_get_active_channel(image_id);

	if (preview) {
		gimp_image_undo_freeze(image_id);
	} else {
		gimp_image_undo_group_start(image_id);
	}

	interpolate(drawable, vals, image_id, preview);

	if (preview) {
		gimp_image_undo_thaw(image_id);
	} else {
		gimp_image_undo_group_end(image_id);
	}

	if (active_channel == -1) {
		gimp_image_unset_active_channel(image_id);
	} else {
		gimp_image_set_active_channel(image_id, active_channel);
	}
}

void invoke_render(InterpolateInvocation *inv, GimpPreview *preview) {
	return render(inv->drawable, inv->vals, preview);
}
