/* GIMP Plug-in Toy
 * Copyright (C) 2011  Rüdiger Schneider <remschneid@web.de> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include "config.h"
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <gdk/gdkkeysyms.h>
#include "main.h"
#include "interface.h"
#include "plugin-intl.h"

gboolean toy_dialog(gint32 image_ID,GimpDrawable *drawable)
{
	GtkWidget *dialog;
	GtkWidget *hbox1;
	GtkWidget *hbox2;
	GtkWidget *vbox;
	GtkWidget *table;
	GtkWidget *frame1;
	GtkWidget *frame2;
	GtkWidget *preview;
	GtkWidget *gradientlabel;
	GtkObject *radius;
	GtkObject *contrast;
	GtkObject *saturation;
	GtkObject *startx;
	GtkObject *starty;
	GtkObject *endx;
	GtkObject *endy;	
	gboolean run,isrgb;
	
	tmpvals.drawable=drawable;
	toy_variables_init();
	isrgb=gimp_drawable_is_rgb(drawable->drawable_id);
	gimp_ui_init(PROCEDURE_NAME, FALSE);
	dialog=gimp_dialog_new(_("Toy"),PROCEDURE_NAME,NULL,0,gimp_standard_help_func,PROCEDURE_NAME,GTK_STOCK_CANCEL,GTK_RESPONSE_CANCEL,GTK_STOCK_OK, GTK_RESPONSE_OK,NULL);	
	g_signal_connect(dialog,"key_press_event",G_CALLBACK(pressed_keys),NULL);
	g_signal_connect(dialog,"key_release_event",G_CALLBACK(released_keys),NULL); 
	gimp_window_set_transient(GTK_WINDOW (dialog));
	hbox1=gtk_hbox_new(FALSE, 12);
	gtk_container_set_border_width(GTK_CONTAINER (hbox1), 12);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox),hbox1);
	gtk_widget_show(hbox1);	
	vbox=gtk_vbox_new(FALSE, 12);
	gtk_container_add(GTK_CONTAINER(hbox1),vbox);
	gtk_widget_show(vbox);
	frame1=gtk_aspect_frame_new(NULL,0.0,0.0,1.0,TRUE);
	gtk_frame_set_shadow_type(GTK_FRAME(frame1),GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(vbox),frame1,FALSE,FALSE,0);
	gtk_widget_show(frame1);
	tmpvals.preview=preview=gimp_preview_area_new();
	gtk_widget_set_size_request(preview,tmpvals.preview_width,tmpvals.preview_height);
	gtk_container_add(GTK_CONTAINER(frame1),preview);
	gtk_widget_show(preview);
	gtk_widget_add_events(preview,GDK_POINTER_MOTION_MASK|GDK_BUTTON1_MOTION_MASK|GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK|GDK_POINTER_MOTION_HINT_MASK);
	g_signal_connect(preview,"realize",G_CALLBACK(toy_realize_callback),NULL);	
	g_signal_connect(preview,"motion_notify_event",G_CALLBACK(motion_and_mouse),NULL);
	g_signal_connect(preview,"button_press_event",G_CALLBACK(motion_and_mouse),NULL);
	g_signal_connect(preview,"button_release_event",G_CALLBACK(mouse_released),NULL);
	g_signal_connect_after(preview, "expose-event",G_CALLBACK (toy_expose_callback),NULL);  	
  	gradientlabel=gimp_hint_box_new(_("Click and drag the handles in the preview to define the depth of field. Shift click and drag the handles to position the focus. Press the space bar to toggle visibility of the gradient handles."));
	gtk_box_pack_end (GTK_BOX(vbox),gradientlabel, FALSE, FALSE, 0);
	gtk_widget_show(gradientlabel);
	hbox2=gtk_hbox_new(FALSE,12);
	gtk_box_pack_start(GTK_BOX (hbox1),hbox2,FALSE,FALSE,0);
	gtk_widget_show(hbox2);
	frame2=gimp_frame_new(_("Picture parameters"));
	gtk_box_pack_start(GTK_BOX (hbox2),frame2,FALSE,FALSE,0);
	gtk_widget_show(frame2);
	table=gtk_table_new((isrgb)?7:6,3,FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table),6);
	gtk_table_set_row_spacings(GTK_TABLE(table),6);
	gtk_container_add(GTK_CONTAINER(frame2),table);
	gtk_widget_show(table);
	radius=gimp_scale_entry_new(GTK_TABLE(table),0,1,_("Blur:"),200,5,toyvals.radius,5.0,100.0,0.1,1.0,1,TRUE,0,0,NULL,NULL);
	g_signal_connect(radius,"value-changed",G_CALLBACK(gimp_double_adjustment_update_r),&toyvals.radius);
	contrast=gimp_scale_entry_new(GTK_TABLE(table),0,2,_("Contrast:"),200,5,toyvals.contrast, 0,127,1,10,0,TRUE,0,0,NULL,NULL);
	g_signal_connect(contrast,"value-changed",G_CALLBACK (gimp_int_adjustment_update_cs),&toyvals.contrast);	
	if (isrgb)
		{
		saturation=gimp_scale_entry_new(GTK_TABLE(table),0,3,_("Saturation:"),200,5,toyvals.saturation, 0,100,1,10,0,TRUE,0,0,NULL,NULL);
		g_signal_connect(saturation,"value-changed",G_CALLBACK(gimp_int_adjustment_update_cs),&toyvals.saturation);
		}
	startx=gimp_scale_entry_new(GTK_TABLE(table),0,4,_("Start X:"),200,5,toyvals.start[0],0.0,1.0,0.001,0.01,3,TRUE,0,0,NULL,NULL);
	handlevals.startx=GTK_ADJUSTMENT(startx);
	g_signal_connect(startx,"value-changed",G_CALLBACK(gimp_double_adjustment_update_m),&toyvals.start[0]);
	starty=gimp_scale_entry_new(GTK_TABLE(table),0,5,_("Start Y:"),200,5,toyvals.start[1],0.0,1.0,0.001,0.01,3,TRUE,0,0,NULL,NULL);
	handlevals.starty=GTK_ADJUSTMENT(starty);
	g_signal_connect(starty,"value-changed",G_CALLBACK(gimp_double_adjustment_update_m),&toyvals.start[1]);
	endx=gimp_scale_entry_new(GTK_TABLE(table),0,6,_("End X:"),200,5,toyvals.end[0],0.0,1.0,0.001,0.01,3,TRUE,0,0,NULL,NULL);
	handlevals.endx=GTK_ADJUSTMENT(endx);
	g_signal_connect(endx,"value-changed",G_CALLBACK(gimp_double_adjustment_update_m),&toyvals.end[0]);
	endy=gimp_scale_entry_new(GTK_TABLE(table),0,7,_("End Y:"),200,5,toyvals.end[1],0.0,1.0,0.001,0.01,3,TRUE,0,0,NULL,NULL);
	handlevals.endy=GTK_ADJUSTMENT(endy);
	g_signal_connect(endy,"value-changed",G_CALLBACK(gimp_double_adjustment_update_m),&toyvals.end[1]);
	gtk_widget_show(dialog);
	toy_value_changed(preview);
	run=(gimp_dialog_run(GIMP_DIALOG(dialog))==GTK_RESPONSE_OK);
	gtk_widget_destroy(dialog);
	tmpvals.preview=NULL;	
	tmpvals.background_id=-1;
	free_buffers_and_tempimage();
	return run;
}

void draw_gradient(GimpDrawable *drawable,GimpPreviewArea *preview)
{
	cairo_t *cr=gdk_cairo_create(GTK_WIDGET(preview)->window);;
	
	if (!cr) {return;}
	if (handlevals.gradient_hidden) {return;}

	gint middle_x=(gint)(ROUND(toyvals.start[0]*tmpvals.preview_width));
	gint middle_y=(gint)(ROUND(toyvals.start[1]*tmpvals.preview_height));
	gint end_x=(gint)(ROUND(toyvals.end[0]*tmpvals.preview_width));
	gint end_y=(gint)(ROUND(toyvals.end[1]*tmpvals.preview_height));
	gint difference_x=middle_x-end_x;
	gint difference_y=middle_y-end_y;
	gint start_x=middle_x+difference_x;
	gint start_y=middle_y+difference_y;
	
	if (!handlevals.show_line)
		{
		gboolean inpic_x,inpic_y;
		gint sharp_x1=middle_x+difference_y;
		gint sharp_y1=middle_y-difference_x;
		inpic_x=((sharp_x1>0)&&(sharp_x1<tmpvals.preview_width));
		inpic_y=((sharp_y1>0)&&(sharp_y1<tmpvals.preview_height));
		// change coordinate until out of preview
		while (inpic_x&&inpic_y)
			{
			sharp_x1=sharp_x1+difference_y;
			sharp_y1=sharp_y1-difference_x;
			inpic_x=((sharp_x1>0)&&(sharp_x1<tmpvals.preview_width));
			inpic_y=((sharp_y1>0)&&(sharp_y1<tmpvals.preview_height));
			}
		gint sharp_x2=middle_x+(middle_x-sharp_x1);
		gint sharp_y2=middle_y+(middle_y-sharp_y1);
		inpic_x=((sharp_x2>0)&&(sharp_x2<tmpvals.preview_width));
		inpic_y=((sharp_y2>0)&&(sharp_y2<tmpvals.preview_height));	
		// change coordinate until out of preview
		while (inpic_x&&inpic_y)
			{
			sharp_x2=sharp_x2-difference_y;
			sharp_y2=sharp_y2+difference_x;
			inpic_x=((sharp_x2>0)&&(sharp_x2<tmpvals.preview_width));
			inpic_y=((sharp_y2>0)&&(sharp_y2<tmpvals.preview_height));
			}
		// draw sharp horizon	
		cairo_set_source_rgb(cr,0.2,0.8,0.2);
		cairo_set_operator(cr,CAIRO_OPERATOR_SOURCE);
		cairo_move_to(cr,sharp_x1,sharp_y1);
		cairo_line_to(cr,sharp_x2,sharp_y2);
		cairo_stroke(cr);
		gint blurred1_x1=sharp_x1-difference_x;
		gint blurred1_y1=sharp_y1-difference_y;
		inpic_x=((blurred1_x1>0)&&(blurred1_x1<tmpvals.preview_width));
		inpic_y=((blurred1_y1>0)&&(blurred1_y1<tmpvals.preview_height));	
		// change coordinate until out of preview
		while (inpic_x&&inpic_y)
			{
			blurred1_x1=blurred1_x1+difference_y;
			blurred1_y1=blurred1_y1-difference_x;
			inpic_x=((blurred1_x1>0)&&(blurred1_x1<tmpvals.preview_width));
			inpic_y=((blurred1_y1>0)&&(blurred1_y1<tmpvals.preview_height));
			}
		gint blurred1_x2=sharp_x2-difference_x;
		gint blurred1_y2=sharp_y2-difference_y;
		inpic_x=((blurred1_x2>0)&&(blurred1_x2<tmpvals.preview_width));
		inpic_y=((blurred1_y2>0)&&(blurred1_y2<tmpvals.preview_height));	
		// change coordinate until out of preview
		while (inpic_x&&inpic_y)
			{
			blurred1_x2=blurred1_x2-difference_y;
			blurred1_y2=blurred1_y2+difference_x;
			inpic_x=((blurred1_x2>0)&&(blurred1_x2<tmpvals.preview_width));
			inpic_y=((blurred1_y2>0)&&(blurred1_y2<tmpvals.preview_height));
			}
		// draw first full blurred horizon
		cairo_set_source_rgb (cr,0.8,0.2,0.2);
		cairo_move_to(cr,blurred1_x1,blurred1_y1);
		cairo_line_to(cr,blurred1_x2,blurred1_y2);
		cairo_stroke(cr);
		gint blurred2_x1=sharp_x1+difference_x;
		gint blurred2_y1=sharp_y1+difference_y;
		inpic_x=((blurred2_x1>0)&&(blurred2_x1<tmpvals.preview_width));
		inpic_y=((blurred2_y1>0)&&(blurred2_y1<tmpvals.preview_height));	
		// change coordinate until out of preview
		while (inpic_x&&inpic_y)
			{
			blurred2_x1=blurred2_x1+difference_y;
			blurred2_y1=blurred2_y1-difference_x;
			inpic_x=((blurred2_x1>0)&&(blurred2_x1<tmpvals.preview_width));
			inpic_y=((blurred2_y1>0)&&(blurred2_y1<tmpvals.preview_height));
			}
		gint blurred2_x2=sharp_x2+difference_x;
		gint blurred2_y2=sharp_y2+difference_y;
		inpic_x=((blurred2_x2>0)&&(blurred2_x2<tmpvals.preview_width));
		inpic_y=((blurred2_y2>0)&&(blurred2_y2<tmpvals.preview_height));	
		// change coordinate until out of preview
		while (inpic_x&&inpic_y)
			{
			blurred2_x2=blurred2_x2-difference_y;
			blurred2_y2=blurred2_y2+difference_x;
			inpic_x=((blurred2_x2>0)&&(blurred2_x2<tmpvals.preview_width));
			inpic_y=((blurred2_y2>0)&&(blurred2_y2<tmpvals.preview_height));
			}
		// draw second full blurred horizon
		cairo_move_to(cr,blurred2_x1,blurred2_y1);
		cairo_line_to(cr,blurred2_x2,blurred2_y2);
		cairo_stroke(cr);
		}
	else 
		{
		cairo_set_source_rgb (cr,1,1,1);
		cairo_set_operator(cr,CAIRO_OPERATOR_ADD);
		cairo_move_to(cr,start_x,start_y);
		cairo_line_to(cr,end_x,end_y);
		cairo_stroke(cr);
		cairo_set_source_surface(cr,handlevals.endhandle,start_x-(handlevals.handlesize/2),start_y-(handlevals.handlesize/2));
		cairo_mask_surface(cr,handlevals.endhandle,start_x-(handlevals.handlesize/2),start_y-(handlevals.handlesize/2));
		}	
	
	// draw movable handles
	cairo_set_operator(cr,CAIRO_OPERATOR_SOURCE);
	cairo_set_source_surface(cr,handlevals.middlehandle,middle_x-(handlevals.handlesize/2),middle_y-(handlevals.handlesize/2));
	cairo_mask_surface(cr,handlevals.middlehandle,middle_x-(handlevals.handlesize/2),middle_y-(handlevals.handlesize/2));
	cairo_set_source_surface(cr,handlevals.starthandle,end_x-(handlevals.handlesize/2),end_y-(handlevals.handlesize/2));
	cairo_mask_surface(cr,handlevals.starthandle,end_x-(handlevals.handlesize/2),end_y-(handlevals.handlesize/2));
	cairo_destroy(cr);
}

void toy_variables_init()
{
	cairo_t *cr;
	gdouble factor_x,factor_y;
	
	factor_x=(gdouble)tmpvals.drawable->width/tmpvals.maxpreview;
	factor_y=(gdouble)tmpvals.drawable->height/tmpvals.maxpreview;
	tmpvals.pre2img=(factor_x>=factor_y)?factor_x:factor_y;
	if ((factor_x<=1.0)&&(factor_y<=1.0)) {tmpvals.pre2img=1.0;}
	tmpvals.preview_width=(gint)(tmpvals.drawable->width/tmpvals.pre2img);
	tmpvals.preview_height=(gint)(tmpvals.drawable->height/tmpvals.pre2img);
	handlevals.coordinates[0]=(gint)(ROUND(toyvals.start[0]*tmpvals.preview_width));
	handlevals.coordinates[1]=(gint)(ROUND(toyvals.start[1]*tmpvals.preview_height));
	handlevals.coordinates[2]=(gint)(ROUND(toyvals.end[0]*tmpvals.preview_width));
	handlevals.coordinates[3]=(gint)(ROUND(toyvals.end[1]*tmpvals.preview_height));
	handlevals.difference_x=handlevals.coordinates[2]-handlevals.coordinates[0];
	handlevals.difference_y=handlevals.coordinates[3]-handlevals.coordinates[1];
	if (!handlevals.middlehandle)
		{
		handlevals.middlehandle=cairo_image_surface_create(CAIRO_FORMAT_RGB24,handlevals.handlesize,handlevals.handlesize);
		cr=cairo_create(handlevals.middlehandle);
		cairo_set_source_rgb (cr,1,1,1);
		cairo_rectangle(cr,0,0,handlevals.handlesize,handlevals.handlesize);
		cairo_fill(cr);
		cairo_set_source_rgb (cr,0.2,0.8,0.2);
		cairo_rectangle(cr,1,1,handlevals.handlesize-2,handlevals.handlesize-2);
		cairo_fill(cr);
		cairo_destroy(cr);
		}	
	if (!handlevals.starthandle)
		{
		handlevals.starthandle=cairo_image_surface_create(CAIRO_FORMAT_RGB24,handlevals.handlesize,handlevals.handlesize);
		cr=cairo_create(handlevals.starthandle);
		cairo_set_source_rgb (cr,1,1,1);
		cairo_rectangle(cr,0,0,handlevals.handlesize,handlevals.handlesize);
		cairo_fill(cr);
		cairo_set_source_rgb (cr,0.8,0.2,0.2);
		cairo_rectangle(cr,1,1,handlevals.handlesize-2,handlevals.handlesize-2);
		cairo_fill(cr);
		cairo_destroy(cr);
		}
	if (!handlevals.endhandle)
		{
		handlevals.endhandle=cairo_image_surface_create(CAIRO_FORMAT_RGB24,handlevals.handlesize,handlevals.handlesize);
		cr=cairo_create(handlevals.endhandle);
		cairo_set_source_rgb (cr,1,1,1);
		cairo_move_to(cr,handlevals.handlesize/2,0);
		cairo_line_to(cr,handlevals.handlesize/2,handlevals.handlesize-1);
		cairo_stroke(cr);
		cairo_move_to(cr,0,handlevals.handlesize/2);
		cairo_line_to(cr,handlevals.handlesize-1,handlevals.handlesize/2);
		cairo_stroke(cr);
		cairo_destroy(cr);
		}
	// To ensure that preview is updated at dialog appearance
	preview_idle=1;	
}

void toy_realize_callback(GtkWidget *widget)
{
	GdkDisplay *display=gtk_widget_get_display(widget);
	GdkCursor *cursor=gdk_cursor_new_for_display(display,GDK_CROSSHAIR);
	gdk_window_set_cursor(widget->window,cursor);
	gdk_cursor_unref(cursor);
}

gboolean toy_expose_callback(GtkWidget *widget,GdkEvent *event)
{
	draw_gradient(tmpvals.drawable,GIMP_PREVIEW_AREA(tmpvals.preview));
	return FALSE;
}

void mouse_released(GtkWidget *widget,GdkEvent *event)
{
	if (handlevals.button_down)
		{
		handlevals.coordinates[0]=CLAMP(handlevals.coordinates[0],0,tmpvals.preview_width-1);
		handlevals.coordinates[1]=CLAMP(handlevals.coordinates[1],0,tmpvals.preview_height-1);
		handlevals.coordinates[2]=CLAMP(handlevals.coordinates[2],0,tmpvals.preview_width-1);
		handlevals.coordinates[3]=CLAMP(handlevals.coordinates[3],0,tmpvals.preview_height-1);
		handlevals.difference_x=handlevals.coordinates[2]-handlevals.coordinates[0];
		handlevals.difference_y=handlevals.coordinates[3]-handlevals.coordinates[1];
		handlevals.button_down=FALSE;
		}
}

gboolean motion_and_mouse(GtkWidget *widget,GdkEvent *event)
{
	GdkModifierType state;
	gint x,y,size=handlevals.handlesize/2;
	gboolean shifted,pressed;
	
	if (handlevals.gradient_hidden) 
		{
		handlevals.difference_x=0;
		handlevals.difference_y=0;
		handlevals.inhandle1=FALSE;
		handlevals.inhandle2=FALSE;
		return TRUE;
		}

	gdk_window_get_pointer(widget->window,&x,&y,&state);
	shifted=(state&GDK_SHIFT_MASK);
	pressed=(state&GDK_BUTTON1_MASK);

	if (handlevals.button_down&&pressed) {recalc_handle_position(x,y,shifted);}
	
	if ((!shifted&&handlevals.button_down)||(!handlevals.button_down))
		{
		handlevals.inhandle1=((x>=handlevals.coordinates[0]-size)&&(x<=handlevals.coordinates[0]+size)&&(y>=handlevals.coordinates[1]-size)&&(y<=handlevals.coordinates[1]+size));
		handlevals.inhandle2=((x>=handlevals.coordinates[2]-size)&&(x<=handlevals.coordinates[2]+size)&&(y>=handlevals.coordinates[3]-size)&&(y<=handlevals.coordinates[3]+size));	
		}
	
	GdkDisplay *display=gtk_widget_get_display(widget);
	GdkCursor *cursor=gdk_cursor_new_for_display(display,((!handlevals.inhandle1)&&(!handlevals.inhandle2))?GDK_CROSSHAIR:(shifted)?GDK_FLEUR:GDK_HAND1);
	gdk_window_set_cursor(widget->window,cursor);
	gdk_cursor_unref(cursor);
	
	if (pressed&&!handlevals.button_down)
		{
		gint mx=(gint)(ROUND(toyvals.start[0]*tmpvals.preview_width));
		gint my=(gint)(ROUND(toyvals.start[1]*tmpvals.preview_height));
		gint ex=(gint)(ROUND(toyvals.end[0]*tmpvals.preview_width));
		gint ey=(gint)(ROUND(toyvals.end[1]*tmpvals.preview_height));
		if (handlevals.inhandle1)
			{
			handlevals.offsetx=x-mx;
			handlevals.offsety=y-my;
			handlevals.button_down=TRUE;
			}
		else if (handlevals.inhandle2)
			{
			handlevals.offsetx=x-ex;
			handlevals.offsety=y-ey;
			handlevals.button_down=TRUE;
			}
		}
		
return FALSE;
}

void recalc_handle_position(gint x,gint y,gboolean both)
{
	gint w=tmpvals.preview_width;
	gint h=tmpvals.preview_height;
	gint active_handle_x=x+handlevals.offsetx;
	gint active_handle_y=y+handlevals.offsety;	
	gint other_handle_x=0,other_handle_y=0;
	
	if (both&&handlevals.inhandle1)
		{
		other_handle_x=active_handle_x+handlevals.difference_x;
		other_handle_y=active_handle_y+handlevals.difference_y;
		}
	else if(both&&handlevals.inhandle2)
		{
		other_handle_x=active_handle_x-handlevals.difference_x;
		other_handle_y=active_handle_y-handlevals.difference_y;
		}
		
	if (handlevals.inhandle1)
		{
		if (both)
			{	
			if (handlevals.difference_x>=0)
				{
				if (active_handle_x>=w-handlevals.difference_x-1) 
					{
					active_handle_x=w-handlevals.difference_x-1;
					other_handle_x=w-1;
					}
				else if (active_handle_x<=0)
					{
					active_handle_x=0;
					other_handle_x=handlevals.difference_x;
					}
				}
			else	//handlevals.difference_x<0
				{
				if (active_handle_x>=w-1) 
					{					
					active_handle_x=w-1;
					other_handle_x=w+handlevals.difference_x-1;
					}
				else if (active_handle_x<=-handlevals.difference_x)
					{
					active_handle_x=-handlevals.difference_x;
					other_handle_x=0;
					}
				}
			if (handlevals.difference_y>=0)
				{
				if (active_handle_y>=h-handlevals.difference_y-1) 
					{
					active_handle_y=h-handlevals.difference_y-1;
					other_handle_y=h-1;
					}
				else if (active_handle_y<=0)
					{
					active_handle_y=0;
					other_handle_y=handlevals.difference_y;	
					}
				}
			else //handlevals.difference_y<0
				{
				if (active_handle_y>=h-1) 
					{					
					active_handle_y=h-1;
					other_handle_y=h+handlevals.difference_y-1;
					}
				else if (active_handle_y<=-handlevals.difference_y)
					{
					active_handle_y=-handlevals.difference_y;
					other_handle_y=0;
					}
				}
			toyvals.end[0]=round_value((gdouble)other_handle_x/(gdouble)tmpvals.preview_width);
			toyvals.end[1]=round_value((gdouble)other_handle_y/(gdouble)tmpvals.preview_height);	
			gtk_adjustment_set_value(handlevals.endx,toyvals.end[0]);
			gtk_adjustment_set_value(handlevals.endy,toyvals.end[1]);
			handlevals.coordinates[2]=other_handle_x;
			handlevals.coordinates[3]=other_handle_y;
			}
		toyvals.start[0]=round_value((gdouble)active_handle_x/(gdouble)tmpvals.preview_width);
		toyvals.start[1]=round_value((gdouble)active_handle_y/(gdouble)tmpvals.preview_height);		
		gtk_adjustment_set_value(handlevals.startx,toyvals.start[0]);
		gtk_adjustment_set_value(handlevals.starty,toyvals.start[1]);
		handlevals.coordinates[0]=active_handle_x;
		handlevals.coordinates[1]=active_handle_y;
		}
	else if (handlevals.inhandle2)
		{
		if (both)
			{
			if (handlevals.difference_x>=0)
				{
				if (active_handle_x>=w-1)
					{
					active_handle_x=w-1;
					other_handle_x=w-handlevals.difference_x-1;
					}
				else if (active_handle_x<=handlevals.difference_x)
					{
					active_handle_x=handlevals.difference_x;
					other_handle_x=0;
					}
				}
			else	//handlevals.difference_x<0
				{
				if (active_handle_x>=w+handlevals.difference_x-1) 
					{					
					active_handle_x=w+handlevals.difference_x-1;
					other_handle_x=w-1;
					}
				else if (active_handle_x<=0)
					{
					active_handle_x=0;
					other_handle_x=-handlevals.difference_x;
					}
				}
			if (handlevals.difference_y>=0)
				{
				if (active_handle_y>=h-1) 
					{
					active_handle_y=h-1;
					other_handle_y=h-handlevals.difference_y-1;
					}
				else if (active_handle_y<=handlevals.difference_y)
					{
					active_handle_y=handlevals.difference_y;
					other_handle_y=0;	
					}
				}
			else //handlevals.difference_y<0
				{
				if (active_handle_y>=h+handlevals.difference_y-1) 
					{					
					active_handle_y=h+handlevals.difference_y-1;
					other_handle_y=h-1;
					}
				else if (active_handle_y<=0)
					{
					active_handle_y=0;
					other_handle_y=-handlevals.difference_y;
					}
				}
			toyvals.start[0]=round_value((gdouble)other_handle_x/(gdouble)tmpvals.preview_width);
			toyvals.start[1]=round_value((gdouble)other_handle_y/(gdouble)tmpvals.preview_height);	
			gtk_adjustment_set_value(handlevals.startx,toyvals.start[0]);
			gtk_adjustment_set_value(handlevals.starty,toyvals.start[1]);
			handlevals.coordinates[0]=other_handle_x;
			handlevals.coordinates[1]=other_handle_y;
			}
		toyvals.end[0]=round_value((gdouble)active_handle_x/(gdouble)tmpvals.preview_width);
		toyvals.end[1]=round_value((gdouble)active_handle_y/(gdouble)tmpvals.preview_height);	
		gtk_adjustment_set_value(handlevals.endx,toyvals.end[0]);
		gtk_adjustment_set_value(handlevals.endy,toyvals.end[1]);
		handlevals.coordinates[2]=active_handle_x;
		handlevals.coordinates[3]=active_handle_y;
		}
}

gdouble round_value(gdouble in)
{
	gint roundedint=(gint)((in+0.0005)*1000.0);
	gdouble out=CLAMP((gdouble)roundedint/1000.0,0.0,1.0);
	return out;
}

gboolean pressed_keys(GtkWidget *widget,GdkEventKey *key)
{
	GdkCursor *cursor;
	GdkDisplay *display;
	GdkModifierType state;
				
	if (key->keyval==GDK_space)
		{	
		handlevals.gradient_hidden=!handlevals.gradient_hidden;
		display=gtk_widget_get_display(tmpvals.preview);
		gdk_window_get_pointer(tmpvals.preview->window,NULL,NULL,&state);
		if ((handlevals.inhandle1||handlevals.inhandle2))
			{				
			if (tmpvals.preview&&handlevals.gradient_hidden)
				{
				cursor=gdk_cursor_new_for_display(display,GDK_CROSSHAIR);
				gdk_window_set_cursor(tmpvals.preview->window,cursor);
				gdk_cursor_unref(cursor);
				}
			else if (tmpvals.preview&&!handlevals.gradient_hidden)
				{
				cursor=gdk_cursor_new_for_display(display,(state&GDK_SHIFT_MASK)?GDK_FLEUR:GDK_HAND1);
				gdk_window_set_cursor(tmpvals.preview->window,cursor);
				gdk_cursor_unref(cursor);
				}
			}
		else if ((!handlevals.inhandle1||!handlevals.inhandle2)&&tmpvals.preview)
			{
			cursor=gdk_cursor_new_for_display(display,GDK_CROSSHAIR);
			gdk_window_set_cursor(tmpvals.preview->window,cursor);
			gdk_cursor_unref(cursor);
			}
		// redraw preview
		preview_idle=1;	
		toy_value_changed(NULL);
		return TRUE;	
		}
	else if 	((key->keyval==GDK_KEY_Shift_L)||(key->keyval==GDK_KEY_Shift_R))
		{
		display=gtk_widget_get_display(tmpvals.preview);
		if ((handlevals.inhandle1||handlevals.inhandle2)&&tmpvals.preview)
			{
			cursor=gdk_cursor_new_for_display(display,GDK_FLEUR);
			gdk_window_set_cursor(tmpvals.preview->window,cursor);
			gdk_cursor_unref(cursor);
			}
		else if ((!handlevals.inhandle1&&!handlevals.inhandle2)&&tmpvals.preview)
			{	
			cursor=gdk_cursor_new_for_display(display,GDK_CROSSHAIR);
			gdk_window_set_cursor(tmpvals.preview->window,cursor);
			gdk_cursor_unref(cursor);
			}	
		return TRUE;	
		}		
	return FALSE;
}

gboolean released_keys(GtkWidget *widget,GdkEventKey *key)
{
	if ((key->keyval==GDK_KEY_Shift_L)||(key->keyval==GDK_KEY_Shift_R))
		{
		if ((handlevals.inhandle1||handlevals.inhandle2)&&tmpvals.preview)
			{	
			GdkDisplay *display=gtk_widget_get_display(tmpvals.preview);
			GdkCursor *cursor=gdk_cursor_new_for_display(display,GDK_HAND1);
			gdk_window_set_cursor(tmpvals.preview->window,cursor);
			gdk_cursor_unref(cursor);
			return TRUE;
			}
		}
	return FALSE;
}
	
void toy_value_changed(GtkWidget *widget)
{
	toy();
	toy_update_preview(0,0,tmpvals.preview_width,tmpvals.preview_height);
	preview_idle=0;
}

void toy_update_preview(gint x0,gint y0,gint x1,gint y1)
{
	if (!preview_idle) {return;}
	gimp_preview_area_draw(GIMP_PREVIEW_AREA(tmpvals.preview),x0,y0,x1-x0,y1-y0,gimp_drawable_type(tmpvals.drawable->drawable_id),tmpvals.preview_buffer,tmpvals.preview_width*gimp_drawable_bpp(tmpvals.drawable->drawable_id));
	draw_gradient(tmpvals.drawable,GIMP_PREVIEW_AREA(tmpvals.preview));
}

void gimp_int_adjustment_update_cs(GtkAdjustment *adjustment,gpointer data)
{
	gint *val=(gint *)data;
	*val=RINT(gtk_adjustment_get_value(adjustment));
	tmpvals.radius_changed=TRUE;
	tmpvals.contrastsaturation_changed=TRUE;
	if (preview_idle) {g_source_remove(preview_idle);}
	preview_idle=g_idle_add_full(G_PRIORITY_LOW,(GSourceFunc)toy_value_changed,NULL,NULL);
}

void gimp_double_adjustment_update_r(GtkAdjustment *adjustment,gpointer data)
{
	gdouble *val=(gdouble*)data;
	*val=gtk_adjustment_get_value(adjustment);
	tmpvals.radius_changed=TRUE;
	if (preview_idle) {g_source_remove(preview_idle);}
	preview_idle=g_idle_add_full(G_PRIORITY_LOW,(GSourceFunc)toy_value_changed,NULL,NULL);
}

void gimp_double_adjustment_update_m(GtkAdjustment *adjustment,gpointer data)
{
	gdouble *val=(gdouble*)data;
	*val=gtk_adjustment_get_value(adjustment);
	if (adjustment==handlevals.startx)
		{
		handlevals.coordinates[0]=(gint)(ROUND(toyvals.start[0]*tmpvals.preview_width));
		handlevals.difference_x=handlevals.coordinates[2]-handlevals.coordinates[0];
		}
	else if (adjustment==handlevals.starty)
		{
		handlevals.coordinates[1]=(gint)(ROUND(toyvals.start[1]*tmpvals.preview_height));
		handlevals.difference_y=handlevals.coordinates[3]-handlevals.coordinates[1];
		}	
	else if (adjustment==handlevals.endx)
		{
		handlevals.coordinates[2]=(gint)(ROUND(toyvals.end[0]*tmpvals.preview_width));
		handlevals.difference_x=handlevals.coordinates[2]-handlevals.coordinates[0];
		}	
	else if (adjustment==handlevals.endy)
		{
		handlevals.coordinates[3]=(gint)(ROUND(toyvals.end[1]*tmpvals.preview_height));
		handlevals.difference_y=handlevals.coordinates[3]-handlevals.coordinates[1];
		}	
	tmpvals.mask_changed=TRUE;
	if (preview_idle) {g_source_remove(preview_idle);}
	preview_idle=g_idle_add_full(G_PRIORITY_LOW,(GSourceFunc)toy_value_changed,NULL,NULL);
}

