/* GIMP Plug-in Toy
 * Copyright (C) 2011  Rüdiger Schneider <remschneid@web.de> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include "config.h"

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "main.h"
#include "render.h"
#include "interface.h"
#include "plugin-intl.h"

static void query(void);
static void run(const gchar *name,gint nparams,const GimpParam *param,gint *nreturn_vals,GimpParam **return_vals);

gint preview_idle=0;
gboolean is27=FALSE;
gchar *oldgradient=NULL;
gchar *toygradient=NULL;
ToyVals toyvals={10,30,25.0,{0.5,0.5},{0.5,1.0},NULL,BLUR_NORMAL};
TmpVals tmpvals={0,0,MAXPREVIEWSIZE,-1,-1,NULL,NULL,NULL,NULL,NULL,NULL,1.0,NULL,FALSE,FALSE,FALSE};
HandleVals handlevals={0,0,13,0,0,0,{0,0,0,0},NULL,NULL,NULL,NULL,NULL,NULL,NULL,FALSE,FALSE,FALSE,FALSE,FALSE};
GimpPlugInInfo PLUG_IN_INFO ={NULL,NULL,query,run};

MAIN ()

static void query(void)
{
	gchar *help_path;
	gchar *help_uri;

	static GimpParamDef args[] =
		{
			{GIMP_PDB_INT32,"run_mode","Interactive, non-interactive"},
			{GIMP_PDB_IMAGE,"image","Input image"},
			{GIMP_PDB_DRAWABLE,"drawable","Input drawable"},
			{GIMP_PDB_INT16,"contrast","Change of contrast (0 - 127)"},
			{GIMP_PDB_INT16,"saturation","Change of saturation (0 - 100)"},
			{GIMP_PDB_FLOAT,"radius","Radius of blur (0-100)"},
			{GIMP_PDB_FLOAT,"startx","X startposition of gradient (0.0-1.0)"},
			{GIMP_PDB_FLOAT,"starty","Y startposition of gradient (0.0-1.0)"},
			{GIMP_PDB_FLOAT,"endx","X endposition of gradient (0.0-1.0)"},
			{GIMP_PDB_FLOAT,"endy","Y endposition of gradient (0.0-1.0)"},
		};

	gimp_install_procedure(PROCEDURE_NAME,
		N_("Creates a toy effect"),
		"This plug-in creates a toy effect on the specified drawable. "
		"Indexed images are not supported.",
		"Rüdiger Schneider",
		"Copyright Rüdiger Schneider",
		"2011",
		N_("Toy..."),
		"RGB*, GRAY*",
		GIMP_PLUGIN,
		G_N_ELEMENTS (args), 0,
		args, NULL);

	gimp_plugin_menu_register (PROCEDURE_NAME,"<Image>/Filters/Blur/");
  
	#ifdef ENABLE_NLS
		gimp_plugin_domain_register (GETTEXT_PACKAGE,LOCALEDIR);
	#endif
	
	help_path=g_build_filename(DATADIR,"help",NULL);
	help_uri=g_filename_to_uri(help_path, NULL, NULL);
	g_free(help_path);
	gimp_plugin_help_register("http://developer.gimp.org/plug-in-toy/help",help_uri);
}


static void run (const gchar *name,gint n_params,const GimpParam *param,gint *nreturn_vals,GimpParam **return_vals)
{
	static GimpParam values[1];
	gint32 image;
	GimpDrawable *drawable;
	GimpRunMode run_mode;
	GimpPDBStatusType  status=GIMP_PDB_SUCCESS;

	INIT_I18N();

	gchar* gimpversion=g_strndup(gimp_version(),3);
	gdouble val=g_ascii_strtod(gimpversion,NULL);
	is27=(val>=2.7)?TRUE:FALSE;
	g_free(gimpversion);

	run_mode=param[0].data.d_int32;
	image=param[1].data.d_int32;
	drawable=gimp_drawable_get(param[2].data.d_drawable);
	
	 if (gimp_drawable_is_rgb(drawable->drawable_id)||gimp_drawable_is_gray(drawable->drawable_id))
    	{	
    	switch (run_mode)
			{
			case GIMP_RUN_INTERACTIVE:
				get_from_gimprc();
				gimp_get_data(PROCEDURE_NAME,&toyvals);
				if (!toy_dialog(param[1].data.d_image,drawable)) {status=GIMP_PDB_EXECUTION_ERROR;}
				break;
			case GIMP_RUN_NONINTERACTIVE:
				if (n_params!=10) {status=GIMP_PDB_CALLING_ERROR;}
				if (status==GIMP_PDB_SUCCESS) 
					{
					tmpvals.drawable=drawable;
					toyvals.contrast=param[3].data.d_int16;
					toyvals.saturation=param[4].data.d_int16;
					toyvals.radius=param[5].data.d_float;
					toyvals.start[0]=param[6].data.d_float;
					toyvals.start[1]=param[7].data.d_float;
					toyvals.end[0]=param[8].data.d_float;
					toyvals.end[1]=param[9].data.d_float;			
					}
				break;
			case GIMP_RUN_WITH_LAST_VALS:
				gimp_get_data(PROCEDURE_NAME,&toyvals);
				break;
			default:
				break;
			}
		}
	else
		{
		status=GIMP_PDB_EXECUTION_ERROR;
		}
	
	*nreturn_vals=1;
	*return_vals=values;
	values[0].type=GIMP_PDB_STATUS;
	values[0].data.d_status=status;
	
	if (status==GIMP_PDB_SUCCESS)
		{
		gimp_image_undo_group_start(image);
		toy();
		gimp_image_undo_group_end(image);
			
		gimp_displays_flush();
		gimp_drawable_detach(drawable);
		
		if (run_mode==GIMP_RUN_INTERACTIVE) 
			{
			gimp_set_data(PROCEDURE_NAME,&toyvals,sizeof (toyvals));
			save_to_gimprc();
			}
		}
		
	free_buffers_and_tempimage();
	restore_original_gradient();
}

void free_buffers_and_tempimage()
{
	if (tmpvals.pic_buffer) 
		{
		g_free(tmpvals.pic_buffer);
		tmpvals.pic_buffer=NULL;
		}
	if (tmpvals.preview_buffer) 
		{
		g_free(tmpvals.preview_buffer);
		tmpvals.preview_buffer=NULL;
		}
	if (tmpvals.tmpimg_id!=-1) 
		{
		gimp_image_delete(tmpvals.tmpimg_id);
		tmpvals.tmpimg_id=-1;
		}
	if (tmpvals.csbuffer) 
		{
		gimp_buffer_delete(tmpvals.csbuffer);
		tmpvals.csbuffer=NULL;
		}
	if (tmpvals.alpbuffer) 
		{
		gimp_buffer_delete(tmpvals.alpbuffer);
		tmpvals.alpbuffer=NULL;
		}
	if (handlevals.starthandle) 
		{
		cairo_surface_destroy(handlevals.starthandle);
		handlevals.starthandle=NULL;
		}
	if (handlevals.middlehandle) 
		{
		cairo_surface_destroy(handlevals.middlehandle);
		handlevals.middlehandle=NULL;
		}
	if (handlevals.endhandle) 
		{
		cairo_surface_destroy(handlevals.endhandle);
		handlevals.endhandle=NULL;
		}
}

void get_from_gimprc()
{
	gchar *value=NULL,*token=NULL;
	
	token=g_new(gchar,30);
	g_return_if_fail(token!=NULL);
	g_snprintf(token,30,"%s_radius",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gdouble val=g_ascii_strtod(value,NULL);
		if ((val>=5.0)&&(val<=100.0)) {toyvals.radius=val;}
		g_free(value);
		}
	g_snprintf(token,30,"%s_contrast",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gint contrast=0;	
		gdouble val=g_ascii_strtod(value,NULL);
		contrast=(gint)val;		
		if ((contrast>=0)&&(contrast<=127)) {toyvals.contrast=contrast;}
		g_free(value);
		}
	g_snprintf(token,30,"%s_saturation",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gint saturation=0;	
		gdouble val=g_ascii_strtod(value,NULL);
		saturation=(gint)val;		
		if ((saturation>=0)&&(saturation<=100)) {toyvals.saturation=saturation;}
		g_free(value);
		}
	g_snprintf(token,30,"%s_startx",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gdouble val=g_ascii_strtod(value,NULL);
		if ((val>=0.0)&&(val<=1.0)) {toyvals.start[0]=val;}
		g_free(value);
		}
	g_snprintf(token,30,"%s_starty",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gdouble val=g_ascii_strtod(value,NULL);
		if ((val>=0.0)&&(val<=1.0)) {toyvals.start[1]=val;}
		g_free(value);
		}
	g_snprintf(token,30,"%s_endx",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gdouble val=g_ascii_strtod(value,NULL);
		if ((val>=0.0)&&(val<=1.0)) {toyvals.end[0]=val;}
		g_free(value);
		}
	g_snprintf(token,30,"%s_endy",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gdouble val=g_ascii_strtod(value,NULL);
		if ((val>=0.0)&&(val<=1.0)) {toyvals.end[1]=val;}
		g_free(value);
		}
	sprintf(token,"%s_showline",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gint8 tmpvalue;	
		gdouble val=g_ascii_strtod(value,NULL);
		tmpvalue=(gint16)val;
		handlevals.show_line=(tmpvalue)?TRUE:FALSE;
		g_free (value);
		}
	sprintf(token,"%s_previewsize",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gdouble val=g_ascii_strtod(value,NULL);
		tmpvals.maxpreview=(gint16)val;
		if ((!tmpvals.maxpreview)||(tmpvals.maxpreview>2048)) {tmpvals.maxpreview=MAXPREVIEWSIZE;}
		g_free (value);
		}		
	sprintf(token,"%s_handlesize",PROCEDURE_NAME);
	value=gimp_gimprc_query(token);
	if (value)
		{
		gint8 tmpvalue;
		gdouble val=g_ascii_strtod(value,NULL);
		tmpvalue=(gint8)val;
		if ((tmpvalue>=5)||(tmpvalue<=255)) {handlevals.handlesize=tmpvalue;}
		g_free (value);
		}
	else 
		{
		sprintf(token,"%s_knotsize",PROCEDURE_NAME);
		value=gimp_gimprc_query(token);
		if (value)
			{
			gint8 tmpvalue;	
			gdouble val=g_ascii_strtod(value,NULL);
			tmpvalue=(gint8)val;
			if ((tmpvalue>=5)||(tmpvalue<=255)) {handlevals.handlesize=tmpvalue;}
			g_free (value);
			}	
		}	
	g_free(token);
}

void save_to_gimprc()
{
	gchar *value=NULL,*token=NULL;

	token=g_new(gchar,30);
	g_return_if_fail(token!=NULL);
	value=g_new(gchar,30);
	g_return_if_fail(value!=NULL);
	
	g_snprintf(token,30,"%s_radius",PROCEDURE_NAME);
	setlocale(LC_NUMERIC, "C");
	g_snprintf(value,30,"%3.1f",toyvals.radius);
	gimp_gimprc_set(token,value);
	g_snprintf(token,30,"%s_contrast",PROCEDURE_NAME);
	g_snprintf(value,30,"%d",toyvals.contrast);
	gimp_gimprc_set(token,value);
	g_snprintf(token,30,"%s_saturation",PROCEDURE_NAME);
	g_snprintf(value,30,"%d",toyvals.saturation);
	gimp_gimprc_set(token,value);
	g_snprintf(token,30,"%s_startx",PROCEDURE_NAME);
	g_snprintf(value,30,"%1.3f",toyvals.start[0]);
	gimp_gimprc_set(token,value);
	g_snprintf(token,30,"%s_starty",PROCEDURE_NAME);
	g_snprintf(value,30,"%1.3f",toyvals.start[1]);
	gimp_gimprc_set(token,value);
	g_snprintf(token,30,"%s_endx",PROCEDURE_NAME);
	g_snprintf(value,30,"%1.3f",toyvals.end[0]);
	gimp_gimprc_set(token,value);
	g_snprintf(token,30,"%s_endy",PROCEDURE_NAME);
	g_snprintf(value,30,"%1.3f",toyvals.end[1]);
	gimp_gimprc_set(token,value);
	g_free(token);
	g_free(value);
}

void restore_original_gradient()
{
	if (oldgradient) {gimp_context_set_gradient(oldgradient);}
	if (toygradient&&oldgradient) {gimp_gradient_delete(toygradient);}
}