/* GIMP Plug-in Toy
 * Copyright (C) 2011  Rüdiger Schneider <remschneid@web.de> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include "config.h"

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "main.h"
#include "render.h"
#include "plugin-intl.h"

GimpLut* gimp_lut_new(void)
{
	GimpLut *lut;
	lut=g_slice_new(GimpLut);
	lut->luts=NULL;
	lut->nchannels=0;
	return lut;
}

void gimp_lut_setup(GimpLut *lut,GimpLutFunc func,gint bytes)
{
	guint v;
	gdouble val;
	
	g_return_if_fail(lut!=NULL);
	if (lut->luts)
		{
		g_free(lut->luts[0]);
		g_free(lut->luts);
		}
	lut->nchannels=1;
	lut->luts=g_new(guchar*,1);
	lut->luts[0]=g_new(guchar,256);
	for (v=0;v<256;v++)
		{
		val=255.0*func(bytes,0,v/255.0)+0.5;
		lut->luts[0][v]=CLAMP0255(val);
		}
}

void gimp_lut_free (GimpLut *lut)
{
	if (!lut) {return;} 
	g_free(lut->luts[0]);
	g_free(lut->luts);
	g_slice_free(GimpLut,lut);
}

void separate_alpha(guchar *buffer,gint width,gint bytes)
{
	gint i,j;

	for (i=0;i<width;i++,buffer+=bytes)
		{
		guchar alpha=buffer[bytes-1];
		switch (alpha)
			{
			case 0:
			case 255:
				break;
			default:
				{
				gdouble recip_alpha=255.0/alpha;
				for (j=0;j<bytes-1;j++)
					{
					gint new_val=ROUND(buffer[j]*recip_alpha);
					buffer[j]=MIN(255,new_val);
					}
				}
				break;
			}
		}
}

void multiply_alpha(guchar *buffer,gint width,gint bytes)
{
	gint i,j;
	for (i=0;i<width;i++,buffer+=bytes)
		{
		gdouble alpha=buffer[bytes-1]*(1.0/255.0);
      for (j=0;j<bytes-1;j++) {buffer[j]=ROUND(buffer[j]*alpha);}
		}
}

gfloat contrast_lut_func(gint bytes,gint channel,gfloat value)
{
	gdouble slant,contrast;
	contrast=(gdouble)toyvals.contrast/127.0;
	if (toyvals.contrast==0) {return value;}
	slant=tan((contrast+1)*G_PI_4);
	value=(value-0.5)*slant+0.5;
	return value;
}

void process_pixel(GimpLut *lut,guchar *buffer,gint bytes)
{
	guchar *lut0=lut->luts[0];	
	switch (bytes)
		{
		case 1:
				*buffer=lut0[*buffer];
			break;
		case 2:
				buffer[0] = lut0[buffer[0]];
				buffer[1] = lut0[buffer[1]];
			break;
		case 3:
		case 4:
				buffer[0]=lut0[buffer[0]];
				buffer[1]=lut0[buffer[1]];
				buffer[2]=lut0[buffer[2]];
				saturation(buffer,bytes);
			break;
		}
}

void handle_layers_and_image(gboolean realimage)
{
	GimpDrawable *draw=tmpvals.drawable;
	gint width=tmpvals.preview_width;
	gint height=tmpvals.preview_height;
	GimpPixelRgn in;
	gint32 new_ID=-1,image_ID=-1,bg_ID=-1,drawable_ID=-1;
	
	 //gimp_drawable_mask_intersect (gint32 drawable_ID,&x,&y,&width,&height);
	if (!realimage)
		{
		if (tmpvals.tmpimg_id==-1)
			{
			tmpvals.radius_changed=TRUE;
			tmpvals.mask_changed=TRUE;
			tmpvals.contrastsaturation_changed=TRUE;
			tmpvals.tmpimg_id=gimp_image_new(width,height,gimp_image_base_type(gimp_drawable_get_image(draw->drawable_id)));
			new_ID=gimp_layer_new_from_drawable(draw->drawable_id,tmpvals.tmpimg_id);
			gimp_image_insert_layer_wrapper(tmpvals.tmpimg_id,new_ID,0,-1);
			gimp_layer_scale(new_ID,width,height,FALSE);
			if (gimp_layer_get_mask(new_ID)!=-1) {gimp_layer_remove_mask(new_ID,GIMP_MASK_DISCARD);}
			bg_ID=gimp_layer_copy(new_ID);
			gimp_image_insert_layer_wrapper(tmpvals.tmpimg_id,bg_ID,0,-1);
			gimp_image_lower_layer_to_bottom(tmpvals.tmpimg_id,bg_ID);
			tmpvals.background_id=bg_ID;
			gimp_layer_add_mask(new_ID,gimp_layer_create_mask(new_ID,GIMP_ADD_BLACK_MASK));
			tmpvals.preview_drawable=gimp_drawable_get(new_ID);
			// Copy source pic to buffer
			gimp_pixel_rgn_init(&in,tmpvals.preview_drawable,0,0,width,height,FALSE,FALSE);
			gimp_pixel_rgn_get_rect(&in,tmpvals.pic_buffer,0,0,width,height);
			}
		}	
	else
		{
		if (tmpvals.background_id==-1)
			{
			gchar* oldname;	
			drawable_ID=tmpvals.drawable->drawable_id;	
			image_ID=gimp_drawable_get_image(drawable_ID);
			bg_ID=gimp_layer_copy(drawable_ID);
			if (gimp_layer_get_mask(drawable_ID)!=-1) {gimp_layer_remove_mask(drawable_ID,GIMP_MASK_DISCARD);}
			gimp_layer_add_mask(drawable_ID,gimp_layer_create_mask(drawable_ID,GIMP_ADD_BLACK_MASK));
			gimp_image_insert_layer_wrapper(image_ID,bg_ID,0,-1);
			oldname=gimp_drawable_get_name(drawable_ID);
			gimp_drawable_set_name(drawable_ID,_("Blurring"));
			gimp_drawable_set_name(bg_ID,oldname);
			gimp_image_lower_layer_to_bottom(image_ID,bg_ID);
			tmpvals.background_id=bg_ID;
			}
		}
}

void toy()
{
	GimpLut *lut=NULL;
	GimpDrawable *draw=NULL;
	gint x1,y1,x2,y2,w=0,h=0,bytes;

	lut=gimp_lut_new();
	bytes=gimp_drawable_bpp(tmpvals.drawable->drawable_id);	
	gimp_lut_setup(lut,(GimpLutFunc)contrast_lut_func,bytes);
	
	if (tmpvals.preview!=NULL) 
		{
		x1=y1=0;
		w=x2=tmpvals.preview_width;
		h=y2=tmpvals.preview_height;
		if (!tmpvals.pic_buffer) {tmpvals.pic_buffer=g_new(guchar,w*h*bytes);}
		if (!tmpvals.preview_buffer) {tmpvals.preview_buffer=g_new(guchar,w*h*bytes);}
		handle_layers_and_image(0);
		draw=tmpvals.preview_drawable;
		}
	else 
		{
		draw=tmpvals.drawable;
		gimp_progress_init(_("Toy"));
		gimp_drawable_mask_bounds(draw->drawable_id,&x1,&y1,&x2,&y2);
		w=x2-x1;
		h=y2-y1;
		}

	gimp_tile_cache_ntiles(2*(draw->width/gimp_tile_width()+1));
   contrastsaturation_and_gauss(lut,draw,x1,y1,w,h);

    if (tmpvals.preview_buffer==NULL)
		{
		gimp_drawable_flush(draw);
		gimp_drawable_merge_shadow(draw->drawable_id,TRUE);
		gimp_drawable_update(draw->drawable_id,x1,y1,w,h);
  		}
	
	gimp_lut_free(lut);
}

void saturation(guchar *buffer,gint bytes)
{
	GimpRGB pixel;
	GimpHSL hsl;
	gdouble value,sat;
	 
	if (toyvals.saturation==0) {return;}
	sat=toyvals.saturation/100.0;
	pixel.r=buffer[0]/255.0;
	pixel.g=buffer[1]/255.0;
	pixel.b=buffer[2]/255.0;
	pixel.a=(bytes==4)?buffer[3]/255.0:1.0;
	gimp_rgb_to_hsl(&pixel,&hsl);
	value=(hsl.s*sat)+hsl.s;
	hsl.s=CLAMP(value,0.0,1.0);
	gimp_hsl_to_rgb(&hsl,&pixel);
	buffer[0]=CLAMP0255(ROUND(pixel.r*255.0));
	buffer[1]=CLAMP0255(ROUND(pixel.g*255.0));
	buffer[2]=CLAMP0255(ROUND(pixel.b*255.0));
}

void contrastsaturation_and_gauss(GimpLut *lut,GimpDrawable *drawable,gint x1,gint y1,gint width,gint height)
{
	GimpPixelRgn  src_rgn,dest_rgn;
	GimpRGB white={1.0,1.0,1.0,1.0},black={0.0,0.0,0.0,1.0};
	gboolean has_alpha;
	gint bytes=drawable->bpp;
	guchar *dest,*src,*sp_p,*sp_m,*buffer;
	gdouble blur,n_p[5],n_m[5],d_p[5],d_m[5],bd_p[5],bd_m[5];
	gdouble *val_p = NULL,*val_m = NULL;
	gdouble *vp,*vm;
	gint i,j,row,col,b,terms,maxupdate;
	gdouble progress,max_progress;
	gint initial_p[4],initial_m[4];
	gdouble std_dev;
	gboolean realimage;
	gint32 floating=0,tmp_ID=-1,mask_ID=-1,drawable_ID=-1;

	progress=0.0;
	realimage=(tmpvals.preview_buffer==NULL);
	has_alpha=gimp_drawable_has_alpha(drawable->drawable_id);
	drawable_ID=drawable->drawable_id;
	if (gimp_drawable_is_layer_mask(drawable_ID)) {return;}
	mask_ID=gimp_layer_get_mask(drawable_ID);	
	val_p=g_new(gdouble,MAX(width,height)*bytes);
	val_m=g_new(gdouble,MAX(width,height)*bytes);
	src=g_new(guchar,MAX(width,height)*bytes);
	dest=g_new(guchar,MAX(width,height)*bytes);
	
	gimp_pixel_rgn_init(&src_rgn,drawable,0,0,drawable->width,drawable->height,FALSE, FALSE);
	gimp_pixel_rgn_init(&dest_rgn,drawable,0,0,drawable->width,drawable->height,TRUE,TRUE);
	max_progress=3*width*height;
	
	// Scale blur radius for preview image if necessary
	blur=	(realimage)?fabs(toyvals.radius):fabs(toyvals.radius/tmpvals.pre2img);
	if (blur<2.0) {blur=2.0;}
	std_dev=sqrt(-(blur*blur)/(2*log(1.0/255.0)));
	find_iir_constants (n_p,n_m,d_p,d_m,bd_p,bd_m,std_dev);
	maxupdate=MAX(width,height)/16;
	if (maxupdate<5) {maxupdate=5;}
	
	//Apply contrast & saturation on picture
	if (realimage)
		{
		for (row=0;row<height;row++)
			{
			buffer=src;
			gimp_pixel_rgn_get_row(&src_rgn,src,x1,row+y1,width);
			if (has_alpha) {multiply_alpha(src,width,bytes);}
			for (col=0;col<width;col++)
				{
				process_pixel(lut,buffer,bytes);
				buffer+=bytes;
         	}
			if (has_alpha) {separate_alpha(src,width,bytes);}
			gimp_pixel_rgn_set_row(&dest_rgn,src,x1,row+y1,width);
			progress+=width;
			if ((row%maxupdate)==0) {gimp_progress_update(progress/max_progress);}
			}
		gimp_drawable_flush(drawable);
		gimp_drawable_merge_shadow(drawable_ID,TRUE);
		handle_layers_and_image(1);
		mask_ID=gimp_layer_get_mask(drawable_ID);
		}
	else 
		{
		// Preview	
		if (tmpvals.contrastsaturation_changed)
			{
			for (row=0;row<height;row++)
				{	
				buffer=src;
				memcpy(src,tmpvals.pic_buffer+row*width*bytes,width*bytes);
				if (has_alpha) {multiply_alpha(src,width,bytes);}
				for (col=0;col<width;col++)
					{
					process_pixel(lut,buffer,bytes);
					buffer+=bytes;
     	   		}
				if (has_alpha) {separate_alpha(src,width,bytes);}
				gimp_pixel_rgn_set_row(&dest_rgn,src,x1,row+y1,width);
				}
			gimp_drawable_flush(drawable);
			gimp_drawable_merge_shadow(drawable_ID,TRUE);
			if (tmpvals.csbuffer)
				{
				gimp_buffer_delete(tmpvals.csbuffer);
				tmpvals.csbuffer=NULL;
				}
			tmpvals.csbuffer=gimp_edit_named_copy(drawable_ID,"csbuffer");
			tmpvals.contrastsaturation_changed=FALSE;
			}
		}
	
	if  ((realimage)||((!realimage)&&(tmpvals.radius_changed)))
		{
		if (!realimage)
			{
			if (tmpvals.csbuffer)
				{
				floating=gimp_edit_named_paste(tmpvals.background_id,tmpvals.csbuffer,TRUE);
				if (floating) {gimp_floating_sel_anchor(floating);}						
				floating=gimp_edit_named_paste(drawable_ID,tmpvals.csbuffer,TRUE);
				if (floating) {gimp_floating_sel_anchor(floating);}
				}
			}
		for (col=0;col<width;col++)
			{
			memset(val_p,0,height*bytes*sizeof(gdouble));
			memset(val_m,0,height*bytes*sizeof(gdouble));
			gimp_pixel_rgn_get_col(&src_rgn,src,col+x1,y1,height);
			if (has_alpha) {multiply_alpha(src,height,bytes);}
			sp_p=buffer=src;
			sp_m=src+(height-1)*bytes;
			vp=val_p;
			vm=val_m+(height-1)*bytes;

			for (i=0;i<bytes;i++)
				{
				initial_p[i]=sp_p[i];
				initial_m[i]=sp_m[i];
				}

			for (row=0;row<height;row++)
				{
				gdouble *vpptr,*vmptr;
				terms=(row<4)?row:4;
				for (b=0;b<bytes;b++)
					{
					vpptr=vp+b;vmptr=vm+b;
					for (i=0;i<=terms;i++)
						{
						*vpptr+=n_p[i]*sp_p[(-i*bytes)+b]- d_p[i]*vp[(-i* bytes)+b];
						*vmptr+=n_m[i]*sp_m[(i*bytes)+b]-d_m[i]*vm[(i*bytes)+b];
						}
					for (j=i;j<=4;j++)
						{
						*vpptr+=(n_p[j]-bd_p[j])*initial_p[b];
						*vmptr+=(n_m[j]-bd_m[j])*initial_m[b];
						}
					}
				sp_p+=bytes;
				sp_m-=bytes;
				vp+=bytes;
				vm-=bytes;
				}

			transfer_pixels(val_p,val_m,dest,bytes,height);
			if (has_alpha) {separate_alpha(dest,height,bytes);}
			if (realimage)
				{
				gimp_pixel_rgn_set_col(&dest_rgn,dest,col+x1,y1,height);
				progress+=height;
				if ((col%maxupdate)==0) {gimp_progress_update(progress/max_progress);}
				}
			else
				{
				for (row=0;row<height;row++) {memcpy(tmpvals.preview_buffer+(row*width+col)*bytes,dest+row*bytes,bytes);}      
				}
			}
			
			gimp_pixel_rgn_init (&src_rgn,drawable,0,0,drawable->width,drawable->height,FALSE,TRUE);
			
			for (row=0;row<height;row++)
				{
				memset(val_p,0,width*bytes*sizeof (gdouble));
				memset(val_m,0,width*bytes*sizeof (gdouble));

				if (realimage)
					{
					gimp_pixel_rgn_get_row(&src_rgn,src,x1,row+y1,width);
         	   }
				else
					{
					memcpy(src,tmpvals.preview_buffer+row*width*bytes,width*bytes);
					}

				if (has_alpha) {multiply_alpha(src,width,bytes);}

				sp_p=src;
				sp_m=src+(width-1)*bytes;
				vp=val_p;
				vm=val_m+(width-1)*bytes;

				for (i=0;i<bytes;i++)
					{
					initial_p[i]=sp_p[i];
					initial_m[i]=sp_m[i];
					}

				for (col=0;col<width;col++)
					{
					gdouble *vpptr,*vmptr;
					terms=(col<4)?col:4;
					for (b=0;b<bytes;b++)
						{
						vpptr=vp+b;vmptr=vm+b;
						for (i=0;i<=terms;i++)
							{
							*vpptr+=n_p[i]*sp_p[(-i*bytes)+b]-d_p[i]*vp[(-i*bytes)+b];
							*vmptr+=n_m[i]*sp_m[(i*bytes)+b]-d_m[i]*vm[(i*bytes)+b];
							}
						for (j=i;j<=4;j++)
							{
							*vpptr+=(n_p[j]-bd_p[j])*initial_p[b];
							*vmptr+=(n_m[j]-bd_m[j])*initial_m[b];
							}
						}
						sp_p+=bytes;
						sp_m-=bytes;
						vp+=bytes;
						vm-=bytes;
					}

				transfer_pixels(val_p,val_m,dest,bytes,width);
				if (has_alpha) {separate_alpha(dest,width,bytes);}
				if (realimage)
					{
					gimp_pixel_rgn_set_row(&dest_rgn,dest,x1,row+y1,width);
					progress+=width;
					if ((row%maxupdate)==0) {gimp_progress_update(progress/max_progress);}
					}
				else
					{
					memcpy(tmpvals.preview_buffer+row*width*bytes,dest,width*bytes);
					}
				}
				
			if (!realimage)
				{
				gimp_pixel_rgn_set_rect(&dest_rgn,tmpvals.preview_buffer,0,0,drawable->width,drawable->height);
				gimp_drawable_flush(drawable);
				gimp_drawable_merge_shadow(drawable_ID,TRUE);
				tmpvals.radius_changed=FALSE;
				}	
			}

	if ((mask_ID!=-1)&&(realimage))
		{
		if (!oldgradient) {oldgradient=gimp_context_get_gradient();}
		if (!toygradient) 
			{
			toygradient=gimp_gradient_new(GRADIENTNAME);
			gimp_gradient_segment_set_left_color(toygradient,0,&black,100);
			gimp_gradient_segment_set_right_color(toygradient,0,&white,100);
			gimp_context_set_gradient(toygradient);	
			}
		gimp_edit_blend(mask_ID,GIMP_CUSTOM_MODE,GIMP_NORMAL_MODE,GIMP_GRADIENT_BILINEAR,100,0,GIMP_REPEAT_NONE,FALSE,FALSE,0,0.0,TRUE,toyvals.start[0]*width,toyvals.start[1]*height,toyvals.end[0]*width,toyvals.end[1]*height);	
		}
	else if ((mask_ID!=-1)&&(!realimage))
		{
		GimpDrawable *preview_drawable=NULL;
		if (!oldgradient) {oldgradient=gimp_context_get_gradient();}
		if (!toygradient) 
			{
			toygradient=gimp_gradient_new(GRADIENTNAME);
			gimp_gradient_segment_set_left_color(toygradient,0,&black,100);
			gimp_gradient_segment_set_right_color(toygradient,0,&white,100);
			gimp_context_set_gradient(toygradient);	
			}
		if (tmpvals.mask_changed) 
			{
			gimp_edit_blend(mask_ID,GIMP_CUSTOM_MODE,GIMP_NORMAL_MODE,GIMP_GRADIENT_BILINEAR,100,0,GIMP_REPEAT_NONE,FALSE,FALSE,0,0.0,TRUE,toyvals.start[0]*width,toyvals.start[1]*height,toyvals.end[0]*width,toyvals.end[1]*height);    
			tmpvals.mask_changed=FALSE;
			}
		if (tmpvals.alpbuffer)
			{
			gimp_buffer_delete(tmpvals.alpbuffer);
			tmpvals.alpbuffer=NULL;
			}
		tmpvals.alpbuffer=gimp_edit_named_copy_visible(tmpvals.tmpimg_id,"albuffer");
		if (tmpvals.alpbuffer)
			{
			tmp_ID=gimp_layer_copy(drawable_ID);
			gimp_image_insert_layer_wrapper(tmpvals.tmpimg_id,tmp_ID,0,0);
			floating=gimp_edit_named_paste(tmp_ID,tmpvals.alpbuffer,TRUE);
			if (floating) {gimp_floating_sel_anchor(floating);}
			gimp_layer_set_opacity(tmp_ID,0);
			preview_drawable=gimp_drawable_get(tmp_ID);
			gimp_pixel_rgn_init(&src_rgn,preview_drawable,0,0,preview_drawable->width,preview_drawable->height,FALSE, FALSE);
			gimp_pixel_rgn_get_rect(&src_rgn,tmpvals.preview_buffer,0,0,width,height);
			gimp_image_remove_layer(tmpvals.tmpimg_id,tmp_ID);
			}
		}
		
	g_free(val_p);
	g_free(val_m);
	g_free(src);
	g_free(dest);
}

void transfer_pixels(const gdouble *src1,const gdouble *src2,guchar *dest,gint bytes,gint width)
{
	gint b;
	gint bend=bytes*width;
	gdouble sum;

	for (b=0;b<bend;b++)
		{
		sum=*src1++ + *src2++;
		sum=CLAMP0255(sum);
		*dest++=(guchar)sum;
		}
}

void find_iir_constants (gdouble *n_p,gdouble *n_m,gdouble *d_p,gdouble *d_m,gdouble *bd_p,gdouble *bd_m,gdouble  std_dev)
{
	gint i;
	const gdouble div=sqrt(2*G_PI)*std_dev;
	const gdouble x0=-1.783/std_dev;
	const gdouble x1=-1.723/std_dev;
	const gdouble x2=0.6318/std_dev;
	const gdouble x3=1.997/std_dev;
	const gdouble x4=1.6803/div;
	const gdouble x5=3.735/div;
	const gdouble x6=-0.6803/div;
	const gdouble x7=-0.2598/div;
	gdouble sum_n_p,sum_n_m,sum_d;
	gdouble a,b;
    
	n_p[0]=x4+x6;
	n_p[1]=(exp(x1)*(x7*sin(x3)-(x6+2*x4)*cos(x3))+exp(x0)*(x5*sin(x2)-(2*x6+x4)*cos (x2)));
	n_p[2]=(2*exp(x0+x1)*((x4+x6)*cos(x3)*cos(x2)-x5*cos(x3)*sin(x2)-x7*cos(x2)*sin(x3))+x6*exp(2*x0)+x4*exp(2*x1));
	n_p[3]=(exp(x1+2*x0)*(x7*sin(x3)-x6*cos(x3))+exp(x0+2*x1)*(x5*sin(x2)-x4*cos(x2)));
	n_p[4]=0.0;
	d_p[0]=0.0;
	d_p[1]=-2*exp(x1)*cos(x3)-2*exp(x0)*cos(x2);
	d_p[2]=4*cos(x3)*cos(x2)*exp(x0+x1)+exp(2*x1)+exp(2*x0);
	d_p[3]=-2*cos(x2)*exp(x0+2*x1)-2*cos(x3)*exp(x1+2*x0);
	d_p[4]=exp(2*x0+2*x1);

	for (i=0;i<=4;i++) {d_m[i]=d_p[i];}

	n_m[0]=0.0;

	for (i=1;i<=4;i++) {n_m[i]=n_p[i]-d_p[i]*n_p[0];}

	sum_n_p=0.0;
	sum_n_m=0.0;
	sum_d=0.0;

	for (i=0;i<=4;i++)
		{
		sum_n_p+=n_p[i];
		sum_n_m+=n_m[i];
		sum_d+=d_p[i];
		}

	a=sum_n_p/(1.0+sum_d);
	b=sum_n_m/(1.0+sum_d);

	for (i=0;i<=4;i++)
		{
		bd_p[i]=d_p[i]*a;
		bd_m[i]=d_m[i]*b;
		}
}

gboolean gimp_image_insert_layer_wrapper(gint32 image_ID,gint32 layer_ID,gint32 parent_ID,gint position)
{
	GimpParam *return_vals;
	gint nreturn_vals;
	gboolean success=TRUE;
  
	if (!is27)
		{
		return_vals=gimp_run_procedure("gimp-image-add-layer",&nreturn_vals,GIMP_PDB_IMAGE, image_ID,GIMP_PDB_LAYER, layer_ID,GIMP_PDB_INT32, position,GIMP_PDB_END);
		}
	else 
		{	
		return_vals=gimp_run_procedure("gimp-image-insert-layer",&nreturn_vals,GIMP_PDB_IMAGE, image_ID,GIMP_PDB_LAYER, layer_ID,GIMP_PDB_LAYER, parent_ID,GIMP_PDB_INT32, position,GIMP_PDB_END);
		}
		
  success=return_vals[0].data.d_status==GIMP_PDB_SUCCESS;
  gimp_destroy_params(return_vals, nreturn_vals);
  return success;
}
