/* 
 * Copyright (C) 2005 Alexandre Heitor Schmidt <alexsmith@solis.coop.br>
 * (the "Author").
 * All Rights Reserved.
 *
 * This file is part of GIMP Image Subtraction Plug-In.
 * Foobar is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * GIMP Image Subtraction Plug-In is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GIMP Image Subtraction Plug-In; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 */

#include "config.h"

#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "main.h"
#include "interface.h"
#include "render.h"

#include "plugin-intl.h"


/*  Constants  */

#define PROCEDURE_NAME   "image_subtraction"

#define DATA_KEY_VALS    "image_subtraction"

#define PARASITE_KEY     "image-subtraction-options"


/*  Local function prototypes  */

static void   query (void);
static void   run   (const gchar      *name,
		             gint              nparams,
		             const GimpParam  *param,
		             gint             *nreturn_vals,
		             GimpParam       **return_vals);


/*  Local variables  */

const PlugInVals default_vals =
{
     0,
    -1,
    -1
};

static PlugInVals         vals;

GimpPlugInInfo PLUG_IN_INFO =
{
    NULL,  /* init_proc  */
    NULL,  /* quit_proc  */
    query, /* query_proc */
    run,   /* run_proc   */
};

MAIN ()

static void
query (void)
{
    gchar *help_path;
    gchar *help_uri;

    static GimpParamDef args[] =
    {
        { GIMP_PDB_INT32,    "run_mode",   "Interactive, non-interactive"    },
        { GIMP_PDB_IMAGE,    "image",      "Input image"                     },
        { GIMP_PDB_DRAWABLE, "drawable",   "Master drawable"                 },
        { GIMP_PDB_DRAWABLE, "drawable",   "Slave drawable"                  },
        { GIMP_PDB_INT32,    "threshold",  "Threshold"                       }
    };

    gimp_plugin_domain_register (PLUGIN_NAME, LOCALEDIR);

    help_path = g_build_filename (DATADIR, "help", NULL);
    help_uri = g_filename_to_uri (help_path, NULL, NULL);
    g_free (help_path);

    gimp_plugin_help_register ("http://www.solis.coop.br/",
                               help_uri);

    gimp_install_procedure (PROCEDURE_NAME,
			  "Subtracts one image/layer/region from another",
			  "Subtracts one image/layer/region from another, creating \
               a new image containing only the difference over a transparent \
               background.",
			  "Alexandre Heitor Schmidt <alexsmith@solis.coop.br>\
               Daniel Afonso Heisler <daniel@solis.coop.br>",
			  "Alexandre Heitor Schmidt <alexsmith@solis.coop.br>",
			  "2005",
			  N_("Image subtraction"),
			  "RGB*",
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (args), 0,
			  args, NULL);

    gimp_plugin_menu_register (PROCEDURE_NAME, "<Image>/Filters/Misc/");
}

static void
run (const gchar      *name,
     gint              n_params,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
    static GimpParam   values[1];
    GimpDrawable      *drawable;
    gint32             image_ID;
    GimpRunMode        run_mode;
    GimpPDBStatusType  status = GIMP_PDB_SUCCESS;

    *nreturn_vals = 1;
    *return_vals  = values;

    /*  Initialize i18n support  */
    bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
    textdomain (GETTEXT_PACKAGE);

    run_mode = param[0].data.d_int32;
    image_ID = param[1].data.d_int32;
    drawable = gimp_drawable_get (param[2].data.d_drawable);

    /*  Initialize with default values  */
    vals = default_vals;
    
    if (strcmp (name, PROCEDURE_NAME) == 0) {
        switch (run_mode) {
            case GIMP_RUN_NONINTERACTIVE:
                if (n_params != 8) {
                    status = GIMP_PDB_CALLING_ERROR;
                } else {
                    vals.threshold = param[3].data.d_int32;
                }
            break;
            case GIMP_RUN_INTERACTIVE:
                /*  Possibly retrieve data  */
                gimp_get_data (DATA_KEY_VALS,    &vals);

                if (! dialog (&vals))
                {
                    status = GIMP_PDB_CANCEL;
                }
            break;
            case GIMP_RUN_WITH_LAST_VALS:
                /*  Possibly retrieve data  */
                gimp_get_data (DATA_KEY_VALS, &vals);
            break;
            default:
            break;
        }
    } else {
        status = GIMP_PDB_CALLING_ERROR;
    }

    if (status == GIMP_PDB_SUCCESS)
    {
        render (&vals);

        if (run_mode != GIMP_RUN_NONINTERACTIVE) {
            gimp_displays_flush ();
        }

        if (run_mode == GIMP_RUN_INTERACTIVE) {
            gimp_set_data (DATA_KEY_VALS,    &vals,    sizeof (vals));
        }

        gimp_drawable_detach (drawable);
    }

    values[0].type = GIMP_PDB_STATUS;
    values[0].data.d_status = status;
}
