/* 
 * Copyright (C) 2005 Alexandre Heitor Schmidt <alexsmith@solis.coop.br>
 * (the "Author").
 * All Rights Reserved.
 *
 * This file is part of GIMP Image Subtraction Plug-In.
 * Foobar is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * GIMP Image Subtraction Plug-In is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GIMP Image Subtraction Plug-In; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 */

#include "config.h"

#include <gtk/gtk.h>
#include <glib/gprintf.h>

#include <libgimp/gimp.h>

#include "main.h"
#include "render.h"

#include "plugin-intl.h"

gboolean isSameColor(guint8 r1, guint8 g1, guint8 b1, 
                     guint8 r2, guint8 g2, guint8 b2,
                     guint8 threshold) {
                         
    gboolean retVal = FALSE;
                         
    /*
    if ( (ABS(r1-r2) <= (guint8)(255 * ((gfloat)threshold / 100))) &&
         (ABS(g1-g2) <= (guint8)(255 * ((gfloat)threshold / 100))) &&
         (ABS(b1-b2) <= (guint8)(255 * ((gfloat)threshold / 100))) ) {
        retVal = TRUE;
    }
    */

    if ( (ABS(r1-r2) +
          ABS(g1-g2) +
          ABS(b1-b2)) <= (guint16)((3 * 255) * ((gfloat)threshold / 100)) ) {
        retVal = TRUE;
    }
                         
    
    return retVal;
}

/*  Public functions  */

void
render (PlugInVals         *vals)
{
    // variables to create the new image
    gint32       new_image_id = -1;
    gint32       new_layer_id = -1;

    // auxiliary images
    gint32       new_image1;
    gint32       new_image2;
    
    // images width
    gint32       layer1_width = gimp_drawable_width(vals->drawable_id1);
    gint32       layer2_width = gimp_drawable_width(vals->drawable_id2);
    gint32       final_width;
    // images height
    gint32       layer1_height = gimp_drawable_height(vals->drawable_id1);
    gint32       layer2_height = gimp_drawable_height(vals->drawable_id2);
    gint32       final_height;

    // final image width will be the greatest width
    if ( layer1_width >= layer2_width ) {
        final_width = layer1_width;
    } else {
        final_width = layer2_width;
    }

    // final image height will be the greatest height
    if ( layer1_height >= layer2_height ) {
        final_height = layer1_height;
    } else {
        final_height = layer2_height;
    }

    // create two new copies of images 1 and 2 and
    // change their dimensions to the width and height
    // of the final image
    gint *layers;
    gint num_layers;
    new_image1 = gimp_image_duplicate(gimp_drawable_get_image(vals->drawable_id1));
    new_image2 = gimp_image_duplicate(gimp_drawable_get_image(vals->drawable_id2));
    
    gimp_image_resize(new_image1,final_width,final_height,0,0);
    layers = gimp_image_get_layers(new_image1,&num_layers);
    while ( num_layers > 0 ) {
        gimp_layer_resize_to_image_size(layers[num_layers-1]);
        num_layers--;
    }
    
    gimp_image_resize(new_image2,final_width,final_height,0,0);
    layers = gimp_image_get_layers(new_image2,&num_layers);
    while ( num_layers > 0 ) {
        gimp_layer_resize_to_image_size(layers[num_layers-1]);
        num_layers--;
    }
    
    // create the new image
    new_image_id = gimp_image_new (final_width, final_height, GIMP_RGB);
    
    // create the initially transparent layer
    new_layer_id = gimp_layer_new (new_image_id, "Background",
                                   final_width, final_height,
                                   GIMP_RGBA_IMAGE,
                                   100.0,
                                   GIMP_NORMAL_MODE);

    // add layer to the newly created image
    gimp_image_add_layer (new_image_id, new_layer_id, 0);

    // obtain the layer1 and layer2 drawables
    GimpDrawable *layer1 = gimp_drawable_get(gimp_image_get_active_drawable(new_image1));
    GimpDrawable *layer2 = gimp_drawable_get(gimp_image_get_active_drawable(new_image2));
    GimpDrawable *layer_final = gimp_drawable_get(new_layer_id);
    
    // declaration of the regions on which we will iterate
    GimpPixelRgn rgn_layer1;
    GimpPixelRgn rgn_layer2;
    GimpPixelRgn rgn_layer_final;

    // initialize regions
    gimp_pixel_rgn_init(&rgn_layer1,layer1,0,0,final_width,final_height,FALSE,FALSE);
    gimp_pixel_rgn_init(&rgn_layer2,layer2,0,0,final_width,final_height,FALSE,FALSE);
    gimp_pixel_rgn_init(&rgn_layer_final,layer_final,0,0,final_width,final_height,FALSE,FALSE);

    // row buffers for layers 1 and 2
    guchar *row_layer1;
    guchar *row_layer2;
    guchar *row_final;
    
    row_layer1 = g_new(guchar, final_width * layer1->bpp);
    row_layer2 = g_new(guchar, final_width * layer2->bpp);
    row_final = g_new(guchar, final_width * layer_final->bpp);

    gint row_num;
    gint col_num;
    gint col_num_layer1;
    gint col_num_layer2;
    
    // pixel RGB colors
    guint8 r1, g1, b1, r2, g2, b2;

    for ( row_num=0; row_num < final_height; row_num++ ) {
        gimp_pixel_rgn_get_row(&rgn_layer1,row_layer1,0,row_num,final_width);
        gimp_pixel_rgn_get_row(&rgn_layer2,row_layer2,0,row_num,final_width);
        
        col_num = 0;
        col_num_layer1 = 0;
        col_num_layer2 = 0;
        while ( col_num < final_width * layer_final->bpp ) {
            r1 = row_layer1[col_num_layer1];
            r2 = row_layer2[col_num_layer2];
            g1 = row_layer1[col_num_layer1+1];
            g2 = row_layer2[col_num_layer2+1];
            b1 = row_layer1[col_num_layer1+2];
            b2 = row_layer2[col_num_layer2+2];
            if ( isSameColor(r1,g1,b1,r2,g2,b2,vals->threshold) ) {
                row_final[col_num]   = 0;
                row_final[col_num+1] = 0;
                row_final[col_num+2] = 0;
                row_final[col_num+3] = 0;
            } else {
                row_final[col_num]   = row_layer1[col_num_layer1];
                row_final[col_num+1] = row_layer1[col_num_layer1+1];
                row_final[col_num+2] = row_layer1[col_num_layer1+2];
                row_final[col_num+3] = 255;
            }

            col_num += layer_final->bpp;
            col_num_layer1 += layer1->bpp;
            col_num_layer2 += layer2->bpp;
        }
        
        gimp_pixel_rgn_set_row(&rgn_layer_final,row_final,0,row_num,final_width);
        
    }

    gimp_drawable_flush(layer_final);

    g_free(row_layer1);
    g_free(row_layer2);
    g_free(row_final);
    
    gimp_image_delete(new_image1);
    gimp_image_delete(new_image2);
    
    gimp_display_new(new_image_id);

}
