#!/usr/bin/python
# -*- coding: utf8 -*-
"""IDE for Gimp
"""
__VERSION__=0.2

import os.path,os
import mimetypes

from localize import _

NAME          = _("Gimp IDE")
VERSION       = str(__VERSION__)
COPYRIGHT     = _(u"Copyright (C) 2009 Dmitry Shamov")
URL           = "demmsnt@gmail.com" # None
DESCRIPTION   = _("Simple plugins IDE for Gimp")


UI_file = os.path.join(os.path.dirname(os.path.abspath(__file__)),'gimpide.glade')

import gtk,gobject,gtk.glade
import pango
import subprocess


#Testing for GTKSourceView2
try:
    import gtksourceview2
    using_source_view = True
    print _("using gtksourceview")
except ImportError:
    using_source_view = False
    print _("not using gtksourceview")
    pass

from sbmgr import StatusBarManager
from dcfg import *
import executor
import plugb
DEFAULT_CID = "GIMP IDE Editor"


class GimpIde:
    """Main class for IDE"""
    filename = None
    language = None
    about_dialog = None
    def __init__(self):
        """Initialize Editor"""
        self.wTree = gtk.glade.XML(UI_file, "mainWindow")
        self.window = self.wTree.get_widget("mainWindow")
        self.text_view = self.wTree.get_widget("textview")
        self.scroll_view = self.wTree.get_widget("scrolledwindow")

        self.statusbar = self.wTree.get_widget("statusbar")
        self.statusbar_manager = StatusBarManager(DEFAULT_CID)
        self.statusbar_manager.set_statusbar(self.statusbar)
        self.statusbar_cid = self.statusbar.get_context_id(DEFAULT_CID)
        
        self.setupSourceView()
        buff = self.text_view.get_buffer()
        self.change_highlight = buff.create_tag("change_highlight", background_set="True",background=CONF_CHANGE_HIGHLIGHT)
        self.wTree.signal_autoconnect(self)
        self.on_new_file(None)

    def set_highlight(self):
        if using_source_view:
            if CONF_HIGHLIGHT_SYNTAX:
                buffer = self.text_view.get_buffer()
                if not self.language: 
                    self.language = 'None'
                #print "language:",self.language,dir(self.language)
                
                if self.language=='None':
                    #Bug workaround
                    self.language=None
                try:
                    buffer.set_language(self.language)
                except:
                    print "TODO",self.language,type(self.language)
                buffer.set_highlight_syntax(CONF_HIGHLIGHT_SYNTAX)

    def check_for_save(self):
        """Check for save changes"""
        ret = False
        buff = self.text_view.get_buffer()
        if buff.get_modified():
            # we need to prompt for save
            message = _("Do you want to save the changes you have made?")
            dialog = gtk.MessageDialog(self.window,
                                       gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                       gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO,
                                       message)
            dialog.set_title(_("Save?"))

            if dialog.run() == gtk.RESPONSE_NO: ret = False
            else: ret = True

            dialog.destroy()
        return ret

    def get_save_filename(self):
        filename = None
        chooser = gtk.FileChooserDialog(_("Save File..."), self.window,
                                        gtk.FILE_CHOOSER_ACTION_SAVE,
                                        (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                         gtk.STOCK_SAVE, gtk.RESPONSE_OK))

        response = chooser.run()
        if response == gtk.RESPONSE_OK: filename = chooser.get_filename()
        if response == gtk.RESPONSE_CANCEL: filename = None
        chooser.destroy()

        return filename

    def write_file(self, filename):
        # add Saving message to status bar and ensure GUI is current
        if filename:
            self.statusbar.push(self.statusbar_cid, _("Saving %s") % filename)
        else:
            self.statusbar.push(self.statusbar_cid, _("Saving %s") % self.filename)

        while gtk.events_pending(): gtk.main_iteration()

        try:
            # disable text view while getting contents of buffer
            buff = self.text_view.get_buffer()
            self.text_view.set_sensitive(False)
            text = buff.get_text(buff.get_start_iter(), buff.get_end_iter())
            self.text_view.set_sensitive(True)
            buff.set_modified(False)

            # set the contents of the file to the text from the buffer
            if filename: fout = open(filename, "w")
            else: fout = open(self.filename, "w")
            fout.write(text)
            fout.close()

            if filename: self.filename = filename
        except:
            # error writing file, show message to user
            self.error_message(_("Could not save file: %s") % filename)

        # clear saving status and restore default
        self.statusbar.pop(self.statusbar_cid)
        self.reset_default_status()

    def running_as_root(self):
        return (os.name == 'posix') and (os.geteuid() == 0)

    def reset_default_status(self):
        if self.filename:
            status = _("File: %s") % self.filename #os.path.basename(self.filename)
            self.window.set_title(_("%s | Gimp IDE") % os.path.basename(self.filename))
        else:
            status = _("File: (UNTITLED)")
            self.window.set_title(_("Untitled | Gimp IDE"))

        #check for root
        #if platform.system() is not "Windows":
        if self.running_as_root():
            old_title = self.window.get_title()
            new_title = old_title + _(" | RUNNING AS ROOT")
            self.window.set_title(new_title)

        self.statusbar.pop(self.statusbar_cid)
        self.statusbar.push(self.statusbar_cid, status)

        self.statusbar_manager.set_filename(self.filename)

        try:
            language_name = self.language.get_name()
        except:
            language_name = self.language
            
        self.statusbar_manager.set_language(language_name)

        buff = self.text_view.get_buffer()
        buff.remove_tag_by_name("change_highlight",buff.get_start_iter(),buff.get_end_iter())

        self.text_view.grab_focus()
    
    def on_new_file(self,menuitem,data=None):
        """new file start """
        buff = self.text_view.get_buffer()
        cycler = True
        while cycler:
            if self.check_for_save():
                fname = self.get_save_filename()
                if fname:
                    self.write_file(fname)
            else:
                cycler=False
        self.filename = None
        # Default script language is Python
        self.language = self.get_language_for_mime_type('text/x-python')
        self.set_highlight()
        buff.set_text("")
        buff.set_modified(False)
        self.reset_default_status()

    def on_save_file(self, menuitem, data=None):
        if self.filename == None:
            filename = self.get_save_filename()
            if filename: self.write_file(filename)
        else: self.write_file(None)

    def on_save_as(self, menuitem, data=None):
        filename = self.get_save_filename()
        if filename: self.write_file(filename)

    def on_plug_builder(self,menuitem,data=None):
        """Show builder dialog"""
        buff = self.text_view.get_buffer()
        text = buff.get_text(buff.get_start_iter(), buff.get_end_iter())
        plugb.PlugInBuilder(UI_file,text)
        
    def on_execute(self,menuitem,data=None):
        #Before execute we need to save file
        def check_saved_need():
            if not self.filename:
                return True
            buff = self.text_view.get_buffer()
            if buff.get_modified():
                return True
            return False
        def ask_for_save():
            message = _("For execute script you need save them. \n Do you want to save the changes you have made?")
            dialog = gtk.MessageDialog(self.window,
                                       gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                       gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO,
                                       message)
            dialog.set_title(_("Save?"))

            if dialog.run() == gtk.RESPONSE_NO: ret = False
            else: ret = True
            dialog.destroy()
            return ret
        execute = True
        while check_saved_need():
            if ask_for_save():
                self.on_save_file(None, data=None)
            else:
                execute = False
                break
        if execute:
            result = executor.execute_file(self.filename)
            if not result.success:
                message = _("Script error %s ") % result
                dialog = gtk.MessageDialog(self.window,
                                           gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                           gtk.MESSAGE_ERROR, gtk.BUTTONS_OK,
                                           message)
                dialog.set_title(_("Error?"))
                dialog.run()
                dialog.destroy()

    def load_file(self, filename):
        # add Loading message to status bar and ensure GUI is current
        self.statusbar.push(self.statusbar_cid, _("Loading %s") % filename)
        while gtk.events_pending(): gtk.main_iteration()

        if using_source_view: self.text_view.get_buffer().begin_not_undoable_action()
        try:
            # get the file contents
            fin = open(filename, "r")
            text = fin.read()
            fin.close()

            # disable the text view while loading the buffer with the text
            self.text_view.set_sensitive(False)
            buff = self.text_view.get_buffer()
            buff.set_text(text)
            # reenable text view and set everything up for the user
            buff.set_modified(False)
            self.text_view.set_sensitive(True)

            # now we can set the current filename since loading was a success
            self.filename = filename

        except:
            # error loading file, show message to user
            self.error_message (_("Could not open file: %s") % filename)

        # syntax highlighting
        if using_source_view:
            if CONF_HIGHLIGHT_SYNTAX:
                buffer = self.text_view.get_buffer()
                f = file(filename)
                path = os.path.dirname(filename)
                mime_type = mimetypes.guess_type(filename)[0]
                language = None

                if mime_type:
                    language = self.get_language_for_mime_type(mime_type)

                    if language: self.language = language.get_name()
                    if not language:
                        #self.error_message('No language found for mime type "%s"' % mime_type)
                        self.language = 'None'
                else:
                    self.error_message(_('Couldn\'t get mime type for file "%s"') % filename)

                buffer.set_language(language)


        buffer.set_highlight_syntax(CONF_HIGHLIGHT_SYNTAX)

        # move insertion point and scroll to top
        buff.place_cursor(buff.get_start_iter())

        if using_source_view: self.text_view.get_buffer().end_not_undoable_action()

        # clear loading status and restore default
        self.statusbar.pop(self.statusbar_cid)
        self.reset_default_status()

    def get_open_filename(self):
        filename = None
        chooser = gtk.FileChooserDialog(_("Open File..."), self.window,
                                        gtk.FILE_CHOOSER_ACTION_OPEN,
                                        (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                         gtk.STOCK_OPEN, gtk.RESPONSE_OK))

        response = chooser.run()
        if response == gtk.RESPONSE_OK: filename = chooser.get_filename()
        chooser.destroy()
        return filename


    def on_open(self, menuitem, data=None):
        if self.check_for_save(): self.on_save_menu_item_activate(None, None)

        filename = self.get_open_filename()
        if filename: self.load_file(filename)

    def error_message(self, message):
        """Show message about error to console and X"""
        # log to terminal window
        self.log(message)
        # create an error message dialog and display modally to the user
        dialog = gtk.MessageDialog(None,
                                   gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                   gtk.MESSAGE_ERROR, gtk.BUTTONS_OK, message)
        dialog.run()
        dialog.destroy()

    def log(self,string):
        """Show log into console"""
        try:
            instance_name = self.window.get_title()
        except:
            instance_name = _('Window title not preset')
        pid = '[%d]'%(os.getpid())

        print "%s %s %s"%(instance_name,pid,string)

    def setupSourceView(self):
        """Setup sourceView if preset"""
        if (using_source_view):
            CONF_SMART_HOME_END_TYPE_MAP = {'Disabled':gtksourceview2.SMART_HOME_END_DISABLED,'Before':gtksourceview2.SMART_HOME_END_BEFORE,'After':gtksourceview2.SMART_HOME_END_AFTER,'Always':gtksourceview2.SMART_HOME_END_ALWAYS}
            self.scroll_view.remove(self.text_view)
            self.text_view = gtksourceview2.View()
            self.text_view.set_auto_indent(CONF_AUTO_INDENT)
            self.text_view.set_highlight_current_line(CONF_HIGHLIGHT_CURRENT_LINE)
            self.text_view.set_show_line_numbers(CONF_SHOW_LINE_NUMBERS)
            self.text_view.set_show_right_margin(CONF_SHOW_RIGHT_MARGIN)
            self.text_view.set_insert_spaces_instead_of_tabs(CONF_SPACES_INSTEAD_OF_TABS)
            self.text_view.set_tab_width(CONF_TAB_WIDTH)
            self.text_view.set_indent_width(CONF_INDENT_WIDTH)
            self.text_view.set_right_margin_position(CONF_RIGHT_MARGIN_POSITION)
            self.text_view.set_smart_home_end(CONF_SMART_HOME_END_TYPE_MAP[CONF_SMART_HOME_END_TYPE[0]])
            self.text_view.set_indent_on_tab(CONF_INDENT_ON_TAB)
            self.text_view.set_buffer(gtksourceview2.Buffer())
            self.text_view.get_buffer().set_highlight_syntax(CONF_HIGHLIGHT_SYNTAX)
            self.text_view.get_buffer().set_highlight_matching_brackets(CONF_HIGHLIGHT_MATCHING_BRACKETS)
            self.text_view.get_buffer().set_max_undo_levels(CONF_MAX_UNDO_LEVELS)
            self.text_view.show()
            self.text_view.set_events(gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.POINTER_MOTION_HINT_MASK |             
                                    gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK)
            self.text_view.set_left_margin(2)
            self.text_view.set_right_margin(2)
            self.scroll_view.add(self.text_view)

            mgr = gtksourceview2.style_scheme_manager_get_default()
            style_scheme = mgr.get_scheme(CONF_STYLE_SCHEME[0])
            if style_scheme:
                self.text_view.get_buffer().set_style_scheme(style_scheme)

    def get_language_for_mime_type(self,mime):
        lang_manager = gtksourceview2.language_manager_get_default()
        lang_ids = lang_manager.get_language_ids()
        for i in lang_ids:
            lang = lang_manager.get_language(i)
            for m in lang.get_mime_types():
                if m == mime:
                    return lang
        return None

    def main(self):
        """main routine"""
        self.window.show_all()
        gtk.main()

    def on_window_destroy(self, widget, data=None):
        gtk.main_quit()
# edit menu routines

    def on_cut(self, menuitem, data=None):
        buff = self.text_view.get_buffer();
        buff.cut_clipboard (gtk.clipboard_get(), True);

    def on_copy(self, menuitem, data=None):
        buff = self.text_view.get_buffer();
        buff.copy_clipboard (gtk.clipboard_get());

    def on_paste(self, menuitem, data=None):
        buff = self.text_view.get_buffer();
        buff.paste_clipboard (gtk.clipboard_get(), None, True);

    def on_delete(self, menuitem, data=None):
        buff = self.text_view.get_buffer();
        gone = buff.get_text(buff.get_selection_bounds()[0],buff.get_selection_bounds()[1])
        buff.delete_selection (False, True);
#AboutDialog
    def on_about_dialog(self, menuitem, data=None):
            if self.about_dialog:
                self.about_dialog.present()
                return

            about_dialog = gtk.AboutDialog()
            about_dialog.set_transient_for(self.window)
            about_dialog.set_destroy_with_parent(True)
            about_dialog.set_name(NAME)
            about_dialog.set_version(VERSION)
            about_dialog.set_copyright(COPYRIGHT)
            about_dialog.set_website(URL)
            about_dialog.set_comments(DESCRIPTION)
            about_dialog.set_logo_icon_name(gtk.STOCK_EDIT)

            # callbacks for destroying the about dialog
            def close(dialog, response, editor):
                editor.about_dialog = None
                dialog.destroy()

            def delete_event(dialog, event, editor):
                editor.about_dialog = None
                return True

            about_dialog.connect("response", close, self)
            about_dialog.connect("delete-event", delete_event, self)

            self.about_dialog = about_dialog
            about_dialog.show()
        
if __name__ == "__main__":
    editor = GimpIde()
    editor.main()
    
