#!/usr/bin/env python

'''
GUI for generic browser/inspector app implemented using GtkBuilder.

This app is independent of whats inspected/displayed.
This app is data driven.
It displays data from a dictionary from an imported module.
A dictionary of views defines the nature of the data.
Different conceptual views of the same object model can be displayed.
A view is defined by a viewspec.
The caller imports modules that create viewspecs,
and puts the viewspecs in a dictionary passed to init the inspector.

GUI semantics:
User selects from a treeview to display documentation in a textview.
Only leaves can be selected.
User can filter the treeview with a search function.

For the Gimp PDB Inspector
Combines two prior versions:  "PDB Browser" and "Plugin Browser"
Shows additional attributes of procedures than prior version.
Has additional GUI improvements:
1) tree model for type/name instead of just list of names
2) other treeviews of the data (future)
3) minor clarification, standardization, simplification of the GUI
'''


# if run standalone, possibly independent of any programs supplying db's
if __name__ == "__main__":
  # since this file is in the package, but imports from the package
  # put the directory above the package in the path
  import sys, os
  sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
  print sys.path


import pygtk
pygtk.require("2.0")
import gtk
import glib

# our own sub modules, must be installed alongside the inspector
# These are independent of the db
from inspector import db_treemodel
from inspector import search


# Make fully qualified path so pygtk finds glade file.
# Get the glade file from same directory as this file (must be together)
import os.path
UI_FILENAME = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'inspector.glade')




class InspectorApp(object):   # new-style class inherits from object
  '''
  Generic GUI app to browse documentation, implemented using GTKBuilder.
  
  Intended to be generic, ie independent of a specific application.
  
  '''

  def __init__(self, dictofviews):
    # Note: use self for all variables accessed across class methods,
    # but not passed into a method, eg in a callback.
    
    # The dictofviews argument was created earlier by the app.
    self.dictofviews = dictofviews  # the dictionary of views that drives this app
    self.currentviewname = dictofviews.keys()[0]  # arbitrarily the first

    '''
    Build search/filter data.
    This app has one search string, not one per view.
    Must be done before callbacks from loading model?
    '''
    self.searchstring = search.FilterString()
    
    '''
    Keep several treemodels, and just swap them in and out of the treeview.
    Instead of: clear them and reload them.
    Data driven construction of the set of treemodels as specified by the dictofviews.
    Also populates and sorts the treemodels.
    Note: ignore any treestore model from glade, don't: model = builder.get_object("treestore1")
    Must be done before signals connected but after search_dict.
    '''
    self.models = db_treemodel.TreeModelDictionary(self.dictofviews)
    
    builder = gtk.Builder()
    builder.add_from_file(UI_FILENAME)
    # Connect some signals defined in the glade file.  Was auto_connect in libglade.
    # See callbacks below, named same as in glade file.
    builder.connect_signals(self)
    self.window = builder.get_object("window1")
    self.treeview = builder.get_object("treeview1") # treeview is used in callbacks
    
    '''
    Selection handling:
    Note a treeselection object is always part of a treeview: no need to create in glade
    Connect selection event to callback function.
    I don't think this is done by connect_signals() above, since the selection objects are not in glade?
    '''
    self.treeview.get_selection().connect("changed", self.on_selection_changed)
    # Connect pre-selection event to callback function that determines what can be selected
    self.treeview.get_selection().set_select_function(self.filter_select)

    textview = builder.get_object("textview1")
    self.buffer = textview.get_buffer() # buffer is used in callbacks
    # Initial text in the textview takes care of itself, either blank or 
    # the selection_changed callback sets it to "nothing selected".

    '''
    "Inspect" (list foo by bar) widget.
    Combobox liststore data and active row can be set in Glade, but we ignore.
    For data driven app, we create a model for the combobox here
    This makes the name of the view appear in the GUI, it should be meaningful.
    TBD Sorted, ordered how?
    '''
    viewtypesmodel = builder.get_object("liststore1")
    viewtypesmodel.clear() # discard rows from glade
    for key in self.dictofviews.keys():  # view name
      viewtypesmodel.append([key])
    combobox = builder.get_object("combobox1")
    combobox.set_active(0)
    

    '''
    Alternative: gtk filtered models, but needs work 
    # Filtered models, wrappers of the above
    self.foo_filtered = type_name_model.filter_new(root=None) # No virtual root
    # !!! Both views use the same searchfunc
    # !!! Can only set this once, see gtk docs
    self.foo.set_visible_func(search.search_func)
    '''
    
    # Set model of treeview to the treemodel of the current viewspec
    # TBD bug it is initially out of sync with the List by widget.
    self.treeview.set_model(self.models[self.currentviewname].treemodel)
    
    # Prepare for later feedback on time consuming operations
    self.busypointer = gtk.gdk.Cursor(gtk.gdk.WATCH)
    
    self.window.show()


  def main(self):
    gtk.main()  # event loop


  '''
  Callback functions
  '''
  
  def on_selection_changed(self, theSelection):
    '''
    Treeview widget, signal=selectionChanged on theSelection which is a gtk.TreeSelection 
    Action: display documentation for selected item in textview.
    Single selection is enforced by default.
    '''
    model, path = theSelection.get_selected()
    if path:
      # Alternative: column 0 drives selection
      # selected_value = model.get_value(path, 0) # column 0
      # Model must take the view value to a model value (what the view represents)
    
      # Alternative: column 1 (hidden?) drives selection (ie is the model value)
      selected_value = model.get_value(path, 1) # column 1
      assert selected_value # not empty string, else filter_select logic is wrong
      # print selected_value
      self.buffer.set_text(str(self.dictofviews[self.currentviewname].db[selected_value]))
    else: # no selection
      self.buffer.set_text("<Nothing selected>")  # Clear textview.
     
     
  def filter_select(self, path):
    '''
    Callback: return boolean indicating whether selection is allowed.
    Let user select only leaves.
    This 'filter' is distinct from the search filtering.
    Filters the user's clicks in the treeview,
    and ignores those clicks that are not in leaves.
    '''
    '''
    Note: tree levels without children are a complication.
    The definition of a leaf here is: row with non-empty second, hidden column.
    If search filtering takes out all leaf rows under a tree level row,
    that tree level row has no children.
    If we choose to display tree level rows without children,
    then definition of leaf is not: has no children.
    '''
    model = self.treeview.get_model() # !!! Get treeview's model, which varies in this app.
    return model.get_value(model.get_iter(path), 1) != ""
    # OLD return not model.iter_has_child(model.get_iter(path)) # no child means leaf
    
    
  def on_combobox1_changed(self, box):
    '''
    "List By" widget, signal=changed, callback function.
    Change treeview to model of user's choice from "List by" combobox.
    Note there are two separate gtk models here: the gtk.ListModel for the combobox, and the gtk.TreeModel for the gtk.Treeview.
    '''
    comboboxmodel = box.get_model() # !!! model for the combobox widget
    index = box.get_active()
    if index >= 0: # !!! It could be 0 for the first row, which is also False
      viewname = comboboxmodel[index][0]
      self.currentviewname = viewname
      
      # Rebuild new current view now,  user might have changed search string after treemodel was built.
      self.models[self.currentviewname].rebuild(self.searchstring.pattern)

      self.treeview.set_model(self.models[viewname].treemodel) # Point the treeview to another treemodel.
      
      # Selection and document might now be invalid (not present in the new view.)
      # But apparently the gtk.TreeSelection takes care of it, ie the selection_change callback is invoked.
      # It might be nicer for user: if the view changed but db is same, 
      # try to keep the selection if it is still in the new current view.
      
  
  def idle_cb(self):
    ''' 
    GTK idle callback (allows gtk main loop to change busy pointer/cursor).
    This is for search (user entered filter).
    '''
    # time consuming operations
    
    # Create filter dictionary
    count = self.models[self.currentviewname].viewspec.filterdict.filter(
      self.dictofviews[self.currentviewname].db, self.searchstring.getpattern())
    print "Count: ", count
    
    # Rebuild just the current view now, rebuild other treemodels when they are exposed.
    self.models[self.currentviewname].rebuild(self.searchstring.pattern)
    
    if not count:
      # Use the textview for feedback to say the filter is too strict.
      # Alternatively, another widget should show the count of matches.
      self.buffer.set_text("<Search found no match; relax the filter.>")
      
    # Adjust selection and the textview if selection has been filtered out
    # TBD if the selection is still unfiltered, don't change textview
    
    self.window.window.set_cursor(None) # Done, revert to previous cursor
    return False  # Removes callback from gtk scheduler


  def on_entry1_activate(self, textentry):
    '''
    "Filter by.." widget, signal=activated, callback function.
    Change treeview to filter model by user's entered text.
    
    Note: on_activated (by return key), not on_changed (for each char changed)
    Future: entry_completion widget for choices of previous searches
    '''
    # print "Text entry activated",textentry.get_text()
    self.searchstring.setstring(textentry.get_text())
    
    # Rest of actions are time consuming, performed in a callback
    self.window.window.set_cursor(self.busypointer) # !!! window.window is the gtk.gdk.Window, an X window
    glib.idle_add(self.idle_cb)
    
    
  def on_window1_destroy( self, object ):
    gtk.main_quit()   


  '''
  Utility functions.
  '''

  def default_textview(self):
    '''
    '''
    # Don't assume the treeview has something to select
    if not self.treeview.get_model().get_iter_first(): # if no first row
      self.buffer.set_text("<Select a menu item or name>")
    else:
      self.buffer.set_text("<Found 0 rows: relax the filter.>")
      

if __name__ == "__main__":
  # standalone app
  
  # ??
  import sys
  sys.path.append(os.path.dirname(os.path.abspath(__file__)))
  print sys.path
  
  app = InspectorApp()
  gtk.main()
  
'''
stuff for busy pointer (cursor)
def idle_cb(gtk_window):
    time.sleep(10)             # your time consuming operation here
    gtk_window.set_cursor(None)

 watch = gtk.gdk.Cursor(gtk.gdk.WATCH)
 gtk_window.set_cursor(watch)
 gobject.idle_add(idle_cb, gtk_window)
'''

