#!/usr/bin/env python

'''
inspect gimp data objects
'''

import gimp
import inspect  # The officially sanctioned way of inspecting (introspecting?) python objects
import inspector.db_treemodel

'''
Don't rely on internal types and names of attributes: use approved ways of introspection.
Get all the attributes (even methods) and test using getattr
and various tricks/heuristics.
'''

  
# TBD use context manager to hide try/excepts

# TBD evidently, the data members of base classes come with the super's dir?
# Is it a difference between dir and __dict__?


def datamembers2(aobject):
  doc = "Doc"
  for key, value in inspect.getmembers(aobject, inspect.isdatadescriptor):
    doc.append("," + key)
  return doc


def datamembers(aobject):
  '''
  Return string of public attributes that are data.
  '''
  if aobject is None:
    doc = "No data available to inspect."
    return doc
  
  doc = "Doc"
  for name in dir(aobject):
    if name.startswith('_'):  # private, not exposed
      continue
    try:
      value = getattr(aobject, name)
    except Exception:
      continue
    if inspect.isbuiltin(value):
      # Ad hoc, not approved: repr(value).startswith('<built-in method'):
      # print "Built-in Method:", name, value
      continue  
    # Not work? if inspect.ismethod(value):
    if repr(value).startswith('<method'):  # isinstance(value, type(document)):  # method
      # print "Method:", name, value
      continue
    doc = doc + "\n" + name + ":" + repr(value)
  return doc
    
    
    
def private(aobject):  
  # Private attributes
  for name in dir(aobject):
    if name.startswith('_'):  # private, not exposed
      print "Private", name


def document(aobject):
  '''
  This is specialized to the object, ie to a PyGimp object.
  For example, I know that a Layer has one base class Drawable
  '''
  aclass = aobject.__class__  # The class object of this object
  print "Class:", aclass
  datamembers(aobject)  # Attributes of object
  
  print "Base classes:", aclass.__bases__
  bases = aclass.__bases__
  if bases:
    abase = bases[0]
    datamembers(abase)  # Attributes of base object
    
  
    
  

  

def methods(aobject):      
  # Built-in methods
  try:
    builtins = inspect.getmembers(aobject, inspect.isbuiltin)
    methods = inspect.getmembers(aobject, inspect.ismethod)
    for name in builtins:
      print "Built-in method:", name
  except gimp.error:
    print "gimp.error passed in ismethod"
    pass


class Document(object):
  '''
  Wrap an object with an API for inspection.
  Mainly, having a repr that shows the data members.
  Also with a path attribute that shows has-a is-part-of relationships
  '''
  
  def __init__(self, name, path, aobject):
    self.name = name
    self.path = path
    self.baseobject = aobject # reference to object being documented
  
  def __repr__(self):
    ''' Document the data members'''
    return datamembers(self.baseobject)


def document_gimp_component(db, component, path):
  ''' Named component.
  Understands that components are in the namespace of their path.
  IE name must be qualified by path.
  '''
  name = path + component.name
  db[name] = Document(name, name, component)
  
  
  
def document_gimp_images(db):
  ''' Understands how to document image and its components.'''
  imagelist = gimp.image_list()
  if not imagelist:
    print "No images to inspect"
  for image in imagelist:
    name = image.name # not the filename, which itself is a path
    db[name] = Document(name, "Images/"+ name, image)
    if not image.layers:
      print name, "has no layers"
    for layer in image.layers:
      document_gimp_component(db, layer, "Images/"+name+"/Layers/")
    # TBD channels is empty???
    if not image.channels:
      print name, "has no channels"
    for channel in image.channels:
      document_gimp_component(db, channel, "Images/"+name+"/Channels/")
    # vectors is undocumented in PyGimp Reference 2010
    for vector in image.vectors:
      document_gimp_component(db, vector, "Images/"+name+"/Vectors/")

    


def document_gimp_gradients(db):
  ''' This understands that gradients have names but no other accessible data.'''
  for name in gimp.gradients_get_list():
    db[name] = Document(name, "Gradient/"+ name, None)

def document_gimp_toplevel_list(db, toplevellist, typename):
  ''' This understands that:
  1) gimp has lists of objects at top level
  2) the objects have names but no other pygimp accessible data.
  '''
  if not toplevellist:
    print "Empty ", typename
  for name in toplevellist:
    db[name] = Document(name, typename + name, None)    


def document_gimp_data(db):
  '''
  Load a dictionary with document objects for current Gimp data.
  This understands the top-level has-a relationships for Gimp data.
  IE images, gradients, etc.
  Replicates Windows>Dockable Dialogs for many things
  '''
  # This documents gimp data that pygimp exposes as objects with data members.
  document_gimp_images(db)
  
  # This lists (without documents) gimp data that pygimp does NOT expose.
  # gimp_foo_get_list() takes a filter string, returns a tuple of (count, list)
  document_gimp_toplevel_list(db, gimp.pdb.gimp_gradients_get_list("")[1] , "Gradients/")
  document_gimp_toplevel_list(db, gimp.pdb.gimp_buffers_get_list("")[1] , "Buffers/")
  document_gimp_toplevel_list(db, gimp.pdb.gimp_brushes_get_list("")[1] , "Brushes/")
  document_gimp_toplevel_list(db, gimp.pdb.gimp_fonts_get_list("")[1] , "Fonts/")
  # palettes, patterns
  
  # TBD parasites:  undocumented parasite_list()
  # TBD colormaps?
  # Displays?
  # TBD images and drawables have component parasite but no way to get them?

  
  
def test():
  
  # Data for images.
  # Exposes has-a (contains) relationships
  imagelist = gimp.image_list()
  if not imagelist:
    print "No images to inspect"
  for image in imagelist:
    print ">>>>>>Image", image.filename
    # print image
    # treemodel row
    document(image)
    layerlist = image.layers
    for layer in layerlist:
      # treemodel row
      # treemodel leaf
      # document is:
      # print all attributes and values
      # methods
      print ">>>>>>Layer"
      print layer
      document(layer)
      # TBD print drawable attributes
   
  # object attributes and methods
  # exposes is-a (inheritance) relationships
  print "Class Structure"
  print ">>>>>>>>>Image methods"
  methods(gimp.Image)
  print ">>>>>>>>>Layer methods"
  methods(gimp.Layer)
  # channel
  # drawable
   
'''
color_cube  contradicts documentation, not there
palette foreground and background

arguments of members
'''

'''
define views on db.
'''

proctypedict2 = {
  type(1) : "Int",
  type('str') : "Char",
  "Unknown" : "Unknown"
  }

'''
Test
# db of objects
mydb = {}
mydb["foo"] = "bar"
mydb["bar"] = 1
# filter dictionary
mydbfilterdict = inspector.search.TreeModelFilterDict(mydb)

# dictionary of views
dictofviews = {}  # Exported, the main product
    
# dictofviews["Menu"] = inspector.db_treemodel.ViewSpec("Menu", "menupath", "SlashPath", None, mypdb)

# Note all views of type Type need a mapping to define the set of values,
# but the mapping is not used for decoding at view loading time,
# but at db loading time, see below.
dictofviews["Test"] = inspector.db_treemodel.ViewSpec("Test", "__class__", "Type", proctypedict2, mydb, mydbfilterdict)
'''

mydb = {}
document_gimp_data(mydb)
mydbfilterdict = inspector.search.TreeModelFilterDict(mydb)
# dictionary of views
dictofviews = {}  # Exported, the main product
dictofviews["Gimp data"] = inspector.db_treemodel.ViewSpec("Gimp data", "path", "SlashPath", None, mydb, mydbfilterdict)      
  
