#!/usr/bin/env python

'''
Glue between Inspector app and pygimp pdb.

Reimplement pdb from pygimp:
  Make it iterable (a full dictionary)
  Add __repr__ method that documents a procedure
  Add extra attributes.

Also define views on the pdb, for use by the Inspector app.

The general API for glue objects between Inspector app and some source of documentation must include:
  dictofviews object a dictionary of viewspecs on a db object,
  dictionary of objects with attributes and repr method (referred to as the db.)
  a filter dictionary: boolean valued with same keys as the db
The viewspecs all refer to the same dictionary of objects 


Notes about the gimp.pdb object and its iterability:

On the PyGimp module: gimp
The gimp module is a Python extension written in C that gives access to the Gimp PDB,
(which give IPC or wire access to Gimp itself.)
The gimp module does not def a class for the PDB, but defines an object with name 'pdb'.
PyGimp documentation says "treat gimp.pdb object as a mapping" however it is not iterable,
meaning the iteration methods __iter__ and __next__ are not implemented by PyGimp.
gimp.pdb[foo] returns an object of type pdbFunc that has attributes and is callable.
!!! gimp.pdb.__members__ returns a list of Gimp PDB procedure names,
but __members__ is deprecated in py3k (Python version 3).

Here we define a class Pdb derived from Python built-in dict.
An object (usually a singleton) of type Pdb is an iterable dictionary,
returning values that are objects of type Procedure
which have the same attributes (or more) as in the Gimp PDB
but are not callable (although they easily could be.)

!!! Don't import gimpfu, which aliases pdb = gimp.pdb




Notes about the attributes of Gimp PDB procedures (there are more than are exposed.)

From the pygimp documentation: attributes of gimp.pdb procedure objects.
Note there are some errors here: type constants names are wrong and omit INTERNAL

proc_name
proc_blurb: A short piece of information about the procedure.
proc_help: More detailed information about the procedure.
proc_author: The author of the procedure.
proc_copyright: The copyright holder for the procedure (usually the same as the author).
proc_date: The date when the procedure was written.
proc_type: The type of procedure. This will be one of PROC_PLUG_IN, PROC_EXTENSION or PROC_TEMPORARY.
nparams: The number of parameters the procedure takes.
nreturn_vals: The number of return values the procedure gives.
params: A description of parameters of the procedure. It takes the form of a tuple of 3-tuples, where each 3-tuple describes a parameter. The items in the 3-tuple are a parameter type (one of the PARAM_* constants), a name for the parameter, and a description of the parameter.
return_vals: A description of the return values. It takes the same form as the params attribute.

But gimp_plugins_query() also returns attributes:
loc, accel, etc. (see below)

So there are several ways to get attributes:
1) look at the source (or call Unix strings() on binary) files
2) call gimp_plugins_query(), which only works for plugins of type 'Plugin' (and is undocumented except from the PDB Browser!)
3) call gimp_query_pdb
'''


import gimp
import gimpenums  # for proctypes
import types
import time
import inspector.db_treemodel


# Dictionaries of types in the conceptual model

'''
This used to map integer types to strings on loading from the PDB.
Programming surprise: some docs say constants are like PROC_PLUG_IN, others say GIMP_PLUG_IN
Constants defined by gimpenums are same as in libgimp but without prefix GIMP_ e.g. GIMP_EXTENSION -> EXTENSION
'''
proctypedict = {
  gimpenums.PLUGIN : "Plugin",
  gimpenums.EXTENSION : "Extension",
  gimpenums.TEMPORARY : "Temporary",
  gimpenums.INTERNAL : "Internal",
  -1 : "Unknown"   # Hope this doesn't conflict or change
  }

# mapping from name to more friendly
# Put in a viewspec and used in loading treemodel from db.
proctypedict2 = {
  "Plugin" : "Plugin",
  "Extension" : "Extension",
  "Temporary" : "Temporary",
  "Internal" : "Internal",
  "Unknown" : "Unknown"
  }

# mapping from name to more friendly
# Put in a viewspec and used in loading treemodel from db.
# Future: a way to show the hierarchy of image types
imagetypedict = { 
  "RGB" : "RGB",
  "RGBA" : "RGBA",
  "RGB*" : "RGB*",
  "GRAY" : "GRAY",
  "GRAYA" : "GRAYA",
  "GRAY*" : "GRAY*",
  "INDEXED" : "INDEXED",
  "INDEXEDA" : "INDEXEDA",
  "INDEXED*" : "INDEXED*",
  "" : "No image needed",  # more friendly displayed string
  "*" : "* (Any image type)",
  "NA" : "Not apply" # OR <Unknown> for INTERNAL procedures, etc.
  # For now, assume that if the imagetype is not accessible, it is not apply.
  # But I don't think it is true.
  # See elsewhere we set imagetypes to "Not apply"
  }
  # TBD more




class Procedure:
  '''
  Procedures in the Gimp PDB.
  Mimics the attributes exposed by the PDB.
  Unifies ALL the exposed attributes of any type of procedure.
  Even attributes that aren't readily available from Gimp.
  Implements repr for str()
  '''
  # TBD catch ValueError on decode ?
  
  # Note it is important to properly default those attributes that we build views on
  def __init__(self, name, accel, loc, time, menupath = "<Unknown>", imagetype="<Unknown>",
      blurb="", help="", author="", copyright="", date="", proctype=-1 ):
      
    # global proctypedict
    
    # attributes returned by gimp_plugin_query
    self.name = name
    self.menupath = menupath
    self.accel = accel
    self.loc = loc
    self.imagetype = imagetype
    self.time = time
    # attributes returned by gimp_procedural_db_proc_info
    self.blurb = blurb
    self.help = help
    self.author = author
    self.copyright = copyright
    self.date = date
    self.type = proctypedict[proctype]
    # other attributes that can be discerned, eg by inference or parsing source files
    self.filename = "Unknown"
    self.language = "Unknown"
    
  
  def __repr__(self):
    '''
    Return text describing procedure.
    
    Future: different formats for different types
    Future: formatted
    Future: highlight the search hits
    '''
    # print self.__dict__
    text = ""
    for attrname, attrvalue in self.__dict__.iteritems():
      if attrvalue is None: # Don't know why pygimp didn't do this earlier?
        attrvalue = ""  # Must be a string
      if not isinstance(attrvalue, types.StringType) :
        print "Non string value for attr:", attrname, "of:", self.name
        attrvalue = "***Non string error***"

      text = text + attrname + ': ' + attrvalue + "\n"
    return text

    
  def update(self, blurb, help, author, copyright, date, thetype):
    '''
    TBD generalize 
    '''
    self.blurb = blurb
    self.help = help
    self.author = author
    self.copyright = copyright
    self.date = date
    self.type = proctypedict[thetype]

    

class Pdb(dict):
  '''
  A read only dictionary of objects of type Procedure mimicing the Gimp PDB.
  For now, it initializes itself with data and you can't setitem.
  '''
  
  def __init__(self):
    
    # Base class init
    dict.__init__(self)
   
    # Fill self with data from Gimp PDB
    
    # Query the plugins, which have different attributes exposed.
    # Empty search string means get all.  Returns count, list, ...
    c1, menupath, c2, accel, c3, loc, c4, imagetype, c5, times, c6, name = gimp.pdb.gimp_plugins_query("")
    
    for i in range(0,len(name)):
      # Create new procedure object
      procedure = Procedure(name[i],  accel[i], loc[i],
        time.strftime("%c", time.localtime(times[i])),  # format time.  TBD convert to UTF8
        menupath[i], imagetype[i])
      dict.__setitem__(self, name[i], procedure)
    
    # Get list of all names in the PDB.
    # Deprecated gimp.pdb.__members__
    # This doesn't work: 'for key in gimp.pdb.keys()' since gimp.pdb is a crippled dictionary
    # Search using reg expr for: name, blurb, help, author, copyright, date, type
    (count, names) = gimp.pdb.gimp_procedural_db_query(".*", ".*", ".*", ".*", ".*", ".*",".*")
    print "Count procedures", len(names)
    for name in names:
      # print "Name", name
      # !!! Note nargs and nreturns are NOT returned via pygimp, ie 8 return values, not 10
      blurb, help, author, copyright, date, thetype, args, returns  = \
        gimp.pdb.gimp_procedural_db_proc_info(name)
      # If its already here, add the extra attributes
      if dict.has_key(self, name):
        procedure = dict.__getitem__(self, name)
        procedure.update(blurb, help, author, copyright, date, thetype)
      else:
        # Use the attributes exposed to us
        # !!! And set some of the others (any that we build a tree model from, eg menupath)
        # so that the tree models work.
        procedure = Procedure(name,  
          "", "", "", # accel, loc, time blank
          imagetype="NA",
          blurb=blurb, help=help, author=author, copyright=copyright, date=date, proctype=thetype)
        dict.__setitem__(self, name, procedure)
    # TBD parameters
    
  def __setitem__(self):
    raise RuntimeError, "Pdb does not allow adding procedures"
  
  # iterator methods, and all other special methods, inherited from base
  # No overriding is necessary.


'''
This is the meat of this glue module:
define views on an augmented PDB.
'''
mypdb = Pdb() # augmented pdb
pdbfilterdict = inspector.search.TreeModelFilterDict(mypdb)

dictofviews = {}  # Exported, the main product
    
dictofviews["Procedures by menu path"] = inspector.db_treemodel.ViewSpec("Procedures by menu path", "menupath", "SlashPath", None, mypdb, pdbfilterdict)

# Note all views of type Type need a mapping to define the set of values,
# but the mapping is not used for decoding at view loading time,
# but at db loading time, see below.
dictofviews["Procedures by type"] = inspector.db_treemodel.ViewSpec("Procedures by type", "type", "Type", proctypedict2, mypdb, pdbfilterdict)

# Uses a mapping for imagetypes to show more friendly displayed string
dictofviews["Procedures by image type"] = inspector.db_treemodel.ViewSpec("Procedures by image type", "imagetype", "Category", imagetypedict, mypdb, pdbfilterdict)



if __name__ == "__main__":
  # test
  # Currently, this doesn't work: abort at wire to gimp
  # That is, gimp must be running.
  import sys
  
  # gimp module is .so in /usr/lib/gimp/2.0/python
  # gimpfu.py in /usr/lib/gimp/2.0/plug-ins
  sys.path.append('/usr/lib/gimp/2.0/python')
  import gimp
  
  # test 
  mypdb = Pdb()
  
      
    
