/*
 * This is copyrighted software.  Originally uploaded at the GIMP Plugin
 * Registry (http://registry.gimp.org/).  This is to be distributed under the
 * terms of version 3 of the GNU General Public License (no other version).
 * For inquiries, email the author at stamit@stamit.gr .  You may not remove
 * this notice.
 */
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <stdio.h>
#include <math.h>
#include "ring.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

typedef enum InterpolateKernelType {
	KERNEL_CONSTANT,
	KERNEL_LINEAR,
	KERNEL_SQUARE,
	KERNEL_INVLINEAR,
	KERNEL_INVSQUARE,
	KERNEL_EXPONENTIAL,
	KERNEL_GAUSSIAN,
	KERNEL_INVNTH,
} InterpolateKernelType;

typedef struct InterpolateOpts {
	gint window;
	gfloat sigma;
	InterpolateKernelType kernel;
	gfloat lowthres;
	gfloat highthres;
	gboolean logthres;
	gboolean halfsel;
	gboolean multmask;

	gboolean calcderivs;

	gint linwindow;
	gboolean lincenter;
	gboolean linsymwin;
	gfloat linsigma;
	InterpolateKernelType linkernel;
} InterpolateOpts;


InterpolateOpts options = {
	64,
	4,
	KERNEL_INVNTH,
	0,
	0.01,
	FALSE,
	FALSE,
	FALSE,

	TRUE,  // calcderivs

	1,  //8,
	TRUE,
	FALSE,
	4.0,
	KERNEL_CONSTANT,  //KERNEL_INVNTH,
};

inline static gfloat gaussian(gfloat x, gfloat s) {
	return exp(-0.5*(x/s)*(x/s))/(s*sqrt(2*M_PI));
}

static gfloat *make_kernel(InterpolateKernelType type, gint window, gfloat s) {
	gint kernel_width = (1+2*window);
	gfloat *kernel = g_new(gfloat, kernel_width*kernel_width);
	for (gint y = -window ; y <= window ; ++y) {
		gfloat *frow = kernel + kernel_width*(window+y);
		for (gint x = -window ; x <= window ; ++x) {
			gfloat d = sqrt(x*x+y*y);
			switch (type) {
			case KERNEL_CONSTANT:
				frow[window+x] = 1.0;
				break;
			case KERNEL_LINEAR:
				frow[window+x] = MAX(0.0,1.0-d/s);
				break;
			case KERNEL_SQUARE:
				frow[window+x] = MAX(0.0,1.0-(d/s)*(d/s));
				break;
			case KERNEL_INVLINEAR:
				frow[window+x] = (d>0.0 ? 1.0/d : -1.0);
				break;
			case KERNEL_INVSQUARE:
				frow[window+x] = (d>0.0 ? 1.0/(d*d) : -1.0);
				break;
			case KERNEL_EXPONENTIAL:
				frow[window+x] = exp(-d/s);
				break;
			case KERNEL_GAUSSIAN:
				frow[window+x] = gaussian(d,s);
				break;
			case KERNEL_INVNTH:
				frow[window+x] = (d>0.0 ? pow(d,-s) : -1.0);
				break;
			}
		}
	}
	return kernel;
}

static void slopes(GimpDrawable *layer, GimpDrawable *mask, gint left, gint top, gint right, gint bottom, gboolean halfsel,
                   gint window, gfloat *kernel, gboolean recenter, gboolean symwin, gfloat **pdx, gfloat **pdy) {
	Ring ring;
	ring_init(&ring, layer, left, top, right, bottom, window);

	/* only white mask pixels will be considered */
	Ring mring;
	ring_init(&mring, mask, left, top, right, bottom, window);

	gint width = right-left;
	gint height = bottom-top;

	gfloat *dx = *pdx = g_new(gfloat, width*height*ring.bpp);
	gfloat *dy = *pdy = g_new(gfloat, width*height*ring.bpp);

	gimp_progress_init("Calculating slopes...");
	for (; ring_loop(&ring) && ring_loop(&mring) ; ring_rotate(&ring), ring_rotate(&mring)) {
		guchar *row = ring_row(&ring,ring.y);
		gfloat *dxrow = dx + ring.bpp*width*(ring.y-top);
		gfloat *dyrow = dy + ring.bpp*width*(ring.y-top);

		for (gint x = left ; x < right ; ++x) {
			//printf("--- SLOPE %d %d\n", x, ring.y);

			guchar *pixel = row + ( (x-left)*ring.bpp );

			gfloat wsum;

			gfloat centerx;
			gfloat centery;
			gfloat centerz[4];

			if (recenter) {
				wsum = 0.0;
				centerx = 0.0;
				centery = 0.0;
				centerz[0] = 0.0;
				centerz[1] = 0.0;
				centerz[2] = 0.0;
				centerz[3] = 0.0;

				gint y2_min = MAX(0,ring.y-window);
				gint y2_max = MIN(ring.y+window,layer->height-1);
				if (symwin) {
					y2_max = MIN(y2_max, ring.y+(ring.y-y2_min));
					y2_min = MAX(y2_min, ring.y-(y2_max-ring.y));
				}
				for (gint y2 = y2_min ; y2 <= y2_max ; ++y2) {
					guchar *row2 = ring_row(&ring,y2);
					guchar *mrow = ring_row(&mring,y2);
	
					gint x2_min = MAX(0,x-window);
					gint x2_max = MIN(x+window,layer->width-1);
					if (symwin) {
						x2_max = MIN(x2_max, x+(x-x2_min));
						x2_min = MAX(x2_min, x-(x2_max-x));
					}
					for (gint x2 = x2_min ; x2 <= x2_max ; ++x2) {
						guchar *mpixel = mrow + ( (x2-left)*mring.bpp );
						if (mpixel[0] && (mpixel[0]==255 || halfsel)) {
							gfloat w = kernel[(1+2*window)*(window+y2-ring.y) + (window+x2-x)];
							if (w>=0) {
								if (halfsel) {
									w *= mpixel[0];
								}
								wsum += w;
								centerx += w*(x2-x);
								centery += w*(y2-ring.y);

								guchar *pixel2 = row2 + ( (x2-left)*ring.bpp );
								centerz[0] += w*pixel2[0];
								if (ring.bpp>=3) {
									centerz[1] += w*pixel2[1];
									centerz[2] += w*pixel2[2];
									if (ring.bpp>=4) {
										centerz[3] += w*pixel2[3];
									}
								}
							}
						}
					}
				}

				if (wsum>0) {
					centerx /= wsum;
					centery /= wsum;
					centerz[0] /= wsum;
					if (ring.bpp>=3) {
						centerz[1] /= wsum;
						centerz[2] /= wsum;
						if (ring.bpp>=4) {
							centerz[3] /= wsum;
						}
					}
				} else {
					centerx = 0;
					centery = 0;
					centerz[0] = pixel[0];
					if (ring.bpp>=3) {
						centerz[1] = pixel[1];
						centerz[2] = pixel[2];
						if (ring.bpp>=4) {
							centerz[3] = pixel[3];
						}
					}
				}

				//printf("(%g;%g;%g) ", centerx, centery, wsum);

			} else {
				centerx = 0;
				centery = 0;
				centerz[0] = pixel[0];
				if (ring.bpp>=3) {
					centerz[1] = pixel[1];
					centerz[2] = pixel[2];
					if (ring.bpp>=4) {
						centerz[3] = pixel[3];
					}
				}
			}


			gfloat dxsum[4], dysum[4];
			wsum = 0.0;
			dxsum[0] = 0.0;
			dxsum[1] = 0.0;
			dxsum[2] = 0.0;
			dxsum[3] = 0.0;
			dysum[0] = 0.0;
			dysum[1] = 0.0;
			dysum[2] = 0.0;
			dysum[3] = 0.0;

			gint y2_min = MAX(0,ring.y-window);
			gint y2_max = MIN(ring.y+window,layer->height-1);
			if (symwin) {
				y2_max = MIN(y2_max, ring.y+(ring.y-y2_min));
				y2_min = MAX(y2_min, ring.y-(y2_max-ring.y));
			}
			for (gint y2 = y2_min ; y2 <= y2_max ; ++y2) {
				guchar *row2 = ring_row(&ring,y2);
				guchar *mrow = ring_row(&mring,y2);

				gint x2_min = MAX(0,x-window);
				gint x2_max = MIN(x+window,layer->width-1);
				if (symwin) {
					x2_max = MIN(x2_max, x+(x-x2_min));
					x2_min = MAX(x2_min, x-(x2_max-x));
				}
				for (gint x2 = x2_min ; x2 <= x2_max ; ++x2) {
					guchar *mpixel = mrow + ( (x2-left)*mring.bpp );
					gfloat deltax = x2-centerx-x;
					gfloat deltay = y2-centery-ring.y;
					gfloat distsquared = (deltax*deltax + deltay*deltay);
					if ( distsquared>0.0 && mpixel[0] && (mpixel[0]==255 || halfsel)) {
						gfloat w = kernel[(1+2*window)*(window+y2-ring.y) + (window+x2-x)];
						if (w>=0) {
							if (halfsel) {
								w *= mpixel[0];
							}
							//printf("IN %d %d %d\n", x2, y2, mpixel[0]);
							guchar *pixel2 = row2 + ( (x2-left)*ring.bpp );
							dxsum[0] += w*(pixel2[0]-centerz[0])*deltax/distsquared;
							if (ring.bpp>=3) {
								dxsum[1] += w*(pixel2[1]-centerz[1])*deltax/distsquared;
								dxsum[2] += w*(pixel2[2]-centerz[2])*deltax/distsquared;
								if (ring.bpp>=4) {
									dxsum[3] += w*(pixel2[3]-centerz[3])*deltax/distsquared;
								}
							}
							dysum[0] += w*(pixel2[0]-centerz[0])*deltay/distsquared;
							if (ring.bpp>=3) {
								dysum[1] += w*(pixel2[1]-centerz[1])*deltay/distsquared;
								dysum[2] += w*(pixel2[2]-centerz[2])*deltay/distsquared;
								if (ring.bpp>=4) {
									dysum[3] += w*(pixel2[3]-centerz[3])*deltay/distsquared;
								}
							}
							wsum += w;
						}
					}
				}
			}

			gfloat *dxpixel = dxrow + ring.bpp*(x-left);
			gfloat *dypixel = dyrow + ring.bpp*(x-left);
			if (wsum>0.0) {
				dxpixel[0] = 2*dxsum[0]/wsum;
				if (ring.bpp>=3) {
					dxpixel[1] = 2*dxsum[1]/wsum;
					dxpixel[2] = 2*dxsum[2]/wsum;
					if (ring.bpp>=4) {
						dxpixel[3] = 2*dxsum[3]/wsum;
					}
				}

				dypixel[0] = 2*dysum[0]/wsum;
				if (ring.bpp>=3) {
					dypixel[1] = 2*dysum[1]/wsum;
					dypixel[2] = 2*dysum[2]/wsum;
					if (ring.bpp>=4) {
						dypixel[3] = 2*dysum[3]/wsum;
					}
				}
			} else {
				dxpixel[0] = 0.0;
				if (ring.bpp>=3) {
					dxpixel[1] = 0.0;
					dxpixel[2] = 0.0;
					if (ring.bpp>=4) {
						dxpixel[3] = 0.0;
					}
				}

				dypixel[0] = 0.0;
				if (ring.bpp>=3) {
					dypixel[1] = 0.0;
					dypixel[2] = 0.0;
					if (ring.bpp>=4) {
						dypixel[3] = 0.0;
					}
				}
			}

			//printf("(%g;%g) ", dxpixel[0], dypixel[0]);
			//printf("(%g;%g)/%g ", dxsum[0], dysum[0], wsum);
		}
		//printf("\n");

		if (!(ring.y%4)) {
			gimp_progress_update(1.0*(ring.y-top+1)/(bottom-top));
		}
	}
	gimp_progress_end();
	ring_fini(&ring);
}

inline static gfloat wsum2sel(gfloat w, gfloat lt, gfloat ht, gboolean logthres) {
	if (logthres) w = log(w);

	gfloat ww;
	if (w<=lt) {
		ww = 0.0;
	} else if (w>=ht) {
		ww = 1.0;
	} else {
		ww = (w-lt)/(ht-lt);
	}
	return 1.0-ww;
}

static void interpolate(GimpDrawable *layer, GimpDrawable *sel, GimpDrawable *mask, InterpolateOpts *opts, gint image_id) {
	gint window = opts->window;
	gboolean calcderivs = opts->calcderivs;


	gint left,top,right,bottom,width,height;
	gimp_drawable_mask_bounds(layer->drawable_id,
	                          &left, &top, &right, &bottom);
	width = right-left;
	height = bottom-top;

	gint inleft = MAX(left-window,0);
	gint intop = MAX(top-window,0);
	gint inwidth = MIN(right+window,layer->width)-inleft;
	gint inheight = MIN(bottom+window,layer->height)-intop;


	/* calculate slopes of sample areas */
	gfloat *dx=0, *dy=0;
	if (calcderivs) {
		gfloat *linkernel = make_kernel(opts->linkernel,opts->linwindow,opts->linsigma);
		slopes(layer, mask, inleft, intop, inleft+inwidth, intop+inheight, opts->halfsel,
		       opts->linwindow, linkernel, opts->lincenter, opts->linsymwin, &dx, &dy);
		g_free(linkernel);
	}

	gint tw = gimp_tile_width();
	gimp_tile_cache_ntiles(2*(layer->width+tw-1)/tw);

	gint channels = gimp_drawable_bpp(layer->drawable_id);
	gint mchannels = gimp_drawable_bpp(mask->drawable_id);


	/* make kernel layer (brush?) */
	gint kernel_width = (1+2*window);
	gfloat *kernel = make_kernel(opts->kernel,window,opts->sigma);


	/* get initial input rows */
	GimpPixelRgn in;
	gimp_pixel_rgn_init(&in, layer, inleft, intop, inwidth, inheight, FALSE, FALSE);

	GimpPixelRgn min;
	gimp_pixel_rgn_init(&min, mask, inleft, intop, inwidth, inheight, FALSE, FALSE);

	gint in_ring = window;
	gint inrow_bytes = channels*inwidth;
	guchar *inrows = g_new(guchar, kernel_width*inrow_bytes);
	gint inmrow_bytes = mchannels*inwidth;
	guchar *inmrows = g_new(guchar, kernel_width*inmrow_bytes);
	for (gint y = MAX(top-window,0) ;
	     y <= MIN(top+window,layer->height-1) ; ++y) {
		gimp_pixel_rgn_get_row(&in,
			inrows + (in_ring+y-top)*inrow_bytes,
			inleft, y, inwidth
		);
		gimp_pixel_rgn_get_row(&min,
			inmrows + (in_ring+y-top)*inmrow_bytes,
			inleft, y, inwidth
		);
	}

	GimpPixelRgn ins;
	gimp_pixel_rgn_init(&ins, sel, left, top, width, height, FALSE, FALSE);
	guchar *insrow = g_new(guchar, mchannels*width);

	/* prepare output drawables */
	GimpPixelRgn out;
	gimp_pixel_rgn_init(&out, layer,
		left,
		top,
		right-left,
		bottom-top,
	TRUE, TRUE);
	guchar *outrow = g_new(guchar, channels*(right-left));

	GimpPixelRgn outs;
	gimp_pixel_rgn_init(&outs, sel,
		left,
		top,
		right-left,
		bottom-top,
	TRUE, TRUE);
	guchar *soutrow = g_new(guchar, mchannels*(right-left));

	/* do the interpolation */
	gimp_progress_init("Interpolating...");
	for (gint y = top ; y < bottom ; ++y) {
		guchar *inmrow = inmrows + ( (in_ring%kernel_width)*inmrow_bytes );
		gimp_pixel_rgn_get_row(&ins, insrow, left, y, width);

		for (gint x = left ; x < right ; ++x) {
			//printf("--- PIXEL %d %d\n", x, y);

			guchar *inmpixel = inmrow + (x-inleft)*mchannels;
			guchar *inspixel = insrow + (x-left)*mchannels;
			guchar *outpixel = outrow + (x-left)*channels;
			guchar *outspixel = soutrow + (x-left)*mchannels;

			gfloat sum[4];
			sum[0] = 0.0;
			sum[1] = 0.0;
			sum[2] = 0.0;
			sum[3] = 0.0;
			gfloat ws = 0.0;

			gint deltay_min = MAX(-window,-y);
			gint deltay_max = MIN(window,layer->height-1-y);
			for (gint deltay = deltay_min ; deltay <= deltay_max ; ++deltay) {
				guchar *inrow2 = inrows + ( ((in_ring+deltay)%kernel_width)*inrow_bytes );
				guchar *inmrow2 = inmrows + ( ((in_ring+deltay)%kernel_width)*inmrow_bytes );
				gfloat *dxrow;
				gfloat *dyrow;

				gint deltax_min = MAX(-window,-x);
				gint deltax_max = MIN(window,layer->width-1-x);

				if (calcderivs) {
					dxrow = dx + channels*inwidth*(y+deltay-intop);
					dyrow = dy + channels*inwidth*(y+deltay-intop);
				}

				for (gint deltax = deltax_min ; deltax <= deltax_max ; ++deltax) {
					guchar *inpixel2 = inrow2 + (x+deltax-inleft)*channels;
					guchar *inmpixel2 = inmrow2 + (x+deltax-inleft)*mchannels;

					gfloat w = kernel[kernel_width*(window+deltay) + window+deltax];
					if (w<0.0 && inmpixel2[0]>0) {
						ws = opts->highthres;
						sum[0] = opts->highthres*inpixel2[0];
						if (channels>=3) {
							sum[1] = opts->highthres*inpixel2[1];
							sum[2] = opts->highthres*inpixel2[2];
							if (channels>=4) {
								sum[3] = opts->highthres*inpixel2[3];
							}
						}
						goto breakout;
					} else if (opts->halfsel) {
						gfloat pw = w*inmpixel2[0];
						if (calcderivs) {
							gfloat *dxpixel = dxrow + (x+deltax-inleft)*channels;
							gfloat *dypixel = dyrow + (x+deltax-inleft)*channels;

							//printf("%d %d %g %g\n", x+deltax, y+deltay, dxpixel[0], dypixel[0]);
							sum[0] += (inpixel2[0] - dxpixel[0]*deltax - dypixel[0]*deltay)*pw;
							if (channels>=3) {
								sum[1] += (inpixel2[1] - dxpixel[1]*deltax - dypixel[1]*deltay)*pw;
								sum[2] += (inpixel2[2] - dxpixel[2]*deltax - dypixel[2]*deltay)*pw;
								if (channels>=4) {
									sum[3] += (inpixel2[3] - dxpixel[3]*deltax - dypixel[3]*deltay)*pw;
								}
							}
						} else {
							sum[0] += inpixel2[0]*pw;
							if (channels>=3) {
								sum[1] += inpixel2[1]*pw;
								sum[2] += inpixel2[2]*pw;
								if (channels>=4) {
									sum[3] += inpixel2[3]*pw;
								}
							}
						}
						ws += pw;
					} else if (inmpixel2[0]==255) {
						if (calcderivs) {
							gfloat *dxpixel = dxrow + (x+deltax-inleft)*channels;
							gfloat *dypixel = dyrow + (x+deltax-inleft)*channels;
							//printf("%d %d %g %g\n", x+deltax, y+deltay, dxpixel[0], dypixel[0]);
							sum[0] += (inpixel2[0] - dxpixel[0]*deltax - dypixel[0]*deltay)*w;
							if (channels>=3) {
								sum[1] += (inpixel2[1] - dxpixel[1]*deltax - dypixel[1]*deltay)*w;
								sum[2] += (inpixel2[2] - dxpixel[2]*deltax - dypixel[2]*deltay)*w;
								if (channels>=4) {
									sum[3] += (inpixel2[3] - dxpixel[3]*deltax - dypixel[3]*deltay)*w;
								}
							}
						} else {
							sum[0] += inpixel2[0]*w;
							if (channels>=3) {
								sum[1] += inpixel2[1]*w;
								sum[2] += inpixel2[2]*w;
								if (channels>=4) {
									sum[3] += inpixel2[3]*w;
								}
							}
						}
						ws += w;
					}
				}
			}

			breakout:
			if (ws>0.0) {
				outpixel[0] = CLAMP0255(sum[0]/ws);
				if (channels>=3) {
					outpixel[1] = CLAMP0255(sum[1]/ws);
					outpixel[2] = CLAMP0255(sum[2]/ws);
					if (channels>=4) {
						outpixel[3] = CLAMP0255(sum[3]/ws);
					}
				}
			} else {
				outpixel[0] = 127;
				if (channels>=3) {
					outpixel[1] = 127;
					outpixel[2] = 127;
					if (channels>=4) {
						outpixel[3] = 127;
					}
				}
			}

			// mask is multiplied with pixels without being normalized to (0.0,1.0), so divide here
			if (opts->halfsel) ws /= 255.0;

			gfloat outsel = wsum2sel(ws, opts->lowthres, opts->highthres, opts->logthres) * inspixel[0];
			if (opts->multmask) outsel *= inmpixel[0]/255.0;
			outspixel[0] = outsel;
		}

		gimp_pixel_rgn_set_row(&out,outrow,left,y,right-left);
		gimp_pixel_rgn_set_row(&outs,soutrow,left,y,right-left);

		/* rotate input row buffer */
		++in_ring;
		if ((y+1)+window < layer->height) {
			gimp_pixel_rgn_get_row(&in,
				inrows + ((in_ring+window)%kernel_width)*inrow_bytes,
				inleft, (y+1)+window, inwidth
			);
			gimp_pixel_rgn_get_row(&min,
				inmrows + ((in_ring+window)%kernel_width)*inmrow_bytes,
				inleft, (y+1)+window, inwidth
			);
		}

		if (!(y%4)) {
			gimp_progress_update(1.0*(y-top+1)/(bottom-top));
		}
	}
	gimp_progress_end();
	g_free(soutrow);
	g_free(outrow);

	g_free(inmrows);
	g_free(inrows);

	gimp_drawable_flush(layer);
	gimp_drawable_merge_shadow(layer->drawable_id, TRUE);
	gimp_drawable_update(layer->drawable_id, left, top,
	                     right-left, bottom-top);

	gimp_selection_none(image_id);

	gimp_drawable_flush(sel);
	gimp_drawable_merge_shadow(sel->drawable_id, TRUE);
	gimp_drawable_update(sel->drawable_id, left, top,
	                     right-left, bottom-top);

	g_free(kernel);

	g_free(dx);
	g_free(dy);
}

static gboolean interpolate_dialog(GimpDrawable *layer, InterpolateOpts *options) {
  gboolean   run;

  gimp_ui_init ("interpolation", FALSE);

  GtkWidget *dialog = gimp_dialog_new (
  	"Interpolation", "interpolation",
                            NULL, 0,
                            gimp_standard_help_func, "plug-in-interpolation",

                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK,     GTK_RESPONSE_OK,

                            NULL);

  GtkWidget *main_vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), main_vbox);
  gtk_widget_show (main_vbox);



  GtkWidget *window_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (window_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), window_hbox);

  GtkWidget *window_label = gtk_label_new_with_mnemonic ("_Window:");
  gtk_widget_show (window_label);
  gtk_box_pack_start (GTK_BOX (window_hbox), window_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (window_label), GTK_JUSTIFY_RIGHT);

  GtkObject *window_adj = gtk_adjustment_new (options->window, 0, 1024, 1, 8, 8);
  GtkWidget *window_spin = gtk_spin_button_new (GTK_ADJUSTMENT (window_adj), 1, 0);
  gtk_widget_show (window_spin);
  gtk_box_pack_start (GTK_BOX (window_hbox), window_spin, FALSE, FALSE, 6);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (window_spin), TRUE);

  gtk_widget_set_tooltip_text(window_spin,
  	"Number of nearby pixels to sample for each output pixel, in order to calculate a weighted average. "
	"1 means a 3x3 rectangle, 2 means a 5x5 rectangle, and so on. "
	"Only pixels with mask values equal to 255 are read. "
	"The first non-RGBA channel is used as the \"input mask\". "
	"If there is no such channel, the inverse of the selection is used."
  );



  GtkWidget *halfsel = gtk_check_button_new_with_mnemonic("Read half-sel.");
  gtk_widget_show (halfsel);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(halfsel), options->halfsel);
  gtk_box_pack_start (GTK_BOX (window_hbox), halfsel, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(halfsel,
  	"Whether to read from input pixels whose corresponding input mask pixels have values less than 255. "
  	"If there is no input mask (first channel), the inverse of the selection is used."
  );



  GtkWidget *kernel_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (kernel_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), kernel_hbox);

  GtkWidget *kernel_label = gtk_label_new_with_mnemonic ("_Kernel:");
  gtk_widget_show (kernel_label);
  gtk_box_pack_start (GTK_BOX (kernel_hbox), kernel_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (kernel_label), GTK_JUSTIFY_RIGHT);

  GtkWidget *kernel_combo = gtk_combo_box_new_text();
  gtk_widget_show (kernel_combo);
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Constant");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Linear");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Square");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Inverse Linear");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Inverse Square");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Exponential");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Gaussian");
  gtk_combo_box_append_text(GTK_COMBO_BOX(kernel_combo), "Inverse to the nth");
  gtk_combo_box_set_active(GTK_COMBO_BOX(kernel_combo), (int)options->kernel);
  gtk_box_pack_start (GTK_BOX (kernel_hbox), kernel_combo, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(kernel_combo,
  	"The kernel has the weights for the averaging of the nearby pixels. "
  	"The weights are functions of the distance D from the center pixel. "
  	"Constant generates a white square of width Window+1+Window. "
  	"Linear is 1-D/s (upside-down cone up to distance 's', then a flat 0). "
  	"Square is 1-(D/s)^2 (again up to distance 's'). "
  	"Inverse Linear is 1/D . Inverse Square is 1/(D^2) . "
  	"Exponential is e^(-D) and the result is similar to smudging from the edges of the mask. "
  	"Gaussian is the standard Gaussian kernel with standard deviation 'Sigma'. "
  	"Inverse to the nth is 1/(D^s)."
  );




  GtkWidget *sigma_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (sigma_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), sigma_hbox);

  GtkWidget *sigma_label = gtk_label_new_with_mnemonic ("_Sigma:");
  gtk_widget_show (sigma_label);
  gtk_box_pack_start (GTK_BOX (sigma_hbox), sigma_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (sigma_label), GTK_JUSTIFY_RIGHT);

  GtkObject *sigma_adj = gtk_adjustment_new (options->sigma, 0.0, 1024, 1, 8, 8);
  GtkWidget *sigma_spin = gtk_spin_button_new (GTK_ADJUSTMENT (sigma_adj), 1, 2);
  gtk_widget_show (sigma_spin);
  gtk_box_pack_start (GTK_BOX (sigma_hbox), sigma_spin, FALSE, FALSE, 6);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (sigma_spin), TRUE);

  gtk_widget_set_tooltip_text(sigma_spin,
	"If kernel is Linear or Square, Sigma is the radius after which the weights are zero. "
	"For Exponential it is a scale factor. For Gaussian it is the standard deviation."
	"For \"Inverse to the nth\" it is the exponent. "
	"Generally, the smaller the 'Sigma' the 'sharper' the result."
  );







  GtkWidget *lowthres_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (lowthres_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), lowthres_hbox);

  GtkWidget *lowthres_label = gtk_label_new_with_mnemonic ("_Low thres.:");
  gtk_widget_show (lowthres_label);
  gtk_box_pack_start (GTK_BOX (lowthres_hbox), lowthres_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (lowthres_label), GTK_JUSTIFY_RIGHT);

  GtkObject *lowthres_adj = gtk_adjustment_new (options->lowthres, -1073741824.0, 1073741824.0, 1, 9, 9);
  GtkWidget *lowthres_spin = gtk_spin_button_new (GTK_ADJUSTMENT (lowthres_adj), 1, 8);
  gtk_widget_show (lowthres_spin);
  gtk_box_pack_start (GTK_BOX (lowthres_hbox), lowthres_spin, FALSE, FALSE, 6);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (lowthres_spin), TRUE);

  gtk_widget_set_tooltip_text(lowthres_spin,
	"If the selection is modified, this is the weight-sum value that will give completely unselected pixels. "
	"This must be smaller than \"High thres.\". "
	"Larger values here and in \"High thres.\" shrink the selection more."
  );


  GtkWidget *highthres_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (highthres_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), highthres_hbox);

  GtkWidget *highthres_label = gtk_label_new_with_mnemonic ("_High thres.:");
  gtk_widget_show (highthres_label);
  gtk_box_pack_start (GTK_BOX (highthres_hbox), highthres_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (highthres_label), GTK_JUSTIFY_RIGHT);

  GtkObject *highthres_adj = gtk_adjustment_new (options->highthres, -1073741824.0, 1073741824.0, 1, 9, 9);
  GtkWidget *highthres_spin = gtk_spin_button_new (GTK_ADJUSTMENT (highthres_adj), 1, 8);
  gtk_widget_show (highthres_spin);
  gtk_box_pack_start (GTK_BOX (highthres_hbox), highthres_spin, FALSE, FALSE, 6);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (highthres_spin), TRUE);

  gtk_widget_set_tooltip_text(highthres_spin,
	"If the selection is modified, this is the weight-sum value that will give completely selected pixels. "
	"This must be larger than \"Low thres.\". "
	"Larger values here and in \"Low thres.\" shrink the selection more."
  );



  GtkWidget *logthres = gtk_check_button_new_with_mnemonic("Log scale new selection");
  gtk_widget_show (logthres);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(logthres), options->logthres);
  gtk_box_pack_start (GTK_BOX (main_vbox), logthres, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(logthres,
	"Use with Exponential and Gaussian kernels. "
	"By default there is a linear relation between the weight-sums and the new selection."
  );



  GtkWidget *multmask = gtk_check_button_new_with_mnemonic("Multiply with mask");
  gtk_widget_show (multmask);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(multmask), options->multmask);
  gtk_box_pack_start (GTK_BOX (main_vbox), multmask, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(multmask,
	"Whether to multiply new selection with mask. "
	"New selection is always multiplied with old selection."
  );




  GtkWidget *calcderivs_check = gtk_check_button_new_with_mnemonic("Calculate slopes");
  gtk_widget_show (calcderivs_check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(calcderivs_check), options->calcderivs);
  gtk_box_pack_start (GTK_BOX (main_vbox), calcderivs_check, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(calcderivs_check,
	"Whether to calculate a plane for each input pixel. "
	"This is how much the intensity near that point increases or decreases, horizontally and vertically. "
	"This changes the way that the pixel affects nearby interpolated pixels. "
  );




  GtkWidget *linwindow_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (linwindow_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), linwindow_hbox);

  GtkWidget *linwindow_label = gtk_label_new_with_mnemonic ("Lin. _Window:");
  gtk_widget_show (linwindow_label);
  gtk_box_pack_start (GTK_BOX (linwindow_hbox), linwindow_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (linwindow_label), GTK_JUSTIFY_RIGHT);

  GtkObject *linwindow_adj = gtk_adjustment_new (options->linwindow, 0, 1024, 1, 8, 8);
  GtkWidget *linwindow_spin = gtk_spin_button_new (GTK_ADJUSTMENT (linwindow_adj), 1, 0);
  gtk_widget_show (linwindow_spin);
  gtk_box_pack_start (GTK_BOX (linwindow_hbox), linwindow_spin, FALSE, FALSE, 6);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (linwindow_spin), TRUE);

  gtk_widget_set_tooltip_text(linwindow_spin,
	"The size of the window to use for calculating the horizontal and vertical angles, as above."
  );

  GtkWidget *linsymwin_check = gtk_check_button_new_with_mnemonic("Symm.win.");
  gtk_widget_show (linsymwin_check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(linsymwin_check), options->linsymwin);
  gtk_box_pack_start (GTK_BOX (linwindow_hbox), linsymwin_check, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(linsymwin_check,
	"Whether to decrease the width or the height of the window near the edges of the image, "
	"so that it is symmetric around the center pixel."
  );



  GtkWidget *lincenter_check = gtk_check_button_new_with_mnemonic("Re-center");
  gtk_widget_show (lincenter_check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lincenter_check), options->lincenter);
  gtk_box_pack_start (GTK_BOX (linwindow_hbox), lincenter_check, FALSE, FALSE, 6);




  GtkWidget *linkernel_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (linkernel_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), linkernel_hbox);

  GtkWidget *linkernel_label = gtk_label_new_with_mnemonic ("Lin. _Kernel:");
  gtk_widget_show (linkernel_label);
  gtk_box_pack_start (GTK_BOX (linkernel_hbox), linkernel_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (linkernel_label), GTK_JUSTIFY_RIGHT);

  GtkWidget *linkernel_combo = gtk_combo_box_new_text();
  gtk_widget_show (linkernel_combo);
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Constant");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Linear");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Square");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Inverse Linear");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Inverse Square");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Exponential");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Gaussian");
  gtk_combo_box_append_text(GTK_COMBO_BOX(linkernel_combo), "Inverse to the nth");
  gtk_combo_box_set_active(GTK_COMBO_BOX(linkernel_combo), (int)options->linkernel);
  gtk_box_pack_start (GTK_BOX (linkernel_hbox), linkernel_combo, FALSE, FALSE, 6);

  gtk_widget_set_tooltip_text(linkernel_combo,
	"The type of kernel, as above."
  );



  GtkWidget *linsigma_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (linsigma_hbox);
  gtk_container_add (GTK_CONTAINER (main_vbox), linsigma_hbox);

  GtkWidget *linsigma_label = gtk_label_new_with_mnemonic ("Lin. _Sigma:");
  gtk_widget_show (linsigma_label);
  gtk_box_pack_start (GTK_BOX (linsigma_hbox), linsigma_label, FALSE, FALSE, 6);
  gtk_label_set_justify (GTK_LABEL (linsigma_label), GTK_JUSTIFY_RIGHT);

  GtkObject *linsigma_adj = gtk_adjustment_new (options->linsigma, 0.0, 1024, 1, 8, 8);
  GtkWidget *linsigma_spin = gtk_spin_button_new (GTK_ADJUSTMENT (linsigma_adj), 1, 2);
  gtk_widget_show (linsigma_spin);
  gtk_box_pack_start (GTK_BOX (linsigma_hbox), linsigma_spin, FALSE, FALSE, 6);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (linsigma_spin), TRUE);

  gtk_widget_set_tooltip_text(linsigma_spin,
	"The parameter for the kernel, as above."
  );





  gtk_widget_show (dialog);

  run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);

  if (run) {
	  options->window = gtk_adjustment_get_value(GTK_ADJUSTMENT(window_adj));
	  options->sigma = gtk_adjustment_get_value(GTK_ADJUSTMENT(sigma_adj));
	  options->kernel = gtk_combo_box_get_active(GTK_COMBO_BOX(kernel_combo));
	  options->lowthres = gtk_adjustment_get_value(GTK_ADJUSTMENT(lowthres_adj));
	  options->highthres = gtk_adjustment_get_value(GTK_ADJUSTMENT(highthres_adj));
          options->logthres = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(logthres));
          options->halfsel = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(halfsel));
          options->multmask = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(multmask));

          options->calcderivs = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(calcderivs_check));
	  options->linwindow = gtk_adjustment_get_value(GTK_ADJUSTMENT(linwindow_adj));
          options->linsymwin = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(linsymwin_check));
          options->lincenter = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lincenter_check));
	  options->linsigma = gtk_adjustment_get_value(GTK_ADJUSTMENT(linsigma_adj));
	  options->linkernel = gtk_combo_box_get_active(GTK_COMBO_BOX(linkernel_combo));
  }

  gtk_widget_destroy (dialog);

  return run;
}

/*static void init() {
}

static void quit() {
}*/

static void query() {
	static GimpParamDef args[] = {
		{
			GIMP_PDB_INT32,
			"run-mode",
			"Run mode"
		},
		{
			GIMP_PDB_IMAGE,
			"image",
			"Input image"
		},
		{
			GIMP_PDB_DRAWABLE,
			"drawable",
			"Input drawable"
		},
		{
			GIMP_PDB_INT32,
			"window",
			"Window size"
		},
		{
			GIMP_PDB_FLOAT,
			"sigma",
			"Gaussian sigma"
		},
		{
			GIMP_PDB_INT32,
			"kernel",
			"Kernel type"
		},
		{
			GIMP_PDB_FLOAT,
			"lowthres",
			"Low sel. threshold"
		},
		{
			GIMP_PDB_FLOAT,
			"highthres",
			"High sel. threshold"
		},
		{
			GIMP_PDB_INT8,
			"logthres",
			"Log scale for new selection"
		},
		{
			GIMP_PDB_INT8,
			"halfsel",
			"Read from half-selected pixels"
		},
	};

	gimp_install_procedure(
		"plug-in-hello",
		"My filter",
		"Does things",
		"stamit@stamit.gr",
		"Copyright stamit.gr",
		"2010",
		"<Image>/Filters/_My filter...",
		"RGB*, GRAY*",
		GIMP_PLUGIN,
		G_N_ELEMENTS(args), 0,
		args, NULL
	);
}

static void run(const gchar *name, gint nparams, const GimpParam *param,
                gint *nreturn_vals, GimpParam **return_vals) {
	GimpPDBStatusType status = GIMP_PDB_SUCCESS;
	static GimpParam values[1];
	*nreturn_vals = 1;
	*return_vals = values;
	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = status;

	gint image_id = param[1].data.d_image;
	GimpDrawable *layer = gimp_drawable_get(param[2].data.d_drawable);

	GimpRunMode run_mode = param[0].data.d_int32;
	switch (run_mode) {
	case GIMP_RUN_WITH_LAST_VALS:
		gimp_get_data("plug-in-interpolate", &options);
		break;
	case GIMP_RUN_INTERACTIVE:
		gimp_get_data("plug-in-interpolate", &options);
		if (!interpolate_dialog(layer,&options)) {
			status = GIMP_PDB_CANCEL;
		}
		break;
	case GIMP_RUN_NONINTERACTIVE:
		if (nparams != 4)
			status = GIMP_PDB_CALLING_ERROR;
		if (status == GIMP_PDB_SUCCESS) {
			options.window = param[3].data.d_int32;
			options.sigma = param[4].data.d_float;
			options.kernel = param[5].data.d_int32;
			options.lowthres = param[6].data.d_float;
			options.highthres = param[7].data.d_float;
			options.halfsel = param[8].data.d_int8;
		}
		break;
	}

	if (status == GIMP_PDB_SUCCESS) {
		gimp_image_undo_group_start(image_id);

		/* get input mask (first channel found, or selection) */
		gint mask_id = 0;
		GimpDrawable *mask = 0;

		gint numchans;
		gint *chans = gimp_image_get_channels(image_id, &numchans);
		if (numchans>0) {
			// TODO search by name
			mask = gimp_drawable_get(chans[0]);
		} else {
			/* by default, read from all pixels that are not selected */
			gimp_selection_invert(image_id);
			mask_id = gimp_selection_save(image_id);
			gimp_selection_invert(image_id);
			mask = gimp_drawable_get(mask_id);
		}

		/* save selection to channel, so we can modify it */
		gint sel_id = gimp_selection_save(image_id);
		GimpDrawable *sel = gimp_drawable_get(sel_id);

		interpolate(layer, sel, mask, &options, image_id);

		gimp_drawable_detach(sel);
		gimp_selection_load(sel_id);
		gimp_image_remove_channel(image_id, sel_id);

		gimp_drawable_detach(mask);
		if (mask_id) {
			gimp_image_remove_channel(image_id, mask_id);
		}

		gimp_image_undo_group_end(image_id);
	}

	gimp_drawable_detach(layer);

	gimp_displays_flush();

	if (run_mode == GIMP_RUN_INTERACTIVE) {
		gimp_set_data("plug-in-interpolate", &options, sizeof(options));
	}
}

GimpPlugInInfo PLUG_IN_INFO = { NULL, NULL, query, run };

MAIN()
