#include <libgimp/gimp.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

static void query (void);
static void run   (const gchar      *name,
                   gint              nparams,
                   const GimpParam  *param,
                   gint             *nreturn_vals,
                   GimpParam       **return_vals);

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,
  NULL,
  query,
  run
};

MAIN()

static void
query (void)
{
  static GimpParamDef args[] =
  {
    { GIMP_PDB_INT32,	 "run-mode",	"Run mode" },
    { GIMP_PDB_IMAGE,    "image",  	"Input image" },
    { GIMP_PDB_DRAWABLE, "drawable",	"Input drawable" }
  };

  gimp_install_procedure (
    "plug-in-iterative-save",
    "Saves to a new filename \n with an increased numeric suffix",
    "Saves the image with an increased numeric suffix (if one exists), or adds it (if the current filename does not have one).",
    "Mircea Purdea",
    "nothing is not",
    "2009.09",
    "Iterative Save",
    "RGB*, GRAY*, INDEXED*",
    GIMP_PLUGIN,
    G_N_ELEMENTS (args), 0,
    args, NULL);

  gimp_plugin_menu_register ("plug-in-iterative-save",
                             "<Image>/File/Save");
}


static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam  values[1];
  GimpPDBStatusType status = GIMP_PDB_SUCCESS;
  GimpRunMode       run_mode;

  /* Setting mandatory output values */
  *nreturn_vals = 1;
  *return_vals  = values;

  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;
    
  gint32	drawableID = param[2].data.d_drawable;
  gint32	imageID = param[1].data.d_image;
  gchar		*filename = gimp_image_get_filename(imageID);
 
  if (filename == 0)
  {
   	values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;    
	if (run_mode != GIMP_RUN_NONINTERACTIVE)
		g_message("%s\n","Please save the file manually. This plugin will only work \nafter you specify an initial filename.");
    	return;
  }  
 
  gchar basename[FILENAME_MAX];
  strcpy(basename, g_filename_display_basename(filename));
  unsigned short int digits = 4;  /* default */
  const unsigned short int digitlimit = 9;
  const gboolean addmoredigitsifneeded = TRUE;
  const gboolean setcleanflag = FALSE;
  unsigned long tempint, number = 0;
  gchar *pchar=0;
  gint i;

  if (digits > digitlimit)
  {
   	values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;    
	if (run_mode != GIMP_RUN_NONINTERACTIVE)
		g_message("%s\n","The digit parameter exceeds the allowed limit.");
    	return;
  }

  gchar *lastdot = strrchr(basename, '.');
  if (! lastdot)
  {
  	values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
	if (run_mode != GIMP_RUN_NONINTERACTIVE)
	    g_message("%s\n","This plugin will not work on files with no explicit extension.");
	    return;
  }    
        
  /* extension */
  tempint = strlen(basename)-(lastdot-basename+1);
  gchar fileext[tempint];
  strncpy(fileext, lastdot+1, tempint);
  fileext[tempint]='\0';
  tempint = 13;
  if (strlen(fileext) == 3)
   	if ((tolower(fileext[0])=='x') && (tolower(fileext[1])=='c') && (tolower(fileext[2])=='f'))
   		tempint = 0;
  if (tempint != 0)
  {	
  	values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
	if (run_mode != GIMP_RUN_NONINTERACTIVE)
		g_message("%s\n","This plugin was designed to work only on XCF files.");
        return;
  }    

  /* separate the name from the extension*/
  tempint = lastdot-basename;
  gchar fname[tempint+digits+2];   /* reserve enough space for number */
  strncpy(fname, basename, tempint);
  fname[tempint]='\0';

  /* check for valid numeric suffix */
  number = 0;
  gint slen = strlen(fname);
  if (strrchr(fname, '_'))
  {
    tempint = 0; number = 1;
    for (i=slen; i>0; i--)
    {   /* parse the suffix */
        if (! isdigit(fname[i-1]))
        {
            if ((fname[i-1] != '_') || (tempint == 0))
                number = 0;
            break;
        }
        else
            ++tempint;
    }
  }
  if (number > 0)
  { /* get the actual number */
    digits = tempint;
    gchar nrstring[digits+1];
    strncpy(nrstring, fname+slen-digits, digits);
    nrstring[digits]='\0';
    sscanf(nrstring, "%lu", &number);

    /* check if number is within the proper range */
    i = log10(number+1) + 1;
    if (i > digits)
        if ((addmoredigitsifneeded) && (digits < digitlimit))
            ++digits;
        else
        {	
		 	values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;	            
			if (run_mode != GIMP_RUN_NONINTERACTIVE)
				g_message("%s\n","Increasing the number would require more digits than those available.");
			return;
		}  

    /* cut off the current number */
    pchar = strrchr(fname, '_');
    fname[pchar-fname]='\0';
    pchar = 0;

    /* append padding and the ++ number*/
    strcat(fname, "_");
    sprintf(nrstring, "%lu", number+1);
    for (i=0; i<(digits-strlen(nrstring)); i++)
        strcat(fname, "0");
    strcat(fname, nrstring);
  }
  else
  { /* append an initial suffix */
    strcat(fname, "_");
    for (i=0; i<digits; i++)
        strcat(fname, "0");
  }

  gchar finalname[FILENAME_MAX];
  strncpy(finalname, filename, strlen(filename)-strlen(basename)); 
  finalname[strlen(filename)-strlen(basename)]='\0';
  strcat(finalname, fname);
  strcat(finalname, ".");
  strcat(finalname, fileext);
  
  if (g_file_test(finalname, G_FILE_TEST_EXISTS))
  {	
 	values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;	            
	if (run_mode != GIMP_RUN_NONINTERACTIVE)
		g_message("File %s already exists.\nPlease perform a manual save if you would like to overwrite it.", finalname);
	return;
  }
  else
  {   	
	gimp_file_save(1, imageID, drawableID, finalname, finalname);
	gimp_image_set_filename(imageID, finalname);
	if (setcleanflag)
		gimp_image_clean_all(imageID);
  }
    
}

