#include "config.h"

#include <string.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "main.h"
#include "openjpeg.h"

#include "plugin-intl.h"

static void opj_image_to_gimp_image (opj_image_t * image,
				     GimpPixelRgn * rgn_in);
static int get_file_format (char *filename);
static void error_callback (const char *msg, void *client_data);
static void warning_callback (const char *msg, void *client_data);
static void info_callback (const char *msg, void *client_data);

gint32
load_jp2 (const gchar * filename, 
          GimpRunMode runmode, 
          gboolean preview)
{
  GimpPixelRgn rgn_in;
  GimpDrawable *drawable;
  gint32 volatile image_ID;
  gint32 layer_ID;
  gint x1, y1, x2, y2, width, height;

  /* openjpeg variables */
  opj_dparameters_t parameters;	/* decompression parameters */
  opj_event_mgr_t event_mgr;	/* event manager */
  opj_image_t *image = NULL;
  FILE *fsrc = NULL;
  unsigned char *src = NULL;
  int file_length;
  opj_dinfo_t *dinfo = NULL;	/* handle to a decompressor */
  opj_cio_t *cio = NULL;
  /* end of variables */

  /* configure the event callbacks (not required) */
  memset (&event_mgr, 0, sizeof (opj_event_mgr_t));
  event_mgr.error_handler = error_callback;
  event_mgr.warning_handler = warning_callback;
  event_mgr.info_handler = info_callback;

  /* set decoding parameters to default values */
  opj_set_default_decoder_parameters (&parameters);
  /* parse input and get user decoding parameters */

  /* read the input file and put it in memory */
  /* ---------------------------------------- */
  strncpy (parameters.infile, filename, MAX_PATH);
  parameters.decod_format = get_file_format (parameters.infile);
  fsrc = fopen (parameters.infile, "rb");
  if (!fsrc)
    {
      fprintf (stderr, "ERROR -> failed to open %s for reading\n",
	       parameters.infile);
      return 1;
    }
  fseek (fsrc, 0, SEEK_END);
  file_length = ftell (fsrc);
  fseek (fsrc, 0, SEEK_SET);
  src = (unsigned char *) malloc (file_length);
  fread (src, 1, file_length, fsrc);
  fclose (fsrc);

  /* decode the code-stream */
  /* ---------------------- */
  switch (parameters.decod_format)
    {
    case J2K_CFMT:
      {
	/* JPEG-2000 codestream */
	/* get a decoder handle */
	dinfo = opj_create_decompress (CODEC_J2K);

	/* catch events using our callbacks and give a local context */
	opj_set_event_mgr ((opj_common_ptr) dinfo, &event_mgr, stderr);

	/* setup the decoder decoding parameters using user parameters */
	opj_setup_decoder (dinfo, &parameters);

	/* open a byte stream */
	cio = opj_cio_open ((opj_common_ptr) dinfo, src, file_length);

	/* decode the stream and fill the image structure */
	image = opj_decode (dinfo, cio);
	if (!image)
	  {
	    fprintf (stderr,
		     "ERROR -> j2k_to_image: failed to decode image!\n");
	    opj_destroy_decompress (dinfo);
	    opj_cio_close (cio);
	    return 1;
	  }

	/* close the byte stream */
	opj_cio_close (cio);
      }
      break;
    case JP2_CFMT:

      {

	/* JPEG 2000 compressed image data */
	/* get a decoder handle */
	dinfo = opj_create_decompress (CODEC_JP2);

	/* catch events using our callbacks and give a local context */
	opj_set_event_mgr ((opj_common_ptr) dinfo, &event_mgr, stderr);

	/* setup the decoder decoding parameters using the current image and using user parameters */
	opj_setup_decoder (dinfo, &parameters);

	/* open a byte stream */
	// INIT_Il8N ();
	cio = opj_cio_open ((opj_common_ptr) dinfo, src, file_length);
	/* decode the stream and fill the image structure */
	image = opj_decode (dinfo, cio);
	if (!image)
	  {
	    fprintf (stderr,
		     "ERROR -> j2k_to_image: failed to decode image!\n");
	    opj_destroy_decompress (dinfo);
	    opj_cio_close (cio);
	    return 1;
	  }

	/* close the byte stream */
	opj_cio_close (cio);
      }
      break;
    case JPT_CFMT:

      {

	/* JPEG 2000, JPIP */

	/* get a decoder handle */
	dinfo = opj_create_decompress (CODEC_JPT);

	/* catch events using our callbacks and give a local context */
	opj_set_event_mgr ((opj_common_ptr) dinfo, &event_mgr, stderr);

	/* setup the decoder decoding parameters using user parameters */
	opj_setup_decoder (dinfo, &parameters);

	/* open a byte stream */
	cio = opj_cio_open ((opj_common_ptr) dinfo, src, file_length);

	/* decode the stream and fill the image structure */
	image = opj_decode (dinfo, cio);
	if (!image)
	  {
	    fprintf (stderr,
		     "ERROR -> j2k_to_image: failed to decode image!\n");
	    opj_destroy_decompress (dinfo);
	    opj_cio_close (cio);
	    return 1;
	  }

	/* close the byte stream */
	opj_cio_close (cio);
      }
      break;
    default:
      fprintf (stderr,
	       "ERROR -> j2k_to_image : Unknown input image format\n");
      return 1;
      break;
    }

  /* free the memory containing the code-stream */
  free (src);
  src = NULL;

  /* create output image */

  width = (gint) (image->x1);
  height = (gint) (image->y1);
  x1 = (gint) (image->x0);
  y1 = (gint) (image->y0);
  x2 = x1 + width;
  y2 = y1 + height;
  image_ID = gimp_image_new (width, height, GIMP_RGB);
  layer_ID = gimp_layer_new (image_ID, _("Background"),
			     width, height, GIMP_RGB, 100, GIMP_NORMAL_MODE);
  gimp_image_add_layer (image_ID, layer_ID, 0);
  drawable = gimp_drawable_get (layer_ID);	// initializes the drawable     
  gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);
  gimp_pixel_rgn_init (&rgn_in, drawable, x1, y1,
		       x2 - x1, y2 - y1, TRUE, TRUE);
  opj_image_to_gimp_image (image, &rgn_in);
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id, x1, y1, x2 - x1, y2 - y1);
  gimp_image_set_filename (image_ID, filename);
  return image_ID;
}

static void
opj_image_to_gimp_image (opj_image_t * image, GimpPixelRgn * rgn_in)
{
  int i, j;
  gint height, width, comps;
  guchar *buf;

  width = (gint) (image->x1);
  height = (gint) (image->y1);
  comps = (gint) (image->numcomps);
  buf = g_new (guchar, 3 * width * height);

  switch (comps)
    {
    case 1:
      for (i = 0; i < width; ++i)
	{
	  for (j = 0; j < height; ++j)
	    {
	      buf[0 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	      buf[1 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	      buf[2 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	    }
	}
      gimp_pixel_rgn_set_rect (rgn_in, buf, 0, 0, width, height);
      g_free (buf);
      break;

    case 2:
      break;

    case 3:
      for (i = 0; i < width; ++i)
	{
	  for (j = 0; j < height; ++j)
	    {
	      buf[0 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	      buf[1 + 3 * (j * width + i)] =
		(guchar) (image->comps[1].data[i + width * j]);
	      buf[2 + 3 * (j * width + i)] =
		(guchar) (image->comps[2].data[i + width * j]);
	    }
	}
      gimp_pixel_rgn_set_rect (rgn_in, buf, 0, 0, width, height);
      g_free (buf);
      break;
    case 4:
      break;
    case 5:
      break;
    case 6:
      break;
    }
}

int
get_file_format (char *filename)
{
  int i;
  static const char *extension[] =
    { "pgx", "pnm", "pgm", "ppm", "bmp", "j2k", "jp2", "jpt" };
  static const int format[] =
    { PGX_DFMT, PXM_DFMT, PXM_DFMT, PXM_DFMT, BMP_DFMT, J2K_CFMT, JP2_CFMT,
    JPT_CFMT
  };
  char *ext = strrchr (filename, '.') + 1;
  if (ext)
    {
      for (i = 0; i < sizeof (format); i++)
	{
	  if (strncmp (ext, extension[i], 3) == 0)
	    {
	      return format[i];
	    }
	}
    }
  return -1;
}

/**
 * sample error callback expecting a FILE* client object
 * */
void
error_callback (const char *msg, void *client_data)
{
  FILE *stream = (FILE *) client_data;
  fprintf (stream, "[ERROR] %s", msg);
}

/**
 * sample warning callback expecting a FILE* client object
 * */
void
warning_callback (const char *msg, void *client_data)
{
  FILE *stream = (FILE *) client_data;
  fprintf (stream, "[WARNING] %s", msg);
}

/**
 * sample debug callback expecting no client object
 * */
void
info_callback (const char *msg, void *client_data)
{
  fprintf (stdout, "[INFO] %s", msg);
}
