/* GIMP Plug-in Template
 * Copyright (C) 2000  Michael Natterer <mitch@gimp.org> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include "config.h"

#include <string.h>
#include <malloc.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "main.h"
#include "interface.h"
#include "render.h"

#include "openjpeg.h"

#include "plugin-intl.h"

/*  Constants  */
/* ----------------------------------------------------------------------- */
#define J2K_CFMT 0
#define JP2_CFMT 1
#define JPT_CFMT 2
#define MJ2_CFMT 3
#define PXM_DFMT 0
#define PGX_DFMT 1
#define BMP_DFMT 2
#define YUV_DFMT 3
/* ----------------------------------------------------------------------- */
#define PROCEDURE_NAME        "gimp_jp2_load"
#define PROCEDURE_NAME_SAVE   "gimp_jp2_save"

#define DATA_KEY_VALS    "plug_in_template"
#define DATA_KEY_UI_VALS "plug_in_template_ui"

#define PARASITE_KEY     "plug-in-template-options"


/*  Local function prototypes  */

static void query (void);
static void run (const gchar * name,
		 gint nparams,
		 const GimpParam * param,
		 gint * nreturn_vals, GimpParam ** return_vals);

static void INIT_Il8N ();

static void opj_image_to_gimp_image (opj_image_t * image,
				     GimpPixelRgn * rgn_in);
static int get_file_format (char *filename);
static void error_callback (const char *msg, void *client_data);
static void warning_callback (const char *msg, void *client_data);
static void info_callback (const char *msg, void *client_data);

/*  Local variables  */

const PlugInVals default_vals = {
  0,
  1,
  2,
  0,
  FALSE
};

const PlugInImageVals default_image_vals = {
  0
};

const PlugInDrawableVals default_drawable_vals = {
  0
};

const PlugInUIVals default_ui_vals = {
  TRUE
};

/* 
static PlugInVals         vals;
static PlugInImageVals    image_vals;
static PlugInDrawableVals drawable_vals;
static PlugInUIVals       ui_vals;
*/


GimpPlugInInfo PLUG_IN_INFO = {
  NULL,				/* init_proc  */
  NULL,				/* quit_proc  */
  query,			/* query_proc */
  run,				/* run_proc   */
};

MAIN ()
     static void query (void)
{
  gchar *help_path;
  gchar *help_uri;

  static GimpParamDef load_args[] = {
    {GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive"},
    {GIMP_PDB_STRING, "filename", "The name of the file to load"},
    {GIMP_PDB_STRING, "raw_filename", "The name of the file to load"},
  };

  static GimpParamDef load_return_vals[] = {
    {GIMP_PDB_IMAGE, "image", "Output image"}
  };
  static GimpParamDef save_args[] = {
    {GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive"},
    {GIMP_PDB_IMAGE, "image", "Input image"},
    {GIMP_PDB_DRAWABLE, "drawable", "Drawable to save"},
    {GIMP_PDB_STRING, "filename",
     "The name of the file to save the image in"},
    {GIMP_PDB_STRING, "raw_filename", "The name entered"},
  };



  gimp_plugin_domain_register (PLUGIN_NAME, LOCALEDIR);

  help_path = g_build_filename (DATADIR, "help", NULL);
  help_uri = g_filename_to_uri (help_path, NULL, NULL);
  g_free (help_path);

  gimp_plugin_help_register
    ("http://developer.gimp.org/plug-in-template/help", help_uri);

  gimp_install_procedure (PROCEDURE_NAME,
			  "loads files in the JPEG2000 file format",
			  "loads files in the JPEG2000 file format",
			  "Divyanshu Vats",
			  "Divyanshu Vats",
			  "2006-2010",
			  N_("JP2 image"),
			  NULL,
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (load_args),
			  G_N_ELEMENTS (load_return_vals),
			  load_args, load_return_vals);

  gimp_register_file_handler_mime (PROCEDURE_NAME, "image/jp2");
  gimp_register_magic_load_handler (PROCEDURE_NAME, "jp2,j2k", "", "");

  gimp_install_procedure (PROCEDURE_NAME_SAVE,
			  "saves files in the JP2 file format",
			  "saves files in the JP2 file format",
			  "Divyanshu Vats",
			  "Divyanshu Vats",
			  "2006-2010",
			  N_("JP2 image"),
			  "INDEXED, GRAY, RGB",
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (save_args), 0, save_args, NULL);

  gimp_register_file_handler_mime (PROCEDURE_NAME_SAVE, "image/jp2");
  gimp_register_save_handler (PROCEDURE_NAME_SAVE, "jp2", "");
}

static void
run (const gchar * name,
     gint n_params,
     const GimpParam * param, gint * nreturn_vals, GimpParam ** return_vals)
{
  static GimpParam values[1];
  GimpRunMode run_mode;
  GimpPDBStatusType status = GIMP_PDB_SUCCESS;

  gint32 image_ID;

  run_mode = param[0].data.d_int32;	// not sure what this does yet
  // was in the jpeg plug-in so
  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;

  if (strcmp (name, PROCEDURE_NAME) == 0)
    {
      INIT_Il8N ();
      image_ID = load_jp2 (param[1].data.d_string, run_mode, FALSE);

      if (image_ID != -1)
	{
	  *nreturn_vals = 2;
	  values[1].type = GIMP_PDB_IMAGE;
	  values[1].data.d_image = image_ID;
	}
      else
	{
	  printf ("did not open image");
	  status = GIMP_PDB_EXECUTION_ERROR;
	}
    }
  else if (strcmp (name, PROCEDURE_NAME_SAVE) == 0)
    {
      image_ID = param[1].dada.d_int32;
      drawable_ID = param[2].data.d_int32;
      status = write_jp2(param[3].data.d_string, image_ID, drawable_ID);
    }
  values[0].data.d_status = status;
  // gimp_displays_flush ();
}

static gint32
load_image (const gchar * filename, GimpRunMode runmode, gboolean preview)
{
  GimpPixelRgn rgn_in;
  GimpDrawable *drawable;
  gint32 volatile image_ID;
  gint32 layer_ID;
  gint x1, y1, x2, y2, width, height;
  // gint temp;

  /* openjpeg variables */
  opj_dparameters_t parameters;	/* decompression parameters */
  opj_event_mgr_t event_mgr;	/* event manager */
  opj_image_t *image = NULL;
  FILE *fsrc = NULL;
  unsigned char *src = NULL;
  int file_length;
  opj_dinfo_t *dinfo = NULL;	/* handle to a decompressor */
  opj_cio_t *cio = NULL;
  /* end of variables */

  /* configure the event callbacks (not required) */
  memset (&event_mgr, 0, sizeof (opj_event_mgr_t));
  event_mgr.error_handler = error_callback;
  event_mgr.warning_handler = warning_callback;
  event_mgr.info_handler = info_callback;

  /* set decoding parameters to default values */
  opj_set_default_decoder_parameters (&parameters);
  /* parse input and get user decoding parameters */

  /* read the input file and put it in memory */
  /* ---------------------------------------- */
  strncpy (parameters.infile, filename, MAX_PATH);
  parameters.decod_format = get_file_format (parameters.infile);
  fsrc = fopen (parameters.infile, "rb");
  printf ("Complete Stage 1\n");
  if (!fsrc)
    {
      fprintf (stderr, "ERROR -> failed to open %s for reading\n",
	       parameters.infile);
      return 1;
    }
  fseek (fsrc, 0, SEEK_END);
  file_length = ftell (fsrc);
  fseek (fsrc, 0, SEEK_SET);
  src = (unsigned char *) malloc (file_length);
  fread (src, 1, file_length, fsrc);
  fclose (fsrc);

  /* decode the code-stream */
  /* ---------------------- */
  printf ("Complete Stage 2\n");
  switch (parameters.decod_format)
    {
    case J2K_CFMT:
      {
	/* JPEG-2000 codestream */
	/* get a decoder handle */
	dinfo = opj_create_decompress (CODEC_J2K);

	/* catch events using our callbacks and give a local context */
	opj_set_event_mgr ((opj_common_ptr) dinfo, &event_mgr, stderr);

	/* setup the decoder decoding parameters using user parameters */
	opj_setup_decoder (dinfo, &parameters);

	/* open a byte stream */
	cio = opj_cio_open ((opj_common_ptr) dinfo, src, file_length);

	/* decode the stream and fill the image structure */
	image = opj_decode (dinfo, cio);
	if (!image)
	  {
	    fprintf (stderr,
		     "ERROR -> j2k_to_image: failed to decode image!\n");
	    opj_destroy_decompress (dinfo);
	    opj_cio_close (cio);
	    return 1;
	  }

	/* close the byte stream */
	opj_cio_close (cio);
      }
      break;
    case JP2_CFMT:

      {

	/* JPEG 2000 compressed image data */
	printf ("JP2 compression \n");
	/* get a decoder handle */
	dinfo = opj_create_decompress (CODEC_JP2);
	printf ("JP2 compression 1 \n");

	/* catch events using our callbacks and give a local context */
	opj_set_event_mgr ((opj_common_ptr) dinfo, &event_mgr, stderr);
	printf ("JP2 compression 2 \n");

	/* setup the decoder decoding parameters using the current image and using user parameters */
	opj_setup_decoder (dinfo, &parameters);
	printf ("JP2 compression 3 \n");

	/* open a byte stream */
	INIT_Il8N ();
	cio = opj_cio_open ((opj_common_ptr) dinfo, src, file_length);
	printf ("JP2 compression 4 \n");
	/* decode the stream and fill the image structure */
	image = opj_decode (dinfo, cio);
	printf ("JP2 compression 5 \n");
	if (!image)
	  {
	    fprintf (stderr,
		     "ERROR -> j2k_to_image: failed to decode image!\n");
	    opj_destroy_decompress (dinfo);
	    opj_cio_close (cio);
	    return 1;
	  }

	/* close the byte stream */
	opj_cio_close (cio);
      }
      break;
    case JPT_CFMT:

      {

	/* JPEG 2000, JPIP */

	/* get a decoder handle */
	dinfo = opj_create_decompress (CODEC_JPT);

	/* catch events using our callbacks and give a local context */
	opj_set_event_mgr ((opj_common_ptr) dinfo, &event_mgr, stderr);

	/* setup the decoder decoding parameters using user parameters */
	opj_setup_decoder (dinfo, &parameters);

	/* open a byte stream */
	cio = opj_cio_open ((opj_common_ptr) dinfo, src, file_length);

	/* decode the stream and fill the image structure */
	image = opj_decode (dinfo, cio);
	if (!image)
	  {
	    fprintf (stderr,
		     "ERROR -> j2k_to_image: failed to decode image!\n");
	    opj_destroy_decompress (dinfo);
	    opj_cio_close (cio);
	    return 1;
	  }

	/* close the byte stream */
	opj_cio_close (cio);
      }
      break;
    default:
      fprintf (stderr,
	       "ERROR -> j2k_to_image : Unknown input image format\n");
      return 1;
      break;
    }

  printf ("Complete Stage 3\n");
  /* free the memory containing the code-stream */
  free (src);
  src = NULL;

  /* create output image */

  width = (gint) (image->x1);
  height = (gint) (image->y1);
  x1 = (gint) (image->x0);
  y1 = (gint) (image->y0);
  x2 = x1 + width;
  y2 = y1 + height;
  image_ID = gimp_image_new (width, height, GIMP_RGB);
  layer_ID = gimp_layer_new (image_ID, _("Background"),
			     width, height, GIMP_RGB, 100, GIMP_NORMAL_MODE);
  gimp_image_add_layer (image_ID, layer_ID, 0);
  drawable = gimp_drawable_get (layer_ID);	// initializes the drawable     
  gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);
  gimp_pixel_rgn_init (&rgn_in, drawable, x1, y1,
		       x2 - x1, y2 - y1, TRUE, TRUE);
  opj_image_to_gimp_image (image, &rgn_in);
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id, x1, y1, x2 - x1, y2 - y1);
  gimp_image_set_filename (image_ID, filename);
  return image_ID;
}

static void
opj_image_to_gimp_image (opj_image_t * image, GimpPixelRgn * rgn_in)
{
  int i, j;
  gint height, width, comps;
  guchar *buf;

  width = (gint) (image->x1);
  height = (gint) (image->y1);
  comps = (gint) (image->numcomps);
  buf = g_new (guchar, 3 * width * height);

  switch (comps)
    {
    case 1:
      for (i = 0; i < width; ++i)
	{
	  for (j = 0; j < height; ++j)
	    {
	      buf[0 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	      buf[1 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	      buf[2 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	    }
	}
      gimp_pixel_rgn_set_rect (rgn_in, buf, 0, 0, width, height);
      g_free (buf);
      break;

    case 2:
      break;

    case 3:
      for (i = 0; i < width; ++i)
	{
	  for (j = 0; j < height; ++j)
	    {
	      buf[0 + 3 * (j * width + i)] =
		(guchar) (image->comps[0].data[i + width * j]);
	      buf[1 + 3 * (j * width + i)] =
		(guchar) (image->comps[1].data[i + width * j]);
	      buf[2 + 3 * (j * width + i)] =
		(guchar) (image->comps[2].data[i + width * j]);
	    }
	}
      gimp_pixel_rgn_set_rect (rgn_in, buf, 0, 0, width, height);
      g_free (buf);
      break;
    case 4:
      break;
    case 5:
      break;
    case 6:
      break;
    }
}

int
get_file_format (char *filename)
{
  int i;
  static const char *extension[] =
    { "pgx", "pnm", "pgm", "ppm", "bmp", "j2k", "jp2", "jpt" };
  static const int format[] =
    { PGX_DFMT, PXM_DFMT, PXM_DFMT, PXM_DFMT, BMP_DFMT, J2K_CFMT, JP2_CFMT,
    JPT_CFMT
  };
  char *ext = strrchr (filename, '.') + 1;
  if (ext)
    {
      for (i = 0; i < sizeof (format); i++)
	{
	  if (strncmp (ext, extension[i], 3) == 0)
	    {
	      return format[i];
	    }
	}
    }
  return -1;
}

void
INIT_Il8N ()
{
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
  textdomain (GETTEXT_PACKAGE);
}

/**
 * sample error callback expecting a FILE* client object
 * */
void
error_callback (const char *msg, void *client_data)
{
  FILE *stream = (FILE *) client_data;
  fprintf (stream, "[ERROR] %s", msg);
}

/**
 * sample warning callback expecting a FILE* client object
 * */
void
warning_callback (const char *msg, void *client_data)
{
  FILE *stream = (FILE *) client_data;
  fprintf (stream, "[WARNING] %s", msg);
}

/**
 * sample debug callback expecting no client object
 * */
void
info_callback (const char *msg, void *client_data)
{
  fprintf (stdout, "[INFO] %s", msg);
}
