/*_ Magie library					                _*/
/*_ Copyright (c) 2007, Ivan Bezdomniy              _*/
/*_ This program is free software. It may be distributed
 * and/or modified under the terms of the Lesser General
 * Public License. See the GNU Lesser General Public
 * License for more details */


/*_ Offline Image library	_*/


#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <stdint.h>

#include <stdio.h>

#ifdef	__OFFIMAGE_INTERNAL
#error	"Internal error, namespace conflict"
#else /*__OFFIMAGE_INTERNAL*/
#define	__OFFIMAGE_INTERNAL
#include "magie/offimage.h"
#undef	__OFFIMAGE_INTERNAL
#endif/*__OFFIMAGE_INTERNAL*/




int		offimage_init(
			OFFIMAGE*	obj,	/* Object size		*/
			int			sx,		/* x-size			*/
			int			sy,		/* y-size			*/
			int			su) {	/* bytes per pixel	*/

	int		padd;

	if (su != OFFI_USIZE) {
		return -1;
	}

	if (obj->data != NULL) {
		return -1;
	}

	obj->sx = 2*OFFI_BNDRY + sx;
	obj->sy	= 2*OFFI_BNDRY + sy;
	obj->su	= su;
	obj->sl	= obj->sx*su;
	padd = obj->sl % OFFI_ALIGN;
	obj->sl+= OFFI_ALIGN - padd;

	obj->data = malloc(obj->sl * obj->sy);

	if (!obj->data) {
		return -1;
	}

	return 0x0;
}


void	offimage_free(
			OFFIMAGE*	obj) {

	if (obj->data == NULL) return;

	free(obj->data);
	obj->data = NULL;
}



int		offimage_extent(
			OFFIMAGE*	obj,
			int			value,
			int			opcode) {

	if (opcode == OFFI_BNDRY_EXT) {
		int			p, t;
	
		/* Extend left/right cuts	*/
		for (p = 0; p < OFFI_LEN_COL(obj); p++) {
			uint16_t*	line;
			line = OFFI_PROW(obj,p);

			for (t = 0; t < OFFI_BNDRY; t++) {
				line[-t -1] = line[0];
				line[OFFI_LEN_ROW(obj) + t] = 
				line[OFFI_LEN_ROW(obj) - 1];
			}
		}

		/* Extend top/down lines 	*/
		for (p = 0; p < OFFI_BNDRY; p++) {
			void*	p_us = OFFI_PROW(obj, 0);
			void*	p_ud = OFFI_PROW(obj, -p -1);
			void*	p_ds = OFFI_PROW(obj, OFFI_LEN_COL(obj) - 1);
			void*	p_dd = OFFI_PROW(obj, OFFI_LEN_COL(obj) + p);

			memcpy(p_ud, p_us, OFFI_PROW_SIZE(obj));
			memcpy(p_dd, p_ds, OFFI_PROW_SIZE(obj));
		}

	} else
	if (opcode == OFFI_BNDRY_SET) {

	} else {
		/* Error, unsupported opcode	*/
	}

	return 0;
}



int		offimage_blur(
			OFFIMAGE*	obj,
			int			rx,		/* X blur radius	*/
			int			ry,		/* Y blur radius	*/
			OFFISTATE*	sco) {	/* State controls	*/

	struct FSM_Gauss	fsm_local;
	struct FSM_Gauss*	fsm = &fsm_local;
	struct State_Blur*	state;
	void*				line;
	void*				vert_local;
	void**				vert = & vert_local;
	int					len_cx;
	int					p;

	state = (struct State_Blur*) ((sco) ? &sco->state : NULL );

	if (sco) {
		/* Interrupt execution requested		*/
		/* Perform sanity checkups				*/
		/* Check state bytes capacity			*/
		if (OFFI_STATEBYTES < OFFI_SIZE_STBLUR)
			return OFFI_R_LACK_SUPP; else
		if (!obj) goto L_STAGE_FF;
		if(!((sco->holder == OFFI_HBLUR && sco->is_stored)||
			(sco->holder == OFFI_HNONE && !sco->is_stored))) {
			/* Invalid state object passed		*/
			return OFFI_R_INVALID_S;
		}

		/* Occupy state	control object			*/
		fsm			= &state->fsm;
		vert		= &state->vert;
		sco->unused	= sco->cycles;

		if (sco->holder == OFFI_HNONE) {
			sco->holder = OFFI_HBLUR;
			sco->is_finished= 0;
			sco->is_released= 0;
			goto L_STAGE_00;
		} else {
			if (sco->is_terminate) goto L_STAGE_05;
			if (sco->stage >= 0x05) {
				if (! sco->is_finished)	return OFFI_R_INVALID_S;
				if (! sco->is_released) return OFFI_R_INVALID_S;
				goto L_STAGE_FF;
			}

			if (sco->is_finished)	return OFFI_R_INVALID_S;
			if (sco->is_released)	return OFFI_R_INVALID_S;
			if (!sco->is_stored)	return OFFI_R_INVALID_S;
			/* Resume stages					*/
			if (sco->stage == 0x02) goto L_STAGE_02;
			if (sco->stage == 0x04) goto L_STAGE_04;
			/* Resume points invalid			*/
			return OFFI_R_INVALID_S;
		}
	}

#define	_CYC_STEP	((sco) ? sco->step : 0)

L_STAGE_00:	/*@@ Common init		*/
	SET_STAGE(0, -1);

	/* NULL Control object, return OK */
	if (!obj) goto L_STAGE_FF;

	/* Reset state object		*/
	fsm->SL = NULL; fsm->SR = NULL; *vert   = NULL;

	/* For FSM conveyer we should have an image with
	 * the boundaries allocated length of FSM_GAUSS_TM
	 * items on each side of an image. This boundaries
	 * are not carry any payload, true image itself
	 * (t.i. payload) resides inside these boundaries */

	if (obj->sx <= 2*_FSM_GAUSS_TM ||
		obj->sy <= 2*_FSM_GAUSS_TM) {
		goto L_STAGE_05;
	}
	
	/* The size of the conveyer state lines should be the
	 * size of an image full line. */
	len_cx = (obj->sx > obj->sy)? obj->sx : obj->sy;
	/* Vertical line with boundaries */
	if (sco) sco->is_stored = 1;
	fsm->SL = (double*)  malloc(len_cx*sizeof(double));
	fsm->SR = (double*)  malloc(len_cx*sizeof(double));
	*vert   = (void*)    malloc(obj->sy*obj->su);

	if (!fsm->SL || !fsm->SR || !vert) {
		goto L_STAGE_05;
	}

	goto L_STAGE_01;

L_STAGE_01:	/*@@ H-Pass init	*/
	SET_STAGE(1, 0);
	if (rx == 0) goto L_STAGE_03;

	fsm_gauss_set(fsm, rx);

L_STAGE_02:	/*@@ H-Pass 		*/
	SET_STAGE(2, -1);
	for (p = _CYC_STEP; p < OFFI_LEN_COL(obj); p++) {
		line = OFFI_FROW(obj, p);
		fsm_gauss_line(fsm, line, obj->sx, obj->su);

		SET_STEP_J(--sco->unused == 0, p + 1);
	}

L_STAGE_03:	/*@@ V-Pass init	*/
	SET_STAGE(3, 0);
	if (ry == 0) goto L_STAGE_05;
	if (rx != ry) fsm_gauss_set(fsm, ry);

L_STAGE_04:	/*@@ V-Pass			*/
	SET_STAGE(4, -1);
	for (p = _CYC_STEP; p < OFFI_LEN_ROW(obj); p++) {
		/* Get vertical line	*/
		offimage_col(obj, *vert, p, OFFI_GET|OFFI_COL_FULL);
		fsm_gauss_line(fsm, *vert, obj->sy, obj->su);
		/* Set vertical line	*/
		offimage_col(obj, *vert, p, OFFI_SET|OFFI_COL_FULL);

		SET_STEP_J(--sco->unused == 0, p + 1);
	}

L_STAGE_05:	/*@@ Release		*/
	SET_STAGE(5, -1);

	if (sco) sco->is_finished = 1;

	NULL_FREE(fsm->SL);
	NULL_FREE(fsm->SR);
	NULL_FREE(*vert);
	if (sco) sco->is_released = 1;

L_STAGE_FF:	/*@@ Exit			*/
	return 0;

#undef	_CYC_STEP
} 


void	offimage_row_p (
			OFFIMAGE*	obj,
			OFFI_PP		line,
			int			row,
			int			opcode) {

	*line.void_pp = NULL;

	if (row < -OFFI_BNDRY ||
		(row >= obj->sy - OFFI_BNDRY)) {
		/* Out of range  */
		return;
	}

	if (opcode & OFFI_ROW_FULL)
		*line.void_pp = OFFI_FROW(obj, row);
	else
		*line.void_pp = OFFI_PROW(obj, row);
}



void	offimage_col(
			OFFIMAGE*	obj,
			void*		line,
			int			col,	/* Column		*/
			int			opcode) {
	int			len;
	void*		point;
	
	/* Column number should be in range 
	 * [-OFFI_BNDRY to sx + OFFI_BNDRY -1]*/
	if (col < -OFFI_BNDRY ||
		col >= obj->sx - OFFI_BNDRY) {
		/* Out of range */
		return;
	}

	if (opcode & OFFI_COL_FULL) {
		/* Use full column data			*/
		len 	= obj->sy;
		point	= OFFI_PX(obj, col, -OFFI_BNDRY);
	} else {
		/* Use only payloaded column	*/
		len 	= OFFI_LEN_COL(obj);
		point	= OFFI_PX(obj, col, 0);
	}

	if (len == 0) return;

	if (OFFI_MOP(opcode) == OFFI_GET) { /* get column	*/
		while(len--) {
			*((uint16_t*)line) = *((uint16_t*)point);
			point += obj->sl;
			line  += sizeof(uint16_t);
		} ;
	} else
	if (OFFI_MOP(opcode) == OFFI_SET) { /* set column	*/
		while(len--) {
			*((uint16_t*)point) = *((uint16_t*)line);
			point += obj->sl;
			line  += sizeof(uint16_t);
		} ;
	}

}



/*_ Internal calls		_*/
void	fsm_gauss_set(
			struct FSM_Gauss*	fsm,
			double				radius) {
	double		x[8];
	double		div;
	double		dev;

	/*_ Calc FSM constants	*/
	radius += 1;

	dev = sqrt(radius*radius/(2*log(255.0)));
	div = sqrt(2*M_PI)*dev;

	x[0] = -1.7830 / dev;
	x[1] = -1.7230 / dev;
	x[2] = +0.6318 / dev;
	x[3] = +1.9970 / dev;
	x[4] = +1.6803 / div;
	x[5] = +3.7350 / div;
	x[6] = -0.6803 / div;
	x[7] = -0.2598 / div;
	
	/* Mean constant M0				*/
	fsm->M0 = x[4] + x[6];
	/* Y-constants Y1, Y2, Y3, Y4	*/
	fsm->Y1 = -2*( exp(x[1])*cos(x[3]) + exp(x[0])*cos(x[2]) );
	fsm->Y2 = 4*exp(x[0] + x[1]) * cos(x[3]) * cos(x[2]);
	fsm->Y2+= exp( x[1]+x[1] ) + exp( x[0]+x[0] );
	fsm->Y3 = -2*(+exp(x[0]+x[1]+x[1])*cos(x[2])
				  +exp(x[0]+x[0]+x[1])*cos(x[3]) );
	fsm->Y4 = exp(x[0]+x[0]+x[1]+x[1]);
	/* X-constants X1, X2, X3		*/
	fsm->X1 = exp(x[1])*(x[7]*sin(x[3]) - (x[6]+2*x[4])*cos(x[3]));
	fsm->X1+= exp(x[0])*(x[5]*sin(x[2]) - (2*x[6]+x[4])*cos(x[2]));
	fsm->X2 = 2*exp(x[0]+x[1])* (
				+ (x[4] + x[6]) * cos(x[3])*cos(x[2])
				- x[5] * cos(x[3]) * sin(x[2])
				- x[7] * cos(x[2]) * sin(x[3]) );
	fsm->X2+= x[6]*exp(2*x[0]) + x[4]*exp(2*x[1]);
	fsm->X3 = exp(x[1]+x[0]+x[0]) * (x[7]*sin(x[3]) - x[6]*cos(x[3]));
	fsm->X3+= exp(x[1]+x[1]+x[0]) * (x[5]*sin(x[2]) - x[4]*cos(x[2]));

	fsm->X1 = fsm->X1 - fsm->Y1*fsm->M0;
	fsm->X2 = fsm->X2 - fsm->Y2*fsm->M0;
	fsm->X3 = fsm->X3 - fsm->Y3*fsm->M0;
	fsm->X4 = 0.0     - fsm->Y4*fsm->M0;

	/* SY constant					*/
	fsm->SY = + fsm->X1 + fsm->X2
			  + fsm->X3 + fsm->X4;
	fsm->SY/= 1.0 + fsm->Y1 + fsm->Y2
				  + fsm->Y3 + fsm->Y4;
}


/* 1D gaussiam blur on place with FSM	*/

void	fsm_gauss_line(
			struct FSM_Gauss*	s,
			void*				buffer,
			int					len,
			int					usize) {
	int			p, t, ou;
	uint16_t*	line;

	line = (uint16_t*)buffer;
	
	/* Each line should have extra 4 items from left
	 * and 4 items from right for boundary history
	 * calculation in the conveyers. So if we have
	 * payload of N items in a line, the total its
	 * length should be N + 2*FSM_GAUSS_TM */
	
	if (usize != sizeof(uint16_t)) return;
	if (len <= 2*_FSM_GAUSS_TM) return;

	/* Fill boundary edges	*/
	for (p = 0; p < _FSM_GAUSS_TM; p++) {
		/* Remember that conveyer line size equals
		 * to the image line size */
		t = len - p - 1;

		/* Set boundaries						*/
		line[p] = line[_FSM_GAUSS_TM];
		line[t] = line[len - _FSM_GAUSS_TM - 1];

		/* Assume we have setted boundaries		*/
		s->SL[p] = s->SY * line[p];
		s->SR[t] = s->SY * line[t];
	}

	/* L->R walk conveyer	*/
	for (p = _FSM_GAUSS_TM; p < len - _FSM_GAUSS_TM; p++) {
		s->SL[p] = s->X1*line[p-1] - s->Y1*s->SL[p-1];
		s->SL[p]+= s->X2*line[p-2] - s->Y2*s->SL[p-2];
		s->SL[p]+= s->X3*line[p-3] - s->Y3*s->SL[p-3];
		s->SL[p]+= s->X4*line[p-4] - s->Y4*s->SL[p-4];
	}

	/* R->L walk conveyer	*/
	for (p = _FSM_GAUSS_TM; p < len - _FSM_GAUSS_TM; p++) {
		t = len - p - 1;
	
		s->SR[t] = s->X1*line[t+1] - s->Y1*s->SR[t+1];
		s->SR[t]+= s->X2*line[t+2] - s->Y2*s->SR[t+2];
		s->SR[t]+= s->X3*line[t+3] - s->Y3*s->SR[t+3];
		s->SR[t]+= s->X4*line[t+4] - s->Y4*s->SR[t+4];
	}

	/* And write result back to the line */
	for (p = _FSM_GAUSS_TM; p < len - _FSM_GAUSS_TM; p++) {
		ou = line[p] * s->M0 + s->SL[p] + s->SR[p];

		if (ou < 0)	 ou = 0; else
		if (ou >_FSM_GAUSS_MAX) ou = _FSM_GAUSS_MAX;

		line[p] = (uint16_t)ou;
	}
}


/* Debugging tool, would be removed	*/
void    offimage_print(OFFIMAGE* obj) {
	int p, t;

	for (p = -OFFI_BNDRY; p < OFFI_LEN_COL(obj)+OFFI_BNDRY; p++) {
		uint16_t*   line;

		line = OFFI_FROW(obj, p);

		printf("%3i/%4i: ", p, (void*)line - obj->data);
		
		for (t = 0; t < obj->sx; t++)
			printf(" %3i", line[t]);

		printf("\n");
	}
	
	printf("OK--\n\n");
}



