/*_ GIMP DR Transformation plug-in.                 _*/
/*_ Copyright (c) 2007, Ivan Bezdomniy              _*/
/*_ This program is free software. It may be distributed
 * and/or modified under the terms of the General Public
 * License. See the GNU General Public License for more
 * details */


#include "config.h"

//#include <unistd.h>
#include <math.h>
#include <stdlib.h>
#include <stdint.h>

#include <libgimp/gimp.h>



#include "main.h"
#define	__RENDER_INTERNAL
#include "render.h"
#undef	__RENDER_INTERNAL
/* Off-line image storage library	*/
#include <magie/offimage.h>
/* Pixel conversion conveyers		*/
#include <magie/pixmodel.h>
/* DR conversion conveyers			*/
#include <magie/drconv.h>

#include "plugin-intl.h"



/*_ Some constants				_*/
#define	RENDER_MAX_BPP		0x04
#define	RENDER_TILE_SIZE	0x40

/*_ Shortcut  macros			_*/
#define	_G_D(_n)	gimp_drawable
#define	_G_P(_n)	gimp_pixel





struct _Render_Control	o;
struct _Progress_Status	p;

void
	r_reset() {
	o.is_gray		= 0;
	o.is_rgb		= 0;
	o.off_im_valid	= 0;
	o.is_configured	= 0;
	o.is_preview	= 0;
}


/*_ Init SYSTEM object			_*/
gboolean
  r_init(
  		gint32		imageID,	/*_ Work image ID		_*/
		gint32		workID,		/*_ Work drawable ID	_*/
		int			flags ) {

	/*_ Check type of work drawable	_*/

	o.is_gray = (gimp_drawable_is_gray(workID) != 0);
	o.is_rgb  = (gimp_drawable_is_rgb(workID) != 0);
	
	if (!o.is_gray && !o.is_rgb) {
		g_error("Invalid type of pixel model\n");
		return FALSE;
	}

	o.imageID	= imageID;
	o.workID	= workID;
	o.workDrw	= gimp_drawable_get (workID);
	o.bpp		= o.workDrw->bpp;
 	o.is_alpha	= (gimp_drawable_has_alpha (o.workDrw->drawable_id));
	o.is_interactive = (flags & R_INIT_INTERACTIVE);
	o.sX		= o.workDrw->width;
	o.sY		= o.workDrw->height;
	o.rX		= 0;
	o.rY		= 0;

	if (o.bpp > RENDER_MAX_BPP) {
		g_error("Pixels bytes exceed allowed value");
		return FALSE;
	}

	gimp_drawable_mask_bounds (
		o.workDrw->drawable_id,
		&o.rM.x1, &o.rM.y1,
		&o.rM.x2, &o.rM.y2);

	gimp_tile_cache_ntiles (
		 3*(o.workDrw->width / gimp_tile_width () + 1));

	gimp_pixel_rgn_init (
		&o.rSrc, o.workDrw,
		0, 0, o.sX, o.sY,
		FALSE, FALSE);

	gimp_pixel_rgn_init (
		&o.rDst, o.workDrw,
		o.rM.x1, o.rM.y1,
		o.rM.x2 - o.rM.x1,
		o.rM.y2 - o.rM.y1,
		TRUE, TRUE);

	/* Make offline images for avg & sqr */
	OFFI_NULL(&o.o_avg);
	OFFI_NULL(&o.o_sqr);

	offimage_init(&o.o_avg, o.sX, o.sY, OFFI_USIZE);
	offimage_init(&o.o_sqr, o.sX, o.sY, OFFI_USIZE);

	if (OFFI_ISNULL(&o.o_avg) ||
		OFFI_ISNULL(&o.o_sqr)) {
		g_error("Can't allocate offline images");
		return FALSE;
	}

	if (o.is_gray) o.model = PIXEL_GRAY_VALUE;
	if (o.is_rgb)  o.model = PIXEL_HSV_VALUE;

	
	return TRUE;
}



gboolean
	r_execute(
		PLUG_PARMS*	plugs,
		void*		preview) {

	if (!plugs) return FALSE;

	/*_ Configure engine				_*/
	gint	rX = plugs->rX;
	gint	rY = plugs->rY;

	if (o.is_gray) o.model = PIXEL_GRAY_VALUE;
	if (o.is_rgb)  o.model = PIXEL_HSV_VALUE;
	
	if (o.rX != rX || o.rY != rY) {
		o.rX = rX;
		o.rY = rY;
		o.off_im_valid = 0;
	}

	/* Save plug-in settings to DRC controls */
	o.vmp.conveyer	= DRT_DRU_PLINEAR;
	o.vmp.parm.T	= plugs->v_center;
	o.vmp.parm.ks	= plugs->v_uniformity;
	o.vmp.parm.kr	= plugs->v_edges;
	o.vmp.parm.kc	= plugs->v_contrast;
	o.vmp.parm.wr	= plugs->v_width;
	o.vmp.parm.As	= plugs->v_assymm;
	o.vmp.parm.Tr	= 0.8;

	/* Validate configuration			*/
	o.is_preview 	= (preview != NULL);
	o.is_configured = 1;

	/*_ Prepare destination region		_*/
	if (o.is_preview) {
		gimp_pixel_rgn_init (
			&o.rPrv, o.workDrw,
			o.rP.x1, o.rP.y1,
			o.rP.x2 - o.rP.x1,
			o.rP.y2 - o.rP.y1,
			FALSE, TRUE);

		o.rOut = &o.rPrv;
		o.rU   = &o.rP;
	} else {
		o.rOut = &o.rDst;
		o.rU   = &o.rM;
	}

	/*_ Init progress status			_*/
	{
		int		parm = _PRGS_STG_02;
		if (!o.off_im_valid) parm |= _PRGS_STG_01;
		r_progress(_PRGS_INIT, parm);
	}

	/*_ If need, make AVG & SQR maps	_*/
	if (!o.off_im_valid) r_value();
	/*_ Run unify engine				_*/
	r_unify();

	r_progress(_PRGS_FIN, 0);

	/*_ Update GUI part					_*/
	if (!o.is_preview) {
		gimp_drawable_flush (o.workDrw);
		gimp_drawable_merge_shadow (o.workID, TRUE);
		gimp_drawable_update(
			o.workID,o.rM.x1, o.rM.y1,
			o.rM.x2 - o.rM.x1,
			o.rM.y2 - o.rM.y1);
	}

	o.is_preview = 0;

	return TRUE;
}



/* Make VALUE from image &store them in AVG & SQR offlines*/
int
  r_value(void){
  	OFFISTATE	sco;
	guchar*		line 	= NULL;
	float*		values	= NULL;
	uint16_t*	line_a;
	uint16_t*	line_s;
	int			cycles;
	int			r, i;

	/* Engine have to be configured	*/
	if (!o.is_configured) return FALSE;

	/* Skip if offline images have been calculated	*/
	if (o.off_im_valid) return TRUE;

	line = g_new(guchar, o.sX * o.bpp);
	values = g_new(float, o.sX);

	if (!line || !values) goto error;

	cycles = r_progress(_PRGS_ENTER, _PRGS_STG_01);
	cycles = (cycles < 0x10) ? 0x10 : cycles;

	for (r = 0; r < o.sY; r++) {
		gimp_pixel_rgn_get_row (
			&o.rSrc, line, 0, r, o.sX);

		offimage_row_p(&o.o_avg, (OFFI_PP)&line_a, r, 0);
		offimage_row_p(&o.o_sqr, (OFFI_PP)&line_s, r, 0);
		pixmodel_rgb_to_v(line,	values,	o.sX, o.model, o.bpp);

		for(i = 0; i < o.sX; i++) {
			line_a[i] = (uint16_t)(values[i]*(128*256-1));
			line_s[i] = (int16_t)(values[i]*values[i]*(128*256-1));
		}
	}

	/* Blur offline images		*/
	/*offimage_extent(&o.o_avg, 0, OFFI_BNDRY_EXT);
	offimage_extent(&o.o_sqr, 0, OFFI_BNDRY_EXT);
*/
	i_oi_blur(&o.o_avg, &sco, o.rX, o.rY, cycles);
	i_oi_blur(&o.o_sqr, &sco, o.rX, o.rY, cycles);

	/* Validate offline images	*/
	o.off_im_valid = 1;

	r_progress(_PRGS_LEAVE, _PRGS_STG_01);

error:
	if (line)	free(line);
	if (values)	free(values);

	return TRUE;
}



int
  i_oi_blur(
  		OFFIMAGE*		obj,
		OFFISTATE*		sco,
		int				rX,
		int				rY,
		int				cycles) {
	int		ret = 0;

	/* Query for interrupted execution	*/
	if (!offimage_blur(NULL, 0, 0, sco)) {
		int		errors = 0;
		int		step;

		step = (2 * cycles *  o.sX * o.sY)/(o.sX + o.sY);

		OFFI_STATE_RESET(sco, cycles);

		while (!OFFI_STATE_IS_FIN(sco)) {
			ret = offimage_blur(obj, rX, rY, sco);

			if (ret != 0) {
				g_debug("OFFIMAGE/I Blur execution gives %i", ret);
				OFFI_STATE_TERM(sco);
				
				if (errors++ > R_LIMIT_ERRORS) {
					g_debug("OFFIMAGE/I Blur errors limit reached");
					return -1;
				}
			}

			if (sco->unused != 0) {
				step = sco->cycles - sco->unused;
				step*= 2 * o.sX * o.sY;
				step/= o.sX + o.sY;
			}

			r_progress(_PRGS_UPDATE, step);
		}

		if (!OFFI_STATE_IS_FINOK(sco)) {
			g_debug("OFFIMAGE/II Blur execution failed");
			return -1;
		}
	} else {
		/* Interrupted execution can't be used	*/
		g_debug("OFFIMAGE/I Blur, query failed");
		ret = offimage_blur(obj, rX, rY, NULL);

		if (ret != 0) {
			g_debug("OFFIMAGE Blur execution gives %i", ret);
		}
	}

	return 0;
}



/* Perform dynamic range transformation	*/
int
  r_unify (void) {	/*_ Source drawable 	_*/
	guchar*		row_w	= NULL;	/* Work image line buffer	*/
	uint16_t*	row_a	= NULL;	/* AVG line pointer			*/
	uint16_t*	row_q	= NULL;	/* SQR line pointer			*/
	float*		values	= NULL;
	float*		current = NULL;
	int			width, height;	/* Work region size			*/
	int			pY;				/* Walk point				*/

	/* Engine have to be configured			*/
	if (!o.is_configured) 		return FALSE;
	/* And offline images have to be exists	*/
	if (!o.off_im_valid)  		return FALSE;

	/* Validate DRC Conveyer				*/
	if (drc_validate(&o.vmp))	return FALSE;

	width	= o.rU->x2 - o.rU->x1;
	height	= o.rU->y2 - o.rU->y1;

	row_w  = g_new(guchar, width*o.bpp);
	values = g_new(float, width);
	current= g_new(float, width); 
	
	if (!row_w || !values) {
		g_error("Can't allocate line buffer");
		goto error;
	}

	/* Enter progress status	*/
	r_progress(_PRGS_ENTER, _PRGS_STG_02);

	for (pY = o.rU->y1; pY < o.rU->y2; pY++){
		/* Get source line			*/
		gimp_pixel_rgn_get_row (
			&o.rSrc, row_w,	o.rU->x1, pY, width);
		/* Get AVG & SQR lines		*/
		offimage_row_p(&o.o_avg, (OFFI_PP)&row_a, pY, 0);
		offimage_row_p(&o.o_sqr, (OFFI_PP)&row_q, pY, 0);

		row_a += o.rU->x1;
		row_q += o.rU->x1;

		/* Make source line values	*/
		pixmodel_rgb_to_v(row_w, current, width, o.model, o.bpp);
	
		/* Execute DRC Conveyer		*/
		drc_conveyer(&o.vmp, current, values, row_a, row_q, width);
	
	/*	int		p;

		for(p = 0; p < width; p++) {
			float	vv, uu;

			uu = ((float)row_q[p])/(128*256 - 1);
			vv = ((float)row_a[p])/(128*256 - 1);
		
			values[p] = uu - vv*vv;
			if (values[p] >= 0) {
				values[p] = sqrtf(fabsf(values[p]));
				if (values[p] > 1) values[p] = 1;
			} else
				values[p] = 0;

		}
	*/

		/* Update pixels values		*/
		pixmodel_rgb_set_v(row_w, values,
				width, o.model, o.bpp, current);
	
		/* Flush work row			*/
		gimp_pixel_rgn_set_row (o.rOut, row_w, o.rU->x1, pY, width);

		/* Update progress status	*/
		r_progress(_PRGS_UPDATE, width);
	}

	/* Leave progress status	*/
	r_progress(_PRGS_LEAVE, _PRGS_STG_02);
	
error:
	/*_ Release all resources		_*/
	if (row_w)	g_free(row_w);
	if (values) g_free(values);
	if (current)g_free(current);

	return TRUE;
}



/*_ Releases ITER object		_*/
void
  r_clean(void){

	if (o.workDrw) {
		gimp_drawable_detach (o.workDrw);
		o.workDrw = NULL;  }

	/* Free offline images	*/
	offimage_free(&o.o_avg);
	offimage_free(&o.o_sqr);

	r_reset();
}



/* Progress status interface		*/
int
	r_progress(
			int		opcode,
			int		parm) {

	switch (opcode) {
	/* Init progress settings		*/
	case _PRGS_INIT: {
		int		max;

		p.weight_summ	= 0;
		p.weight_cur	= 0;
		p.weight_step	= 0;
		p.accum			= 0;
		p.weight[0]		= 0;
		p.weight[1]		= 0;
		/* Sanity cheks for stages flags	*/
		if (parm & (~_PRGS_STAGESM)) return -1;
		p.mask			= (o.is_preview) ? 0 : parm;
		
		/* All stages disabled, nothing to do*/
		if (!(p.mask & _PRGS_STAGESM)) return 0;

		/* Weights for stages. Algorithms complexity
		 * are apriori calculated values for each satge */
		if (parm & _PRGS_STG_01) {
			/* 2x gaussian blurs			*/
			p.weight[0] = (o.rX > 0) ? 1 : 0;
			p.weight[0]+= (o.rY > 0) ? 1 : 0;
			p.weight[0]*= 2*(o.sX * o.sY);
		}

		if (parm & _PRGS_STG_02) {
			/* Simple line-walking conveyer	*/
			p.weight[1] = o.rU->y2 - o.rU->y1;
			p.weight[1]*= o.rU->x2 - o.rU->x1;
		}

		p.weight_summ = p.weight[0] + p.weight[1];
	
		p.weight_step = p.weight_summ / _PRGS_GRANULAR;
		/* Limit step to minimum	*/
		max = (o.sX > o.sY) ? o.sX : o.sY ;
		
		if (p.weight_step < max * _PRGS_MIN_DIV)
			p.weight_step = max * _PRGS_MIN_DIV;

		if (o.is_interactive)
			gimp_progress_init (_PRGS_T_DRCI);
		else
			gimp_progress_init (_PRGS_T_DRCM);

		return 0; }

	/* Enter new stage				*/
	case _PRGS_ENTER: {
		int		steps;
		if ( (p.mask & parm) == 0) return -1;

		if (parm == _PRGS_STG_01) {
			p.weight_cur= 0;
			steps = 2 * p.weight_step / (o.sX + o.sY);

			if (o.is_interactive)
				gimp_progress_set_text(_PRGS_T_DRCT);
		} else
		if (parm == _PRGS_STG_02) {
			p.weight_cur= p.weight[0];
			steps = p.weight_step / (o.rU->x2 - o.rU->x1);
			
			if (o.is_interactive)
				gimp_progress_set_text(_PRGS_T_DRCU);
		} else
			return 0;
			
		p.accum	= 0;
		p.stage = parm;

		steps = (steps <= 0) ? 1 : steps;

		return steps; }

	/* Leave stage					*/
	case _PRGS_LEAVE: {
		p.stage = 0;

		/* This code is for the debugging purpose	*/
		int		collect = 0;

		if (parm == _PRGS_STG_01) collect += p.weight[0];
		if (parm == _PRGS_STG_02) collect += p.weight[1];

		if (p.weight_summ > collect) {
			g_debug("Progress weight overflow, (%i, %i)",
				collect, p.weight_summ);
		}
		return 0; }

	/* Update progress accumulator	*/
	case _PRGS_UPDATE: {
		int			mplex;
		double		value;

		if (p.stage == 0) return 0;

		p.accum 		+= parm;
		p.weight_cur	+= parm;

		mplex = p.accum / p.weight_step;

		if (mplex== 0) return 0;

		/* Do here progress update*/
		value	 = p.weight_cur;
		value	/= p.weight_summ;
		p.accum -= p.weight_step * mplex;
		
		gimp_progress_update((gdouble)value);

		return 0; }

	case _PRGS_FIN:
		/*gimp_progress_cancel();*/
		return 0;

	}

	return -1;
}




