#!/usr/bin/env python

'''
  The heart of the "Make shortcut..." GIMP plugin.
  Generates another plugin (in Python language) that is a shortcut to a target plugin.
  Shortcut means: 
  1) alias or link to target plugin
  2) no options dialog (standard, current, or preset options for the target plugin.)

  Copyright (C) 2010 Lloyd Konneker      bootch at nc.rr.com

    License:
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
'''

'''
Terminology:
This plugin: what you are reading now, that generates code.
Wrapping plugin: the plugin code generated by this plugin, iow the shortcut plugin.
Wrapped plugin: the plugin that the shortcut calls, iow the target.
'''
'''
I tried to simply register an existing plugin under a new menu item.
It doesn't work, disallowed by GIMP, can only register menu at certain times in life of plugin, init etc.
gimp.pdb.gimp_plugin_menu_register("python_fu_impscriptor", "<Image>/Filters/Map/Foo");
'''

# TBD
# gui improve
# check for file, menu exists already
# shortcuts independent of image, without parameters

# Later: 
# let user enter menu path
# created date

# domain

from gimpfu import *
import os
import stat
from string import Template


# Constants, at least in this version.
# Next version might let user chose the menu path for the shortcut
SHORTCUT_FILENAME_PREFIX = "python-fu-shortcut-to-"
SHORTCUT_MENU_PATH_PREFIX = "<Image>/Shortcuts/"


# Globals
imagetypemap = {}
menupathmap = {}


'''
Template for the wrapping plugin.
See standard Python documentation for the Template module. Briefly: $placeholder
No copyright or license for the wrapping plugin.
'''
template = Template(
r'''#!/usr/bin/env python

# This plugin is a shortcut to another plugin iow a wrapper of a wrapped plugin.
# This plugin was created by the GIMP plugin "Make shortcut..."
# See plugin-make-shortcut.py


def plugin_main($wrappingmainformalparams):
  # Call the wrapped plugin.
  # If the wrapped plugin requires (image, drawable ) pass them through.
  # Other parameters are dummy parameters, ignored by wrapped since using LAST_VALS.
  # run-mode=RUN_WITH_LAST_VALS tells wrapped plugin to use its standard(default) or current parameters 
  $calltowrappedplugin  # substituted 

if __name__ == "__main__": # invoked at top level, from GIMP
  from gimpfu import *    
  
  register(
    "$wrappingprocedurename",  # substituted
    "A shortcut to $wrappedmenupath",
    "This is a shortcut created locally using Make shortcut...",
    "Anonymous",
    "Uncopyrighted",
    "No copyright date",
    "$wrappinglabel",  # substituted
    "$wrappingimagetype",  # substituted, same as wrapped
    [], # Wrapping takes no extra parameters (besides possibly image)
    [],
    plugin_main,
    $wrappingmenuarg # substituted
    )  
    # domain=("gimp20-python", gimp.locale_directory))
  
  main()
'''
)




def generate(wrappedprocname, wrappingmenuitem):
  '''
  Generate file of python code for a plugin that is a shortcut, calling a target plugin.
  
  wrappedprocname: of wrapped plugin, user chose.
  wrappingmenuitem: of shortcut plugin, user chose.
  '''
  
  # Make global maps from procname to foo
  global imagetypemap
  global menupathmap
  make_procname_maps()
  
  global template
  substitutions = {}
  
  
  '''
  Create map for template substitution (actual parameters to the template.)
  '''
  # A string to call the wrapped plugin
  substitutions["wrappingmainformalparams"] = make_wrapping_main_params(wrappedprocname)
  substitutions["calltowrappedplugin"] = make_call_string(wrappedprocname)
  # Image type for the wrapping plugin that is same image type as the wrapped plugin.
  substitutions["wrappingimagetype"] = imagetypemap[wrappedprocname]
  # make standard name for the wrapping plugin procedure.
  # !!! Note this is a string, don't need to substitute underbars for dash
  substitutions["wrappingprocedurename"] = SHORTCUT_FILENAME_PREFIX + wrappedprocname
  substitutions["wrappedmenupath"] = menupathmap[wrappedprocname]
  '''
  The label and menu parameters work together.
  !!! A side effect of the menu parameter is to omit the image and drawable parameters to a plugin.
  !!! Obscure wierdness in gimpfu.
  Set them so that image and drawable parameters are passed to the wrapping plugin,
  if the wrapped plugin needs them.
  If we don't do this, then gimpfu might open a dialog for the image parameter
  in some circumstances.
  '''
  if does_wrapped_take_image(wrappedprocname):
    # label comprises menu path cat menu item
    # menu keyword arg omitted
    # !!! gimpfu will pass image and drawable to wrapping plugin
    substitutions["wrappinglabel"] = SHORTCUT_MENU_PATH_PREFIX + wrappingmenuitem
    substitutions["wrappingmenuarg"] = ""
  else:
    # label is just menu item
    # menu keyword is the menupath
    # !!! gimpfu will NOT pass image and drawable to wrapping plugin
    substitutions["wrappinglabel"] = wrappingmenuitem
    substitutions["wrappingmenuarg"] = 'menu="' + SHORTCUT_MENU_PATH_PREFIX + '"'
  
  # Make substitutions in the python template.
  substitutedtemplate = template.substitute(substitutions)
  
  # Make standard file name for the wrapper plugin:
  # Gimp std + my std + name
  filename = "plugin-" + "shortcut-to-" + wrappedprocname
  # Make file path to local plugins.
  # Pygimp knows parent directory + standard directory name + filename + standard suffix
  filepath = gimp.directory + "/plug-ins/" + filename + ".py"
  print filepath
  
  # Write template to file.
  # tbd warn of overwrite
  with open(filepath, "w") as f:
    f.write(substitutedtemplate)
  
  # Make executable. (Linux, not needed for Windows?)
  os.chmod(filepath, stat.S_IRWXU)



def make_procname_maps():
  '''
  !!! Note:
    Pygimp pdb does not expose the imagetype as attribute of a PDB function.
    gimp-procedural-db-query also does not return the imagetype.
    gimp_plugins_query() DOES return the image type, but can't query by procname.
    So we must query all the plugins, and create a mapping from procname to imagetype.
  '''
  global imagetypemap
  global menupathmap
  
  # Empty search string means get all.  Returns count, list, ...
  c1, menupath, c2, accel, c3, loc, c4, imagetype, c5, times, c6, name = gimp.pdb.gimp_plugins_query("")
  for i in range(0,len(name)):
    imagetypemap[name[i]]=imagetype[i]
    menupathmap[name[i]]=menupath[i]



def does_wrapped_take_image(procname):
  '''
  Return whether the wrapped plugin takes an image.
  i.e. if its second parameter has type PF_IMAGE
  (Also, it has the name "image")
  '''
  return pdb[procname].params[1][0] == PF_IMAGE



def make_wrapping_main_params(procname):
  '''
  Return string for the formal parameters of the wrapping plugin main() procedure.
  Depends on whether the wrapped plugin takes the same parameters (which are passed through).
  Note the string must match in the wrapping main formal parameters and the call to the wrapped plugin
  (for pass through to work.)
  '''
  if does_wrapped_take_image(procname):
    return "image, drawable"
  else:
    # Also, the imagetypes param to register() should be empty,
    # meaning the wrapped plugin is enabled always (whether an image is open or not.)
    return ""



def make_imagetype_string(procname):
  '''
  Return the imagetype string eg. "RGB" for a plugin.
  '''
  return make_procname_to_imagetype_map()[procname]





def make_call_string(procname):
  '''
  Make Python code for a call to the named plugin, with don't care parameters.
  Note that in the PDB, the first three parameters are always run-mode, image, and drawable.
  But Pygimp requires run-mode as a keyword arg
  '''
  # Substitute _ for - in name
  # Because in Python, - is the subtraction operator, can't be used in names.
  underbar_name = procname.replace("-", "_")
  # return "gimp.pdb." + underbar_name + "(image, drawable, " + make_NC_params(procname) + "run_mode=RUN_WITH_LAST_VALS)"
  # Plugin that doesn't have an image
  return "gimp.pdb." + underbar_name + "( " + make_NC_params(procname) + "run_mode=RUN_WITH_LAST_VALS)"
  
  
  

'''
Here I started an elegant way to create the mapping, but gave up on it.
How do you instantiate a type without knowing what parameters __call__ takes??
_instance_mapping = { }
for (typeconstant, type) in gimpfu._obj_mapping.items():
  _instance_mapping[typeconstant] = type.__call__()
  
'''
'''
Derived from gimpfu.py.
Map parameter type to instance of that parameter type.
Instances are don't care values OR standard values for actual values to a plugin call.
EG "image" is a standard actual value for the first parameter of type PF_IMAGE.
'''
_instance_mapping = {
    PF_INT8        : 0,
    PF_INT16       : 0,
    PF_INT32       : 0,
    PF_FLOAT       : 0.0,
    PF_STRING      : '"foo"',  # At shortcut time: a don't care string (can't be empty.)
    PF_COLOR       : (0,0,0), # a don't care color
    PF_REGION      : 0,
    PF_DISPLAY     : gimp.Display,
    PF_IMAGE       : "image", # At shortcut time: not a string , but a Python reference
    PF_LAYER       : gimp.Layer,  # TBD ??? I think this should be an int (an ID of a layer)
    PF_CHANNEL     : gimp.Channel,
    PF_DRAWABLE    : "drawable",
    PF_VECTORS     : gimp.Vectors,
}
# TBD If the plugin requires layers, will they default properly?



def make_NC_params(procname):
  ''' For a call to a plugin, make a string of actual parameters with don't care values.'''
  ret = ""
  
  # !!!Note pygimp provides pdb[name].params attribute, a list of parameter definitions.
  # !!!Don't need to call gimp.pdb.gimp_procedural_db_proc_info(name)
  #
  # Omit first standard params: run-mode
  # image, drawable are in the mapping:
  # They are mapped not to an instance, but to an actual parameter
  # that will be passed through from the wrapping plugin to the wrapped plugin.
  for param in pdb[procname].params[1:]:
    paramtype = param[0]  # first item is the type
    # map param type to instance of the type
    # !!! Note want __str__ not __repr__ here so an "image" parameter is not quoted
    ret += _instance_mapping[paramtype].__str__() + ", "
  # No need to elide the trailing comma, later we append a keyword actual param run_mode
  return ret





'''
  These are three test instances: python, scheme, C.  
  To prove that LAST_VALS ignores the parameters, etc.
  !!! Note, must pass all parameters, even if LAST_VALS means ignore them
  # pdb.plug_in_resynthesizer2(timg, tdrawable, 0,0, 1, tdrawable, -1, -1, 0.0, 0.117, 16, 500, run_mode=RUN_WITH_LAST_VALS)
  # pdb.script_fu_add_bevel(timg, tdrawable, 5.0, 0, 0, run_mode=RUN_WITH_LAST_VALS)
  # pdb.plug_in_nova(timg, tdrawable, 0,0,(10,10,10),0, 0, 0, run_mode=RUN_WITH_LAST_VALS)
'''




