#!/usr/bin/env python

'''
GUI for "Make shortcut..." GIMP plugin.
Uses GtkBuilder.

GUI semantics:
User selects from a treeview to choose a plugin by menu item.
Only leaves can be selected.

User keys in text for shortcut's new menu item.

Notes about creating the glade file:

Add a treestore object.
Add two columns to the treestore (of type gchararray.)
Add a treeview object (asks for treestore object)
  Set General>Search Column to 1 (2nd column, indicates boolean for filtering/search)
  (This is a hidden column, here always True.  Missing this, no text displays in treeview.)
Add a treeviewcolumn to the treeview:
  RMB/Edit... the treeview
  Choose Hierarchy tab
  Choose the Add button in the left frame to add a treecolumn.
  Set the Clickable property
  Add cellrenderer child to treecolumn:
    Select treecolumn
    RMB Choose Add child text
    Set the General>Text property to 0 (with gchar..)
    (This tells the renderer which column of the model (the treestor) to use)

Note the glade file has the help string in it.
Probably it should be moved here for easier update to it?
'''


# if run standalone, possibly independent of any programs supplying db's
if __name__ == "__main__":
  # since this file is in the package, but imports from the package
  # put the directory above the package in the path
  import sys, os
  sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
  print sys.path


import pygtk
pygtk.require("2.0")
import gtk
import glib

# Our own sub modules, installed in same directory as this file.
# These are independent of the db
from makeshortcut import db_treemodel
from makeshortcut import generateshortcut


# Make fully qualified path so pygtk finds glade file.
# Get the glade file from same directory as this file (must be together)
import os.path
UI_FILENAME = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'makeshortcut.glade')


# global GUI state
selected_value = "" # plugin name


class MakeShortcutApp(object):   # new-style class inherits from object
  '''
  '''

  def __init__(self, dictofviews):
    # Note: use self for all variables accessed across class methods,
    # but not passed into a method, eg in a callback.
    
    # The dictofviews argument was created earlier by the app.
    self.dictofviews = dictofviews  # the dictionary of views that drives this app
    self.currentviewname = dictofviews.keys()[0]  # arbitrarily the first

    '''
    Keep several treemodels, and just swap them in and out of the treeview.
    Instead of: clear them and reload them.
    Data driven construction of the set of treemodels as specified by the dictofviews.
    Also populates and sorts the treemodels.
    Note: ignore any treestore model from glade, don't: model = builder.get_object("treestore1")
    Must be done before signals connected but after search_dict.
    '''
    self.models = db_treemodel.TreeModelDictionary(self.dictofviews)
    
    builder = gtk.Builder()
    builder.add_from_file(UI_FILENAME)
    # Connect some signals defined in the glade file.  Was auto_connect in libglade.
    # See callbacks below, named same as in glade file.
    builder.connect_signals(self)
    
    # Get references to widgets will refer to later
    self.dialog = builder.get_object("dialog1")
    self.buttonOK = builder.get_object("button2")
    self.helpMessage = builder.get_object("messagedialog1")
    self.textentry = builder.get_object("entry1")
    self.treeview = builder.get_object("treeview1")
    
    '''
    Selection handling:
    Note a treeselection object is always part of a treeview: no need to create in glade
    Connect selection event to callback function.
    I don't think this is done by connect_signals() above, since the selection objects are not in glade?
    '''
    self.treeview.get_selection().connect("changed", self.on_selection_changed)
    # Connect pre-selection event to callback function that determines what can be selected
    self.treeview.get_selection().set_select_function(self.filter_select)
        
    # Set model of treeview to the treemodel of the current viewspec
    # TBD bug it is initially out of sync with the List by widget.
    self.treeview.set_model(self.models[self.currentviewname].treemodel)
    
    # Initial GUI state
    self.buttonOK.set_sensitive(False) # Can't OK unless text entered
    # The following doesn't work
    # builder.get_object("button1").grab_default()   # Return defaults to cancel
    # Initially, the focus is in the text entry and there is no default button?

    self.dialog.show()


  def main(self):
    gtk.main()  # event loop


  '''
  Callback functions
  '''

  '''
  Callbacks for the main app, a dialog box.
  '''

  def on_dialog1_response(self, widget, responseID):
    '''
    Callback when user clicks on any button in "internal action_area" of dialog.
    In Glade: Add toplevel DialogBox.  
      Set dialog1>Signals>response to "on_dialog1_response" (one of the choices
      and this callback method should be named the same.)
      Add buttons to internal_action area.
      For each button, set General>Response ID property to correspond, eg. 1.
      No need to set button Signals properties.
    This is a dispatcher.
    '''
    if responseID == 1: # same as responseID property set in Glade on Cancel button
      print "Cancel"
      gtk.main_quit()
    elif responseID == 2: # OK
      # Crux
      self.doshortcut()
      gtk.main_quit()
    elif responseID == 3: # Help
      self.helpMessage.show()
    elif responseID == -4:  # on destroy, not set in Glade but sent by pygtk
      pass
    else :
      print "Unhandled response ID for dialog:", responseID
    

  def on_dialog1_destroy(self, widget):
    '''
    Callback when user clicks on the dialog window's close icon "X".
    Note minimize, and maximize take care of themselves.
    Window will close but app will not quit unless call gtk.main_quit().
    '''
    gtk.main_quit()
    
    
  def on_messagedialog1_response(self, widget, responseID):
    '''
    Callback when user clicks on OK button of Help dialog.
    '''
    if responseID == 1: # same as responseID property set in Glade on Cancel button
        widget.hide()


  '''
  Callbacks for the text entry box for menu item (name) of shortcut
  '''        
  
  # TBD not currently connected, and does nothing   
  def on_entry1_activate(self, textentry):
    '''
    "Filter by.." widget, signal=activated, callback function.
    Action: set focus to the OK button.
    Note: on_activated (by return key), not on_changed (for each char changed)
    '''
    print "Text entry activated",textentry.get_text()

    
  def on_entry1_changed(self, textentrywidget):
    print "Changed"
    # Enable (set the "sensitive" state of) the OK button
    # according to state of user's actions.
    self.buttonOK.set_sensitive(self.isOKenabled())
  
  ''' 
  Note the text entry cannot be activated by hitting Return key
  and OK button does not have focus always?
  '''
  
  

  '''
  Callbacks for the treeview to select a target plugin
  '''
 
  def on_selection_changed(self, theSelection):
    '''
    Treeview widget, signal=selectionChanged on theSelection which is a gtk.TreeSelection 
    Action: keep a state variable for the GUI showing user has made selection.
    Single selection is enforced by default.
    '''
    global selected_value
    
    model, path = theSelection.get_selected()
    if path:
      # Alternative: column 0 drives selection
      # selected_value = model.get_value(path, 0) # column 0
      # Model must take the view value to a model value (what the view represents)
    
      # Alternative: column 1 (hidden?) drives selection (ie is the model value)
      selected_value = model.get_value(path, 1) # column 1
      assert selected_value # not empty string, else filter_select logic is wrong
    else: # no selection
      selected_value = None
    print "Selection", selected_value
     
     
  def filter_select(self, path):
    '''
    Callback: return boolean indicating whether selection is allowed.
    Let user select only leaves.
    Filters the user's clicks in the treeview,
    and ignores those clicks that are not in leaves.
    '''
    '''
    Note: tree levels without children are a complication.
    The definition of a leaf here is: row with non-empty second, hidden column.
    If search filtering takes out all leaf rows under a tree level row,
    that tree level row has no children.
    If we choose to display tree level rows without children,
    then definition of leaf is not: has no children.
    '''
    model = self.treeview.get_model() # !!! Get treeview's model, which varies in this app.
    return model.get_value(model.get_iter(path), 1) != ""
    # OLD return not model.iter_has_child(model.get_iter(path)) # no child means leaf


 


  '''
  Utility functions.
  '''
  def isOKenabled(self):
    ''' GUI state logic.
    OK button enabled only if user has entered required fields.
    '''
    return self.textentry.get_text() is not None and selected_value is not None
  
  
  def doshortcut(self):
    '''
    Called on OK button.
    '''
    global selected_value
    
    generateshortcut.generate(selected_value, self.textentry.get_text())
    
    

if __name__ == "__main__":
  # standalone app
  
  # ??
  import sys
  sys.path.append(os.path.dirname(os.path.abspath(__file__)))
  print sys.path
  
  app = InspectorApp()
  gtk.main()
  

