#!/usr/bin/env python

'''
Glue between Makeshortcut plugin and pygimp pdb.

Derived from inspector.inspectpdb.py (see it for more notes.)

Reimplement pdb from pygimp:
  Make it iterable (a full dictionary)
  Add __repr__ method that documents a procedure
  Add extra attributes.
  Limit it to plugins.

Also define views on the db.

The general API for glue objects between a treeview and its data:
  dictofviews object a dictionary of viewspecs on a db object,
  dictionary of objects with attributes and repr method (referred to as the db.)
  a filter dictionary: boolean valued with same keys as the db
Many viewspecs can all refer to the same dictionary of objects 
'''


import gimp
import gimpenums  # for proctypes
import types
import time

# our own submodule, installed in same directory as this
import db_treemodel


# Dictionaries of types in the conceptual model

'''
This used to map integer types to strings on loading from the PDB.
Programming surprise: some docs say constants are like PROC_PLUG_IN, others say GIMP_PLUG_IN
Constants defined by gimpenums are same as in libgimp but without prefix GIMP_ e.g. GIMP_EXTENSION -> EXTENSION
'''
proctypedict = {
  gimpenums.PLUGIN : "Plugin",
  gimpenums.EXTENSION : "Extension",
  gimpenums.TEMPORARY : "Temporary",
  gimpenums.INTERNAL : "Internal",
  -1 : "Unknown"   # Hope this doesn't conflict or change
  }




class Procedure:
  '''
  Procedures in the Gimp PDB.
  Mimics the attributes exposed by the PDB.
  Unifies ALL the exposed attributes of any type of procedure.
  Even attributes that aren't readily available from Gimp.
  Implements repr for str()
  '''
  # TBD catch ValueError on decode ?
  
  # Note it is important to properly default those attributes that we build views on
  def __init__(self, name, accel, loc, time, menupath = "<Unknown>", imagetype="<Unknown>",
      blurb="", help="", author="", copyright="", date="", proctype=-1 ):
      
    # global proctypedict
    
    # attributes returned by gimp_plugin_query
    self.name = name
    self.menupath = menupath
    self.accel = accel
    self.loc = loc
    self.imagetype = imagetype
    self.time = time
    # attributes returned by gimp_procedural_db_proc_info
    self.blurb = blurb
    self.help = help
    self.author = author
    self.copyright = copyright
    self.date = date
    self.type = proctypedict[proctype]
    # other attributes that can be discerned, eg by inference or parsing source files
    self.filename = "Unknown"
    self.language = "Unknown"
    
  
  def __repr__(self):
    '''
    Return text describing procedure.
    
    Future: different formats for different types
    Future: formatted
    Future: highlight the search hits
    '''
    # print self.__dict__
    text = ""
    for attrname, attrvalue in self.__dict__.iteritems():
      if attrvalue is None: # Don't know why pygimp didn't do this earlier?
        attrvalue = ""  # Must be a string
      if not isinstance(attrvalue, types.StringType) :
        print "Non string value for attr:", attrname, "of:", self.name
        attrvalue = "***Non string error***"

      text = text + attrname + ': ' + attrvalue + "\n"
    return text

    
  def update(self, blurb, help, author, copyright, date, thetype):
    '''
    TBD generalize 
    '''
    self.blurb = blurb
    self.help = help
    self.author = author
    self.copyright = copyright
    self.date = date
    self.type = proctypedict[thetype]

    

class Pdb(dict):
  '''
  A read only dictionary of objects of type Procedure mimicing the Gimp PDB.
  For now, it initializes itself with data and you can't setitem.
  '''
  
  def __init__(self):
    
    # Base class init
    dict.__init__(self)
   
    # Fill self with data from Gimp PDB
    
    # Query the plugins, which have different attributes exposed.
    # Empty search string means get all.  Returns count, list, ...
    c1, menupath, c2, accel, c3, loc, c4, imagetype, c5, times, c6, name = gimp.pdb.gimp_plugins_query("")
    
    for i in range(0,len(name)):
      # Create new procedure object
      procedure = Procedure(name[i],  accel[i], loc[i],
        time.strftime("%c", time.localtime(times[i])),  # format time.  TBD convert to UTF8
        menupath[i], imagetype[i])
     
      # Note about future development:
      # pygimp wraps pdb.gimp_procedural_db_get_data as gimp.pygimp_get_data(name[i])
      # data will be the default parameters for plugins written in C.
        
      dict.__setitem__(self, name[i], procedure)
    
  def __setitem__(self):
    raise RuntimeError, "Pdb does not allow adding procedures"
  
  # iterator methods, and all other special methods, inherited from base
  # No overriding is necessary.


'''
This is the meat of this glue module:
define views on an augmented PDB.
'''
plugindb = Pdb() # db of plugins

# A map that defines what rows appear in the gtk.treeview
# TBD make it show only plugins that are not shortcuts !
# ie no need for a shortcut to a shortcut.
pluginfilterdict = {}
for name, value in plugindb.iteritems():
  pluginfilterdict[name] = True # show all

dictofviews = {}  # Exported, the main product
    
dictofviews["Procedures by menu path"] = db_treemodel.ViewSpec("Procedures by menu path", "menupath", "SlashPath", None, plugindb, pluginfilterdict)


if __name__ == "__main__":
  # test
  # Currently, this doesn't work: abort at wire to gimp
  # That is, gimp must be running.
  import sys
  
  # gimp module is .so in /usr/lib/gimp/2.0/python
  # gimpfu.py in /usr/lib/gimp/2.0/plug-ins
  sys.path.append('/usr/lib/gimp/2.0/python')
  import gimp
  
  # test 
  mypdb = Pdb()
  
      
    
