#!/usr/bin/env python
# 
# -------------------------------------------------------------------------------------
#
# Copyright (c) 2013, Franco De Fonzo
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without modification, 
# are permitted provided that the following conditions are met:
#
#    - Redistributions of source code must retain the above copyright notice, this 
#    list of conditions and the following disclaimer.
#    - Redistributions in binary form must reproduce the above copyright notice, 
#    this list of conditions and the following disclaimer in the documentation and/or 
#    other materials provided with the distribution.
#    - Neither the name of the author nor the names of its contributors may be used 
#    to endorse or promote products derived from this software without specific prior 
#    written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
# SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
# BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN 
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH 
# DAMAGE.
#
# -------------------------------------------------------------------------------------
#
# This file is a Python plug-in for GIMP.
#
# Creates from a given sentence outlined text, in separate layers.
# It can be executed by selecting the menu option: 'Filters/Text/Outlined text'
#
# This tells Python to load the Gimp module
from gimpfu import *

def outlined_text(theImage, theLayer, string, theFont, fontSize, textColor, outlineColor, outlineThickness, delExistLayers) :
	letter = ''
	word = ''
	listToWrite = []
	for letter in string:
		if letter != ' ' :
			word = word + letter
		else :
			listToWrite.append(word)
			word = ''
	listToWrite.append(word)
	#
	imgWidth = pdb.gimp_image_width(theImage)	
	imgHeight = pdb.gimp_image_height(theImage)
	spacing = (outlineThickness * 2) 
	xTxt = 0
	yTxt = 5
	txtDimensions = [0, 0]
	dimCar = 0
	if delExistLayers == False :
		# Hide all layers	
		for i in range(pdb.gimp_image_get_layers(theImage)[0]):
			pdb.gimp_item_set_visible(theImage.layers[i], False)
		# Show background
		howManyLayers = pdb.gimp_image_get_layers(theImage)[0]
		pdb.gimp_item_set_visible(theImage.layers[howManyLayers - 1], True)		
		#
	for txtToWrite in listToWrite :
		safeMargin = (len(txtToWrite) * dimCar)
		xTxt = xTxt + spacing + txtDimensions[0]
		if xTxt > (imgWidth - (safeMargin * 2)):
			xTxt = spacing
			yTxt = yTxt + spacing + txtDimensions[1]
			if yTxt > (imgHeight - (txtDimensions[1] * 2)):
				pdb.gimp_message ("ERROR: Image canvas too small")
				for i in range(pdb.gimp_image_get_layers(theImage)[0]):
					pdb.gimp_item_set_visible(theImage.layers[i], True)	
				return			
		txtDimensions = single_text(theImage, txtToWrite, theFont, fontSize, textColor, outlineColor, outlineThickness, xTxt, yTxt)
		dimCar = (txtDimensions[0] / len(txtToWrite))	
	for i in range(pdb.gimp_image_get_layers(theImage)[0]):
		pdb.gimp_item_set_visible(theImage.layers[i], True)	
		
def single_text(theImage, txtToWrite, theFont, fontSize, textColor, outlineColor, outlineThickness, xTxt, yTxt) :
	imgWidth = pdb.gimp_image_width(theImage)	
	imgHeight = pdb.gimp_image_height(theImage)
	theOutlineLayer = pdb.gimp_layer_new(theImage, imgWidth, imgHeight, 0, "Outline", 100, 0)
	if (outlineColor[0],outlineColor[1],outlineColor[2]) != (255, 255, 255) :
		bkgLayerOutline = (255, 255, 255)
	else :
		bkgLayerOutline = (0, 0, 0)
	pdb.gimp_context_set_background (bkgLayerOutline)
	pdb.gimp_context_set_foreground (textColor)
	pdb.gimp_image_add_layer(theImage, theOutlineLayer, 0)
	pdb.gimp_drawable_fill(theOutlineLayer, 1) # BACKGROUND-FILL (1)
	theText = pdb.gimp_text_fontname (theImage, theOutlineLayer, xTxt, yTxt, txtToWrite, 0, True, fontSize, 0, theFont)
	pdb.gimp_floating_sel_to_layer(theText)
	# Extracts vectors from the text 
	theVector = pdb.gimp_vectors_new_from_text_layer(theImage, theText)	
	# New Path with vectors extracted from the text
	pdb.gimp_image_insert_vectors(theImage, theVector, None, 0)
	pdb.gimp_context_set_foreground(outlineColor)
	# Set up Brush
	brushTemp = pdb.gimp_brush_new("BrushTemporaneo")
	pdb.gimp_brush_set_shape(brushTemp, 1) # BRUSH-GENERATED-SQUARE (1)
	pdb.gimp_brush_set_hardness(brushTemp, 1)
	pdb.gimp_brush_set_radius(brushTemp, outlineThickness)
	pdb.gimp_brush_set_spacing(brushTemp, 0)
	pdb.gimp_brush_set_spikes(brushTemp, 2)
	pdb.gimp_brush_set_aspect_ratio(brushTemp, 1)
	pdb.gimp_brush_set_angle(brushTemp, 0)
	pdb.gimp_context_set_brush(brushTemp)
	#
	pdb.gimp_edit_stroke_vectors(theOutlineLayer, theVector)
	pdb.gimp_image_remove_vectors(theImage, theVector)
	pdb.gimp_brush_delete(brushTemp)
	# Hide background
	howManyLayers = pdb.gimp_image_get_layers(theImage)[0]
	pdb.gimp_item_set_visible(theImage.layers[howManyLayers - 1], False)
	#	
	pdb.gimp_image_set_active_layer(theImage, theText)
	newLayer = pdb.gimp_image_merge_visible_layers (theImage, 1) # CLIP-TO-IMAGE (1)
	pdb.gimp_item_set_name(newLayer, txtToWrite)
	# Select layer main color
	pdb.gimp_context_set_antialias(False)
	pdb.gimp_context_set_feather(False)
	pdb.gimp_context_set_feather_radius(0, 0)
	pdb.gimp_context_set_sample_merged(True)
	pdb.gimp_context_set_sample_criterion(0) #  SELECT-CRITERION-COMPOSITE (0)
	pdb.gimp_context_set_sample_threshold_int(30)
	pdb.gimp_context_set_sample_transparent(True)
	pdb.gimp_image_select_color(theImage, 2, newLayer, bkgLayerOutline) # CHANNEL-OP-REPLACE (2)
	#
	pdb.gimp_selection_grow(theImage, 2)
	pdb.gimp_edit_clear(newLayer)
	pdb.gimp_selection_invert(theImage)
	selectionBounds = pdb.gimp_selection_bounds(theImage)
	# Selection bounds
	x1 = selectionBounds [1]
	y1 = selectionBounds [2]
	x2 = selectionBounds [3]
	y2 = selectionBounds [4]	
	selectionWidth = x2 - x1
	selectionHeight = y2 - y1
	pdb.gimp_layer_resize(newLayer, selectionWidth, selectionHeight, (0 - x1), (0 - y1))
	pdb.gimp_item_set_visible(newLayer, False)
	# Show background
	howManyLayers = pdb.gimp_image_get_layers(theImage)[0]
	pdb.gimp_item_set_visible(theImage.layers[howManyLayers - 1], True)
	#
	pdb.gimp_selection_none(theImage)
	return (selectionWidth, selectionHeight)
	
register(
        "python_fu_outlined_text",
        "Outlined text",
        "Creates from a given sentence outlined text, in separate layers",
        "FDF",
        "Open source (BSD 3-clause license)",
        "November 1, 2013",
        "<Image>/Filters/Text/Outlined text",
		"RGB, RGB*",
        [
				(PF_STRING, "string", "Enter your text:", "Your text here"),
                (PF_FONT, "theFont", "Font", "Arial Heavy"),
                (PF_SPINNER, "fontSize", "Font Size", 20, (1, 1000, 1)),
                (PF_COLOR, "textColor", "Text Color", (255, 0, 0)),
				(PF_COLOR, "outlineColor", "Outline Color", (255, 255, 0)),
                (PF_SPINNER, "outlineThickness", "Outline Size", 4, (1, 20, 1)),
				(PF_TOGGLE, "delExistLayers", "Delete existing layers?", False)
        ],
        [],
        outlined_text
		)
		
main()
