#!/usr/bin/env python

# -----------------------------------------------------------------------------
#
#   parse_subtitles
#   ===============
#   Python-Fu script to create subtitle images for DVDs from "SubRip" files
#   (*.srt).  Can be used in batch mode; just adapt the files 'gimprc' and
#   the Linux shell script 'run_parse_subtitles'.  Extension to support other
#   filetypes should be easy.
#
#   Copyright (c) 2006-2007 Martin Zuther (http://www.mzuther.de/)
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# -----------------------------------------------------------------------------


import re,os
from gimpfu import *


# common data registered to the PDB
authors = 'Martin Zuther'
copyright = '2006-2007'
version = '0.19'
description_single = 'Creates a single subtitle for PAL and NTSC with given ' \
    + 'options.\n\nMostly called from batch mode or used for debugging...'
description = 'Creates subtitles for PAL and NTSC by reading a ' \
    'subtitle file.'


def print_message(message, indentation_level):
    """Indent message and output on console.

    Keyword arguments:
    message -- message to be printed
    indentation_level -- level of indentation for given message

    Return value:
    None

    """
    indentation_depth = 4  # Python style... ;)
    print ' ' * indentation_level * indentation_depth + message


def parse_subrip_file(input_file):
    """Parse a SubRip file and return its contents as a list of dictionaries.

    Keyword arguments:
    input_file -- path to a SubRip file (string)

    Return value:
    List of subtitles formatted as dictionaries:
        number -- number of subtitle (integer)
        start -- start time of the subtitle (string;  timecode 'HH:MM:SS,mmm')
        end -- end time of the subtitle (string;  timecode 'HH:MM:SS,mmm')
        lines -- contains the actual subtitle text (string; lines are
            separated by newline characters '\n')

    """
    # open SubRip file (*.srt)
    print_message('Parsing SubRip file...', 1)
    subrip_file = open(input_file, 'rU')
    try:
        unparsed_subtitles = subrip_file.read()
    finally:
        subrip_file.close()

    # file is openend in universal file mode, so there's only one kind of
    # newline character
    newline_character = '\n'

    # parse SubRip file (double newlines signal next subtitle)
    subtitles = unparsed_subtitles.split(newline_character * 2)
    print_message('Found %d subtitle(s).\n' % len(subtitles), 1)

    # create list for return value
    parsed_subtitles = []

    # parse subtitle information
    for subtitle in subtitles:
        number, time, lines = \
            subtitle.split(newline_character, 2)
        start, end = time.split(' --> ')

        # create dictionary for subtitle
        parsed_subtitles.append(
            {
                "number":int(number),
                "start":start,
                "end":end,
                "lines":lines
            }
        )

    return parsed_subtitles


def parse_subtitles(input_file, input_file_format, path, filename_prefix,
                    image_format, text_color, shadow_color, shadow_size,
                    safe_area, font_name, font_size, antialias,
                    display_images):
    """Create subtitles from a file with given options.

    Parse the given file to extract subtitle information.  Then create all
    subtitles using the function 'parse_single_subtitle'.

    Keyword arguments:
    input_file -- path to a subtitle file (string)
    input_file_format -- format of input file (string; in the moment, only
        the "SubRip" format is implemented)
    path -- path where the created subfiles are saved (string)
    filename_prefix -- prefix that is added to every subtitle picture (string)
    image_format -- dimensions of subtitle (string):
        "PAL":  720x576 pixels
        "NTSC": 720x486 pixels
    text_color -- text color for subtitle (tuple; (R, G, B))
    shadow_color -- color of shadow around subtitle text (tuple; (R, G, B))
    shadow_size -- size of shadow (integer; in pixels; 0 for no shadow)
    safe_area -- area at subtitle's bottom that is left empty (integer; in
        percent)
    font_name -- name of to be used (string)
    font_size -- size of font (integer; in pixels)
    antialias -- antialias the font (boolean; adds only one addtitional color)
    display_images -- display images when finished (boolean; turn off when
        the GIMP is called in batch mode with the option '--no-interface')

    Return value:
    None

    """
    print_message('', 1)
    print_message('"parse_subtitles" v' + version, 1)
    print_message('Copyright (c) ' + copyright + ' ' + authors, 1)
    print_message('', 1)
    print_message('This Python-Fu script is licensed under the GPL v2.', 1)
    print_message('', 1)

    # print parameters of function call
    for parameter in parameters_batch:
        print_message('%-22s %s' % (parameter[2], str(eval(parameter[1]))), 2)

    print_message('', 1)

    # check parameters for validity (most of the work has already been
    # done by the GIMP)
    if not os.path.exists(path):
        raise IOError('Path "' + path + '" does not exist.')
    elif not os.path.isdir(path):
        raise IOError('"' + path + '" is not a directory.')

    if not os.path.exists(input_file):
        raise IOError('File "' + input_file + '" does not exist.')
    elif not os.path.isfile(input_file):
        raise IOError('"' + input_file + '" is not a file.')

    # parse input file
    if input_file_format == 'SubRip':
        subtitles = parse_subrip_file(input_file)
    else:
        raise ValueError('file format unknown')

    # create subtitles
    for subtitle in subtitles:
        parse_single_subtitle(path, filename_prefix, image_format, text_color,
                              shadow_color, shadow_size, safe_area, font_name,
                              font_size, antialias, display_images,
                              subtitle['lines'], subtitle['start'])

    print_message('Done.', 1)
    print_message('', 1)


def parse_single_subtitle(path, filename_prefix, image_format, text_color,
                          shadow_color, shadow_size, safe_area, font_name,
                          font_size, antialias, display_images, lines,
                          timecode):
    """Create a single subtitle with the given options.

    Create a transparent subtitle with the given text in the given
    color and font.  Surround the text with a shadow and antialias if
    needed.  Save the subtitle as PNG file.

    Keyword arguments:
    path -- path where the created subfiles are saved (string)
    filename_prefix -- prefix that is added to every subtitle picture (string)
    image_format -- dimensions of subtitle (string):
        "PAL":  720x576 pixels
        "NTSC": 720x486 pixels
    text_color -- text color for subtitle (tuple; (R, G, B))
    shadow_color -- color of shadow around subtitle text (tuple; (R, G, B))
    shadow_size -- size of shadow (integer; in pixels; 0 for no shadow)
    safe_area -- area at subtitle's bottom that is left empty (integer; in
        percent)
    font_name -- name of to be used (string)
    font_size -- size of font (integer; in pixels)
    antialias -- antialias the font (boolean; adds only one addtitional color)
    display_images -- display images when finished (boolean; turn off when
        the GIMP is called in batch mode with the option '--no-interface')
    lines -- contains the actual subtitle text (string; lines are separated
        by newline characters '\n')
    timecode -- contains the subtitle's start position (string; timecode
        'HH:MM:SS,mmm')

    Return value:
    None

    """
    print_message('Creating subtitle "' + timecode + '"...', 1)

    # create mixed color between text and shadow color (for antialiasing)
    colormix_red = (text_color[0] + shadow_color[0]) / 2
    colormix_green = (text_color[1] + shadow_color[1]) / 2
    colormix_blue = (text_color[2] + shadow_color[2]) / 2

    # create custom palette
    palette = pdb.gimp_palette_new('parse-subtitles-temp')
    pdb.gimp_palette_add_entry(palette, 'Text color', text_color)
    pdb.gimp_palette_add_entry(palette, 'AntiAlias color',
                               (colormix_red, colormix_green, colormix_blue))
    pdb.gimp_palette_add_entry(palette, 'Shadow color', shadow_color)

    # initialize image format (PAL or NTSC)
    if image_format == 'PAL':
        image_width = 720
        image_height = 576
    elif image_format == 'NTSC':
        image_width = 720
        image_height = 486
    else:
        raise ValueError('image format unknown')

    # parse text lines
    lines = lines.split('\n')

    # create image and background layer
    image = pdb.gimp_image_new(image_width, image_height, RGB)
    background_layer = pdb.gimp_layer_new(image, image_width, image_height,
                                          RGB_IMAGE, 'Background', 100,
                                          NORMAL_MODE)

    # add background with alpha channel to image
    pdb.gimp_layer_add_alpha(background_layer)
    pdb.gimp_image_add_layer(image, background_layer, -1)

    # empty background layer
    pdb.gimp_selection_all(image)
    pdb.gimp_edit_clear(background_layer)
    pdb.gimp_selection_none(image)

    # initialize vertical start position
    vertical_position = image_height - (len(lines) * font_size * 1.25)
    vertical_position -= image_height * safe_area / 100.0

    for line in lines:
        vertical_position += font_size * 1.25
        parse_subtitles_create_layer(image, vertical_position, text_color,
                                  shadow_color, shadow_size, font_name,
                                  font_size, antialias, line)

    # merge all layers with background
    pdb.gimp_image_merge_visible_layers(image, CLIP_TO_IMAGE)

    # convert to maximum number of allowed colors (minus one for alpha channel)
    pdb.gimp_image_convert_indexed(image, FS_DITHER, CUSTOM_PALETTE, 3, 0, 1,
                                   'parse-subtitles-temp')

    # display new image (always respect a customer's wishes...)
    if display_images:
        display = pdb.gimp_display_new(image)
    else:
        display = -1

    # check and parse timecode
    if (len(timecode) != 12) or \
            (re.match('\d\d:\d\d:\d\d,\d\d\d', timecode) == None):
        raise ValueError('"' + timecode + '" is invalid timecode')

    timecode = timecode[0:2] + timecode[3:5] + timecode[6:8] + '_' \
        + timecode[9:12]

    # hand the filename over to the GIMP ...
    output_file = path + '/' + filename_prefix + timecode + '.png'
    pdb.gimp_image_set_filename(image, output_file)

    # ... and save the image
    print_message("Saving as '" + output_file + "'...", 1)
    drawable = pdb.gimp_image_active_drawable(image)
    png_save_parameters = [1, 9, 0, 0, 0, 1, 1, 0, 0]
    pdb.file_png_save2(image, drawable, output_file, output_file,
                       *png_save_parameters)

    # clear dirty flag
    pdb.gimp_image_clean_all(image)

    # delete custom palette
    pdb.gimp_palette_delete(palette)

    if not display_images:
        pdb.gimp_image_delete(image)

    print_message('', 1)


def parse_subtitles_create_layer(image, vertical_position, text_color,
                                 shadow_color, shadow_size, font_name,
                                 font_size, antialias, line):
    """Add a new layer with a single line of the subtitle's text.

    Keyword arguments:
    image -- the image to which the new layer should be attached (image)
    vertical_position -- vertical position where the line should be inserted
        (integer, in pixels from top)
    text_color -- text color for subtitle (tuple; (R, G, B))
    shadow_color -- color of shadow around subtitle text (tuple; (R, G, B))
    shadow_size -- size of shadow (integer; in pixels; 0 for no shadow)
    font_name -- name of to be used (string)
    font_size -- size of font (integer; in pixels)
    antialias -- antialias the font (boolean; adds only one addtitional color)
    line -- contains a single line of the subtitle's text (string)

    Return value:
    None

    """
    # initialize colors
    pdb.gimp_palette_set_foreground(text_color)
    pdb.gimp_palette_set_background(shadow_color)

    # initialize image and text sizes
    image_width = pdb.gimp_image_width(image)
    image_height = pdb.gimp_image_height(image)

    # create and center text layer
    layer_text = pdb.gimp_text_fontname(image, None, 0, vertical_position,
                                        line, -1, antialias, font_size,
                                        PIXELS, font_name)
    text_width = pdb.gimp_drawable_width(layer_text)
    horizontal_position = (image_width - text_width) / 2
    pdb.gimp_layer_set_offsets(layer_text, horizontal_position,
                               vertical_position)

    # create shadow
    if shadow_size > 0:
        # create new layer for shadow
        shadow_layer = pdb.gimp_layer_new(image, image_width, image_height,
                                          RGB_IMAGE, 'Shadow', 100,
                                          NORMAL_MODE)
        pdb.gimp_image_add_layer(image, shadow_layer, -1)
        pdb.gimp_image_lower_layer(image, shadow_layer)

        # remove random data from shadow layer ('gimp_edit_clear' sets
        # selected area to background color if no alpha layer exists)
        pdb.gimp_selection_all(image)
        pdb.gimp_edit_clear(shadow_layer)
        pdb.gimp_selection_none(image)

        # add alpha channel to shadow layer
        pdb.gimp_layer_add_alpha(shadow_layer)

        # create shadow
        pdb.gimp_selection_layer_alpha(layer_text)
        pdb.gimp_selection_grow(image, shadow_size)
        pdb.gimp_selection_invert(image)
        pdb.gimp_edit_clear(shadow_layer)
        pdb.gimp_selection_none(image)

        # merge both layers
        layer_text = pdb.gimp_image_merge_down(image, layer_text,
                                               CLIP_TO_IMAGE)
        pdb.gimp_drawable_set_name(layer_text, line)


# parameters for 'parse_subtitles'
parameters_batch = [
    (
        PF_FILE,
        'input_file',
        'Input file:',
        os.path.normpath(os.path.expanduser('~') + '/subtitles/subtitles.srt')
    ),
    (
        PF_RADIO,
        'input_file_format',
        'Input file format:',
        'SubRip',
        (
            ('SubRip file', 'SubRip'),
        )
    ),
    (
        PF_STRING,
        'path',
        'Output path:',
        os.path.normpath(os.path.expanduser('~') + '/subtitles')
    ),
    (
        PF_STRING,
        'filename_prefix',
        'Filename prefix:',
        'subtitle_'
    ),
    (
        PF_RADIO,
        'image_format',
        'Image format:',
        'NTSC',
        (
            ('PAL (720 x 576)', 'PAL'),
            ('NTSC (720 x 486)', 'NTSC')
        )
    ),
    (
        PF_COLOR,
        'text_color',
        'Text color:',
        (255, 255, 0)
    ),
    (
        PF_COLOR,
        'shadow_color',
        'Shadow color:',
        (0, 0, 0)
    ),
    (
        PF_ADJUSTMENT,
        'shadow_size',
        'Shadow size (pixels):',
        2,
        (0, 5, 1)
    ),
    (
        PF_ADJUSTMENT,
        'safe_area',
        'Safe area (percent):',
        10,
        (0, 25, 1)
    ),
    (
        PF_FONT,
        'font_name',
        'Font:',
        'Sans'
    ),
    (
        PF_ADJUSTMENT,
        'font_size',
        'Font size (pixels):',
        20,
        (8, 40, 1)
    ),
    (
        PF_TOGGLE,
        'antialias',
        'Antialias:',
        True
    ),
    (
        PF_TOGGLE,
        'display_images',
        'Display images:',
        False
    )
]


# parameters for 'parse_single_subtitle'
parameters_single = parameters_batch[2:]
parameters_single.extend([
    (
        PF_TEXT,
        'lines',
        'Text',
        'Short line\n"A long line with quotes and stuff..."\nAnother line!'
    ),
    (
        PF_TEXT,
        'timecode',
        'Subtitle timecode:',
        '01:02:03,045'
    )
])


# register plugin 'parse_subtitles' to PDB
register(
    'parse_subtitles',
    description,
    description,
    authors,
    authors,
    copyright,
    '<Toolbox>/Xtns/Languages/Python-Fu/Video/_Parse subtitles...',
    '',
    parameters_batch,
    [],
    parse_subtitles
)


# register plugin 'parse_single_subtitle' to PDB
register(
    'parse_subtitles_single',
    description_single,
    description_single,
    authors,
    authors,
    copyright,
    '<Toolbox>/Xtns/Languages/Python-Fu/Video/Parse _single subtitle...',
    '',
    parameters_single,
    [],
    parse_single_subtitle
)


# finish plugin registration
main()
