/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * PDS/EDR NASA image format loader plug-in by
 * Holger Isenberg, web@areo.info, http://areo.info
 *
 * The reading code is based on the pnm reading code by
 * Erik Nygren (nygren@mit.edu).
 *
 * This software uses source code from the IMQ decompressor by
 *   Malin Space Science Systems
 *   Mike Caplinger, MOC GDS Design Scientist
 *   SCCS @(#)readmoc.c  1.2 04/10/00
 *   Derived from the GDS readmsdp program.
 *   http://www.msss.com/mocsoft/
 *
 * PDS format is the image format used by NASA Planetary Data System
 * and by ESA Planetary Science Archives. PDS images from various
 * space missions are available at
 *   http://pds.jpl.nasa.gov
 *   http://www.rssd.esa.int/index.php?project=PSA
 *
 * PDS-files usually have the filename-suffix ".img" or ".imq" if compressed.
 * Note, that this suffix is also used for other non-PDS file-formats.
 *
 * debug:
 * # G_MESSAGES_DEBUG=LibGimp GIMP_PLUGIN_DEBUG=pds,pid,run gimp
 * # gdb src/pds $(ps x|grep pds|grep -v grep|cut -d" " -f1)
 * see http://developer.gimp.org/debug-plug-ins.txt
 *
 * Changes: see ChangeLog
 *
 */

#include "headers.h"

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <libgimp/gimpcompat.h>
#include "plugin-intl.h"

#include <errno.h>
#include <fcntl.h>
#include <ctype.h>

/*
#include <limits.h>
#include <sys/stat.h>
#include <math.h>
#include <libintl.h>
*/

#ifdef HAVE_UNISTD_H
char console=1;
#else
/* Non-Unix */
char console=0;
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

#define DATA_KEY_VALS PLUGIN_NAME


extern int errno;
extern unsigned char* decode_imq(char* f_in);		/* imported from readmoc.c */


typedef struct _PDSScanner {
	int fd[6];			/* The file descriptor of the gray,red,green,blue file being read */
	int rgb;			/* 0: grayscale only */
	int cur_fd;			/* currently used fd */
	char cur;			/* The current character in the input stream */
	int eof;				/* Have we reached end of file? */
	char *inbuf;			/* Input buffer - initially 0 */
	int inbufsize;		/* Size of input buffer */
	int inbufvalidsize;	/* Size of input buffer with valid data */
	int inbufpos;			/* Position in input buffer */
} PDSScanner;

typedef struct _PDSInfo {
	int line_samples, lines;	/* The size of the image */
	int record_bytes;		/* blocksize */
	int image_ptr;			/* block where image-data begins */
	char *image_fname;		/* filename, if separate img file */
	int sample_type;		/* 0: LSB-first, 1: MSB-first, 2: SUN_REAL(4 bytes) */
	int sample_bytes;		/* 1, 2 or 4 bytes/sample */
	int np;					/* number of bitplanes (bands) */
	int sample_bits;
	char sample_bit_mask[40];
	int line_prefix_bytes;
	int line_suffix_bytes;
	char filter[256];
	char spacecraft[256];
	char instrument[256];
	char encoding_type[40];
	int viking_cam;			/* Viking Camera:  L1,C1, L1,C2, L2,C1, L2,C2 */
	int viking_adj;			/* Viking Color Adjustment */
	float viking_colmul[4][3];	/* Viking Color Correction values: C'=x*C */
	float viking_colsub[4];		/* B'=B-x*R */
	char minmax;
	long minval;
	long maxval;			/* For ascii image files, the max value which we need to normalize to */
	int bands;
	char storage_type[256];
	
	/* QUBE */
	int qube_rgb;			/* load into rgb-channels */
	int qube_axes;			/* number of qube-axes */
	int qube_bands;			/* number of bands, if qube-image */
	int qube_filters[100];		/* used filters */
	int qube_band_nr[100];		/* filter numbers */
	char* qube_band_wl[100];	/* filter center wavelength */
	int suffix_items_line;
	int suffix_items_band;
	int suffix_bytes;
	unsigned char* imgbuf;		/* IMG-data in memory instead of file */
	long imgbuf_pos;
	jmp_buf jmpbuf;			/* Where to jump to on an error loading */
} PDSInfo;

#define BUFLEN 10240


/* Declare some local functions.
 */
static void query();
static void run(const gchar* name, gint nparams, const GimpParam* param,
			gint* nreturn_vals, GimpParam** return_vals);
static gint32 load_image (char* filename, GimpRunMode run_mode);
static void pds_load (PDSScanner* scan, PDSInfo* info,
			GimpPixelRgn* pixel_rgn, int mode);
/* mode -1: load as grayscale or Viking-RGB
 * mode n: load grayscale into RGB-channel n (0-2)
 */

static void pdsscanner_destroy(PDSScanner * s);
static void pdsscanner_getchar(PDSScanner * s);
static char pdsscanner_getint(char* buf, char* key, int* value);
static char pdsscanner_getlong(char* buf, char* key, long* value);
static char pdsscanner_getstr(char* buf, char* key, char* str);
static char pdsscanner_getstrtuple(char* buf, char* key, char* str[]);

static PDSScanner * pdsscanner_create  (int fd[], int rgb);

#define pdsscanner_eof(s) ((s)->eof)
static int pdsscanner_fd(PDSScanner* s, int i)
{
	s->cur_fd=i;
	return s->fd[i];
}

/* Checks for a fatal error */
#define CHECK_FOR_ERROR(predicate, jmpbuf, errmsg) if ((predicate)) { gimp_message(errmsg); getchar(); longjmp((jmpbuf),1); }

GimpPlugInInfo PLUG_IN_INFO =
{
	NULL,	/* init_proc */
	NULL,	/* quit_proc */
	query,	/* query_proc */
	run,	/* run_proc */
};

#define QUBELOAD_NOT 0
#define QUBELOAD_RGB 1
#define QUBELOAD_LAYERS 2
#define QUBELOAD_IMGS 3

typedef struct {
	guint qube_loadmode;	/* QUBELOAD_NOT, QUBELOAD_LAYERS, QUBELOAD_RGB,  QUBELOAD_IMGS */
	guint qube_load[3];	/* which band to load to the RGB channels */
	guint qube_stretch;	/* FALSE: 8 bit = 16bit>>8, TRUE:  linear contrast stretch */
} PDSLoadVals;

static PDSLoadVals plvals = {
	QUBELOAD_NOT,
	{2,1,0},
	FALSE
};

/* globals */
GtkWidget *frame_color;

MAIN ()

static void
query ()
{
	static GimpParamDef load_args[] =
	{
		{ GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
		{ GIMP_PDB_STRING, "filename", "The name of the file to load" },
		{ GIMP_PDB_STRING, "raw_filename", "The name of the file to load" },
		{ GIMP_PDB_INT32, "loadmode", "0: no QUBE, 1: load as RGB layered, 2: load as gray layered, 3: load single images" },
		{ GIMP_PDB_INT32, "red_band", "which QUBE-band to load as red (0-9)" },
		{ GIMP_PDB_INT32, "green_band", "which QUBE-band to load as green (0-9)" },
		{ GIMP_PDB_INT32, "blue_band", "which QUBE-band to load as blue (0-9)" },
		{ GIMP_PDB_INT32, "conv16to8", "16bit to 8 conv. 0: remove lower 8bits, 1: linear contrast stretch" }
	};
	static GimpParamDef load_return_vals[] =
	{
		{ GIMP_PDB_IMAGE, "image", "Output image" },
	};
	static int nload_args = sizeof (load_args) / sizeof (load_args[0]);
	static int nload_return_vals = sizeof (load_return_vals) / sizeof (load_return_vals[0]);
	static char str[256];

	snprintf(str, 256, "Loads files of the NASA PDS image format (.img, .imq, .qub, .blu, .red, .grn, ...).\n\
Version %s\nDownload newest version from http://registry.gimp.org", PLUGIN_VERSION);
	gimp_install_procedure (PLUGIN_NAME,
				/* text */ str,
				/* additional info */ "PDS-Plugin uses source code of the IMQ decompressor by \
Malin Space Science Systems (MSSS),\n\
http://www.msss.com/mocsoft/\n\n\
Report errors or questions about PDS-plugin to author:\n",
				/* author */ "Holger Isenberg, web@areo.info, http://areo.info",
				/* copyright */ "Free for everyone's use.\n\
You may not remove or alter any proprietary notices contained in the software \
especially the notices by MSSS and you may not charge any third party for the software.",
				/* date */ "first release 1999",
				/* name */ "<Load>/PDS",
				NULL,
				GIMP_PLUGIN,
				nload_args, nload_return_vals,
				load_args, load_return_vals);

	gimp_register_magic_load_handler (PLUGIN_NAME, "img,imq,pds,edr,bb1,bb2,bb3,bb4,ir1,ir2,ir3,n07,n15,sur,sun,red,grn,blu,sgr,vio,qub,lbl", "", "0,string,PDS_VERSION_ID");
}

static void run (
	   const gchar    *name,
       gint      nparams,
       const GimpParam  *param,
       gint     *nreturn_vals,
       GimpParam **return_vals
	   )
{
	static GimpParam values[2];
	GimpRunMode run_mode;
	/* GStatusType status = STATUS_SUCCESS; */
	gint32 image_ID;

	/*  Initialize i18n support  */
	#ifdef GETTEXT_PACKAGE
	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	#endif
	textdomain (GETTEXT_PACKAGE);
	#endif

	run_mode = (GimpRunMode)param[0].data.d_int32;

	*return_vals = values;
	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
	values[1].type = GIMP_PDB_IMAGE;
	values[1].data.d_image = -1;
	image_ID = -1;

	if (strcmp (name, PLUGIN_NAME) == 0) {
		switch (run_mode) {
		    case GIMP_RUN_NONINTERACTIVE:
			    switch(nparams) {
				case 8: plvals.qube_stretch = param[7].data.d_int32;
				case 7: plvals.qube_load[2] = param[6].data.d_int32;
				case 6: plvals.qube_load[1] = param[5].data.d_int32;
				case 5: plvals.qube_load[0] = param[4].data.d_int32;
				case 4: plvals.qube_loadmode = param[3].data.d_int32;
				default: ;
			    }
			    image_ID = load_image (param[1].data.d_string, run_mode);
			    break;
		    case GIMP_RUN_INTERACTIVE:
			    /*  Possibly retrieve data  */
			    gimp_get_data (DATA_KEY_VALS, &plvals);
			    image_ID = load_image (param[1].data.d_string, run_mode);
			    gimp_set_data (DATA_KEY_VALS, &plvals, sizeof (plvals));
			    break;
		    case GIMP_RUN_WITH_LAST_VALS:
			    /*  Possibly retrieve data  */
			    gimp_get_data (DATA_KEY_VALS, &plvals);
			    image_ID = load_image (param[1].data.d_string, run_mode);
			    break;
		    default:
			    break;
		}

		if (image_ID != -1)
		{
			values[0].data.d_status = GIMP_PDB_SUCCESS;
			values[1].data.d_image = image_ID;
		}
		else
		{
			values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
		}
		*nreturn_vals = 2;
	}
}

void qube_menuhandler (GtkWidget* widget, gpointer func_data)
{
	int data = GPOINTER_TO_INT(func_data);
	plvals.qube_load[data/1000] = data % 1000;
}


void
dialog_color_update(GtkWidget *widget, gpointer data)
{
	gimp_radio_button_update(widget, data);
	if(plvals.qube_loadmode==QUBELOAD_RGB)
		gtk_widget_set_sensitive(frame_color, TRUE);
	else
		gtk_widget_set_sensitive(frame_color, FALSE);
}

static gboolean
load_dialog (PDSInfo* pdsinfo)
{
	GtkWidget *dialog;
	GtkWidget *main_vbox;
	GtkWidget *hbox;
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *table;
	GtkWidget *item;
	GtkWidget* qub_selector[3];
	GtkWidget* qub_bands_menu[3];
	int i,j;
	gboolean run;

	gimp_ui_init (PLUGIN_NAME, FALSE);
	dialog = gimp_dialog_new (_("Load NASA-PDS"), PLUGIN_NAME,
				NULL, GTK_DIALOG_MODAL,
				gimp_standard_help_func, "filters/pds.html",
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				NULL);
/*
				  gimp_standard_help_func, "filters/pds.html",
				  GTK_WIN_POS_MOUSE,
				  FALSE, TRUE, FALSE,
				  _("OK"), gtk_main_quit,
				  NULL, NULL, NULL, TRUE, FALSE,
				  NULL, NULL,
				  NULL, 1, NULL, FALSE, TRUE,
				  NULL);
*/
	g_signal_connect (G_OBJECT(dialog), "destroy", G_CALLBACK (gtk_main_quit), NULL);

	main_vbox = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 6);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), main_vbox,
			    FALSE, FALSE, 0);
	gtk_widget_show (main_vbox);

	hbox = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);

	frame = gimp_radio_group_new2 (TRUE, _("Load as"),
				       (GCallback)dialog_color_update,
				       &plvals.qube_loadmode, (gpointer) plvals.qube_loadmode,
				       _("Color layered"),       (gpointer) QUBELOAD_RGB, NULL,
				       _("Gray layered"),      (gpointer) QUBELOAD_LAYERS, NULL,
				       _("Single Images"),     (gpointer) QUBELOAD_IMGS, NULL,
				       NULL);
	gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
	gtk_widget_show (frame);

	frame_color = gtk_frame_new (_("Assign colors"));
	gtk_frame_set_shadow_type (GTK_FRAME (frame_color), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start (GTK_BOX (hbox), frame_color, TRUE, TRUE, 0);
	vbox = gtk_vbox_new (FALSE, 3);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 3);
	gtk_container_add (GTK_CONTAINER (frame_color), vbox);
	table = gtk_table_new (3, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 2);
	gtk_table_set_col_spacings (GTK_TABLE (table), 4);
	gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
	gtk_widget_show (table);

	if(plvals.qube_load[2] > (guint) pdsinfo->qube_bands
	   || plvals.qube_load[1] > (guint) pdsinfo->qube_bands
	   || plvals.qube_load[0] > (guint) pdsinfo->qube_bands) {
		plvals.qube_load[0]=2;
		plvals.qube_load[1]=1;
		plvals.qube_load[2]=0;
	}
/*
	if(pdsinfo->qube_bands==10) {
		plvals.qube_load[0]=8;
		plvals.qube_load[1]=4;
		plvals.qube_load[2]=2;
	}
*/
	for(i=0; i<3; i++) {
		qub_bands_menu[i] = gtk_menu_new();
		for(j=0; j<pdsinfo->qube_bands; j++) {
			item = gtk_menu_item_new_with_label(pdsinfo->qube_band_wl[j]);
			gtk_menu_shell_append((GtkMenuShell *)(qub_bands_menu[i]), item);
			g_signal_connect(G_OBJECT(item), "activate",
					    G_CALLBACK(qube_menuhandler),
					    GINT_TO_POINTER(i*1000+j));
		}
		qub_selector[i] = gtk_option_menu_new();
		gtk_option_menu_set_menu(GTK_OPTION_MENU(qub_selector[i]), qub_bands_menu[i]);
		gtk_option_menu_set_history(GTK_OPTION_MENU(qub_selector[i]), plvals.qube_load[i]);
		switch(i) {
		    case 0:
			    gimp_table_attach_aligned (GTK_TABLE (table), 0, i,_("Red:"), 1.0, 0.5, qub_selector[i], 1, TRUE);
			    break;
		    case 1:
			    gimp_table_attach_aligned (GTK_TABLE (table), 0, i,_("Green:"), 1.0, 0.5, qub_selector[i], 1, TRUE);
			    break;
		    case 2:
			    gimp_table_attach_aligned (GTK_TABLE (table), 0, i,_("Blue:"), 1.0, 0.5, qub_selector[i], 1, TRUE);
			    break;
		}
		gtk_widget_show_all(qub_selector[i]);
	}
	gtk_widget_show (vbox);
	gtk_widget_show (frame_color);

	hbox = gtk_hbox_new (TRUE, 6);
	gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);

	frame = gimp_radio_group_new2 (TRUE, _("16bit to 8bit conversion"),
 		  (GCallback)gimp_radio_button_update,
     	  &plvals.qube_stretch, (gpointer) &plvals.qube_stretch,
     	  _("Ignore lower 8 bits"), (gpointer) FALSE, NULL,
     	  _("Linear contrast stretch (disables calibration)"), (gpointer) TRUE, NULL,
     	  NULL);
	gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
	gtk_widget_show (frame);
	gtk_widget_set_sensitive(frame, pdsinfo->sample_bytes>1);

	gtk_widget_show (dialog);
	run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);
	gtk_widget_destroy (dialog);

	return run;
}


static gint32
load_image (char *filename, GimpRunMode run_mode)
{
	GimpPixelRgn pixel_rgn[100];
	gint32 image_ID = 0;
	gint32 layer_ID;
	GimpDrawable *drawable;
	int fd[6]={0,0,0,0,0,0};		/* File descriptors for gray or RGB +IR */
	char *temp;
	char* file_ir3 = NULL;
	char* file_red = NULL;
	char* file_grn = NULL;
	char* file_blu = NULL;
	char buf[BUFLEN];		/* buffer for random things like scanning */
	char buf2[BUFLEN];		/* buffer for random things like scanning */
	char* tuplebuf[100];
	PDSInfo *pdsinfo;
	char sample_type[256];
	PDSScanner *scan;
	int i,j,f;
	int rgb=1;
	off_t seek_offs=0;
	off_t seek_l=0;
	off_t l;
	char* imagename;
	char* imgtype[3]={
		"EDR gray", "EDR VL", "EDR VO"
	};
	char objname[256];
	objname[0] = '\0';
	
	/* check for LBL/IMG file combination and select LBL then */
	if ( !casecmp(filename+strlen(filename)-4, ".img") ) {
		strncpy(buf, filename, strlen(filename)-4);
		strcpy(buf2, buf);
		strcat(buf2, ".lbl");
		if(g_file_test(buf2, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR))
			filename = strdup(buf2);
		else {
			strcpy(buf2, buf);
			strcat(buf2, ".LBL");
			if(g_file_test(buf2, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR))
				filename = strdup(buf2);
		}
	}
			
	if( !casecmp(filename+strlen(filename)-4, ".red")
	    || !casecmp(filename+strlen(filename)-4, ".grn")
	    || !casecmp(filename+strlen(filename)-4, ".blu"))
		rgb=1;
	else if( !casecmp(filename+strlen(filename)-4, ".vio")
	    || !casecmp(filename+strlen(filename)-4, ".sgr"))
		rgb=2;
	else rgb=0;

	/* allocate the necessary structures */
	pdsinfo = (PDSInfo *) g_malloc(sizeof(PDSInfo));
	if (pdsinfo == NULL)
		return -1;

	if((rgb != 0) && !casecmp(filename+strlen(filename)-4, ".qub")) {
		fd[0] = open(filename, O_RDONLY|O_BINARY);
		fd[1] = open(filename, O_RDONLY|O_BINARY);
		fd[2] = open(filename, O_RDONLY|O_BINARY);
	} else if (rgb != 0) {
		file_ir3 = (char *)g_malloc (strlen (filename) + 11);
		file_red = (char *)g_malloc (strlen (filename) + 11);
		file_grn = (char *)g_malloc (strlen (filename) + 11);
		file_blu = (char *)g_malloc (strlen (filename) + 11);
		if (file_ir3 == NULL || file_red == NULL || file_grn == NULL || file_blu == NULL)
			return -1;

		strcpy(file_red, filename);
		file_red[strlen(filename)-4]='\0';
		strcat(file_red, ".red");
		strcpy(file_grn, filename);
		file_grn[strlen(filename)-4]='\0';
		strcpy(file_blu, filename);
		file_blu[strlen(filename)-4]='\0';
		strcpy(file_ir3, filename);
		file_ir3[strlen(filename)-7]='\0';
		if(console) fprintf(stderr, "ir3: (%s)\n", file_ir3);
		snprintf(file_ir3, strlen(filename)+11, "%s%03d.ir3", file_ir3, atoi(filename+strlen(filename)-7)+1);
		if(rgb==1) {
		  strcat(file_grn, ".grn");
		  strcat(file_blu, ".blu");
		} else {
		  strcat(file_grn, ".sgr");
		  strcat(file_blu, ".vio");
		}
		if ((fd[0] = open(file_red, O_RDONLY|O_BINARY)) == -1
		    || (fd[1] = open(file_grn, O_RDONLY|O_BINARY)) == -1
		    || (fd[2] = open(file_blu, O_RDONLY|O_BINARY)) == -1) {
			rgb=0;
			pdsinfo->viking_adj=0;
			pdsinfo->viking_cam=-1;
		}
		/* Viking color adjustment with IR disabled
		if((rgb==1) && ((fd[3] = open(file_ir3, O_RDONLY|O_BINARY)) !=-1) && !strcmp(filename, file_blu))
			pdsinfo->viking_adj=1;
		else
			pdsinfo->viking_adj=0;
		*/
	} else {
		pdsinfo->viking_cam=-1;
		pdsinfo->viking_adj=0;
	}

	if (rgb==0) {
		/* open the file */
		fd[0] = open(filename, O_RDONLY|O_BINARY);
		if (fd[0] == -1)
		{
			/* gimp_message("pds filter: can't open file\n"); */
			return -1;
		}
	}

	scan = NULL;
	/* set error handling */
	if (setjmp (pdsinfo->jmpbuf))
	{
		/* If we get here, we had a problem reading the file */
		if (scan)
			pdsscanner_destroy (scan);
		for(f=0; f<4; f++)
			if(fd[f]) close (fd[f]);
		g_free (pdsinfo);
		return -1;
	}

	if (!(scan = pdsscanner_create(fd, rgb)))
		longjmp(pdsinfo->jmpbuf,1);

	pdsinfo->record_bytes=0;
	pdsinfo->image_ptr=0;
	pdsinfo->image_fname = strdup("");
	pdsinfo->np=1;
	pdsinfo->lines=0;
	pdsinfo->line_samples=0;
	pdsinfo->qube_axes=0;
	pdsinfo->qube_rgb=0;
	pdsinfo->qube_bands=0;
	pdsinfo->line_prefix_bytes=0;
	pdsinfo->line_suffix_bytes=0;
	pdsinfo->suffix_items_line=0;
	pdsinfo->suffix_items_band=0;
	pdsinfo->suffix_bytes=0;
	pdsinfo->sample_bits=8;
	strcpy(pdsinfo->sample_bit_mask, "");
	pdsinfo->sample_type=0;
	pdsinfo->sample_bytes=1;
	pdsinfo->minmax=0;
	pdsinfo->minval=0L;
	pdsinfo->maxval=255L;
	strcpy(pdsinfo->filter, "");
	strcpy(pdsinfo->spacecraft, "");
	strcpy(pdsinfo->instrument, "");
	strcpy(pdsinfo->encoding_type, "");
	strcpy(pdsinfo->storage_type, "");
	pdsinfo->viking_cam=-1;
	pdsinfo->viking_adj=0;
	pdsinfo->imgbuf=NULL;
	pdsinfo->imgbuf_pos=0;
	for(i=0; i<100; i++) pdsinfo->qube_band_wl[i] = NULL;
	
	do {
		do pdsscanner_getchar(scan); while (isspace(scan->cur) || scan->cur=='\0');
		i=0;
		do {
			do {
				buf[i++]=scan->cur;
				do pdsscanner_getchar(scan); while (scan->cur=='\r');
			} while (scan->cur !='\n' && scan->cur!='\0');
		} while (buf[i-1] == ',');
		buf[i]='\0';
		if(console) fprintf(stderr, "%s\n", buf);

		if(pdsscanner_getint(buf, "RECORD_BYTES", &(pdsinfo->record_bytes))) continue;
		if(pdsinfo->image_ptr == 0 && pdsscanner_getstrtuple(buf, "^IMAGE", tuplebuf)) {
			if(tuplebuf[0][0] == '\"') {
				gchar* path = g_path_get_dirname(filename);
				pdsinfo->image_fname = (char*) g_malloc(strlen(path) + 1 + strlen(tuplebuf[0]));
				strcpy(pdsinfo->image_fname, path);
				strcat(pdsinfo->image_fname, G_DIR_SEPARATOR_S);
				strncat(pdsinfo->image_fname, tuplebuf[0]+1, strlen(tuplebuf[0])-2);
				pdsinfo->image_ptr = 1;
				continue;
			}
		} else {
			if(pdsscanner_getint(buf, "^IMAGE", &(pdsinfo->image_ptr))) continue;
		}
		if(pdsscanner_getstr(buf, "OBJECT", objname)) continue;
		if(objname[0] == '\0' || !casecmp(objname, "IMAGE")) {
			if(pdsscanner_getint(buf, "LINES", &(pdsinfo->lines))) continue;
			if(pdsscanner_getint(buf, "LINE_SAMPLES", &(pdsinfo->line_samples))) continue;
			if(pdsscanner_getint(buf, "LINE_PREFIX_BYTES", &(pdsinfo->line_prefix_bytes))) continue;
			if(pdsscanner_getint(buf, "LINE_SUFFIX_BYTES", &(pdsinfo->line_suffix_bytes))) continue;
			if(pdsscanner_getstr(buf, "SAMPLE_TYPE", sample_type)) continue;
			if(pdsscanner_getstr(buf, "SAMPLE_BIT_MASK", pdsinfo->sample_bit_mask)) continue;
			if(pdsscanner_getint(buf, "SAMPLE_BITS", &(pdsinfo->sample_bits))) continue;
			if(pdsscanner_getlong(buf, "MAXIMUM", &(pdsinfo->maxval))) {
				pdsinfo->minmax = 1;
				continue;
			}
		}
		if(pdsscanner_getstr(buf, "FILTER_NAME", pdsinfo->filter)) continue;
		if(pdsscanner_getstr(buf, "SPACECRAFT_NAME", pdsinfo->spacecraft)) continue;
		if(pdsscanner_getstr(buf, "INSTRUMENT_NAME", pdsinfo->instrument)) continue;
		if(pdsscanner_getstr(buf, "ENCODING_TYPE", pdsinfo->encoding_type)) continue;

		/* QUBE */
		if(pdsscanner_getint(buf, "^SPECTRAL_QUBE", &(pdsinfo->image_ptr))) continue;
		if(pdsscanner_getint(buf, "AXES", &(pdsinfo->qube_axes))) continue;
		if(pdsscanner_getstrtuple(buf, "CORE_ITEMS", tuplebuf)) {
			pdsinfo->line_samples = atoi(tuplebuf[0]);
			pdsinfo->lines = atoi(tuplebuf[1]);
			pdsinfo->qube_bands = atoi(tuplebuf[2]);
		}
		if(pdsscanner_getstrtuple(buf, "SUFFIX_ITEMS", tuplebuf)) {
			pdsinfo->suffix_items_line = atoi(tuplebuf[0]);
			pdsinfo->suffix_items_band = atoi(tuplebuf[1]);
		}
		if(pdsscanner_getstr(buf, "CORE_ITEM_TYPE", sample_type)) continue;
		if(pdsscanner_getstr(buf, "DETECTOR_ID", buf2)) {
			if(!casecmp(buf2, "vis"))
				pdsinfo->qube_rgb=1;
			continue;
		}
		if(pdsscanner_getint(buf, "CORE_ITEM_BYTES", &(pdsinfo->sample_bytes))) continue;
		if(pdsscanner_getint(buf, "SUFFIX_BYTES", &(pdsinfo->suffix_bytes))) continue;
		if(pdsscanner_getstrtuple(buf, "BAND_BIN_FILTER_NUMBER", tuplebuf))
			for(i=0; i<pdsinfo->qube_bands; i++)
				pdsinfo->qube_filters[i] = atoi(tuplebuf[i]);
		if(pdsscanner_getstrtuple(buf, "BAND_BIN_BAND_NUMBER", tuplebuf))
			for(i=0; i<pdsinfo->qube_bands; i++)
				pdsinfo->qube_band_nr[i] = atoi(tuplebuf[i]);
		if(pdsscanner_getstrtuple(buf, "BAND_BIN_CENTER", tuplebuf))
			for(i=0; i<pdsinfo->qube_bands; i++)
				pdsinfo->qube_band_wl[i] = tuplebuf[i];
		
		/* MSL RGB */
		if(pdsscanner_getint(buf, "BANDS", &(pdsinfo->bands))) continue;
		if(pdsscanner_getstr(buf, "INSTRUMENT_HOST_ID", pdsinfo->spacecraft)) continue;
		if(pdsscanner_getstr(buf, "BAND_STORAGE_TYPE", pdsinfo->storage_type)) continue;
		
	} while (strcmp(buf, "END") && strncmp(buf, "END ", 4));
	
	if(pdsinfo->qube_bands>1 && run_mode==GIMP_RUN_INTERACTIVE) {
		if(plvals.qube_loadmode==QUBELOAD_NOT)
			plvals.qube_loadmode=QUBELOAD_RGB;
		if(!load_dialog(pdsinfo)) {
			return -1;
		}
	} else {
		pdsinfo->qube_rgb=0;
	}
	
	// MSL RGB
	if(pdsinfo->bands == 3 && !strcmp(pdsinfo->spacecraft, "MSL")) {
		rgb=1;
	}

	if(plvals.qube_loadmode==QUBELOAD_RGB) rgb=1;
	pdsinfo->np=1+2*(rgb>0);

	temp = (char *)g_malloc (strlen (filename) + 50);
	if(temp == NULL)
		return -1;
	snprintf (temp, strlen(filename)+50, "Loading %s %s", imgtype[rgb], filename);
	gimp_progress_init (temp);
	g_free (temp);

	if(strstr(pdsinfo->encoding_type, "MOC")>0) {
		strcpy(pdsinfo->encoding_type, "");
		for(f=0; f<4; f++)
			if(fd[f]) close (fd[f]);
		pdsinfo->imgbuf=decode_imq(filename);
		fd[0] = 0;
		if (pdsinfo->imgbuf==NULL)
		{
			/* gimp_message("pds filter: can't allocate buffer for moc-decompression\n"); */
			return -1;
		}
	}

	if(strlen(pdsinfo->image_fname)>0) {
		close(fd[0]);
		fd[0] = open(pdsinfo->image_fname, O_RDONLY|O_BINARY);
		if(rgb) {
			fd[1] = open(pdsinfo->image_fname, O_RDONLY|O_BINARY);
			fd[2] = open(pdsinfo->image_fname, O_RDONLY|O_BINARY);
		}
		if(!(rgb && fd[0]>0 && fd[1]>0 && fd[2]>0) && !(!rgb && fd[0])) {
			snprintf(temp, 256, "pds filter: failed to open separate image file \"%s\"", pdsinfo->image_fname);
			CHECK_FOR_ERROR(-1, pdsinfo->jmpbuf, temp);
		}
		scan = pdsscanner_create(fd, rgb);
	}

	/* seek forward to image data start */
	for(f=0;f<(pdsinfo->np+pdsinfo->viking_adj); f++) {
		seek_offs=0;
		if(fd[f]) {
			if(pdsinfo->qube_bands>1)
				seek_offs+=(pdsinfo->sample_bytes*pdsinfo->lines*pdsinfo->line_samples
					+pdsinfo->suffix_bytes
					*(pdsinfo->suffix_items_band*(pdsinfo->line_samples+pdsinfo->suffix_items_line)
					+pdsinfo->lines*pdsinfo->suffix_items_line))
					*plvals.qube_load[f];
			else if(strlen(pdsinfo->image_fname) > 0)
				seek_l=f*(pdsinfo->sample_bytes*pdsinfo->lines*pdsinfo->line_samples + pdsinfo->suffix_bytes);
			else
				seek_l=pdsinfo->record_bytes*(pdsinfo->image_ptr-1)+seek_offs;
			l=lseek(fd[f], seek_l, SEEK_SET);
			if( l != seek_l ) {
				printf("lseek failed: f=%d, seek_l=%d, errno %d: %s\n", f, (int)seek_l, errno, strerror(errno));
				CHECK_FOR_ERROR(-1, pdsinfo->jmpbuf, "pds filter: premature end of file.");
			}
		}
	}
	
	/* Create a new image of the proper size and associate the filename with it.
	 */
	if(strstr(pdsinfo->spacecraft, "VIKING_LANDER_")) {
		pdsinfo->viking_cam=(pdsinfo->spacecraft[strlen("VIKING_LANDER_")]-'1')*2;
		if(strstr(pdsinfo->instrument, "CAMERA_"))
			pdsinfo->viking_cam+=(pdsinfo->instrument[strlen("CAMERA_")]-'1');
	}
	//pdsinfo->line_samples+=pdsinfo->line_suffix_bytes;
	//pdsinfo->maxval=(1<<pdsinfo->sample_bits)-1;
	if(!strcmp(pdsinfo->sample_bit_mask, "2#11111100")
	   || !strcmp(pdsinfo->sample_bit_mask, "2#0000000011111111"))
		pdsinfo->maxval=255;
	else if(pdsinfo->maxval<255) pdsinfo->maxval=255;
	//else if (pdsinfo->maxval<4095) pdsinfo->maxval=4095;
	if(strstr(sample_type, "MSB_") || strstr(sample_type, "SUN_INTEGER") || strstr(sample_type, "SUN_UNSIGNED_INTEGER")) pdsinfo->sample_type=1;
	else if(strstr(sample_type, "SUN_REAL")) pdsinfo->sample_type=2;

	if(console) {
		fprintf(stderr, "/* GIMP-PDS: %dx%d %dbit %s [%ld..%ld] %s",
			pdsinfo->line_samples, pdsinfo->lines,
			pdsinfo->sample_bits, sample_type,
			pdsinfo->minval, pdsinfo->maxval,
			pdsinfo->np>=3?"RGB":"grayscale");
		if(pdsinfo->viking_cam>=0)
			fprintf(stderr, ", viking camera %d", pdsinfo->viking_cam);
		if(pdsinfo->viking_adj)
			fprintf(stderr, " with ir3 '%s'", file_ir3);
		if(pdsinfo->qube_bands > 0)
			fprintf(stderr, ", %d bands", pdsinfo->qube_bands);
		//if(pdsinfo->line_suffix_bytes>0)
		//	fprintf(stderr, ", %d line suffix bytes", pdsinfo->line_suffix_bytes);
		fprintf(stderr, " */\n");
	}

	if (plvals.qube_loadmode != QUBELOAD_IMGS) {
		/* create GIMP image */
		image_ID = gimp_image_new (pdsinfo->line_samples, pdsinfo->lines, (pdsinfo->np >= 3 || plvals.qube_loadmode==QUBELOAD_RGB) ? GIMP_RGB : GIMP_GRAY);
		if(rgb || (strlen(pdsinfo->filter)==0) ) {
			imagename=filename;
			if(console)
				fprintf(stderr, "/* GIMP-PDS: loading rgb... */\n");
		} else {
			imagename=(char *)g_malloc(strlen(filename)+strlen(pdsinfo->filter)+3);
			CHECK_FOR_ERROR(imagename==NULL, pdsinfo->jmpbuf, "cannot allocate name-buffer\n");
			snprintf(imagename, strlen(filename)+strlen(pdsinfo->filter)+3, "%s %s", filename, pdsinfo->filter);
			if(console)
				fprintf(stderr, "/* GIMP-PDS: loading grayscale... */\n");
		}
		gimp_image_set_filename (image_ID, imagename);
		scan->rgb=(rgb>0);
	}

	for(i=0; i<1 || ((i<pdsinfo->qube_bands) && plvals.qube_loadmode); i++) {
		strcpy(buf, "");
		if(pdsinfo->qube_band_wl[i] == NULL) {
			pdsinfo->qube_band_wl[i] = strdup("xxx");
			sprintf(pdsinfo->qube_band_wl[i], "%d", i+1);
		}
		if(plvals.qube_loadmode) {
				for(j=0; j<(int)strlen(pdsinfo->qube_band_wl[i]); j++)
				if(isdigit(pdsinfo->qube_band_wl[i][j]))
					strncat(buf, pdsinfo->qube_band_wl[i]+j, 1);
		}
		if(plvals.qube_loadmode == QUBELOAD_IMGS) {
			image_ID = gimp_image_new (pdsinfo->line_samples, pdsinfo->lines, GIMP_GRAY);
			imagename=(char *)g_malloc(strlen(filename)+strlen(buf)+3);
			CHECK_FOR_ERROR(imagename==NULL, pdsinfo->jmpbuf, "cannot allocate name-buffer\n");
			strcpy(imagename, filename);
			imagename[strlen(filename)-4]='\0';
			strcat(imagename, "_");
			strcat(imagename, buf);
			gimp_image_set_filename (image_ID, imagename);
		} else if(plvals.qube_loadmode == QUBELOAD_RGB) {
			if(i != plvals.qube_load[0] && i != plvals.qube_load[1] && i != plvals.qube_load[2])
				continue;
		}
		layer_ID = gimp_layer_new (image_ID, buf, pdsinfo->line_samples, pdsinfo->lines,
					   ((pdsinfo->np >= 3) || (plvals.qube_loadmode == QUBELOAD_RGB)) ? GIMP_RGB_IMAGE : GIMP_GRAY_IMAGE,
					   100, (plvals.qube_loadmode == QUBELOAD_RGB)? GIMP_ADDITION_MODE: GIMP_NORMAL_MODE);
		gimp_image_insert_layer (image_ID, layer_ID, 0, 0);
		drawable = gimp_drawable_get (layer_ID);
		gimp_pixel_rgn_init (&pixel_rgn[i], drawable, 0, 0, drawable->width, drawable->height, TRUE, FALSE);
		if(plvals.qube_loadmode!=QUBELOAD_NOT) {
			seek_l=pdsinfo->record_bytes*(pdsinfo->image_ptr-1)
				+i*(pdsinfo->sample_bytes*pdsinfo->lines*pdsinfo->line_samples
				    +pdsinfo->suffix_bytes
					*(pdsinfo->suffix_items_band*(pdsinfo->line_samples+pdsinfo->suffix_items_line)
						+pdsinfo->lines*pdsinfo->suffix_items_line));
			if(console)
				fprintf(stderr, "/* GIMP-PDS: loading band %s... */\n", pdsinfo->qube_band_wl[i]);
			l=lseek(fd[0], seek_l, SEEK_SET);
			CHECK_FOR_ERROR(l!=seek_l, pdsinfo->jmpbuf, "premature end of band data\n");
			if(plvals.qube_loadmode == QUBELOAD_LAYERS && i>pdsinfo->qube_bands/2)
				gimp_layer_set_show_mask(layer_ID, FALSE);
		}

		// load as rgb assigned or grayscale
		if(plvals.qube_loadmode==QUBELOAD_RGB) {
			gimp_layer_set_show_mask(layer_ID, FALSE);
			l=3;
			for(j=0; j<3; j++)
				if(plvals.qube_load[j]==(guint)i) {
					gimp_layer_set_show_mask(layer_ID, TRUE);
					l=j;
				}
		} else l=-1;
		int imgpos = lseek(pdsscanner_fd(scan, 0), 0L, SEEK_CUR);
		if(pdsinfo->np == 1 && (pdsinfo->minmax == 1 && pdsinfo->sample_bits > 8)) {
			fprintf(stderr, "/* GIMP-PDS: contrast stretching from [%ld..%ld] */\n", pdsinfo->minval, pdsinfo->maxval);
		}
		pds_load(scan, pdsinfo, &pixel_rgn[i], l);			
		
		/* if for a grayscale image no max and min values are given, read again and maximize contrast */
		if(pdsinfo->np == 1 && (pdsinfo->minmax == 0 && pdsinfo->sample_bits > 8)) {
			lseek(pdsscanner_fd(scan, 0), imgpos, SEEK_SET);
			pds_load(scan, pdsinfo, &pixel_rgn[i], l);
			fprintf(stderr, "/* GIMP-PDS: no contrast stretching */\n");
		}
		
		/* Tell the GIMP to display the image. */
		gimp_drawable_flush(drawable);
		gimp_drawable_detach(drawable);
		if(plvals.qube_loadmode == QUBELOAD_IMGS) gimp_display_new(image_ID);
	}

	/* Destroy the scanner */
	pdsscanner_destroy(scan);

	/* free the structures */
	if(pdsinfo->imgbuf != NULL)	g_free(pdsinfo->imgbuf);
	g_free (pdsinfo);
	for(f=0; f<4; f++) if(fd[f]) close (fd[f]);

	return image_ID;
}


static void
pds_load(PDSScanner *scan, PDSInfo    *info, GimpPixelRgn  *pixel_rgn, int mode)
{
	unsigned char *data, *d, *line_data = NULL, *line_data_ir3 = NULL;
	int x, y, i;
	int start, end, scanlines;
	int fd=-1;
	off_t fd_pos=0;
	int color;
	int bytes,m;
	int value;
	short int shortval;
	int adj;
	union { float f; unsigned long l; } px_union;
	unsigned long* px_ulong;
	short* px_short;
	float min_flt=FLT_MAX, max_flt=.0;
	short int min_int=32767, max_int=-32768;
	char outbuf[BUFLEN];		/* for error messages */
	int samples;
	off_t rbytes;
	long newmax=info->minval;
	long newmin=info->maxval;

	bytes=info->line_samples*info->sample_bits/8;
	CHECK_FOR_ERROR( (data = (unsigned char *)g_malloc (gimp_tile_height () * info->line_samples*info->np)) == NULL,
						info->jmpbuf,
						"pds filter: cannot allocate data buffer\n");
	CHECK_FOR_ERROR( (line_data = (unsigned char *)g_malloc (gimp_tile_height() * bytes * info->sample_bytes)) == NULL,
						info->jmpbuf,
						"pds filter: cannot allocate line_data buffer\n");
	
	// Viking adjust by IR3 (optional)
	if(info->viking_adj) {
		CHECK_FOR_ERROR( (line_data_ir3 = (unsigned char *)g_malloc (gimp_tile_height () * bytes)) == NULL,
				 info->jmpbuf,
				 "pds filter: cannot allocate line buffers\n");
	}
	
	// 16 bit int, 32 bit IEEE float (Themis IR/VIS), get value interval
	if (info->sample_bytes>1 && (strstr(info->instrument, "THERMAL")>=0))  {
		if(console)
			fprintf(stderr, "/* GIMP-PDS: calculating value interval... */\n");
		fd = pdsscanner_fd(scan, 0);
		if(fd_pos==0)
			fd_pos=lseek(fd, 0L, SEEK_CUR);
		for (y = 0; y < info->lines; y++) {
			CHECK_FOR_ERROR(lseek(fd, info->line_prefix_bytes, SEEK_CUR) < 0,
					info->jmpbuf, "premature end of file while reading line prefix bytes\n");
			CHECK_FOR_ERROR((bytes*info->sample_bytes != read(fd, line_data, bytes*info->sample_bytes)),
					info->jmpbuf, "premature end of line data while scaling\n");
			CHECK_FOR_ERROR(lseek(fd, info->suffix_items_line*info->suffix_bytes, SEEK_CUR) <= 0,
					info->jmpbuf, "premature end of file while reading line-suffix\n");
			CHECK_FOR_ERROR(lseek(fd, info->line_suffix_bytes, SEEK_CUR) <= 0,
					info->jmpbuf, "premature end of file while reading line suffix bytes\n");
			for(m=0; m<bytes; m++) {
				if (info->sample_type==2)  {
					// 4byte ieee float
					px_ulong = (unsigned long*)(line_data+m*4);
					px_union.l = g_ntohl(*px_ulong);
					if(px_union.f>max_flt) max_flt=px_union.f;
					else if(px_union.f<min_flt && px_union.f>=.0) min_flt=px_union.f;
				} else {
					// 2byte signed short
					px_short = (short int*)(line_data+m*2);
					shortval = (short int)g_ntohs(*px_short);
					if(info->sample_type==0) shortval = ((shortval & 0xff) <<8) | ((shortval & 0xff00) >>8);
					if(shortval>max_int) max_int=shortval;
					else if(shortval<min_int && shortval>(-32768+4096)) min_int=shortval;
				}
			}
		}
		lseek(fd, fd_pos, SEEK_SET);
	}
				
	for (y = 0; y < info->lines;) {
		start = y;
		end = y + gimp_tile_height ();
		end = MIN (end, info->lines);
		scanlines = end - start;
		d=data;

		for (i = 0; i < scanlines; i++) {
			if(info->viking_adj) {
				fd = pdsscanner_fd(scan, 3);
				CHECK_FOR_ERROR((bytes != read(fd, line_data_ir3, bytes)),
						info->jmpbuf, "premature end of ir3-data\n");
			}
			for (color=0; color<info->np; color++) {
				if(mode>=0 && color>0) continue;
				if(info->imgbuf != NULL) {
					for(m=0; m<bytes; m++)
						line_data[m]=info->imgbuf[info->imgbuf_pos++];
				} else {
					fd = pdsscanner_fd(scan, color);
					CHECK_FOR_ERROR(lseek(fd, info->line_prefix_bytes, SEEK_CUR) < 0,
							info->jmpbuf, "premature end of file while reading line prefix bytes\n");
					samples = bytes*info->sample_bytes;
					rbytes = read(fd, line_data, samples);
					if((int)rbytes < samples) {
						sprintf(outbuf, "premature end of file while loading bitmap. Only %d of %d bytes read. errno=%d\n",
							(int)rbytes, samples, errno);
						CHECK_FOR_ERROR(TRUE, info->jmpbuf, outbuf);
					}
					CHECK_FOR_ERROR(lseek(fd, info->suffix_items_line*info->suffix_bytes, SEEK_CUR) <= 0,
							info->jmpbuf, "premature end of file while reading line-suffix\n");
					CHECK_FOR_ERROR(lseek(fd, info->line_suffix_bytes, SEEK_CUR) <= 0,
							info->jmpbuf, "premature end of file while reading line suffix bytes\n");
					if(info->sample_bytes==2)  {
						// signed short int
						for(m=0; m<bytes; m++) {
							px_short = (short int*)(line_data+m*2);
							shortval = (short int)g_ntohs(*px_short);
							if(info->sample_type==0) shortval = ((shortval & 0xff) <<8) | ((shortval & 0xff00) >>8);
							if(plvals.qube_stretch) {
								if(shortval>max_int) shortval=max_int;
								else if(shortval<min_int) shortval=min_int;
								line_data[m] = (unsigned char)((float)(shortval-min_int)/(max_int-min_int)*255.0);
							} else {
								line_data[m] = (unsigned char)((shortval-32768)>>8);
							}
						}
					} else if (info->sample_type==2)  {
						// ieee float
						for(m=0; m<bytes; m++) {
							px_ulong = (unsigned long*)(line_data+m*info->sample_bytes);
							px_union.l = g_ntohl(*px_ulong);
							line_data[m] = (unsigned char)((px_union.f-min_flt)/max_flt*255);
						}
					}
				}
				if(info->sample_bits<=8) {
					for(x=0; x<info->line_samples; x++)
						if(info->viking_cam>=0) {
							if(info->viking_adj) adj=line_data_ir3[x];
							else adj=0;
							switch(color) {
								/* Viking
								   R':=R -0.1*IR3
								   G':=1.4*G -0.1*IR3
								   B':=1.7*B -0.1*R -0.1*IR3
								*/
							    case 0:
								    value=info->sample_type?(g_ntohs(line_data[x]<<8)):line_data[x];
								    d[info->np*x+color]=(value>=0)?value:0;
								    d[info->np*x+color]=(value<=255)?value:255;
								    break;
							    case 1:
								    //value=info->sample_type?(g_ntohs(line_data[x]<<8)):1.4*line_data[x] -.07*adj;
								    value=info->sample_type?(g_ntohs(line_data[x]<<8)):line_data[x];
								    d[info->np*x+color]=(value>=0)?value:0;
								    d[info->np*x+color]=(value<=255)?value:255;
								    break;
							    case 2:
								    //value=info->sample_type?(g_ntohs(line_data[x]<<8)):1.7*line_data[x]-.1*d[3*x]-.1*adj;
								    value=info->sample_type?(g_ntohs(line_data[x]<<8)):line_data[x];
								    d[info->np*x+color]=(value>=0)?value:0;
								    d[info->np*x+color]=(value<=255)?value:255;
								    break;
							}
						} else {
							if(mode>2) {
								d[(info->np) * x ]=info->sample_type?(g_ntohs(line_data[x]<<8)):line_data[x];
								d[(info->np) * x +1]=info->sample_type?(g_ntohs(line_data[x]<<8)):line_data[x];
								d[(info->np) * x +2]=info->sample_type?(g_ntohs(line_data[x]<<8)):line_data[x];
							} else if (mode >= 0) {
								d[(info->np) * x ]=0;
								d[(info->np) * x +1]=0;
								d[(info->np) * x +2]=0;
								d[(info->np) * x +mode]=line_data[x];
							} else {
								d[(info->np) * x + color]=line_data[x];
							}
						}
				} else {
					for(x=0; x<info->line_samples; x++) {
						value=info->sample_type?
							(line_data[x*2]<<8)+line_data[x*2+1]
							:line_data[x*2]+(line_data[x*2+1]<<8);
						if(value > newmax)
							newmax = value;
						else if(value < newmin)
							newmin = value;
						if(mode>2) {
							d[(info->np) * x]=(unsigned char)(255.0*(double)(value)/info->maxval);
							d[(info->np) * x +1]=(unsigned char)(255.0*(double)(value)/info->maxval);
							d[(info->np) * x +2]=(unsigned char)(255.0*(double)(value)/info->maxval);
						} else if(mode>=0) {
							d[(info->np) * x ] = 0;
							d[(info->np) * x +1] = 0;
							d[(info->np) * x +2] = 0;
							d[(info->np) * x +mode]=(unsigned char)(255.0*(double)(value)/info->maxval);
						} else {
							d[(info->np) * x + color]=(unsigned char)(255.0*(double)(value)/info->maxval);
							//TEST fprintf(stderr, "%d/%d(%d) ", value, info->maxval, d[(info->np) * x + color]);
						}
					}
				}
			}
			d += info->line_samples*info->np;
		}
		if(info->imgbuf != NULL)
			gimp_progress_update((double) y / (double) info->lines*0.5+.5);
		else if(plvals.qube_loadmode!=QUBELOAD_NOT)
			gimp_progress_update(lseek(fd, 0L, SEEK_CUR) / (double)(info->line_samples*info->lines*info->qube_bands));
		else
			gimp_progress_update ((double) y / (double) info->lines);
		gimp_pixel_rgn_set_rect (pixel_rgn, data, 0, y, info->line_samples, scanlines);
		y += scanlines;
	}
	g_free (data);
	g_free (line_data);
	if(line_data_ir3 != NULL)
		g_free (line_data_ir3);

	info->minval = newmin;
	info->maxval = newmax;
}

/**************** FILE SCANNER UTILITIES **************/

/* pdsscanner_create ---
 *    Creates a new scanner based on a file descriptor.  The
 *    look ahead buffer is one character initially.
 */
static PDSScanner *
pdsscanner_create (int fd[], int rgb)
{
	PDSScanner *s;

	s = (PDSScanner *)g_malloc(sizeof(PDSScanner));
	if (s == NULL)
		return(NULL);
	s->fd[0] = fd[0];
	s->fd[1] = fd[1];
	s->fd[2] = fd[2];
	s->fd[3] = fd[3];
	s->cur_fd=0;
	s->rgb=(rgb>0);
	s->inbuf = 0;
	s->eof = !read(s->fd[s->cur_fd], &(s->cur), 1);
	return(s);
}

/* pdsscanner_destroy ---
 *    Destroys a scanner and its resources.  Doesn't close the fd.
 */
static void
pdsscanner_destroy (PDSScanner *s)
{
	if (s->inbuf) g_free(s->inbuf);
	g_free(s);
}

/* pdsscanner_getint ---
 *    Gets the next 16bit int value.
 */
static char
pdsscanner_getint (char* buf, char* key, int* value)
{
	char* k;
	k = (char *)malloc(strlen(key)+2);
	if(k == NULL)
		return 0;
	strcpy(k, key);
	strcat(k, " ");
	for(; isspace (buf[0]); buf++);
	if(strstr(buf, k)==buf) {
		*value=atoi(strstr(buf, "=")+1);
		free(k);
		return 1;
	} else return 0;
}

/* pdsscanner_getlong ---
 *    Gets the next 32bit int value.
 */
static char
pdsscanner_getlong (char* buf, char* key, long* value)
{
	char* k;
	k = (char *)malloc(strlen(key)+2);
	if(k == NULL)
		return 0;
	strcpy(k, key);
	strcat(k, " ");
	for(; isspace (buf[0]); buf++);
	if(strstr(buf, k)==buf) {
		*value=(long)(atof(strstr(buf, "=")+1) + .5);
		free(k);
		return 1;
	} else return 0;
}

/* pdsscanner_getstr ---
 *    Gets the next string.
 */
static char
pdsscanner_getstr (char* buf, char* key, char* str)
{
	char* pos;
	int i,j;
	for(; isspace (buf[0]); buf++);
	if((pos=strstr(buf, key))==buf) {
		pos=strstr(buf, "=")+1;
		for(i=0; pos[i] && !(pos[i]=='\"' || pos[i]!=' '); i++);
		for(j=0; pos[i+j+1] && pos[i+j+1]!='\"' && !isspace(pos[i+j+1]); j++);
		if(pos[i]=='\"')
			strncpy(str, pos+1+i, j);
		else
			strncpy(str, pos+i, j+1);
		str[j+1]=0;
		return 1;
	} else
		return 0;
}

/* pdsscanner_getchar ---
 *    Reads a character from the input stream
 */
static void
pdsscanner_getchar (PDSScanner *s)
{
	if (s->inbuf)
	{
		s->cur = s->inbuf[s->inbufpos++];
		if (s->inbufpos >= s->inbufvalidsize)
		{
			if (s->inbufsize > s->inbufvalidsize)
				s->eof = 1;
			else
				s->inbufvalidsize = read(s->fd[s->cur_fd], s->inbuf, s->inbufsize);
			s->inbufpos = 0;
		}
	}
	else
		s->eof = !read(s->fd[s->cur_fd], &(s->cur), 1);
}


static char
pdsscanner_getstrtuple (char* buf, char* key, char* str[])
{
	char value[1024];
	char* token;
	int i,j;
	char* pos;
	for(; isspace (buf[0]); buf++);
	if((pos=strstr(buf, key))==buf) {
		pos=strstr(buf, "=")+1;
		int lp = strlen(pos);
		for(i=0; i<lp && pos[i]!='('; i++);
		for(j=0; i+j+1<lp && pos[i+j+1]!=')'; j++);
		if(i==lp) return 0;
		strncpy(value, pos+i, j+1);
		value[j+1]=0;
		token=strtok(value, ",");
		for(j=0; isspace(token[j]); j++);
		str[0]=strdup(token+i);
		for(i=1; (token=strtok(NULL, ",")); i++) {
			for(j=0; isspace(token[j]); j++);
			str[i]=strdup(token+j);
		}
		return 1;
	} else
		return 0;
}



