/*
NOTICE

The software accompanying this notice (the "Software") is provided to you
free of charge to facilitate your use of the data collected by the Mars
Orbiter Camera (the "MOC Data").  Malin Space Science Systems ("MSSS")
grants to you (either as an individual or entity) a personal,
non-transferable, and non-exclusive right (i) to use and reproduce the
Software solely for the purpose of accessing the MOC Data; (ii) to modify
the source code of the Software as necessary to maintain or adapt the
Software to run on alternate computer platforms; and (iii) to compile, use
and reproduce the modified versions of the Software solely for the purpose
of accessing the MOC Data.  In addition, you may distribute the Software,
including any modifications thereof, solely for use with the MOC Data,
provided that (i) you must include this notice with all copies of the
Software to be distributed; (ii) you may not remove or alter any
proprietary notices contained in the Software; (iii) you may not charge any
third party for the Software; and (iv) you will not export the Software
without the appropriate United States and foreign government licenses.

You acknowledge that no title to the intellectual property in the Software
is transferred to you.  You further acknowledge that title and full
ownership rights to the Software will remain the exclusive property of MSSS
or its suppliers, and you will not acquire any rights to the Software
except as expressly set forth above.  The Software is provided to you AS
IS.  MSSS MAKES NO WARRANTY, EXPRESS OR IMPLIED, WITH RESPECT TO THE
SOFTWARE, AND SPECIFICALLY DISCLAIMS THE IMPLIED WARRANTIES OF
NON-INFRINGEMENT OF THIRD PARTY RIGHTS, MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE.  SOME JURISDICTIONS DO NOT ALLOW THE EXCLUSION OR
LIMITATION OF INCIDENTAL OR CONSEQUENTIAL DAMAGES, SO SUCH LIMITATIONS OR
EXCLUSIONS MAY NOT APPLY TO YOU.

Your use or reproduction of the Software constitutes your agreement to the
terms of this Notice.  If you do not agree with the terms of this notice,
promptly return or destroy all copies of the Software in your possession.

Copyright (C) 1999 Malin Space Science Systems.  All Rights Reserved.
*/
//static char *sccsid = "@(#)readmoc.c	1.2 04/10/00";

/*
    SDP interpretation program
    Mike Caplinger, MOC GDS Design Scientist
    SCCS @(#)readmoc.c	1.2 04/10/00

    Reads and decompresses MOC SDP files to create PDS images.
    Derived from the GDS readmsdp program.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <libgimp/gimp.h>
#include "array.h"
#include "fs.h"
#include "image_io.h"
#include "msdp.h"
#include "bytecopy.h"

extern void decodeLoad(char *);
extern void decodeInit(int);
extern uint8 *predictive_decomp_main(uint8 *, int, uint32, uint32, uint8,
	   uint16, int, int, int *);
/* HIC
   int in, out;
*/
int in;
unsigned char* out;

struct image_header inf;
FILE *infile;
int errors;
int test_pred;
int rawencode = 0;

#ifndef VERBOSE
#define VERBOSE 0
#endif
int verbose=VERBOSE;

int frag_offset[128];

char infname[256], outfname[256];
int mbr = 0;

uint8 *decode(char *);
uint8 *decode_header(struct msdp_header, uint8*, int, long *, int);
int init_output(struct msdp_header);
extern uint CS8EACC2(register uint8 *, uint);

#define FRAGSIZE (256*1024)

static char decode_file[128];

unsigned long moc_sync;

char label[256];

int status;
#define STAT_SHORT 2
#define STAT_BADSEQ 4
#define STAT_BADCS 8

enum { RAW = 0, PRED, XFORM } compress = RAW;

/***********************************************
 * memory-write added by Holger Isenberg
 */
long mem_write_count=0;
long mem_write_max=0;
void mem_write(unsigned char* mem, pixel* buf, long len)
{
	long i;
	for(i=0; i<len && mem_write_count<mem_write_max; i++)
		mem[mem_write_count++]=buf[i];
	gimp_progress_update((double)mem_write_count/(double)mem_write_max*.5);
}
/***********************************************/

unsigned char* decode_imq(char* f_in)
{
    int height, width;
//    int frag_lines, n_frags;
    pixel *frag;
//    int f;
//    float quant;
    int actual_height;
    int total_image = 0;
    int total = 0;
    int cs_check = 1;
    int pad_cs = 0;
//    int i;
//    char s[8];
    int multi = 0;
    int sequence = -1, processor = 0, n_processors = 1;
    int last_frag = -1;

    moc_sync = 0xf0ca;

    strcpy(infname, f_in);

    if(verbose) fprintf(stderr, "decode_imq %s\n", infname);

#ifdef WIN32
    infile = fopen(infname, "rb");
#else
	infile = fopen(infname, "r");
#endif
    if(infile == 0) {
	fprintf(stderr, "Can't open %s\n", infname);
	return NULL;
    }

    while(1) {
	int count;
	long len;
	static int first = 1;
	struct msdp_header h, lasth;
	int datlen;
	uint8 *indat, *chunk;

	lasth = h;
	fseek(infile, total+2048, 0);
	count = fread(&h, sizeof(h), 1, infile);
	if(count && MAKELONG(h.len) == 0) {
	    /* simulate the EOF even though there's padding */
	    count = 0;
	    h = lasth;
	}
	if(count == 0 && compress == PRED && (h.status&2) == 0) {
	    /* image was short -- last flag missing */
	    h.status = 2;
	    frag = decode_header(h, indat, 0, &len, mbr);
	    mem_write(out, frag, len);
	    total_image += len;
	}
	if(count == 0) break;
	if(MAKELONG(h.len) == 0) break;
	sequence += 1;

	if(first && !multi) {
//	    int edit[2];

	    width = h.edit_length*16;
	    if(init_output(h)==0) return NULL;
	    first = 0;
	    height = MAKESHORT(h.down_total)*16;
	}
	h.edit_length = width/16;
	if(mbr) width = 512;

	datlen = MAKELONG(h.len);

	if(sequence%n_processors != processor) {
	    total += sizeof(struct msdp_header) + datlen + 1;
	    continue;
	}

	if(!multi && MAKESHORT(h.fragment) != last_frag+1) {
	    int n_pad = MAKESHORT(h.fragment)-last_frag-1;
	    char *frag = (char*)malloc(240*1024);
	    /* don't pad predictively-compressed data */
	    if(!(h.compression[0] & 3) && n_pad > 0) {
		errors += 1;
		status |= STAT_BADSEQ;
		bytezero(frag, 240*1024);
		total_image += n_pad*240*1024;
		if(verbose) fprintf(stderr, "padding %d frags\n", n_pad);
		while(n_pad--) mem_write(out, (pixel *)frag, 240*1024);
	    }
	    free(frag);
	}
	last_frag = MAKESHORT(h.fragment);

	if(verbose) fprintf(stderr, "id %d/%d, len %d\n", MAKESHORT(h.id),
		MAKESHORT(h.fragment), MAKELONG(h.len));
	chunk = (uint8 *) malloc(datlen+sizeof(struct msdp_header)+1);
	indat = chunk+sizeof(struct msdp_header);
	count = fread(indat, 1, datlen, infile);
	if(count != datlen) {
	    if(verbose) fprintf(stderr,
		    "Error: short read (%d) of data part of fragment\n",
		    count);
	    errors += 1;
	    break;
	}

	/* check MSDP checksum */
	if(cs_check) {
	    bytecopy((char *)&h, (char *)chunk, sizeof(h));
	    fread(chunk+datlen+sizeof(h), 1, 1, infile);
	    if(!CS8EACC2(chunk, datlen+sizeof(h)+1)) {
		if(verbose) fprintf(stderr, "Error: bad MSDP checksum\n");
		status |= STAT_BADCS;
		errors += 1;
		if(pad_cs) {
		    char *frag = (char*)malloc(240*1024);
		    bytezero(frag, 240*1024);
		    total_image += 240*1024;
		    total += sizeof(struct msdp_header) + datlen + 1;
		    if(verbose) fprintf(stderr, "trashing bad frag\n");
		    mem_write(out, (pixel *)frag, 240*1024);
		    free(frag);
		    continue;
		}
	    }
	}

	frag = decode_header(h, indat, datlen, &len, mbr);
	total_image += len;
	if(verbose) fprintf(stderr, "fragment len %d => %d\n", datlen, len);
	total += sizeof(struct msdp_header) + datlen + 1;
	mem_write(out, frag, len);
	if(0) free(frag);
	free(chunk);
	if(h.status&2) break;
    }

    actual_height = total_image/width;
    if(!multi && actual_height != height) {
	if(verbose) fprintf(stderr,
		"Error: total MSDP height (%d) != actual height (%d)\n",
		height, actual_height);
	/* HIC
	(void) write_header(width, actual_height, infile, outfname);
	*/
	errors += 1;
	status |= STAT_SHORT;
    }

    if(status && verbose) fprintf(stderr, "error status %c%c%c%c\n",
		       compress==RAW?'r':(compress==PRED?'p':'t'),
		       status&STAT_BADCS?'c':'-',
		       status&STAT_BADSEQ?'n':'-',
		       status&STAT_SHORT?'s':'-'
		       );
/* HIC
   if(errors) exit((compress << 4) | status | (errors?1:0));
   else exit(0);
*/
    if(verbose) {
	    if(errors)
		    fprintf(stderr, "decode_imq error: %d\n", (compress << 4) | status | (errors?1:0));
	    else
		    fprintf(stderr, "decode_imq ok\n");
#ifdef WIN32
		fprintf(stderr,"\nPress <Enter> to continue.");
		getchar();
#endif
	}
    return out;
}


uint8 *decode_header(struct msdp_header h, uint8 *data, int datlen, long *len, int mbr)
{
    int height, width;
    unsigned int xcomp, pcomp, spacing, levels;
    uint8 *transform_decomp_main(uint8 *, int, int, int, uint32, uint32, uint32);
    uint8 *image;
    static Array *tbuf;
    static int init_decode;
    int huffman_table;

    if(mbr) {
	width = 512;
	height = 480;
	xcomp = 0;
	pcomp = 0;
    }
    else {
	width = h.edit_length*16;
	height = MAKESHORT(h.down_length)*16;
	xcomp = (h.compression[0] >> 2) & 3;
	pcomp = (h.compression[0] & 3);
	spacing = h.compression[4] | (h.compression[5] << 8);
	levels = (h.compression[1] >> 5)+1;
	huffman_table = h.compression[1]&0xf;
    }

    *len = width*height;

    if(pcomp && xcomp) {
	fprintf(stderr, "error: both pcomp and xcomp set\n");
	exit(1);
    }
    if(pcomp) compress = PRED;
    if(xcomp) compress = XFORM;

    if(!rawencode && pcomp == 0 && xcomp == 0) {
	/* raw image */
	image = data;
	if(datlen > *len) {
	    if(verbose) fprintf(stderr, "Warning: MSDP line count (%d) < implied (%d), using latter\n", height, datlen/width);
	    *len = datlen;
	    height = datlen/width;
	}
	if(verbose) fprintf(stderr, "%d wide by %d high ", width, height);
	if(verbose) fprintf(stderr, "raw fragment%s\n", mbr?" (MBR)":"");
    }
    else
      if(verbose) fprintf(stderr, "%d wide by %d high ", width, height);

    if(xcomp > 0) {
	/* transform compressed; 2 = DCT, 1 = WHT */
	if(verbose) fprintf(stderr, "%s transformed fragment (%d groups, %.2f requant)\n",
		xcomp == 2 ? "dct" : "wht", levels, spacing/16.0);
	image = transform_decomp_main(data, datlen, height, width,
				      xcomp-1, spacing, levels);
    }

    if(rawencode || pcomp > 0) {
	/* predictively compressed */
	if(rawencode)
	  if(verbose) fprintf(stderr, "raw encoded fragment\n");
	else
	  if(verbose) fprintf(stderr, "%s%s predictive fragment, table %d\n",
		  pcomp&1 ? "x" : "", (pcomp&2)>>1 ? "y" : "",
		  huffman_table);

	/* set up decode arrays */
	if(!init_decode) {
	    if(*decode_file) decodeLoad(decode_file);
	    else decodeInit(huffman_table);
	}

	if(test_pred) {
	    int dummy;

	    image = predictive_decomp_main(data, datlen, height, width,
					   (moc_sync != 0), moc_sync,
					   pcomp&1, (pcomp&2) >> 1,
					   &dummy);
	}
	else {
	    /* squirrel data away */
	    if(!tbuf) tbuf = array_new(datlen*8);
	    if(datlen && !array_append(tbuf, (char *)data, datlen)) {
		fprintf(stderr, "can't allocate temp space (%d bytes)\n, datlen");
		exit(1);
	    }
	    image = 0;
	    *len = 0;

	    if(h.status & 2) {
		int got_height;
		extern int pred_past_eof;
		int want_h = MAKESHORT(h.down_total)*16;;

		if(verbose) fprintf(stderr, "decompressing %d wide by %d high image\n",
			width, want_h);
		image =
		  predictive_decomp_main((uint8 *)array_data(tbuf),
					 array_len(tbuf),
					 want_h, width,
					 (moc_sync != 0), moc_sync,
					 pcomp&1, (pcomp&2) >> 1,
					 &got_height);
		/* This is tricky.  We can get bad moc_sync even without
		   checksum errors if anomaly 8 occurs.  We want to
		   distinguish between this case and the case where
		   we just ran out of fragments during the NA image.
		   So if we run into a moc_sync error and haven't run off
		   the end of the image, we force BADCS on. */
		if(got_height != want_h && !pred_past_eof)
		  status |= STAT_BADCS;
		*len = got_height * width;
	    }
	}
    }
    return image;
}

int worklist_init() {
    struct msdp_header h;
    int count;
    int frag;
    int height, width, xcomp;
    int datlen;

    while(1) {
	count = fread(&h, sizeof(h), 1, infile);
	if(count == 0) break;
	xcomp = (h.compression[0] >> 2) & 3;
	if(!xcomp) return 0;
	height = MAKESHORT(h.down_length)*16;
	width = h.edit_length*16;
	datlen = MAKELONG(h.len);
	if(init_output(h)==0) return 0;
	frag_offset[frag+1] = frag_offset[frag]+height*width;
	frag += 1;
	fseek(infile, datlen+1, 1);
    }
    return 1;
}

int init_output(struct msdp_header h)
{
    int height, width;
    char s[8];
    int i;
    char buf[1024];

    height = MAKESHORT(h.down_total)*16;
    width = h.edit_length*16;
    sprintf(label, "decompressed-from %s\nid %d time %u:%d\ngain 0x%x \
offset %d\nstart %d cross %d down %d\ncmd ", infname,
	    MAKESHORT(h.id),
	    MAKELONG(h.time+1), h.time[0],
	    h.gain, h.offset,
	    h.edit_start*16,
	    h.edit_length*16,
	    MAKESHORT(h.down_total)*16);
    switch(h.cmd[0]) {
      case 1:
      case 2:
	strcat(label, "na ");
	break;
      case 3:
      case 4:
	strcat(label, "wa ");
	break;
      case 5:
      case 6:
	strcat(label, "global-map ");
	break;
      case 0x15:
      case 0x16:
	strcat(label, "mbr ");
	mbr = 1;
	width = 512;
	height = 512;
	break;
      case 0xd:
	strcat(label, "read-memory ");
	break;
      default:
	strcat(label, "unknown ");
	break;
    }
    for(i = 0; i < 17; i++) {
	sprintf(s, "%02x", h.cmd[i]);
	strcat(label, s);
    }

    sprintf(buf, "\nsensor %d clocking %d system-id 0x%x",
	    MAKESHORT(h.sensors),
	    MAKESHORT(h.other+1),
	    h.other[3]);
    strcat(label, buf);

    /* HIC
       out = write_header(width, height, infile, outfname);
    */
    mem_write_max = width*height;
    out = (unsigned char*)g_malloc(mem_write_max);
    if(out==NULL) {
	    if(verbose)
		    fprintf(stderr, "Cannot allocate %ld bytes for image size %ldx%ld\n",
			    width*height, width, height);
	    return 0;
    } else return 1;
}
