;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; GIMP - The GNU Image Manipulation Program                               ;;;
;;; Copyright (C) 1995 Spencer Kimball and Peter Mattis                     ;;;
;;;                                                                         ;;;
;;; This program is free software: you can redistribute it and/or modify    ;;;
;;; it under the terms of the GNU General Public License as published by    ;;;
;;; the Free Software Foundation, either version 3 of the License, or       ;;;
;;; (at your option) any later version.                                     ;;;
;;;                                                                         ;;;
;;; This program is distributed in the hope that it will be useful,         ;;;
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of          ;;;
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           ;;;
;;; GNU General Public License for more details.                            ;;;
;;;                                                                         ;;;
;;; You should have received a copy of the GNU General Public License       ;;;
;;; along with this program.  If not, see <http://www.gnu.org/licenses/>.   ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Remove Holes (from Selection)                                           ;;;
;;;                                                                         ;;;
;;; remove-holes-f-s.scm - version 1.06                                     ;;;
;;; Copyright (C) 2010-2014 Gino D                                          ;;;
;;; https://sites.google.com/site/ginodonig/gimp-scripts                    ;;;
;;;                                                                         ;;;
;;; This script enables you to remove the holes of a selection, if by hole  ;;;
;;; we mean every unselected or partially selected area of the image which  ;;;
;;; is surrounded by a fully selected area.                                 ;;;
;;;                                                                         ;;;
;;; ....................................................................... ;;;
;;;                                                                         ;;;
;;; VERSION HISTORY                                                         ;;;
;;;                                                                         ;;;
;;; 1.00 - January 2010                                                     ;;;
;;;  * First release.                                                       ;;;
;;;                                                                         ;;;
;;; 1.01 - January 2010                                                     ;;;
;;;  * Added the ability to finally feather the selection.                  ;;;
;;;                                                                         ;;;
;;; 1.02 - July 2010                                                        ;;;
;;;  * Only reshaped and cleaned up the code.                               ;;;
;;;                                                                         ;;;
;;; 1.03 - January 2011                                                     ;;;
;;;  * Improved the method of removal, by resorting to a temporary layer    ;;;
;;;    instead of manipulating the canvas size.                             ;;;
;;;  * Reshaped and cleaned up the code.                                    ;;;
;;;                                                                         ;;;
;;; 1.04 - August 2011                                                      ;;;
;;;  * Fixed a bug that made the script crash when operating on non-RGB     ;;;
;;;    images.                                                              ;;;
;;;  * As for normalizing the selection, added a new code snippet           ;;;
;;;    consisting only of GIMP internal procedures, so that doing without   ;;;
;;;    the "plug-in-c-astretch" command.                                    ;;;
;;;  * Fixed a small flaw in the display of the progress bar.               ;;;
;;;  * Corrected the year range in my copyright notice.                     ;;;
;;;                                                                         ;;;
;;; 1.05 - December 2012                                                    ;;;
;;;  * Made the script fully compatible with GIMP 2.8, while maintaining    ;;;
;;;    the backward compatibility with the previous versions beginning from ;;;
;;;    GIMP 2.6.10.                                                         ;;;
;;;  * Changed the method of removal, which now uses a temporary image      ;;;
;;;    rather than a temporary layer of the active image.                   ;;;
;;;  * Lots of improvements, cleanups and minor bug fixes regarding both    ;;;
;;;    the code and the dialog window.                                      ;;;
;;;                                                                         ;;;
;;; 1.06 - May 2014                                                         ;;;
;;;  * Renamed some options of the script.                                  ;;;
;;;  * Fixed some minor bugs; enhanced and cleaned up the code.             ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(define *remove-holes-f-s-2.8.0* 1)

(let* ((gimp-version-comparison
        (lambda (predicate version-a version-b)
          (letrec ((func
                    (lambda (string1 p q)
                      (cond ((= (string-length string1) p)
                             (* (expt 100 q) (string->atom string1)))
                            ((eqv? (string-ref string1 p)
                                   (integer->char 46))
                             (+ (* (expt 100 q)
                                   (string->atom (substring string1 0 p)))
                                (func (substring string1
                                                 (+ p 1)
                                                 (string-length string1))
                                      0
                                      (- q 1))))
                            (else (func string1 (+ p 1) q))))))
            (predicate (func version-a 0 2)
                       (func version-b 0 2))))))
  (cond ((gimp-version-comparison < (car (gimp-version)) "2.6.10")
         (quit))
        ((gimp-version-comparison < (car (gimp-version)) "2.8.0")
         (set! *remove-holes-f-s-2.8.0* 0))))

(define (script-fu-remove-holes-f-s img1
                                    img1-drw1
                                    normalize-selection
                                    sharpen-selection
                                    b-w-threshold
                                    feather-selection
                                    radius-of-feather)
  (cond ((= *remove-holes-f-s-2.8.0* 0)
         (define (gimp-context-get-sample-threshold-int))
         (define (gimp-context-set-sample-threshold-int sample-threshold)
           (set! gimp-context-get-sample-threshold-int
                 (lambda () (cons sample-threshold ()))))
         (define (gimp-context-get-antialias))
         (define (gimp-context-set-antialias antialias)
           (set! gimp-context-get-antialias
                 (lambda () (cons antialias ()))))
         (define (gimp-context-get-feather))
         (define (gimp-context-set-feather feather)
           (set! gimp-context-get-feather
                 (lambda () (cons feather ()))))
         (define (gimp-context-get-feather-radius))
         (define (gimp-context-set-feather-radius feather-radius-x
                                                  feather-radius-y)
           (set! gimp-context-get-feather-radius
                 (lambda ()
                   (cons feather-radius-x
                         (cons feather-radius-y
                               ())))))
         (define (gimp-context-get-sample-merged))
         (define (gimp-context-set-sample-merged sample-merged)
           (set! gimp-context-get-sample-merged
                 (lambda () (cons sample-merged ()))))
         (define (gimp-context-get-sample-transparent))
         (define (gimp-context-set-sample-transparent sample-transparent)
           (set! gimp-context-get-sample-transparent
                 (lambda () (cons sample-transparent ()))))
         (define (gimp-context-get-sample-criterion))
         (define (gimp-context-set-sample-criterion sample-criterion)
           (set! gimp-context-get-sample-criterion
                 (lambda () (cons sample-criterion ()))))
         (define (gimp-image-select-contiguous-color image
                                                     operation
                                                     drawable
                                                     x
                                                     y)
           (gimp-fuzzy-select-full
            drawable
            x
            y
            (car (gimp-context-get-sample-threshold-int))
            operation
            (car (gimp-context-get-antialias))
            (car (gimp-context-get-feather))
            (car (gimp-context-get-radius-of-feather))
            (car (cdr (gimp-context-get-radius-of-feather)))
            (car (gimp-context-get-sample-merged))
            (car (gimp-context-get-sample-transparent))
            (car (gimp-context-get-sample-criterion))))))
  (define (round2 num int)
    (let* ((rnd (/ (truncate (+ (* (expt 10 int) num)
                                (if (< num 0) -0.5 0.5)))
                   (expt 10 int))))
      (if (= int 0)
          (inexact->exact rnd)
          rnd)))
  (let* ((b-w-threshold (round2 b-w-threshold 0))
         (radius-of-feather (round2 radius-of-feather 3))
         (img1-type (car (gimp-image-base-type img1)))
         (img1-width (car (gimp-image-width img1)))
         (img1-height (car (gimp-image-height img1)))
         (img1-sln (car (gimp-image-get-selection img1)))
         (img2)
         (img2-sln))
    (gimp-context-push)
    (gimp-context-set-sample-threshold-int 254)
    (gimp-context-set-antialias FALSE)
    (gimp-context-set-feather FALSE)
    (gimp-context-set-feather-radius 0 0)
    (gimp-context-set-sample-merged FALSE)
    (gimp-context-set-sample-transparent FALSE)
    (gimp-context-set-sample-criterion SELECT-CRITERION-COMPOSITE)
    (gimp-image-undo-group-start img1)
    (cond ((= (car (gimp-selection-is-empty img1)) TRUE)
           (gimp-image-undo-group-end img1)
           (gimp-displays-flush)
           (gimp-context-pop)
           (quit))
          ((= normalize-selection TRUE)
           (gimp-progress-init " " 0)
           (gimp-progress-set-text " ")
           (let a-loop ((a-percent (/ 1 9))
                        (a-input-low 0)
                        (a-input-high 255)
                        (a-radius 128)
                        (a-count-low ())
                        (a-count-high ()))
             (gimp-progress-update a-percent)
             (cond ((>= a-radius 1)
                    (set! a-count-low
                          (vector-ref
                           (apply vector
                                  (gimp-histogram img1-sln
                                                  HISTOGRAM-VALUE
                                                  a-input-low
                                                  (+ a-input-low a-radius -1)))
                           4))
                    (set! a-count-high
                          (vector-ref
                           (apply vector
                                  (gimp-histogram img1-sln
                                                  HISTOGRAM-VALUE
                                                  (- a-input-high a-radius -1)
                                                  a-input-high))
                           4))
                    (cond ((> (* a-count-low a-count-high) 0)
                           (a-loop (+ a-percent (/ 1 9))
                                   a-input-low
                                   a-input-high
                                   (/ a-radius 2)
                                   a-count-low
                                   a-count-high))
                          ((> a-count-low 0)
                           (a-loop (+ a-percent (/ 1 9))
                                   a-input-low
                                   (- a-input-high a-radius)
                                   (/ a-radius 2)
                                   a-count-low
                                   a-count-high))
                          ((> a-count-high 0)
                           (a-loop (+ a-percent (/ 1 9))
                                   (+ a-input-low a-radius)
                                   a-input-high
                                   (/ a-radius 2)
                                   a-count-low
                                   a-count-high))
                          (else
                           (a-loop (+ a-percent (/ 1 9))
                                   (+ a-input-low a-radius)
                                   (- a-input-high a-radius)
                                   (/ a-radius 2)
                                   a-count-low
                                   a-count-high))))
                   ((> a-input-high a-input-low)
                    (gimp-levels img1-sln
                                 HISTOGRAM-VALUE
                                 a-input-low
                                 a-input-high
                                 1
                                 0
                                 255))))))
    (if (= sharpen-selection TRUE)
        (gimp-threshold img1-sln b-w-threshold 255))
    (set! img2
          (car (gimp-image-new (+ img1-width 2)
                               (+ img1-height 2)
                               img1-type)))
    (set! img2-sln (car (gimp-image-get-selection img2)))
    (gimp-channel-combine-masks img2-sln
                                img1-sln
                                CHANNEL-OP-REPLACE
                                1
                                1)
    (gimp-image-select-contiguous-color img2
                                        CHANNEL-OP-REPLACE
                                        img2-sln
                                        0
                                        0)
    (gimp-selection-invert img2)
    (gimp-channel-combine-masks img1-sln
                                img2-sln
                                CHANNEL-OP-ADD
                                -1
                                -1)
    (if (= feather-selection TRUE)
        (gimp-selection-feather img1 radius-of-feather))
    (gimp-image-delete img2)
    (gimp-image-undo-group-end img1)
    (gimp-displays-flush)
    (gimp-context-pop)))

(script-fu-register
 "script-fu-remove-holes-f-s"
 _"Remove Holes..."
 _"Remove holes from the selection"
 "Gino D <ginodonig@gmail.com>"
 "Gino D"
 "2010-2014"
 "*"
 SF-IMAGE       "Image"                             0
 SF-DRAWABLE    "Drawable"                          0
 SF-TOGGLE     _"Normalize initial selection"       FALSE
 SF-TOGGLE     _"Sharpen initial selection"         FALSE
 SF-ADJUSTMENT _"    Threshold"                     '(127 0 255 1 1 0 0)
 SF-TOGGLE     _"Feather resultant selection"       FALSE
 SF-ADJUSTMENT _"    Radius of feather (in pixels)" '(5 0 32767 1 1 3 1))

(script-fu-menu-register "script-fu-remove-holes-f-s"
                         "<Image>/Select")