#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# [SNIPPET_NAME: Script-Fu Composer]
# [SNIPPET_CATEGORIES: PyGTK, GimpFu]
# [SNIPPET_DESCRIPTION: Complete RAD tool for Gimp to build new script with currently exists]
# [SNIPPET_AUTHOR: Jeremy Czajkowski <jeremy.cz@wp.pl>]
# [SNIPPET_LICENSE: Creative commons <CC-BY-SA>]
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

import ctypes
import datetime
import json
import os
import re
import unicodedata

import gobject
import gtk
import pygtk

pygtk.require('2.0')

from gimpfu import *
import gtk.gdk


# Dictionaries are sometimes found in other languages as associative memories or associative arrays.
BLOCK_TYPE = {}
BLOCK_PREFIX = {}
ALLOCATIONS = {}
PROPS = {}
VISIBLED_NPARAMS = {}
OUTPUT = {}
OUTPUT["active"] = False

# Do NOT TOUCH this global variables
BLOCK_TYPE["count"] = 0
BLOCK_TYPE["start_block"] = None
BUILD_EXCLUDE_TYPE = ['run-mode', 'enum']
HIDDEN_PARAM_TYPE = ['run-mode', 'image', 'drawable']

LAST_INDEX = 0
MAX_COUNT = 6
AMOUNT_OFENTRY = 5

WIDGET_BORDER_COLOR = "blue"
SC_PROCEDURE_NAME = "python-fu-composer"
SC_TITLE = "Script-Fu Composer"

SC_LANG_LABELS = {"eng": "English", "pol":"Polski", "deu":"Deutsch"}
SC_LANG_FILES = ['scriptcomposer', 'preferencesmanager', 'messages', 'registrationmanager']

SC_ABSDIR = os.path.dirname(os.path.abspath(__file__)).lower()
SC_RESDIR = os.path.join(SC_ABSDIR, '_gc')

SC_CONFIGFILE = os.path.join(SC_RESDIR, "preferences")


class TreeViewGroup(gtk.VPaned):
    SCROLL_STEP = 9
    COUNT = {}
    
    def __init__(self, width, height):
        gtk.VPaned.__init__(self)
        self.width = width
        self.height = height
        self.set_size_request(width, height)

        cell_1P = gtk.CellRendererPixbuf()
        cell_1T = gtk.CellRendererText()
        cell_2P = gtk.CellRendererPixbuf()
        cell_2T = gtk.CellRendererText()
        cell_3 = gtk.CellRendererText()
        model_1 = gtk.ListStore(gtk.gdk.Pixbuf, gobject.TYPE_STRING)
        model_2 = gtk.ListStore(gtk.gdk.Pixbuf, gobject.TYPE_STRING)
        model_3 = gtk.ListStore(gobject.TYPE_STRING)

        column_1P = gtk.TreeViewColumn("", cell_1P, pixbuf=0)
        column_1T = gtk.TreeViewColumn("", cell_1T, text=1)
        column_2P = gtk.TreeViewColumn("", cell_2P, pixbuf=0)
        column_2T = gtk.TreeViewColumn("", cell_2T, text=1)
        column_3 = gtk.TreeViewColumn("", cell_3, text=0)

        column_1P.set_clickable(True)
        column_1P.connect("clicked", self.vpaned_clicked, (15, self.height - 45))
        column_1T.set_clickable(True)
        column_1T.connect("clicked", self.vpaned_clicked, (15, self.height - 45))
        column_2P.set_clickable(True)
        column_2P.connect("clicked", self.vpaned_clicked, (15, self.height - 45))
        column_2T.set_clickable(True)
        column_2T.connect("clicked", self.vpaned_clicked, (15, self.height - 45))        
        column_3.set_clickable(True)
        column_3.connect("clicked", self.self_clicked)

        tree_1 = gtk.TreeView(model_1)
        tree_1.set_size_request(self.width, self.height - 50)
        tree_1.connect("scroll-event" , self.tree_scroll, 0)
        
        tree_2 = gtk.TreeView(model_2)
        tree_2.set_size_request(self.width, 15)
        tree_2.connect("scroll-event" , self.tree_scroll, 1)
        
        tree_3 = gtk.TreeView(model_3)

        self.vpaned = gtk.VPaned()
        self.vpaned.add(tree_1)
        self.vpaned.add(tree_2)
        self.pack1(self.vpaned)
        self.pack2(tree_3)

        tree_1.append_column(column_1P)
        tree_1.append_column(column_1T)
        tree_2.append_column(column_2P)
        tree_2.append_column(column_2T)
        tree_3.append_column(column_3)

        self.show_all()
        
    def get_treeview(self, x):
      return {
              0: self.vpaned.get_child1(),
              1: self.vpaned.get_child2(),
              2: self.get_child2()
              }.get(x, None)

    def set_count(self, ident, value):
      self.COUNT[ident] = value

    def tree_scroll(self, widget, event, ident):
        try:
          start, end = widget.get_visible_range()
          try:
            count = self.COUNT[ident]
          except:
            count = 0  
          widget.get_selection().unselect_all()
        except Exception, e:  
          pdb.gimp_message(e)
          pass
        else:
          if event.direction == gtk.gdk.SCROLL_UP:
            y = start[0] - self.SCROLL_STEP
            if y < 0:
              y = 0
          elif event.direction == gtk.gdk.SCROLL_DOWN:
            y = end[0] + self.SCROLL_STEP
            if y > count:
              y = count - 1
            
          widget.scroll_to_cell((y,))  
          
    def self_clicked(self, column):
        self.get_child1().set_position(15)
        if self.get_position() == 35:
          self.set_position(self.height - 25)
        else:
          self.set_position(35)

    def vpaned_clicked(self, column, param):
      pane = column.get_tree_view().parent
      self.set_position(self.height - 25)
      if pane.get_position() == param[1]:
        pane.set_position(param[0])
      else:
        pane.set_position(param[1])


class FlowChain(gtk.Image):
    IMG_DIR = os.path.join(SC_RESDIR, "chain.png")
    IMG_SIZE = 13;

    def __init__(self, layout, origin, width):
        gtk.Image.__init__(self)
        self.layout = layout
        self.input = None
        self.output = None

        self.context = gtk.gdk.pixbuf_new_from_file(self.IMG_DIR)
        pixbuf = self.context.scale_simple(width, self.IMG_SIZE, gtk.gdk.INTERP_HYPER)
        self.set_from_pixbuf(pixbuf)

        self.event_box = gtk.EventBox()
        self.event_box.set_visible_window(False)
        self.event_box.add(self)
        self.event_box.add_events(gtk.gdk.BUTTON_RELEASE_MASK)
        self.connect("destroy", self.destroy_event)
        self.event_box.connect("button-release-event", self.release)
        self.layout.put(self.event_box, origin[0], origin[1])
        self.layout.show_all()

    def destroy_event(self, widget, data=None):
         if not self.input is None:
           self.input.flowchain_out = None
         if not self.output is None:
           self.output.flowchain_in = None

    def release(self, widget, event):
        if event.button == 3:
           self.destroy()

    def redraw(self, origin):
        x1 = self.input.allocation[0]
        x2 = self.output.allocation[0]
        width = int(x2 - x1) - self.input.allocation[2]
        self.layout.move(self.event_box, origin[0], origin[1])

        pixbuf = self.context.scale_simple(width, self.IMG_SIZE, gtk.gdk.INTERP_HYPER)
        self.set_from_pixbuf(pixbuf)


class ScriptBlock(gtk.Fixed):
    BLOCK_TERMINATOR = ['none', 'control_play.png', 'control_stop.png']
    DISABLED_IMG = ['none', 'hide_in', 'hide_out']
    IN_ACTIVE = [0, 2]
    OUT_ACTIVE = [0, 1]

    def __init__(self, layout, caption, informer, inspector, procedure_prefix, bg_name):
        gtk.Fixed.__init__(self)
        self.drag = False
        self.drag_x = 0
        self.drag_y = 0
        self.layout = layout
        self.x = 10
        self.y = 10

        self.mode_flag = 0
        self.caption = caption
        self.inspector = inspector
        self.informer = informer
        self.flowchain_in = None
        self.flowchain_out = None
        self.BLOCK_BG = os.path.join(SC_RESDIR, bg_name)

        jsonFile = i18n_langfile('messages', get_lang_code())
        self.i18n_message = json.loads(open(jsonFile).read())

        self.labelfield = gtk.Label(caption)
        self.labelfield.set_single_line_mode(True)

        # Pango uses a constant called pango.SCALE(1024) to convert between pixels and pango's native unit
        # get_pango_context().get_font_description().get_size() / 1024
        font_size = self.labelfield.get_layout().get_pixel_size()

        self.height = 83
        if font_size[0] > 58:
           self.width = 58 * 2

           end = len(caption)
           while end > 0:
            self.labelfield.set_label(caption[0:end])
            font_size = self.labelfield.get_layout().get_pixel_size()
            if font_size[0] < self.width:
              break
            else:
              end = end - 1

        else:
           self.width = 58

        self.set_size_request(self.width, self.height)

        try:
          position = caption.index("-")
        except ValueError:
          position = len(caption)

        self.set_name("%s-block-%d" % (caption[0:position], BLOCK_TYPE["count"]))
        self.procedure_prefix = procedure_prefix
        
        self.x, self.y = detectWidgetXY((self.x, self.y, self.width, self.height))
        if (self.x + self.width) >= self.layout.parent.get_allocation()[3]:
          self.informer.set_label(self.i18n_message["WorkspaceRegionWarning"])
        else:
          ALLOCATIONS[self.get_name()] = (self.x, self.y, self.width, self.height)
          if BLOCK_TYPE.has_key(self.caption):
           LAST_INDEX = int(BLOCK_TYPE[self.caption])
          else:
           LAST_INDEX = 0

          if LAST_INDEX < MAX_COUNT:
            LAST_INDEX = int(LAST_INDEX) + 1
            BLOCK_TYPE[self.caption] = LAST_INDEX
            BLOCK_TYPE["count"] = int(BLOCK_TYPE["count"]) + 1

            self.background = gtk.Image()
            self.cloak = gtk.Image()
            self.mode_image = gtk.Image()
            if os.path.exists(self.BLOCK_BG):
              buf = gtk.gdk.pixbuf_new_from_file(self.BLOCK_BG)
              scaled_buf = buf.scale_simple(self.width, self.height, gtk.gdk.INTERP_BILINEAR)
              self.background.set_from_pixbuf(scaled_buf)

            self.add(self.background)
            self.put(self.labelfield, 3, 0)
            self.put(self.cloak, 0, 0)
            self.put(self.mode_image, int(self.width / 2) - 12, self.height - 26)

            self.event_box = gtk.EventBox()
            self.event_box.set_visible_window(False)
            self.event_box.add(self)
            self.event_box.add_events(gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK)
            self.event_box.connect("button-press-event", self.click)
            self.event_box.connect("button-release-event", self.release)
            self.event_box.connect("motion-notify-event", self.mousemove)
            self.connect("destroy", self.destroy_event)

            self.layout.put(self.event_box, self.x, self.y)
            self.layout.show_all()

            if not OUTPUT["active"]:
              try:
                proc_info(self.informer, self.procedure_prefix + self.caption)
              except Exception, e:
                pdb.gimp_message(e)
                
              try:  
                proc_params(self.inspector, self.procedure_prefix + self.caption, self.get_name())
              except  Exception, e:
                pdb.gimp_message(e)  
          else:
             info = self.i18n_message["TooManyCousinsException"]
             self.informer.set_label(info)

    def destroy_event(self, widget, data=None):
        if BLOCK_TYPE.has_key(self.caption):
                 BLOCK_TYPE[self.caption] = int(BLOCK_TYPE[self.caption]) - 1
                 BLOCK_TYPE["count"] = int(BLOCK_TYPE["count"]) - 1

        ALLOCATIONS.pop(self.get_name())
        if PROPS.has_key(self.caption):
          PROPS.pop(self.caption)
          
        self.informer.set_label(self.i18n_message["SelectItemInformation"])
        if self.mode_flag == 1:
          BLOCK_TYPE["start_block"] = None
          

    def click(self, widget, event):
        # Left button
        if event.button == 1:
           self.area = detectWidgetArea(self.get_pointer(), self.width, self.height)
           if self.area == 0:
             self.drag = True
             self.drag_x = event.x
             self.drag_y = event.y

             if not OUTPUT["active"]:
               proc_info(self.informer, self.procedure_prefix + self.caption)
               proc_params(self.inspector, self.procedure_prefix + self.caption, self.get_name())
           else:
             self.drag = False
             if self.area == 1 and self.mode_flag in self.IN_ACTIVE:
              x1 = OUTPUT["allocation"][0]
              x2 = self.get_allocation()[0]
              dist = int(x2 - x1) - OUTPUT["allocation"][2]

              if OUTPUT["widget"] is not None:
                if (self.flowchain_in is None) and (OUTPUT["widget"].flowchain_out is None):
                  if dist < 0:
                    self.informer.set_label(self.i18n_message["DistanceValueException"])
                  else:
                    if self.flowchain_out is None:
                      self.layout.move(self.event_box, self.x, int(OUTPUT["allocation"][1]) - 10)
                      self.update_xy()
                      ALLOCATIONS[self.get_name()] = self.get_allocation()
  
                      chain = FlowChain(self.layout, OUTPUT["pointer"], dist)
                      self.flowchain_in = chain
                      OUTPUT["active"] = False
                      OUTPUT["widget"].flowchain_out = chain
                      OUTPUT["widget"].set_mode(get_counter(self.counter_file, "mode_flag"))
                      chain.input = OUTPUT["widget"]
                      chain.output = self
                      self.informer.set_label(self.i18n_message["InputSucessfullyInformation"])
                else:
                  self.informer.set_label(self.i18n_message["ReservedConnectorException"])

             if self.area == 2 and self.mode_flag in self.OUT_ACTIVE:
              if self.flowchain_out is None:
                OUTPUT["active"] = True
                OUTPUT["widget"] = self
                OUTPUT["pointer"] = self.get_flowchain_xy()
                OUTPUT["allocation"] = self.get_allocation()
                set_counter(self.counter_file, "mode_flag", self.mode_flag)
                self.set_mode(3)
                self.informer.set_label(self.i18n_message["SelectInputInformation"])

        # Middle button or press Mouse wheel
        if event.button == 2:
            if OUTPUT["active"]:
               OUTPUT["widget"] = None
               OUTPUT["active"] = False
               self.set_mode(get_counter(self.counter_file, "mode_flag"))
               self.informer.set_label(self.i18n_message["CancelActionWarning"])
            else:
              self.mode_flag = self.mode_flag + 1
              if int(self.mode_flag) > 2:
                self.mode_flag = 0
              
              self.set_mode(self.mode_flag)
                

        # Right button
        if event.button == 3:
           self.informer.set_label(self.i18n_message["EmptyDescription"])
           for child in self.inspector.get_children():
             self.inspector.remove(child)

           if not self.flowchain_out is None:
             self.flowchain_out.destroy()
           if not self.flowchain_in is None:
             self.flowchain_in.destroy()

           self.destroy()

    def release(self, widget, event):
        if self.drag:
          self.drag = False
          self.window.set_cursor(None)
          self.get_toplevel().showbuttons()

          if self.x < 1:
            new_x = 10
          else:
              if self.x > self.layout.get_allocation()[2] - self.width:
                new_x = self.layout.get_allocation()[2] - self.width
              else:
               new_x = self.x
          if self.y < 1:
            new_y = 10
          else:
              if self.y > self.layout.get_allocation()[3] - self.height:
                new_y = self.layout.get_allocation()[3] - int("%.0f" % (self.height * 1.1))
              else:
               new_y = self.y

          self.layout.move(self.event_box, new_x, new_y)
          self.update_xy()

          ALLOCATIONS[self.get_name()] = self.get_allocation()

          if not self.flowchain_out is None:
            self.flowchain_out.redraw(self.get_flowchain_xy())

    def mousemove(self, widget, event):
        if self.drag:
            self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))
            self.get_toplevel().hidebuttons()

            in_exist = not self.flowchain_in is None
            out_exist = not self.flowchain_out is None
            if in_exist or out_exist:
              new_y = self.y
            else:
              new_y = self.y + int(event.y - self.drag_y)

            if out_exist:
              too_close = self.flowchain_out.get_pixbuf().get_width() < self.flowchain_out.IMG_SIZE
              if too_close:
                self.drag_x = self.drag_x + self.flowchain_out.IMG_SIZE

            if in_exist:
              too_close = self.flowchain_in.input.flowchain_out.get_pixbuf().get_width() < self.flowchain_in.IMG_SIZE
              if too_close:
                self.drag_x = self.drag_x - self.flowchain_in.IMG_SIZE

            if self.x < 1:
              new_x = 10
            else:
              if self.x > self.layout.get_allocation()[2] - self.width:
                new_x = self.layout.get_allocation()[2] - self.width
              else:
                new_x = self.x + int(event.x - self.drag_x)

            self.layout.move(self.event_box, new_x, new_y)
            self.update_xy()

            if out_exist:
              self.flowchain_out.redraw(self.get_flowchain_xy())

            if in_exist:
              if not self.flowchain_in.output is None:
                pt = self.flowchain_in.input.get_flowchain_xy()
                self.flowchain_in.input.flowchain_out.redraw(pt)

    def set_mode(self, mode):
      """
      0: both connector available, state of normal work 
      1: recognized as start block
      2: recognized as finishing block
      3: waiting to complete operation of flow-chain connection
      """
      stock_id = self.get_modestock_id(mode)
      if stock_id == 0:
        self.mode_image.clear()
      else:  
        self.mode_image.set_from_stock(stock_id, gtk.ICON_SIZE_LARGE_TOOLBAR)
      
      if mode > 2:
        cloak_id = 0
      else:
        cloak_id = mode  
      file = os.path.join(SC_RESDIR, "%s%d.png" % (self.DISABLED_IMG[cloak_id], self.width))
      if os.path.exists(file):
        self.cloak.set_from_file(file)
      else:
        self.cloak.clear()

      self.mode_flag = int(mode)
      if int(mode) == 1:
        if BLOCK_TYPE["start_block"] is None:
          BLOCK_TYPE["start_block"] = self
        elif BLOCK_TYPE["start_block"] != self:
          BLOCK_TYPE["start_block"].set_mode(0)
          BLOCK_TYPE["start_block"] = self
          

    def get_flowchain_xy(self):
      return self.translate_coordinates(self.layout, self.width - 1, int(self.height / 2) - 5)

    def get_modestock_id(self, x):
      return {
              1: gtk.STOCK_MEDIA_PLAY,
              2: gtk.STOCK_MEDIA_STOP,
              3: gtk.STOCK_MEDIA_RECORD
              }.get(x, 0)

    def update_xy(self):
      self.x, self.y = self.layout.child_get(self.event_box, 'x', 'y')


class ScriptComposer(gtk.Window):
    __gsignals__ = {
       "configure-event" : "override"
      }

    WIDGET_BACKGROUND_COLOR = "white"
    WIDGET_BORDER_SIZE = 10

    SCRIPT_PREFIX = "script-fu-"
    IPROC_PREFIX = "gimp-"
    TEMPLATE_FILE = 'scriptblank'
    TOTAL_COUNT = 15
    SCRIPT_COUNT = 1
    
    WIDTH = 800
    HEIGHT = 600

    def __init__(self):
        gtk.Window.__init__(self)

        self.i18n = []
        self.i18n_message = []
        self.outFile = ""
        self.language = ""
        self.editable = True
        self.datetime = datetime.datetime.now()
        # Generates a unique filename using the temporary path supplied in the user's gimprc.
        # Used by RegistrationManager class
        self.tempfile = pdb.gimp_temp_name("json").lower() 
        # Store all of counters
        # Used by ScriptBlock class
        self.counter_file = pdb.gimp_temp_name("txt").lower()
        self.set_border_width(self.WIDGET_BORDER_SIZE)
        self.set_size_request(self.WIDTH, self.HEIGHT)

        self.connect("delete-event", self.delete_event)
        self.connect("destroy", self.destroy_event)
   
        aval_lang = []
        try:
          self.language = L10n_const("language")
          isDir = os.path.isdir(i18n_langdir(self.language))
          if (len(self.language) != 3) or (not isDir):
            self.language = ""
            entries = os.listdir(os.path.join(SC_RESDIR, 'language'))
            for name in entries:
              dir = i18n_langdir(name)
              if os.path.isdir(dir):
                correctLDir = True
                for fname in SC_LANG_FILES:
                  file = i18n_langfile(fname, name)
                  if not os.path.exists(file):
                    correctLDir = False
                    break
                if correctLDir:
                  aval_lang.append(name)  
        except Exception, e:
          pdb.gimp_message(e)    
          self.destroy()
        else:
          if self.language == "":
              height = 50
              self.label_field = gtk.Label("")
              self.label_field.set_size_request(self.WIDTH - 100, height)
              self.label_field.set_line_wrap(True)
              
              frame = gtk.Frame("  ***  ")
              frame.set_shadow_type(gtk.SHADOW_ETCHED_IN)
              frame.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse(WIDGET_BORDER_COLOR))
              frame.get_label_widget().modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("RED"))
      
              vbox = gtk.VBox()
              bbox = gtk.VButtonBox()
              bbox.set_layout(gtk.BUTTONBOX_START)
              fbutton = gtk.Button("Invisible")
              bbox.add(fbutton)
              for lang in aval_lang:
                try:
                    label = i18n_const(self.__class__.__name__, lang, "languagedesc")
                except Exception, e:
                    err_file = i18n_langfile(self.__class__.__name__, lang)
                    pdb.gimp_message("%s in %s" % (e, err_file))    
                    label == ""
                else:
                   button = gtk.Button(label)
                   button.set_size_request(self.WIDTH / 2, height)
                   button.set_image(gtk.image_new_from_file(os.path.join(SC_RESDIR, 'language', lang, 'flag.jpg')))
                   button.add_events(gtk.gdk.ALL_EVENTS_MASK)
                   button.connect("clicked", self.set_language_clicked, lang)
               
                   button.get_image().show()
                   bbox.add(button)
                     
              vbox.add(bbox)
              vbox.add(self.label_field)
              frame.add(vbox)
              
              self.add(frame)

              self.show_all()
              fbutton.hide()

        if len(self.language) == 3:
          try:
            langFile = i18n_langfile(self.__class__.__name__, self.language)
            langData = open(langFile).read().decode("windows-1250")
            self.i18n = json.loads(langData)
          except Exception, e:
            pdb.gimp_message("%s in %s" % (e, langFile))
            
          try:  
            msgFile = i18n_langfile("messages", self.language)
            msgData = open(msgFile).read().decode("windows-1250")
            self.i18n_message = json.loads(msgData)
          except Exception, e:
            pdb.gimp_message("%s in %s" % (e, msgFile))  
          
          try:
            self.set_title(self.i18n["windowtitle"])
          except:
            self.set_title(SC_TITLE)  
              
          self.list = self.create_paletteview()
           
          i_view = gtk.Viewport()
          i_view.set_shadow_type(gtk.SHADOW_NONE)
          i_view.set_size_request(200, 200)
           
          self.inspector = gtk.Fixed()
          i_view.add(self.inspector)
           
          i_window = gtk.ScrolledWindow()
          i_window.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
          i_window.add(i_view)
           
          self.vpaned = gtk.VPaned()
          self.hpaned = gtk.HPaned()
          self.add(self.hpaned)
           
          vbox = gtk.VBox()
          vbox.add(self.list)
          vbox.add(gtk.HSeparator())
          vbox.add(i_window)
          try:
            button = gtk.Button(label=self.i18n["manualabel"])
          except:
            button = gtk.Button(stock=gtk.STOCK_HELP)  
            button.get_image().show()                    
             
          vbox.add(button)
          button.add_events(gtk.gdk.ALL_EVENTS_MASK)  
          button.connect("clicked", self.manual_clicked)  

          try:
            vbox_title = self.i18n["leftboxtitle"]
          except:
            vbox_title = ""  
          vbox_frame = gtk.Frame(vbox_title)
          vbox_frame.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
          vbox_frame.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse(WIDGET_BORDER_COLOR))
          vbox_frame.add(vbox)
          
          self.hpaned.add1(vbox_frame)
          self.hpaned.add2(self.vpaned)
          self.vpaned.set_size_request(550, 600)
          
          self.buildbutton = self.create_build()
          self.configurebutton = self.create_configure()
          self.registerbutton = self.create_register()
          
          self.layout = gtk.Fixed()
          self.layout.set_size_request(550, 500)
          self.layout.put(self.configurebutton, 550 - self.WIDGET_BORDER_SIZE, 1)
          self.layout.put(self.registerbutton, 550 - self.WIDGET_BORDER_SIZE, 52 + 10)
          self.layout.put(self.buildbutton, 550 - self.WIDGET_BORDER_SIZE, 2 * (52 + 10))

          self.label_box = gtk.EventBox()
          self.label_field = gtk.Label()
          self.label_field.set_line_wrap(True)
          self.label_field.set_alignment(xalign=1, yalign=0.1)
          try:
            self.label_field.set_label(self.i18n_message["SelectItemInformation"])
          except:
            pass  
          self.label_field.set_size_request(550, 100)
          self.label_box.add(self.label_field)
          self.label_box.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse(self.WIDGET_BACKGROUND_COLOR))

          try:
            frame_title = self.i18n["bottomframetitle"]
          except:
             frame_title = "" 
          frame = gtk.Frame(frame_title)
          frame.add(self.label_box)
          frame.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
          frame.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse(WIDGET_BORDER_COLOR))

          self.vpaned.add1(self.layout)
          self.vpaned.add2(frame)
          self.show_all()
          self.label_field.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.PIRATE))
          self.connect("show-help", self.help_event)
          
          self.show_all()
          
    def main(self):
        gtk.main()

    def set_language_clicked(self, widget, langCode):
        try:
          langFile = i18n_langfile("messages", langCode)
          i18n = json.loads(open(langFile).read())
          label = i18n["DoneDefLangInformation"]
        except IOError, e:
          self.label_field.set_text("Input/Output ERROR => %s" % e)
        except (KeyError, ValueError, TypeError):
          self.label_field.set_text("CONTENT ERROR => Integrity internal error of language pack. Choose other language")
        else:
          try:
            cxt = json.loads(open(SC_CONFIGFILE).read())
            cxt["language"] = langCode
          except Exception, e:
            self.label_field.set_text(e)
          else:  
            try:
              with open(SC_CONFIGFILE, 'w') as f:
                json.dump(cxt, f)
            except IOError:
              self.label_field.set_text("Input/Output ERROR => %s" % e)
            else:  
              self.label_field.set_text(label)    
      
    def create_register(self):
      image = gtk.Image()
      file = os.path.join(SC_RESDIR, "register.png")
      image.set_from_file(file)
      
      box = gtk.EventBox()
      box.add_events(gtk.gdk.KEY_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.LEAVE_NOTIFY_MASK)
      box.set_tooltip_text(self.i18n["registertooltip"])
      
      box.connect("motion-notify-event", self.motion_event)
      box.connect("leave-notify-event", self.leave_event)
      box.connect("button-release-event", self.showregistrationform)      
        
      box.add(image)
      return box
      
    def create_paletteview(self):
        tree_group = TreeViewGroup(200, 350)
        
        prefix = [self.SCRIPT_PREFIX,self.IPROC_PREFIX]
        title  = ["treeviewtitle","i_treeviewtitle"]
        bg_name= ['script.png','brush.png']
        for x in range (0, 2):
          tree = tree_group.get_treeview(x)
          tree.get_column(1).set_title(self.i18n[title[x]])
          model = tree.get_model()
          model.set_sort_column_id(1, gtk.SORT_ASCENDING)
          model.set_default_sort_func(None)
          nproc = 0
          
          for name in pdb.query():
            if name.startswith(prefix[x]):
                proc = slugify(name).replace(prefix[x], "")
                nproc += 1
                nparams = pdb[name].nparams
                for p in pdb[name].params:
                   if p[1] in HIDDEN_PARAM_TYPE:
                     nparams -= 1
                fname = get_nparams_icon(bg_name[x],nparams)
                if os.path.exists(fname):
                  pixbuf = gtk.gdk.pixbuf_new_from_file(fname)
                else:
                  pixbuf = None  
                   
                model.append([pixbuf, proc])          
                VISIBLED_NPARAMS[name] = nparams   
          
          tree_group.set_count(x, nproc)
          tree.add_events(gtk.gdk.KEY_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.LEAVE_NOTIFY_MASK)
    
          tree.connect("motion-notify-event", self.motion_event)
          tree.connect("leave-notify-event", self.leave_event)
          
        tree_group.get_treeview(0).connect('row-activated', self.row_activated_0)
        tree_group.get_treeview(1).connect('row-activated', self.row_activated_1)
       
        return tree_group

    def create_build(self):
        image = gtk.Image()
        file = os.path.join(SC_RESDIR, "build.png")
        image.set_from_file(file)

        box = gtk.EventBox()
        box.add_events(gtk.gdk.KEY_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.LEAVE_NOTIFY_MASK)
        box.set_tooltip_text(self.i18n["buildtooltip"])

        box.connect("motion-notify-event", self.motion_event)

        box.connect("leave-notify-event", self.leave_event)
        box.connect("button-release-event", self.buildscript)

        box.add(image)
        return box

    def create_configure(self):
        image = gtk.Image()
        file = os.path.join(SC_RESDIR, "configure.png")
        image.set_from_file(file)

        box = gtk.EventBox()
        box.add_events(gtk.gdk.KEY_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.LEAVE_NOTIFY_MASK)
        box.set_tooltip_text(self.i18n["configuretooltip"])

        box.connect("motion-notify-event", self.motion_event)
        box.connect("leave-notify-event", self.leave_event)
        box.connect("button-release-event", self.showmanager)

        box.add(image)
        return box

    def do_configure_event(self, event):
        gtk.Window.do_configure_event(self, event)

        parent = self.buildbutton.parent
        wt, ht = self.translate_coordinates(parent, event.width, event.height)
        x = wt - self.configurebutton.allocation[2] - self.WIDGET_BORDER_SIZE

        parent.put(self.configurebutton, x, 1)
        parent.put(self.buildbutton, x, 2 * 62)
        parent.put(self.registerbutton, x, 62)

    def leave_event(self, widget, event):
      widget.window.set_cursor(None)

    def motion_event(self, widget, event):
        hand = gtk.gdk.Cursor(gtk.gdk.HAND1)
        widget.window.set_cursor(hand)

    def delete_event(self, widget, event, data=None):
        return False

    def destroy_event(self, widget, data=None):
        try:
          os.unlink(self.tempfile)
          os.unlink(self.counter_file)
        except IOError:
          pass  
        gtk.main_quit()

    def manual_clicked(self, widget):
      self.help_event(widget , gtk.WIDGET_HELP_WHATS_THIS)

    def help_event(self, widget, help_type):
      try:
        pdb.plug_in_web_browser(get_helpfile(self.language))
      except Exception, e:
        pdb.gimp_message("%s" % e)
      return True
      
    def hidebuttons(self):
      self.buildbutton.hide()
      self.configurebutton.hide()
      self.registerbutton.hide()

    def showbuttons(self):
      self.configurebutton.show()
      self.buildbutton.show()
      self.registerbutton.show()

    def showregistrationform(self, widget, event):
      self.editable = False
      for child in self.layout.get_children():
        child.hide()
        
      self.layout.add(RegistrationManager(self.i18n["registermanagertitle"], self.tempfile))
      
    def showmanager(self, widget, event):
      self.editable = False
      for child in self.layout.get_children():
        child.hide()

      self.layout.add(PreferencesManager(self.i18n["preferencesmanagertitle"]))

    def buildscript(self, widget, event):
        block = BLOCK_TYPE["start_block"]
        if block is not None:
          copyright_person = get_logon_name()
          template_file = os.path.join(SC_RESDIR, self.TEMPLATE_FILE)
          dirname = L10n_const('storage')
          if not os.path.exists(template_file):
            self.label_field.set_label(self.i18n_message["TemplateNotFoundException"].format(*[self.TEMPLATE_FILE, SC_RESDIR]))
          elif not os.path.exists(dirname):
            self.label_field.set_label(self.i18n_message["StorageNotFoundException"])
          else:
            template = open(template_file).read()
            scriptlabel = ""
            scriptdescription = ""
            if os.path.exists(self.tempfile):
              try:
                reg_data = json.loads(open(self.tempfile).read())
              except:
                pass
              else:
                if reg_data.has_key("scriptlabel"):
                  scriptlabel = reg_data["scriptlabel"]
                if reg_data.has_key("scriptdescription"):  
                  scriptdescription = reg_data["scriptdescription"]
                  
            cmp = block
            values, params = list_params(cmp.get_name(), self.counter_file)
            script = "(%s%s %s)\n" % (get_block_prefix(cmp.get_name()), cmp.caption, values)
            in_params, fields = dialog_fields(params)
            cmp = cmp.flowchain_out
            if cmp is not None:
              while True:
                if cmp is not None:
                  cmp = cmp.output
                  if cmp is not None:
                    cmp_values, cmp_params = list_params(cmp.get_name(), self.counter_file)
                    script = script + "(%s%s %s)\n" % (get_block_prefix(cmp.get_name()), cmp.caption, cmp_values)
                    cmp_inparams, cmp_fields = dialog_fields(cmp_params)
                    in_params = "%s %s" % (in_params, cmp_inparams)
                    fields.extend(cmp_fields)
                    cmp = cmp.flowchain_out
                    if cmp is None:
                      scriptname = slugify(self.__class__.__name__)
                      # http://docs.python.org/2/library/datetime.html#strftime-strptime-behavior
                      monthname = self.datetime.strftime("%B")
                      if self.outFile == "":
                        basename = "%s-%d%d" % (scriptname, self.datetime.year, self.datetime.month)
                        self.outFile = os.path.join(dirname, "%s-%d.scm" % (basename, self.SCRIPT_COUNT))
                        if os.path.exists(self.outFile):
                          while os.path.exists(self.outFile):
                            self.SCRIPT_COUNT = self.SCRIPT_COUNT + 1
                            self.outFile = os.path.join(dirname, "%s-%d.scm" % (basename, self.SCRIPT_COUNT))
                            
                        self.get_toplevel().set_title("%s :: %s-%d.scm" % (i18n_const("scriptcomposer", get_lang_code(), "windowtitle"), basename, self.SCRIPT_COUNT))

                      if scriptlabel == "":
                        scriptlabel = str(self.SCRIPT_COUNT)
                        
                      data = [scriptname, self.SCRIPT_COUNT, in_params, script, self.datetime.year, monthname, scriptlabel, scriptdescription, copyright_person, '\n'.join(fields)]
                      with open(self.outFile, 'w') as f :
                        f.write(template.format(*data))
                        f.close()
  
                      self.label_field.set_label(self.i18n_message["BuildSuccessFullyInformation"])
                      break
                    
    def row_activated_0(self, treeview, path, view_column):
      if self.editable:
           (model, iter) = treeview.get_selection().get_selected()

           if iter is not None:
             name_of_data = model.get_value(iter, 1)
           # Check number of all modules
             if BLOCK_TYPE["count"] >= self.TOTAL_COUNT:
                self.label_field.set_label(self.i18n_message["TooManyObjectsException"])
             else:
           # Let's creat!e block for our module
                block = ScriptBlock(self.layout, name_of_data, self.label_field, self.inspector, self.SCRIPT_PREFIX, "script_blank.jpg")
                block.counter_file = self.counter_file
  
                if BLOCK_TYPE["count"] <= 1:
                  block.set_mode(1)

    def row_activated_1(self, treeview, path, view_column):
      if self.editable:
           (model, iter) = treeview.get_selection().get_selected()

           if iter is not None:
             name_of_data = model.get_value(iter, 1)
           # Check number of all modules
             if BLOCK_TYPE["count"] >= self.TOTAL_COUNT:
                self.label_field.set_label(self.i18n_message["TooManyObjectsException"])
             else:
           # Let's creat!e block for our module
                block = ScriptBlock(self.layout, name_of_data, self.label_field, self.inspector, self.IPROC_PREFIX, "procedure_blank.jpg")
                block.counter_file = self.counter_file
  
                if BLOCK_TYPE["count"] <= 1:
                  block.set_mode(1)

class RegistrationManager(gtk.Frame):
    WIDGET_BORDER_SIZE = 10
    ENTRY = {}
    
    def __init__(self, caption, storage):
      gtk.Frame.__init__(self, caption)
      self.set_size_request(550, 130)
      self.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
      self.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse(WIDGET_BORDER_COLOR))
      self.connect("destroy", self.destroy_event)
      
      self.normal_mode = True
      self.storage = storage
      langFile = i18n_langfile(self.__class__.__name__, get_lang_code())
      try:
        self.i18n = json.loads(open(langFile).read())
      except IOError:
        self.normal_mode = False
        label = gtk.Label("Need " + langFile)  
        button = gtk.Button(stock=gtk.STOCK_CANCEL)
        hbox = gtk.HBox()
        hbox.pack_start(label)
        hbox.pack_end(button)
        hbox.set_border_width(self.WIDGET_BORDER_SIZE)
        
        button.connect("clicked", self.confirm)
        self.add(hbox)
        self.set_size_request(550, 60)               
      else:
        n_box = gtk.HBox()
        n_box.set_border_width(self.WIDGET_BORDER_SIZE)
        label = gtk.Label(self.i18n["caption"]["scriptlabel"])
        self.n_entry = gtk.Entry()        
        self.n_entry.connect("changed", self.changed_event)  
        self.n_entry.set_max_length(25)
        n_box.pack_start(label)
        n_box.pack_end(self.n_entry)
        label.set_size_request(90, 20)
        
        d_box = gtk.HBox()
        self.d_entry = gtk.Entry()
        self.d_entry.set_max_length(125)   
        label = gtk.Label(self.i18n["caption"]["scriptdescription"])
        d_box.set_border_width(self.WIDGET_BORDER_SIZE)
        d_box.add(label)     
        d_box.add(self.d_entry)
        label.set_size_request(90, 20)
        
        m_button = gtk.Button(label=self.i18n["caption"]["close"])
        m_button.set_image(gtk.image_new_from_stock(gtk.STOCK_APPLY, gtk.ICON_SIZE_BUTTON))
        m_button.get_image().show()        
        m_button.connect("clicked", self.confirm)
        
        vbox = gtk.VBox()
        vbox.add(n_box)
        vbox.add(d_box)
        vbox.add(m_button)
        self.add(vbox)
        
        self.ENTRY["scriptlabel"] = self.n_entry
        self.ENTRY["scriptdescription"] = self.d_entry  
          
      self.propagate_config()
      self.show_all()

    '''
    @see: http://stackoverflow.com/questions/2726839/creating-a-pygtk-text-field-that-only-accepts-number
    @author: Ivo Wetzel
    '''
    def changed_event(self, widget):
      text = widget.get_text().lstrip()
      widget.set_text("".join([i for i in text if i in "abcdefghijklmnopqrtsuvwxyzABCDEFGHIJKLMNOPQRTSUVWXYZ0123456789 -_"]))      
      
    def destroy_event(self, widget):
      if self.normal_mode:
          data = {}
          for name in self.ENTRY:
            if self.ENTRY[name] is not None:
              data[name] = self.ENTRY[name].get_text()
          
          try:
            with open(self.storage, 'w') as f:
              json.dump(data, f)
          except IOError:
            pass
      
    def propagate_config(self):
      try:
        data = json.loads(open(self.storage).read())
      except IOError:
        pass
      else:  
        for name in self.ENTRY:
          try:
            value = data[name]
          except:
            value = ""
            
          if self.ENTRY[name] is not None:
            self.ENTRY[name].set_text(value)
        
    def confirm(self, button):
       self.get_toplevel().editable = True
       for child in self.parent.get_children():
         child.show()

       self.destroy()     


class PreferencesManager(gtk.Frame):
    WIDGET_BORDER_SIZE = 10

    def __init__(self, caption):
      gtk.Frame.__init__(self, caption)
      self.set_size_request(550, 150)
      self.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
      self.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse(WIDGET_BORDER_COLOR))

      self.l_combo_active = -1
      try:
        langFile = i18n_langfile(self.__class__.__name__, get_lang_code())
        langData = open(langFile).read().decode("windows-1250")
        self.i18n = json.loads(langData)
      except IOError:
        label = gtk.Label("Need " + langFile)
        button = gtk.Button(stock=gtk.STOCK_CANCEL)
        hbox = gtk.HBox()
        hbox.pack_start(label)
        hbox.pack_end(button)
        hbox.set_border_width(self.WIDGET_BORDER_SIZE)
        
        button.connect("clicked", self.clicked_event)
        self.add(hbox)
        self.set_size_request(550, 60)
      else:
        s_box = gtk.HBox()
        s_box.set_border_width(self.WIDGET_BORDER_SIZE)
        s_label = gtk.Label(self.i18n["caption"]["storage"])
        s_label.set_size_request(90, 20)
        
        self.s_entry = gtk.Entry()
        self.s_entry.connect("button-release-event", self.set_dir)
        self.s_entry.connect("key-press-event", self.set_dir)
  
        s_box.add(s_label)
        s_box.add(self.s_entry)
  
        l_box = gtk.HBox()
        l_box.set_border_width(self.WIDGET_BORDER_SIZE)
        l_label = gtk.Label(self.i18n["caption"]["language"])
        self.l_image = gtk.image_new_from_file(os.path.join(SC_RESDIR, 'language', get_lang_code(), 'flag.jpg'))
        self.l_combo = gtk.combo_box_new_text ()
        
        l_box.add(l_label)
        l_box.add(self.l_image)
        l_box.add(self.l_combo)
        self.l_combo.set_size_request(175, 25)
  
        b_box = gtk.HBox()
        b_button = gtk.Button(self.i18n["caption"]["close"])
        b_cancelbutton = gtk.Button(label=self.i18n["caption"]["cancel"])
        """
        You can show explicitly the button image, justly, 
        Gtk+ developers do not recommend doing this because 
        it's overrides the Gtk+ user configuration.
        
        http://stackoverflow.com/questions/2188659/stock-icons-not-shown-on-buttons
        """
        b_button.set_image(gtk.image_new_from_stock(gtk.STOCK_APPLY, gtk.ICON_SIZE_BUTTON))
        b_button.get_image().show()
        b_button.connect("clicked", self.clicked_event)
        
        b_cancelbutton.set_image(gtk.image_new_from_stock(gtk.STOCK_CANCEL, gtk.ICON_SIZE_BUTTON))
        b_cancelbutton.get_image().show()
        b_cancelbutton.connect("clicked", self.cancel_event)
        b_box.add(b_button)
        b_box.add(b_cancelbutton)
  
        m_box = gtk.VBox()
        m_box.add(s_box)
        m_box.add(l_box)
        m_box.add(b_box)
  
        self.add(m_box)
        self.propagate_config()
      self.show_all()

    def cancel_event(self, button):
      self.get_toplevel().editable = True
      for child in self.get_toplevel().layout.get_children():
        child.show()
        
      self.destroy()  
      
    def clicked_event(self, button):
      self.dump_config()
      for child in self.get_children():
        child.destroy()

      msg = i18n_const("messages", get_lang_code(), "SettingUpdatedInformation")
      label = gtk.Label(msg)
      self.add(label)
      label.show()

    def propagate_config(self):
      if os.path.exists(SC_CONFIGFILE):
        data = json.loads(open(SC_CONFIGFILE).read())
        # Storage
        try: 
          storage = data["storage"]
        except (KeyError, ValueError, TypeError):
          storage = ""
        self.s_entry.set_text(storage)    
        # Language Packs
        l_entries = os.listdir(os.path.join(SC_RESDIR, 'language'))
        index = 0
        for code in l_entries:
          try:
            lang = SC_LANG_LABELS[code.lower()]
          except:
            continue
          else:    
            correctPack = True
            for fname in SC_LANG_FILES:
              file = i18n_langfile(fname, code)
              if not os.path.exists(file):
                correctPack = False
                break
               
            if correctPack:  
              self.l_combo.append_text(lang)
              if get_lang_code() == code.lower():
                self.l_combo.set_active(index)
              index += 1
        
    def dump_config(self):
      data = {}
      try:
        data["storage"] = self.s_entry.get_text().lower()
        text = self.l_combo.get_active_text()
        
        for code in SC_LANG_LABELS:
          if text == SC_LANG_LABELS[code]:
            data["language"] = code
            break
      except:
        pass
      else:  
        try:
          with open(SC_CONFIGFILE, 'w') as f:
            json.dump(data, f)
        except IOError:
          pass    

    def set_dir(self, widget, event):
      MessageBox = ctypes.windll.user32.MessageBoxA
      response = ""

      # Check for new pygtk: this is new class in PyGtk 2.4
      if (gtk.pygtk_version < (2, 3, 90)) :
              MessageBox(None, self.i18n["versionException"], self.i18n["title"]["versionexception"], 0)
              return False

      dialog = gtk.FileChooserDialog(
              self.i18n["title"]["choosestorage"],
              None,
              gtk.FILE_CHOOSER_ACTION_OPEN,
              (
                      gtk.STOCK_CANCEL,
                      gtk.RESPONSE_CANCEL,
                      gtk.STOCK_OPEN,
                      gtk.RESPONSE_OK
              )
      )
      dialog.set_default_response(gtk.RESPONSE_OK)
      dialog.set_action(gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)

      if (dialog.run() == gtk.RESPONSE_OK) :
              response = dialog.get_filename()
              widget.set_text(response)
      dialog.destroy()

      return True


def runner():
    ScriptComposer().main()

    return None


def i18n_langdir(locale):
  return os.path.join(SC_RESDIR, 'language', locale)

def i18n_langfile(basename, locale):
  langFile = os.path.join(SC_RESDIR, 'language', locale, "%s%s" % (basename, ".json"))
  langFile = re.sub('\\\\', '/', langFile)
  return langFile.strip().lower()

def i18n_const(basename, locale, prop):
  file = i18n_langfile(basename, locale)
  
  try:
    text = open(file).read()
    text = text.decode("windows-1250")
    matrix = json.loads(text)
  except IOError:
    return ""

  try:
    return matrix[prop]
  except:
    return ""


def L10n_const(prop):
  try:
    matrix = json.loads(open(SC_CONFIGFILE).read())
  except IOError:
    return ""

  try:
    return matrix[prop]
  except (KeyError, ValueError, TypeError):
    return ""

def get_lang_code():
  return L10n_const("language")

def get_logon_name():
  result = ""
  try:
    GetUserNameEx = ctypes.windll.secur32.GetUserNameExW
    NameDisplay = 2
 
    size = ctypes.pointer(ctypes.c_ulong(0))
    GetUserNameEx(NameDisplay, None, size)
 
    nameBuffer = ctypes.create_unicode_buffer(size.contents.value)
    GetUserNameEx(NameDisplay, nameBuffer, size)
  except:
    return ""
  else:  
    try:
      index = nameBuffer.value.index("\\")
      name = nameBuffer.value[index + 1:]
    except:
      name = nameBuffer.value
  
  return name

def get_nparams_icon(bg_name,x):
  dir = os.path.join(SC_RESDIR, 'numbers')
  file = '%s%s%sicon_%d.png' % (dir, os.path.sep, bg_name[0], x)
  file = re.sub('\\\\', '/', file)
  try:
    if os.path.exists(file):
      return file
    else:
      if not os.path.exists(dir):
        os.mkdir(dir)
      
      bg_file = os.path.join(SC_RESDIR, bg_name)
      
      background = pdb.file_png_load(bg_file, bg_name)
      textLay = pdb.gimp_text_layer_new(background, slugify(x), 'Sans', 18, PIXELS)
      background.add_layer(textLay)
      
      b_width = pdb.gimp_image_width(background)
      d_width = pdb.gimp_drawable_width(textLay)
      b_height = pdb.gimp_image_height(background)
      d_height = pdb.gimp_drawable_height(textLay)
      
      if x <= 9:
        d_x = int(b_width / 2 - d_width) + 2
      else:  
        d_x = int(b_width / 2 - d_width / 2)
      d_y = int(b_height / 2 - d_height / 2)
      
      pdb.gimp_layer_set_offsets(textLay, d_x, d_y)
      pdb.gimp_image_flatten(background)
      drawable = pdb.gimp_image_active_drawable(background)
      pdb.file_png_save_defaults(background, drawable, file, file)
      
      return file
  except:
    return ""   
  
def get_helpfile(locale):
  file = os.path.join(SC_RESDIR, 'help', "run-%s.html" % (locale))  
  if not os.path.exists(file):
    file = os.path.join(SC_RESDIR, 'help', "run-eng.html")  
  return file  

def get_inc_counter(file, name):
  c = get_counter(file, name)
  c = c +1
  set_counter(file, name, c)
  return c
  
def get_counter(file, name):
  if os.path.exists(file):
    try:
      data = json.loads(open(file).read())
    except:
      result = 0
    else:
      try:
        result = int(data[name])
      except:
        result = 0     
  else:
    result = 0     
   
  return result  

def set_counter(file, name, value):
  if os.path.exists(file):
    try:
      data = json.loads(open(file).read())
    except:
      data = {}
  else:
     data = {}
      
  data[name] = value
  try:
    with open(file, 'w') as f:
      json.dump(data, f)
  except IOError:
    pass       
        
def get_block_prefix(block_name):
  try:
    return BLOCK_PREFIX[block_name]
  except:
    return ""
          
def set_block_prefix(script_name):
  if script_name.startswith("script-fu-"):
    value = "script-fu-"
  elif script_name.startswith("gimp-"):  
    value = "gimp-"
  else:
    value = ""  
  
  return value
  

def detectWidgetArea(pointer, width, height):
  """
   Usage of widget area
   O: Everywhere; Nothing special, can get drag & drop action
   1: Narrow range of area; Input connection of flow-chain
   2: Narrow range of area; Output connection of flow-chain
  """
  area = 0
  conector_y = pointer[1] > 35 and pointer[1] < 50
  conector_xin = pointer[0] > 0 and pointer[0] < 10
  conector_xout = pointer[0] > 45 and pointer[0] < width

  if conector_y and conector_xin:
    area = 1
  if conector_y and conector_xout:
    area = 2

  return area

def detectWidgetXY(source):
    XY = (source[0], source[1])
    X = 10
    for allocation in ALLOCATIONS:
      rect = ALLOCATIONS[allocation]
      if rect[1] <= XY[1]:
        region = ((rect[0], rect[1]), (rect[0] + rect[2], rect[1]), (rect[0] + rect[2], rect[3]), (rect[0], rect[1] + rect[3]))
        X = X + rect[2] + 20
        if point_in_convex_polygon(region, XY):
          XY = (X, source[1])

    return XY

def point_in_convex_polygon(points, (x, y)):
  """
  Returns true if point (x,y) lie inside
  of convex polygon given with points
  """
  side = None
  n = len(points)
  for i in xrange(n):
    xi, yi = points[i]
    xj, yj = points[(i + 1) % n]

    d = (x - xi) * (yj - yi) - (y - yi) * (xj - xi)
    if d == 0.0:
      continue
    else:
      if side is None:
        side = d > 0.0

      # point have to lie at the same side of all lines
      elif (d > 0.0) != side:
        return False

  return True

def default_value(x):
  return {
# # Fundamental         
           'image' : 0
          , 'drawable': 0 
          , 'string': '""'
  # red green blue
          , 'color' : "'(255 255 255)"
          , 'toggle': 'TRUE'
# # Extended  
  # value lower upper step_inc page_inc decimal_part type
          , 'font' : '"Sans"'
          , 'brush' : "'(\"Circle (03)\" 100 44 0)"
          , 'pattern' : '"Ice"'
          , 'gradient' : '"FG to Transparent"'
          , 'palette' : '""'
# # Additional  
          , 'filename' : '""'
          , 'dirname': '""'
  # default first second third n-option
          , 'option' : "'()"
# # Advanced
          , 'value'  : '""' 
          }.get(x, None)

def dialog_fields(data):
  result = ""
  fields = []
  try:
    for p in data:
      def_value = default_value(p[0].lower())
      if def_value is not None:
        p_name = slugify(p[1])
        result = '%s %s' % (result, p_name)
        field = '\tSF-%s "%s" %s' % (p[0].upper(), p[2], def_value)
        fields.append(field)
  except Exception, e:
    result = ""
    fields = []
      
  return (result.strip(), fields)

def proc_info(widget, name):
     proc = pdb[name]
     formated_information = "%s\r\n%s" % (proc.proc_author, proc.proc_blurb)
     widget.set_label(formated_information)

def proc_params(widget, script_name, block_name):
     params = pdb[script_name].params
     try:
       nparams = VISIBLED_NPARAMS[script_name]
     except:  
       nparams = pdb[script_name].nparams
       
     attach = 0
     ht = 0

     BLOCK_PREFIX[block_name] = set_block_prefix(script_name)
     
     if not PROPS.has_key(block_name):
        PROPS[block_name] = {}
        PROPS[block_name]["script_name"] = script_name
        PROPS[block_name]["values"] = {}
        PROPS[block_name]["names"] = {}

     widget.parent.get_vadjustment().set_value(0)
     for child in widget.get_children():
        if child.__class__.__name__ == "Entry":
          try:
            position = child.get_name().index("_")
          except ValueError:
            position = len(child.get_name())

          b_name = child.get_name()[0:position]
          if PROPS.has_key(b_name):
            args = PROPS[b_name]
            args["values"][attach] = child.get_text()

            attach = attach + 1

        widget.remove(child)

     attach = 0
     if nparams <= 0:
        message = i18n_const('messages', get_lang_code(), "EmptyPropertiesListInformation")
        label = gtk.Label(message)
        label.set_size_request(180, 50)
        label.set_line_wrap(True)
        widget.put(label, 5, 20 + 1)
     else:
       for p in params:
         p_type = p[1]
         if (p_type in HIDDEN_PARAM_TYPE):
           continue
         else:
           if len(p[2]) > 28:
             label_text = p[2][0:25] + "..."
           else:
             label_text = p[2][0:28]

           label = gtk.Label(label_text)
           widget.put(label, 5, (20 * ht * attach) + 1)

           entry = gtk.Entry()
           entry.set_max_length(20)
           entry.set_name("%s_%d" % (block_name, attach))
           entry.connect("grab-focus", entry_params_focus, (attach, AMOUNT_OFENTRY))

           args = PROPS[block_name]["values"]
           if args.has_key(attach):
             p_secured_description = slugify(p[2]) 
             value = str(args[attach])
             entry.set_text(value)
             if not (p_type in BUILD_EXCLUDE_TYPE):
               PROPS[block_name]["names"][p_secured_description] = attach

           widget.put(entry, 5, (20 * ht * attach) + 1 + 20 - 4)
           attach = attach + 1
           ht = 2
       
       widget.put(gtk.HSeparator(), 5, (20 * ht * attach) + 10)
       
     widget.show_all()
      
def list_params(block_name, counter_file):
    user = PROPS[block_name]
    all = pdb[user["script_name"]].params
    result = ""
    p_value = ""
    params = []
    ident = get_inc_counter(counter_file, "list_params")
    for p in all:
      if (p[1] in BUILD_EXCLUDE_TYPE):
        continue
      else:
        isPresent = False
        hasValue = False
        p_name = slugify(p[2])
        if user["names"].has_key(p_name):
            index = user["names"][p_name]
            p_value = str(user["values"][index]).strip()
            if "" == p_value:
              p_value = p_name
              hasValue = False
              isPresent = True
            else:
              hasValue = True  
        else:  
          p_value = p_name
          isPresent = True
        
        if isPresent:    
          p_value = "_%d%s" % (ident, p_value)
        
        result = result + " %s" % p_value 
        if not hasValue:
          params.append((p[1], p_value, p[2]))
      
    return (result.strip(), params)

def slugify(value):
    """
    Normalizes string, converts to lowercase, removes non-alpha characters,
    and converts spaces to hyphens.

    Based on code in: Django
    """
    value = unicodedata.normalize('NFKD', unicode(value))
    value = value.encode('ascii', 'ignore')
    value = re.sub('[^\w\s-]', '', value).strip().lower()
    value = re.sub('[-\s]+', '-', value)

    return value

def entry_params_focus(widget, data):
   try: 
     quotient = (data[0] * data[1]) // data[1]
     if quotient in [5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 65, 70, 75, 80, 85, 90, 95, 100]:
        win = widget.parent.parent
        adj = win.get_vadjustment()
        adj.set_value(widget.allocation[1] - 20)
        widget.queue_resize()
   except Exception, e:
      pdb.gimp_message(e)
      pass  
    

register(
         SC_PROCEDURE_NAME
        , "Complete RAD tool for Gimp to build new script with currently exists."
        , "Complete RAD tool for Gimp to build new script with currently exists."
        , "Jeremy Czajkowski"
        , "Jeremy Czajkowski"
        , "September 2013"
        , "<Toolbox>/Help/Composer"
        , None
        , []
        , []
        , runner)

main()
