/* GIMP Plug-in Viewable Dialog Demo 2: Drawable
 * Copyright (C) 2011  Randall Sawyer <srandallsawyer@gmail.com> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <glib/gprintf.h>
#include <srs_gimpviewabledialog.h>

#define PLUGIN_NAME      "gimp-plugin-gvd-demo-drawable"
#define PLUGIN_BLURB     "Viewable dialog for GIMP plugins demo #2"
#define PLUGIN_HELP      "Demonstrates viewable dialog for plugins referring to a drawable - a layer, channel, or layer mask."
#define PLUGIN_AUTHOR    "Randall Sawyer"
#define PLUGIN_COPYRIGHT "Copyright (C) 2011 Randall Sawyer"
#define PLUGIN_YEAR      "2011"
#define PLUGIN_MENU_ITEM "Demo2: Drawable"

#define LAYER_TYPE      0
#define LAYER_MASK_TYPE 1
#define CHANNEL_TYPE    2

/*  Plugin Function Declarations  */

static void query      (void);

static void run        (const gchar      *name,
                        gint              nparams,
                        const GimpParam  *param,
                        gint             *nreturn_vals,
                        GimpParam       **return_vals);

/*  Private Function Declarations */

static void       drawable_dialog    (gint32  drawable_ID);

static gint       drawable_type_get  (gint32  drawable_ID);

static GtkWidget* greeting_label_new (gint    type);

/*  Private Data  */

static const gchar* drawable_icon[3] =
{
  GIMP_STOCK_LAYER,
  GIMP_STOCK_LAYER_MASK,
  GIMP_STOCK_CHANNEL
};

static const gchar* drawable_name[3] =
{
  "layer",
  "layer mask",
  "channel"
};

/*  Plug-in Required  */

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};

MAIN ()

/*  Plugin Function Definitions  */

static void
query (void)
{
  static const GimpParamDef args[] =
    {
      { GIMP_PDB_INT32,     "run mode", "Run mode"        },
      { GIMP_PDB_IMAGE,     "image",    "Input image"     },
      { GIMP_PDB_DRAWABLE,  "drawable", "Input drawable"  }
    };

  gimp_install_procedure (PLUGIN_NAME,
                          PLUGIN_BLURB,
                          PLUGIN_HELP,
                          PLUGIN_AUTHOR,
                          PLUGIN_COPYRIGHT,
                          PLUGIN_YEAR,
                          PLUGIN_MENU_ITEM,
                          "GRAY*, RGB*",
                          GIMP_PLUGIN,
                          G_N_ELEMENTS(args), 0,
                          args, NULL);

  gimp_plugin_menu_register   (PLUGIN_NAME, "<Image>/Windows/Viewable Dialogs");
}

static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static  GimpParam values[1];
  GimpRunMode       run_mode;
  gint32            drawable_ID;
  GimpPDBStatusType status;

  *nreturn_vals = 1;
  *return_vals  = &values[0];

  run_mode = param[0].data.d_int32;
  drawable_ID = param[2].data.d_drawable;

  if (run_mode == GIMP_RUN_INTERACTIVE)
  {
    drawable_dialog (drawable_ID);
    status = GIMP_PDB_SUCCESS;
  }
  else
    status = GIMP_PDB_PASS_THROUGH;

  /*  Status is the only return value */
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;
}

/*  Private Function Definitions  */

static void
drawable_dialog (gint32 drawable_ID)
{
  GtkWidget  *dialog;
  GtkWidget  *icon;
  GtkWidget  *content_box;
  GtkWidget  *label;
  gint        type = drawable_type_get (drawable_ID);
  
  /* User calls gimp_ui_init() before instantiating dialog */
  gimp_ui_init (PLUGIN_NAME, FALSE);

  /* Arguments 'icon' and 'content_box' are defined before being passed.
     'content_box' can be GtkHBox, GtkVBox, GtkFrame, etc.  */
  icon = gtk_image_new_from_stock (drawable_icon[type],
                                   GTK_ICON_SIZE_LARGE_TOOLBAR);
  content_box = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (content_box), 16);

  /* This plugin is drawable-oriented:  be sure to pass GVD_DRAWABLE.
     For 9th argument ('thumb_size'), pass the same icon  size which was used
     to define 'icon'.  Pass your va_arg button arguments at the end, just as
     you would when calling 'gimp_dialog_new()'.  */
  dialog = srs_gimp_viewable_dialog_new ("Demo 2: Drawable", PLUGIN_NAME, NULL, 0,

                                         GVD_DRAWABLE, drawable_ID,

                                         icon,
                                         "Dialog refers to drawable",
                                         GTK_ICON_SIZE_LARGE_TOOLBAR,
                                         content_box,

                                         gimp_standard_help_func, PLUGIN_NAME,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);

  /* At this point, 'content_box' is defined, packed, and visible.  Put all of
     your user-inteface widgets into it.  */
  label = greeting_label_new (type);
  gtk_container_add (GTK_CONTAINER (content_box), label);
  gtk_widget_show (label);

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_widget_show (dialog);

  gimp_dialog_run (GIMP_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

static gint
drawable_type_get (gint32 drawable_ID)
{
  if (gimp_drawable_is_layer(drawable_ID))
    return LAYER_TYPE;
  if (gimp_drawable_is_layer_mask(drawable_ID))
    return LAYER_MASK_TYPE;
  /* (gimp_drawable_is_channel(drawable_ID)) */
    return CHANNEL_TYPE;
}

static GtkWidget* greeting_label_new (gint type)
{
  GtkWidget *label;
  gchar     *text;

  label = gtk_label_new (NULL);
  text = g_strdup_printf (
          "This <span style=\"italic\" weight=\"bold\">Viewable Dialog</span> "
          "refers to a <span style=\"italic\" weight=\"bold\">%s</span>.\n"
          "[Note the icon in the upper left-hand corner of this dialog.]\n"
          "Now, activate a <span style=\"italic\" weight=\"bold\">%s</span>, "
          "or a <span style=\"italic\" weight=\"bold\">%s</span>.\n"
          "Then, open another new one of these dialog windows.",
          drawable_name[type], drawable_name[(type+1)%3],
          drawable_name[(type+2)%3]);
  gtk_label_set_markup (GTK_LABEL (label), text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.5, 0.5);
  g_free (text);

  return label;
}
