/* GIMP srs-gimpviewabledialog-0.1
 * File: srs_gimp_viewable_dialog.c v 0.1
 * Author: Randall Sawyer <srandallsawyer@gmail.com>
 * Copyright (C) 2011  Randall Sawyer
 * All Rights Reserved.
 * Date: 2011-03-03
 *
 * This is the C source file for code which emulates the built-in GIMP
 * 'GimpViewableDialog' widget - a dialog window with a raised frame at the top
 * which contains an icon, a description in a "larger" bold font, the name of
 * the active image or drawable in a "smaller" font, and a thumbnail icon of the
 * image or drawable (the "viewable").  The purpose of this reverse engineering
 * is to provide plugin developers the means to create viewable-specific dialog
 * windows which have the same "look and feel" as those used by the GIMP's
 * built-in tools.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include "srs_gimpviewabledialog.h"

/* Private Function Declarations */

static GtkWidget* desc_label_new (const gchar  *desc);

static GtkWidget* view_label_new (GVDType       view_type,
                                  gint32        view_id);

static GtkWidget* thumbnail_new  (GVDType       view_type,
                                  gint32        view_id,
                                  GtkIconSize   thumb_size);

/* Public Function Definition */

GtkWidget *
srs_gimp_viewable_dialog_new (const gchar      *title,
                              const gchar      *role,
                              GtkWidget        *parent,
                              GtkDialogFlags    flags,

                              GVDType           view_type,
                              gint32            view_id,

                              GtkWidget        *icon,
                              const gchar      *desc,
                              GtkIconSize       thumb_size,
                              GtkWidget        *content_box,

                              GimpHelpFunc      help_func,
                              const gchar      *help_id,
                              ...)
{
  GtkWidget  *dialog;
  GtkWidget  *frame;
  GtkWidget  *hbox;
  GtkWidget  *vbox;
  GtkWidget  *label;
  GtkWidget  *thumb;

  va_list     args;

  dialog = gimp_dialog_new (title, role, parent, flags,
                            help_func, help_id, NULL);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), frame,
                      FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_widget_show (hbox);

  gtk_misc_set_alignment (GTK_MISC (icon), 0.5, 0.0);
  gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, 0);
  gtk_widget_show (icon);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (hbox), vbox);
  gtk_widget_show (vbox);

  label = desc_label_new (desc);
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  label = view_label_new (view_type, view_id);
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  thumb = thumbnail_new (view_type, view_id, thumb_size);
  gtk_container_add (GTK_CONTAINER (hbox), thumb);
  gtk_widget_show (thumb);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                      content_box, FALSE, FALSE, 0);
  gtk_widget_show (content_box);

  va_start (args, help_id);
  gimp_dialog_add_buttons_valist (GIMP_DIALOG (dialog), args);
  va_end (args);

  return  GTK_WIDGET (dialog);
}

/* Private Function Definitions */

static GtkWidget*
desc_label_new (const gchar *desc)
{
  GtkWidget *label;
  gchar     *text;

  label = gtk_label_new (NULL);
  text = g_markup_printf_escaped (
          "<span size=\"larger\" weight=\"bold\">%s</span>", desc);
  gtk_label_set_markup (GTK_LABEL (label), text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  g_free (text);
  
  return label;
}

static GtkWidget*
view_label_new (GVDType view_type,
                gint32  view_id)
{
  GtkWidget *label;
  gchar     *text;

  label = gtk_label_new (NULL);
  if (view_type == GVD_IMAGE)
    text = g_markup_printf_escaped (
            "<span size=\"smaller\">%s</span>",
            gimp_image_get_name (view_id));
  else  /* (GVDType == GVD_IMAGE) */
    text = g_markup_printf_escaped (
            "<span size=\"smaller\">%s (%s)</span>",
            gimp_drawable_get_name (view_id),
            gimp_image_get_name (gimp_drawable_get_image (view_id)));
  gtk_label_set_markup (GTK_LABEL (label), text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_label_set_ellipsize (GTK_LABEL (label), PANGO_ELLIPSIZE_END);
  g_free (text);
  
  return label;
}

static GtkWidget*
thumbnail_new  (GVDType     view_type,
                gint32      view_id,
                GtkIconSize thumb_size)
{
  GtkWidget  *align;
  GtkWidget  *frame;
  GtkWidget  *image;
  GdkPixbuf  *pixbuf;
  gint        icon_dim[2];

  gtk_icon_size_lookup (thumb_size, &icon_dim[0], &icon_dim[1]);
  if (icon_dim[1] > icon_dim[0])
    icon_dim[0] = icon_dim[1];

  if (view_type == GVD_IMAGE)
    pixbuf = gimp_image_get_thumbnail (view_id, icon_dim[0], icon_dim[0],
                                       GIMP_PIXBUF_SMALL_CHECKS);
  else /* (view_type == GVD_VIEW_DRAWABLE) */
    pixbuf = gimp_drawable_get_thumbnail (view_id, icon_dim[0], icon_dim[0],
                                       GIMP_PIXBUF_SMALL_CHECKS);

  align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);

  frame = gtk_aspect_frame_new (NULL, 0.0, 0.0, 1.0, TRUE);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (align), frame);
  gtk_widget_show (frame);

  image = gtk_image_new_from_pixbuf (pixbuf);
  gtk_container_add (GTK_CONTAINER (frame), image);
  gtk_widget_show (image);

  g_object_unref (pixbuf);
  
  return align;
}
