#define __TEXTUREWERKE_COLOR_CPP__

//
// Texture workshop for GIMP
//
// Copyright (C) Lauris Kaplinski 2007-2011
//
// Licensed under GNU GPL
//

static const int debug = 1;

#include <string.h>

#include <libgimp/gimp.h>

#include "texturewerke.h"

#define PADDING 4

static void
build_histogram (double histogram[], float *buffer, int width, int height, float *mask)
{
	memset (histogram, 0, 256 * sizeof (double));
	if (!mask) {
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				float val = buffer[y * width + x];
				histogram[(int) (val * 255.9999)] += 1;
			}
		}
	} else {
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				float val = buffer[y * width + x];
				float mval = mask[y * width + x];
				histogram[(int) (val * 255.9999)] += mval;
			}
		}
	}
	// Normalize
	double weight = 0;
	for (int i = 0; i < 256; i++) weight += histogram[i];
	// Use factor 100 to get nicer values
	// weight /= 100;
	for (int i = 0; i < 256; i++) histogram[i] /= weight;
}

static double
poly4 (double x, const double p[])
{
	double v = p[4] * x * x * x *x + p[3] * x * x * x + p[2] * x * x + p[1] * x + p[0];
	if (v < 0) return 0;
	if (v > 1) return 1;
	return v;
}

// p - parameters
// v - values
// m - number of parameters
// n - number of samples in set

static void
lm_func (double *p, double *v, int m, int n, void *data)
{
	double *histogram = (double *) data;
	memset (v, 0, n * sizeof (double));

	for (int i = 0; i < n; i++) {
		// Source (center)
		double x = ((double) i) / (n - 1);
		// Source weigth
		double weight = histogram[i];
		// Destination
		double y = poly4 (x, p);
		// Destinations
		int ylower = (int) (y * (n - 1.0001));
		int yupper = ylower + 1;
		double t = y * (n - 1) - ylower;
		double s = 1 - t;
		v[ylower] += s * weight;
		v[yupper] += t * weight;
	}
	for (int i = 1; i < n; i++) v[i] = v[i - 1] + v[i];
}


unsigned int
color_process (GimpDrawable *drawable, GimpDrawable *mask, HParams *params)
{
	// Template
	GimpDrawable *template_drawable = gimp_drawable_get (params->other_layer_id);
	if (!template_drawable) return false;
	if (template_drawable->bpp != drawable->bpp) {
		gimp_drawable_detach (template_drawable);
		return false;
	}
	unsigned char *template_pixels = g_new (unsigned char, template_drawable->width * template_drawable->height * template_drawable->bpp);
	GimpPixelRgn region;
	gimp_pixel_rgn_init (&region, template_drawable, 0, 0, template_drawable->width, template_drawable->height, 0, 0);
	gimp_pixel_rgn_get_rect (&region, template_pixels, 0, 0, template_drawable->width, template_drawable->height);
	GimpDrawable *template_mask = (params->other_mask_id >= 0) ? gimp_drawable_get (params->other_mask_id) : NULL;
	float *template_mask_buffer = NULL;
	if (template_mask) {
		template_mask_buffer = g_new (float, template_drawable->width * template_drawable->height);
		get_buffer (template_mask_buffer, template_mask, 0);
	}

	// Drawable
	unsigned char *drawable_pixels = g_new (unsigned char, drawable->width * drawable->height * drawable->bpp);
	gimp_pixel_rgn_init (&region, drawable, 0, 0, drawable->width, drawable->height, 0, 0);
	gimp_pixel_rgn_get_rect (&region, drawable_pixels, 0, 0, drawable->width, drawable->height);
	float *mask_buffer = NULL;
	if (mask) {
		mask_buffer = g_new (float, drawable->width * drawable->height);
		get_buffer (mask_buffer, mask, 0);
	}

	// Work buffers
	int maxwidth = (drawable->width > template_drawable->width) ? drawable->width : template_drawable->width;
	int maxheight = (drawable->height > template_drawable->height) ? drawable->height : template_drawable->height;
	float *template_buffer = g_new (float, template_drawable->width * template_drawable->height);
	float *buffer = g_new (float, drawable->width * drawable->height);
	float *tmp[3] = { 0 };
	if (params->color_use_blur) {
		for (int i = 0; i < 3; i++) tmp[i] = g_new (float, maxwidth * maxheight);
	}
	for (unsigned int ch = 0; ch < drawable->bpp; ch++) {
		// Template
		if (params->color_use_blur) {
			get_buffer (tmp[2], template_drawable->width, template_drawable->height, template_pixels, template_drawable->width * template_drawable->bpp, template_drawable->bpp, ch);
			blur (template_buffer, template_drawable->width, template_drawable->height, NULL, tmp[2], template_mask_buffer, &tmp[0], params->color_blur_x, params->color_blur_y, 0, 1);
		} else {
			get_buffer (template_buffer, template_drawable->width, template_drawable->height, template_pixels, template_drawable->width * template_drawable->bpp, template_drawable->bpp, ch);
		}
		double template_histogram[256];
		build_histogram (template_histogram, template_buffer, template_drawable->width, template_drawable->height, template_mask_buffer);
		// Buffer
		float *original = buffer;
		if (params->color_use_blur) {
			get_buffer (tmp[2], drawable->width, drawable->height, drawable_pixels, drawable->width * drawable->bpp, drawable->bpp, ch);
			blur (buffer, drawable->width, drawable->height, NULL, tmp[2], mask_buffer, &tmp[0], params->color_blur_x, params->color_blur_y, 0, 1);
			original = tmp[2];
		} else {
			get_buffer (buffer, drawable->width, drawable->height, drawable_pixels, drawable->width * drawable->bpp, drawable->bpp, ch);
		}
		double histogram[256];
		build_histogram (histogram, buffer, drawable->width, drawable->height, mask_buffer);

		// Cumulative histograms
		double tchist[256];
		for (int i = 0; i < 256; i++) tchist[i] = template_histogram[i];
		for (int i = 1; i < 256; i++) tchist[i] = tchist[i - 1] + tchist[i];
		double chist[256];
		for (int i = 0; i < 256; i++) chist[i] = histogram[i];
		for (int i = 1; i < 256; i++) chist[i] = chist[i - 1] + chist[i];
		// Transfer function
		double f[256];
		int s0 = 0;
		while (s0 < 256) {
			// Source range
			int s1 = s0 + 1;
			while ((s1 < 256) && (chist[s1] == chist[s0])) s1 += 1;
			// Target values
			double v0 = 0;
			double v1 = 0;
			// Destination range
			if (chist[s0] < tchist[0]) {
				// NOP
			} else if (chist[s0] >= tchist[255]) {
				v0 = v1 = 1.0;
			} else {
				for (int j = 0; j < 256; j++) {
					if (tchist[j] == chist[s0]) {
						int d0 = j;
						int d1 = d0;
						while ((d1 < 256) && (tchist[d1] == tchist[d0])) d1 += 1;
						v0 = d0 / 255.0;
						v1 = d1 / 255.0;
						break;
					} else if ((j > 0) && (tchist[j] > chist[s0])) {
						double a = tchist[j - 1];
						double b = tchist[j];
						double t = (chist[s0] - a) / (b - a);
						v0 = v1 = (j + t) / 255.0;
						break;
					}
				}
			}
			if (s1 == (s0 + 1)) {
				f[s0] = (v0 + v1) / 2;
			} else {
				double delta = (double) (s1 - 1) - s0;
				for (int j = s0; j < s1; j++) f[j] = v0 + (j - s0) * (v1 - v0) / delta;
			}
			s0 = s1;
		}

		for (unsigned int r = 0; r < drawable->height; r++) {
			for (unsigned int c = 0; c < drawable->width; c++) {
				float v = original[r * drawable->width + c];
				if (v < 0) v = 0;
				if (v > 1) v = 1;
				int i = (int) ( v * 255.9999);
				buffer[r * drawable->width + c] = (float) f[i];
			}
		}
		apply_buffer (drawable_pixels, drawable->width, drawable->height, drawable->width * drawable->bpp, drawable->bpp, ch, buffer);
	}
	// Copy adjusted image back to drawable
	gimp_pixel_rgn_init (&region, drawable, 0, 0, drawable->width, drawable->height, 1, 1);
	gimp_pixel_rgn_set_rect (&region, drawable_pixels, 0, 0, drawable->width, drawable->height);

	// Free buffers
	if (params->color_use_blur) {
		for (int i = 0; i < 3; i++) g_free (tmp[i]);
	}
	g_free (buffer);
	g_free (template_buffer);
	if (mask_buffer) g_free (mask_buffer);
	if (template_mask_buffer) g_free (template_mask_buffer);
	if (template_mask) gimp_drawable_detach (template_mask);
	gimp_drawable_detach (template_drawable);
	g_free (template_pixels);
	g_free (drawable_pixels);

	// Update original and release drawables
	gimp_drawable_merge_shadow (drawable->drawable_id, true);
	gimp_drawable_update (drawable->drawable_id, 0, 0, drawable->width, drawable->height);

	return true;
}
