//
// Fourier fitting
//

#define _USE_MATH_DEFINES
#include <math.h>

#include <stdlib.h>
#include <string.h>

#include <libgimp/gimp.h>

#ifndef M_2PI
#define M_2PI (2 * M_PI)
#endif

static void realFT (float *dr, int drs, const float *sr, int srs, int width, int height, int nxfreqs, int nyfreqs, float *scratch, int *dw, int *dh);
static void realFTR (float *dr, int drs, int width, int height, const float *sr, int srs, int sw, int sh, float *scratch);
static void realFTM (float *dr, int drs, const float *sr, int srs, int width, int height, const float *mask, int nxfreqs, int nyfreqs, float *scratch, int *dw, int *dh);

// Symmetric Fourier implementation

void
process_ft (guchar *dpx, int drs, int dbpp, const guchar *spx, int srs, int sbpp, int width, int height, const guchar *mpx, int mrs, int mbpp, float xsigma, float ysigma)
{
	// Floating point image domain
	float *id = (float *) g_malloc (width * height * sizeof (float));
	// Frequency domain
	float *fd = (float *) g_malloc (4 * width * height * sizeof (float));
	// Reverse transformed image domain
	float *idn = (float *) g_malloc (width * height * sizeof (float));
	float *scratch = (float *) g_malloc (4 * width * height * sizeof (float));
	float *md = NULL;

	int irs = width;
	int frs = 2 * width;
	int irsn = width;

	// Fill in id
	for (int y = 0; y < height; y++) {
		const guchar *s = spx + y * srs;
		for (int x = 0; x < width; x++) {
			id[y * width + x] = s[0];
			s += sbpp;
		}
	}

	if (mpx) {
		// Fill mask
		md = (float *) g_malloc (width * height * sizeof (float));
		for (int y = 0; y < height; y++) {
			const guchar *s = mpx + y * mrs;
			for (int x = 0; x < width; x++) {
				md[y * width + x] = s[0] / 255.0f;
				s += mbpp;
			}
		}
	}

	int freqw, freqh;

	if (md) {
		realFTM (fd, frs, id, irs, width, height, md, (int) (2 * xsigma), (int) (2 * ysigma), scratch, &freqw, &freqh);
	} else {
		realFT (fd, frs, id, irs, width, height, (int) (2 * xsigma), (int) (2 * ysigma), scratch, &freqw, &freqh);
	}

	// Clip
	int freqcx = freqw / 2;
	int freqcy = freqh / 2;
	// Clear zero frequency
	fd[freqcy * frs + freqcx] = 0;
	for (int y = 0; y < freqh; y++) {
		for (int x = 0; x < freqw; x++) {
			// Apply filter
			float nx = (x - freqcx) / xsigma;
			float ny = (y - freqcy) / ysigma;
			float dist = sqrt (nx * nx + ny * ny);
			float q = exp (-dist * dist / 2);
			fd[y * frs + x] *= q;
		}
	}

	realFTR (idn, irsn, width, height, fd, frs, freqw, freqh, scratch);
	// baseFT (fRe2, fIm2, irs2, FRe, FIm, frs, 2 * width - 1, 2 * height - 1, width, height, 1, scratch);
	// calculateAmp(N, M, fAmp2[0][0], fRe2[0][0], fIm2[0][0]);

	for (int y = 0; y < height; y++) {
		guchar *d = dpx + y * drs;
		for (int x = 0; x < width; x++) {
			float val = id[y * irs + x] - idn[y * irsn + x];
			// float val = fd[y * irsn + x];
			if (val < 0) val = 0;
			if (val > 255) val = 255;
			d[0] = (guchar) val;
			d += dbpp;
		}
	}

	if (md) g_free (md);
	g_free (scratch);
	g_free (idn);
	g_free (fd);
	g_free (id);
}


static void
realFT (float *dr, int drs, const float *sr, int srs, int width, int height, int nxfreqs, int nyfreqs, float *scratch, int *dw, int *dh)
{
	// Symmetrical width and height
	int width2 = 2 * width - 1;
	int height2 = 2 * height - 1;
	// Maximum frequency numbers
	int maxxfreq = nxfreqs - 1;
	if (maxxfreq > (width2 / 2)) maxxfreq = width2 / 2;
	int freqw = 2 * maxxfreq + 1;
	int freqxc = maxxfreq;
	int maxyfreq = nyfreqs - 1;
	if (maxyfreq > (height2 / 2)) maxyfreq = height2 / 2;
	int freqh = 2 * maxyfreq + 1;
	int freqyc = maxyfreq;
	// We need at least freqw X height scratch buffer
	memset (scratch, 0, freqw * height * sizeof (float));
	float *rr = scratch;
	// Scale factor based on symmetrical image
	float scale = 1 / sqrt ((float) width2 * height2);

	// Fill frequency buffer by rows
	for (int y = 0; y < height; y++) {
		for (int xfreq = -maxxfreq; xfreq <= maxxfreq; xfreq++) {
			int u2 = (width2 + xfreq) % width2;
			int uf2 = freqxc + xfreq;
			// Start from zero frequency
			rr[y * width2 + uf2] = sr[y * srs + 0];
			for (int x = 1; x < width; x++) {
				float a = (float) (-M_2PI * u2 * x / width2);
				float ca = cos (a);
				rr[y * width2 + uf2] += 2 * sr[y * srs + x] * ca;
			}
		}
	}

	// Transform buffer by columns
	for (int xfreq = -maxxfreq; xfreq <= maxxfreq; xfreq++) {
		int u2 = (width2 + xfreq) % width2;
		int uf2 = freqxc + xfreq;
		for (int yfreq = -maxyfreq; yfreq <= maxyfreq; yfreq++) {
			int v2 = (height2 + yfreq) % height2;
			int vf2 = freqyc + yfreq;
			// Start from zero frequency
			dr[vf2 * drs + uf2] = rr[0 * width2 + uf2];
			for (int y = 1; y < height; y++) {
				float a = (float) (-M_2PI * v2 * y / height2);
				float ca = cos(a);
				dr[vf2 * drs + uf2] += 2 * rr[y * width2 + uf2] * ca;
			}
			dr[vf2 * drs + uf2] *= scale;
		}
	}

	*dw = freqw;
	*dh = freqh;
}

static void
realFTR (float *dr, int drs, int width, int height, const float *sr, int srs, int sw, int sh, float *scratch)
{
	// Symmetrical width and height
	int width2 = 2 * width - 1;
	int height2 = 2 * height - 1;
	// Maximum frequency numbers
	int freqw = sw;
	int maxxfreq = freqw / 2;
	int freqxc = maxxfreq;
	int freqh = sh;
	int maxyfreq = freqh / 2;
	int freqyc = maxyfreq;
	// We need at least width X freqh scratch buffer
	memset (scratch, 0, width * freqh * sizeof (float));
	float *rr = scratch;
	// Scale factor based on symmetrical image
	float scale = 1 / sqrt ((float) width2 * height2);

	for (int yfreq = -maxyfreq; yfreq <= maxyfreq; yfreq++) {
		// int vf2 = (freqh + yfreq) % freqh;
		int vf2 = freqyc + yfreq;
		for (int x = 0; x < width; x++) {
			rr[vf2 * width + x] = 0;
			for (int xfreq = -maxxfreq; xfreq <= maxxfreq; xfreq++) {
				int u2 = (width2 + xfreq) % width2;
				// int uf2 = (freqw + xfreq) % freqw;
				int uf2 = freqxc + xfreq;
				float a = (float) (-M_2PI * x * u2 / width2);
				float ca = cos (a);
				rr[vf2 * width + x] += sr[vf2 * srs + uf2] * ca;
			}
		}
	}

	for (int x = 0; x < width; x++) {
		for (int y = 0; y < height; y++) {
			dr[y * drs + x] = 0;
			for (int yfreq = -maxyfreq; yfreq <= maxyfreq; yfreq++) {
				int v2 = (height2 + yfreq) % height2;
				// int vf2 = (freqh + yfreq) % freqh;
				int vf2 = freqyc + yfreq;
				float a = (float) (-M_2PI * y * v2 / height2);
				float ca = cos (a);
				dr[y * drs + x] += rr[vf2 * width + x] * ca;
			}
			dr[y * drs + x] *= scale;
		}
	}
}

static void
realFTM (float *dr, int drs, const float *sr, int srs, int width, int height, const float *mask, int nxfreqs, int nyfreqs, float *scratch, int *dw, int *dh)
{
	// Symmetrical width and height
	int width2 = 2 * width - 1;
	int height2 = 2 * height - 1;
	// Maximum frequency numbers
	int maxxfreq = nxfreqs - 1;
	if (maxxfreq > (width2 / 2)) maxxfreq = width2 / 2;
	int freqw = 2 * maxxfreq + 1;
	int freqxc = maxxfreq;
	int maxyfreq = nyfreqs - 1;
	if (maxyfreq > (height2 / 2)) maxyfreq = height2 / 2;
	int freqh = 2 * maxyfreq + 1;
	int freqyc = maxyfreq;
	// We need at least freqw X height scratch buffer
	memset (scratch, 0, freqw * height * sizeof (float));
	float *rr = scratch;
	// Scale factor based on symmetrical image
	float scale = 1 / sqrt ((float) width2 * height2);

	float *rm = (float *) malloc (4 * width * height * sizeof (float));

	// Fill frequency buffer by rows
	for (int y = 0; y < height; y++) {
		const float *m = (mask) ? mask + y * width : NULL;
		for (int xfreq = -maxxfreq; xfreq <= maxxfreq; xfreq++) {
			int u2 = (width2 + xfreq) % width2;
			int uf2 = freqxc + xfreq;
			// Start from zero frequency
			float sum = 0;
			float msum = 0;
			float mval = (mask) ? m[0] : 1;
			rr[y * width2 + uf2] = mval * sr[y * srs + 0];
			sum += 1;
			msum += mval;
			for (int x = 1; x < width; x++) {
				float a = (float) (-M_2PI * u2 * x / width2);
				float ca = cos (a);
				mval = (mask) ? m[x] : 1;
				rr[y * width2 + uf2] += mval * 2 * sr[y * srs + x] * ca;
				float scale = fabs (ca);
				sum += scale;
				msum += scale * mval;
			}
			float mw = msum / sum;
			rr[y * width2 + uf2] /= (mw) ? mw : 1;
			rm[y * width2 + uf2] = mw;
		}
	}

	// Transform buffer by columns
	for (int xfreq = -maxxfreq; xfreq <= maxxfreq; xfreq++) {
		int u2 = (width2 + xfreq) % width2;
		int uf2 = freqxc + xfreq;
		for (int yfreq = -maxyfreq; yfreq <= maxyfreq; yfreq++) {
			int v2 = (height2 + yfreq) % height2;
			int vf2 = freqyc + yfreq;
			// Start from zero frequency
			dr[vf2 * drs + uf2] = rr[0 * width2 + uf2];
			for (int y = 1; y < height; y++) {
				float a = (float) (-M_2PI * v2 * y / height2);
				float ca = cos(a);
				dr[vf2 * drs + uf2] += 2 * rr[y * width2 + uf2] * ca;
			}
			dr[vf2 * drs + uf2] *= scale;
		}
	}

	*dw = freqw;
	*dh = freqh;
}
